/***************************************************************************
                          manager.cpp -  description
                             -------------------
    begin                : Fri Aug 05 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "manager.h"

#include "igdcontrolpoint.h"
#include "ssdpconnection.h"

#include <qtimer.h>

#include <kdebug.h>
#include "../../kmessdebug.h"

#ifdef KMESSDEBUG_UPNP
#define KMESSDEBUG_UPNP_GENERAL
#endif


namespace UPnP
{

// Set the static variable
Manager* Manager::instance_(0);


// The constructor
Manager::Manager()
  : activeIgdControlPoint_(0)
  , broadcastFailed_(false)
  , ssdpConnection_(0)
  , ssdpTimer_(0)
{

}



// The destructor
Manager::~Manager()
{
  delete ssdpTimer_;
  delete ssdpConnection_;
  instance_ = 0;  // Unregister the instance
}



// Initialize the manager, detect all devices
void Manager::initialize()
{
#ifdef KMESSTEST
  ASSERT( ssdpConnection_ == 0 );
#endif
#ifdef KMESSDEBUG_UPNP_GENERAL
  kdDebug() << "UPnP::Manager: Initiating a broadcast to detect UPnP devices..." << endl;
#endif


  // Create the SSDP object to detect devices
  ssdpConnection_ = new SsdpConnection();
  connect(ssdpConnection_, SIGNAL(     deviceFound(const QString&,int,const QString&) ) ,
          this,              SLOT( slotDeviceFound(const QString&,int,const QString&) ) );

  // Create a timer
  ssdpTimer_      = new QTimer(this);
  connect(ssdpTimer_, SIGNAL(timeout()), this, SLOT(slotBroadcastTimeout()));

  // Start a UPnP broadcast
  broadcastFailed_ = false;
  ssdpConnection_->queryDevices();
  ssdpTimer_->start(2000, true);
}



// Return the instance of the manager class
Manager * Manager::instance()
{
  // Create when it's required
  if(instance_ == 0)
  {
    instance_ = new Manager();
    instance_->initialize();
  }

  return instance_;
}



// Return the external IP address
QString Manager::getExternalIpAddress() const
{
  // Do not expose activeIgd_;
  return (activeIgdControlPoint_ != 0 ? activeIgdControlPoint_->getExternalIpAddress() : QString::null);
}



// Return true if a controlable gateway is available
bool Manager::isGatewayAvailable()
{
  return (activeIgdControlPoint_ != 0 &&
          activeIgdControlPoint_->isGatewayAvailable());
}



// The broadcast failed
void Manager::slotBroadcastTimeout()
{
#ifdef KMESSTEST
  ASSERT( ! broadcastFailed_ );
#endif
#ifdef KMESSDEBUG_UPNP_GENERAL
  kdDebug() << "UPnP::Manager: Timeout, no broadcast response received!" << endl;
#endif

  broadcastFailed_ = true;
}



// A device was discovered by the SSDP broadcast
void Manager::slotDeviceFound(const QString &hostname, int port, const QString &rootUrl)
{
#ifdef KMESSDEBUG_UPNP_GENERAL
  kdDebug() << "UPnP::Manager: Device found, initializing IgdControlPoint to query it." << endl;
#endif

  IgdControlPoint *controlPoint = new IgdControlPoint(hostname, port, rootUrl);
  igdControlPoints_.append(controlPoint);

  if(activeIgdControlPoint_ == 0)
  {
    activeIgdControlPoint_ = controlPoint;
    activeIgdControlPoint_->initialize();
  }
}



}  // end of namespace

#include "manager.moc"
