/***************************************************************************
                          settingsdialog.cpp  -  description
                             -------------------
    begin                : Sat Jan 11 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "settingsdialog.h"

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcstring.h>
#include <qcolor.h>
#include <qcombobox.h>
#include <qfont.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpixmap.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qtextedit.h>
#include <qtoolbutton.h>

#include <kapp.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdirselectdialog.h>
#include <kfiledialog.h>
#include <kfontdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>
#include <ktextbrowser.h>
#include <krun.h>
#include <kurl.h>
#include <kurllabel.h>
#include <khtmlview.h>

#include "../account.h"
#include "../emoticonmanager.h"
#include "../kmessdebug.h"
#include "../chat/chatmessage.h"
#include "../chat/chatmessagestyle.h"
#include "../chat/chatmessageview.h"
#include "accountswidget.h"
#include "alertswidget.h"
#include "chattingwidget.h"
#include "chatloggingwidget.h"
#include "emailwidget.h"
#include "emoticonwidget.h"

// Initialize the instance to zero
SettingsDialog* SettingsDialog::instance_(0);

// The constructor
SettingsDialog::SettingsDialog(QWidget *parent, const char *name )
  : KDialogBase( IconList, i18n("Settings"),                 // dialogFase, caption
                 (Help | Ok | Apply | Cancel | User1), Ok,   // buttons. defaultbutton
                 parent, name,                               // parent, qobject name
                 false, true,                                // modal, bottom separator
                 i18n( "&Delete" )                           // User1 gui item
               ),
    account_(0),
    chatStyle_(0)
{
  KIconLoader   *loader = KGlobal::iconLoader();
  KStandardDirs *dirs   = KGlobal::dirs();
  QPixmap        pixmap;

  // Create the various widgets
  config_ = kapp->config();
  // Load the options from the configuration file
  readOptions();

  // "Manually" set up the look of the dialog.
  QFrame *accountsFrame    = addPage( i18n("Account"),      i18n("Account"),  loader->loadIcon("identity", KIcon::NoGroup, KIcon::SizeMedium) );
  QFrame *alertsFrame      = addPage( i18n("Alerts"),       i18n("Alerts"),    loader->loadIcon("date", KIcon::NoGroup, KIcon::SizeMedium) );
  QFrame *emoticonFrame    = addPage( i18n("Emoticons"),    i18n("Emoticons"), loader->loadIcon("emoticons", KIcon::NoGroup, KIcon::SizeMedium) );
  QFrame *chattingFrame    = addPage( i18n("Chatting"),     i18n("Chatting"),  loader->loadIcon("fonts", KIcon::NoGroup, KIcon::SizeMedium) );
  QFrame *chatLoggingFrame = addPage( i18n("Chat Logging"), i18n("Chat Logging"), loader->loadIcon("filesaveas", KIcon::NoGroup, KIcon::SizeMedium) );
  QFrame *emailFrame       = addPage( i18n("Email"),        i18n("Email Notification"),  loader->loadIcon("mail_generic", KIcon::NoGroup, KIcon::SizeMedium) );

  // Create the layouts to maximize all child widgets
  QVBoxLayout *accountsLayout    = new QVBoxLayout( accountsFrame );
  QVBoxLayout *alertsLayout      = new QVBoxLayout( alertsFrame );
  QVBoxLayout *emoticonLayout    = new QVBoxLayout( emoticonFrame );
  QVBoxLayout *chattingLayout    = new QVBoxLayout( chattingFrame );
  QVBoxLayout *chatLoggingLayout = new QVBoxLayout( chatLoggingFrame );
  QVBoxLayout *emailLayout       = new QVBoxLayout( emailFrame );

  // Create the widgets that belong in the frames
  accountsWidget_    = new AccountsWidget( accountsFrame , "accounts" );
  alertsWidget_      = new AlertsWidget( alertsFrame,  "alerts" );
  emoticonWidget_    = new EmoticonWidget( emoticonFrame,  "emoticons" );
  chattingWidget_    = new ChattingWidget( chattingFrame , "chatting" );
  chatLoggingWidget_ = new ChatLoggingWidget( chatLoggingFrame , "chatLogging" );
  emailWidget_       = new EmailWidget( emailFrame, "email" );

  // Add the widgets to the layouts
  accountsLayout    ->addWidget( accountsWidget_    );
  alertsLayout      ->addWidget( alertsWidget_      );
  emoticonLayout    ->addWidget( emoticonWidget_    );
  chattingLayout    ->addWidget( chattingWidget_    );
  chatLoggingLayout ->addWidget( chatLoggingWidget_ );
  emailLayout       ->addWidget( emailWidget_       );

  // Connect alert widget signals
  connect( alertsWidget_->showChatStartNotificationsCheckBox_,  SIGNAL(              toggled(bool) ),
           this,                                                SLOT  ( notificationsToggled()     ) );

  // Connect chat widget signals
  connect( chattingWidget_->contactFontColorButton_,SIGNAL(        changed(const QColor&) ) ,
           this,                                    SLOT  (           updatePreview()     ) );
  connect( chattingWidget_->fontColorButton_,       SIGNAL(        changed(const QColor&) ) ,
           this,                                    SLOT  (           updatePreview()     ) );
  connect( chattingWidget_->useContactFontCheckBox_,SIGNAL(                 toggled(bool) ),
           this,                                    SLOT  (   useContactFontToggled(bool) ) );
  connect( chattingWidget_->fontButton_,            SIGNAL(                 clicked()     ),
           this,                                    SLOT  (             fontClicked()     ) );
  connect( chattingWidget_->contactFontButton_,     SIGNAL(                 clicked()     ),
           this,                                    SLOT  (      contactFontClicked()     ) );
  connect( chattingWidget_->chatStyle_,             SIGNAL(           activated(const QString&) ) ,
           this,                                    SLOT  (slotChatStyleChanged(const QString&) ) );
  connect( chattingWidget_->useEmoticonsCheckBox_,  SIGNAL(                 toggled(bool) ) ,
           this,                                    SLOT  (           updatePreview()     ) );
  connect( chattingWidget_->useFontEffectsCheckBox_,SIGNAL(                 toggled(bool) ) ,
           this,                                    SLOT  (           updatePreview()     ) );
  connect( chattingWidget_->showTimeCheckbox_,      SIGNAL(                 toggled(bool) ) ,
           this,                                    SLOT  (           updatePreview()     ) );
  connect( chattingWidget_->groupFollowupCheckbox_, SIGNAL(                 toggled(bool) ) ,
           this,                                    SLOT  (           updatePreview()     ) );

  // Connect chat logging widget signals
  connect( chatLoggingWidget_->chatSavePathButton_, SIGNAL(                 clicked()     ),
           this,                                    SLOT  (         chooseDirectory()     ) );

  chatLoggingWidget_->chatSavePathEdit_->setReadOnly( true );


  // Set up the emoticons in the "chatting" widget
  QString emoticonDir  = CurrentAccount::instance()->getEmoticonStyle();
  pixmap.load( dirs->findResource( "emoticons", emoticonDir + "/teeth.png" ) );
  chattingWidget_->pixmapLabel1_->setPixmap( pixmap );
  pixmap.load( dirs->findResource( "emoticons", emoticonDir + "/rainbow.png" ) );
  chattingWidget_->pixmapLabel2_->setPixmap( pixmap );
  pixmap.load( dirs->findResource( "emoticons", emoticonDir + "/dog.png" ) );
  chattingWidget_->pixmapLabel3_->setPixmap( pixmap );

  // Configure the preview form of the "chatting" widget
  // Insert a KHTMLPart in the placeholder
  chatMessageView_ = new ChatMessageView(chattingWidget_->khtmlPlaceholder_, "ChatMessageView");

  // Create a layout to maximize the KHTMLPart
  QBoxLayout *layout = new QHBoxLayout( chattingWidget_->khtmlPlaceholder_ );
  layout->addWidget( chatMessageView_->view() );   // Stretches widget

  // Get all available chat styles for the "chatting" widget.
  // Avoid duplicate entries.
  QStringList styleDirs = dirs->findDirs("data", "kmess/styles");
  QStringList styleNames;

  for( QStringList::Iterator dirIt = styleDirs.begin(); dirIt != styleDirs.end(); ++dirIt )
  {
    QDir styleDir(*dirIt);
    QStringList styles = styleDir.entryList(QDir::Dirs, QDir::Name | QDir::IgnoreCase);
    for( QStringList::Iterator fileIt = styles.begin(); fileIt != styles.end(); ++fileIt )
    {
      QString styleName = *fileIt;
      QString xslFile = styleDir.path() + "/" + styleName + "/" + styleName + ".xsl";  // e.g. share/apps/kmess/styles/Demo/Demo.xsl
      if( QFile::exists(xslFile) && ! styleNames.contains(styleName) )
      {
        styleNames.append(styleName);
      }
    }
  }

  // Add all items to the chat style list.
  for( QStringList::Iterator styleIt = styleNames.begin(); styleIt != styleNames.end(); ++styleIt )
  {
    chattingWidget_->chatStyle_->insertItem( *styleIt );
  }

  // Prepare the XSL transformation
  chatStyle_ = new ChatMessageStyle();
}



// The destructor
SettingsDialog::~SettingsDialog()
{
  saveOptions();
#ifdef KMESSDEBUG_SETTINGSDIALOG
  kdDebug() << "DESTROYED SettingsDialog " << endl;
#endif

  delete chatStyle_;
  instance_ = 0;
}



// Show the settings dialog for the given account
void SettingsDialog::changeAccountSettings(Account *account, bool isCurrentAccount)
{
  account_ = account;
  if ( account != 0 )
  {
    // Load account information into the widgets
    loadAccountsSettings( account, isCurrentAccount );
    loadAlertsSettings( account );
    loadChattingSettings( account );
    loadChatLoggingSettings( account );
    loadEmailSettings( account );
    loadEmoticonSettings( account );
  }
}



// Get a directory from the user.
void SettingsDialog::chooseDirectory()
{
  QString startDir, dir;

  startDir = chatLoggingWidget_->chatSavePathEdit_->text();

  dir = KFileDialog::getExistingDirectory( startDir );

  if ( dir != "" )
  {
    chatLoggingWidget_->chatSavePathEdit_->setText( dir );
  }
}



// Save options before closing.
void SettingsDialog::closeEvent(QCloseEvent *event)
{
  saveOptions();
  event->accept();
}



// The contact font button was pressed.  Show a font dialog to get a new font.
void SettingsDialog::contactFontClicked()
{
  QFont   font;
  QString fontFamily;
  bool    accepted;

  // Get a default font from the font button
  fontFamily = chattingWidget_->contactFontButton_->text().replace("&", QString::null); // remove accel character
  font       = chattingWidget_->contactFontButton_->font();

  // Get a new font and font family from the user
  accepted = getFont( font, fontFamily );

  // If it was accepted, set the font to the button
  if ( accepted )
  {
    // Reset size, as it can't be used in the chat message view.
    font.setPointSize(10);

    chattingWidget_->contactFontButton_->setText( fontFamily );
    chattingWidget_->contactFontButton_->setFont( font );

    // Update the preview
    chatStyle_->setContactFont( font );
    updatePreview();
  }
}



// The font button was pressed.  Show a font dialog to get a new font.
void SettingsDialog::fontClicked()
{
  QFont   font;
  QString fontFamily;
  bool    accepted;

  // Get a default font from the font button
  fontFamily = chattingWidget_->fontButton_->text().replace("&", QString::null); // remove accel character
  font       = chattingWidget_->fontButton_->font();

  // Get a new font and font family from the user
  accepted = getFont( font, fontFamily );

  // If it was accepted, set the font to the button
  if ( accepted )
  {
    // Reset size, as it can't be used in the chat message view.
    font.setPointSize(10);

    chattingWidget_->fontButton_->setText( fontFamily );
    chattingWidget_->fontButton_->setFont( font );

    // Update the preview
    updatePreview();
  }
}



// Get a font and cleaned-up font family from a dialog
bool SettingsDialog::getFont(QFont &font, QString &fontFamily) const
{
  bool  accepted;
  int   leftBracket, result;
  QFont selection;

  selection = font;

  result = KFontDialog::getFont( selection );
  accepted = ( result == KFontDialog::Accepted );

  if ( accepted )
  {
    fontFamily = selection.family();
    // KDE3 seems to want to throw some stuff in square brackets at the
    //  end of font names.  Strip that stuff off.
    // I'm assuming no real proper font has "[" in the name.
    leftBracket = fontFamily.find( "[" );
    if ( leftBracket >= 0 )
    {
      fontFamily = fontFamily.left( leftBracket );
    }
    fontFamily = fontFamily.stripWhiteSpace();
    font = selection;
  }
  return accepted;
}



// Return a singleton instance of the settings dialog window
SettingsDialog* SettingsDialog::instance( QWidget *parent )
{
  // If the instance is null, create a new dialog and return that.
  if ( instance_ == 0 )
  {
    instance_ = new SettingsDialog( parent, "SettingsDialog" );
  }

  return instance_;
}



// Load account information into the account widget
void SettingsDialog::loadAccountsSettings(Account *account, bool isCurrentAccount)
{
  accountsWidget_->loadSettings( account, isCurrentAccount );

  // Disable the "remove account" button for guest accounts.
  if( account->isGuestAccount() )
  {
    enableButton( KDialogBase::User1, false );
  }
}



// Load account information into the alerts widget
void SettingsDialog::loadAlertsSettings(Account *account)
{
  alertsWidget_->showOnlineNotificationsCheckBox_->setChecked( account->getNotifyContactsOnline() );
  alertsWidget_->showChatStartNotificationsCheckBox_->setChecked( account->getNotifyContactsChatStart() );
  alertsWidget_->showChatMessagesNotificationsCheckBox_->setChecked( account->getNotifyContactsChatMessages() );
  alertsWidget_->showStatusNotificationsCheckBox_->setChecked( account->getNotifyContactsStatus() );
  alertsWidget_->showOfflineNotificationsCheckBox_->setChecked( account->getNotifyContactsOffline() );
  alertsWidget_->showEmailsNotificationsCheckBox_->setChecked( account->getNotifyEmails() );

  alertsWidget_->notificationDurationSpinBox_->setValue( account->getNotificationDuration() );

  notificationsToggled();

  alertsWidget_->useIdleTimerCheckBox_->setChecked( account->getUseIdleTimer() );
  alertsWidget_->idleTimeSpinBox_->setValue( account->getIdleTime() );
  alertsWidget_->shakeNudgeCheckBox_->setChecked( account->getShakeNudge() );
  alertsWidget_->nowListeningCheckBox_->setChecked( account->getShowNowListening() );
}



// Load account information into the chatting widget
void SettingsDialog::loadChattingSettings(Account *account)
{
  QColor color;
  QFont  font;

  // Set up the values
  color.setNamedColor( account->getFontColor() );
  font = account->getFont();
  font.setPointSize(10);  // Reset size, as it can't be used in the chat message view.

  chattingWidget_->fontButton_->setText( font.family() );
  chattingWidget_->fontButton_->setFont( font );
  chattingWidget_->fontColorButton_->setColor( color );

  color.setNamedColor( account->getContactFontColor() );
  font = account->getContactFont();
  font.setPointSize(10);  // Reset size, as it can't be used in the chat message view.

  chattingWidget_->contactFontButton_->setText( font.family() );
  chattingWidget_->contactFontButton_->setFont( font );
  chattingWidget_->contactFontColorButton_->setColor( color );

  chattingWidget_->useContactFontCheckBox_->setChecked( account->getUseContactFont() );
  useContactFontToggled( account->getUseContactFont() );
  chattingWidget_->useEmoticonsCheckBox_->setChecked( account->getUseEmoticons() );
  chattingWidget_->showTimeCheckbox_->setChecked( account->getShowMessageTime() );
  chattingWidget_->useFontEffectsCheckBox_->setChecked( account->getUseFontEffects() );
  chattingWidget_->groupFollowupCheckbox_->setChecked( account->getGroupFollowupMessages() );

  // Select the correct chat style,
  // find the current style in the listbox.
  QListBox *chatStyleListBox = chattingWidget_->chatStyle_->listBox();
  if( KMESS_NULL(chatStyleListBox) ) return;   // listBox() can return null

  QListBoxItem *item = chatStyleListBox->findItem( account->getChatStyle(), Qt::ExactMatch | Qt::CaseSensitive );
  if(item == 0)
  {
    kdWarning() << "Current chat style was not found, attempting to revert to the default setting." << endl;
    item = chatStyleListBox->findItem("Default", Qt::ExactMatch);
  }

  if(item != 0)
  {
    // Somehow the QListBox doesn't update the parent QCombobox,
    // but by selecting the item we can request the index with currentItem()
    chatStyleListBox->setCurrentItem( item );
    chattingWidget_->chatStyle_->setCurrentItem( chatStyleListBox->currentItem() );
    slotChatStyleChanged( item->text() );
  }

  // Also load contact font settings into the chatStyle object,
  // other settings are set in updatePreview()
  chatStyle_->setContactFont(font);
  chatStyle_->setContactFontColor(color.name());
}



// Load account information into the chat logging widget
void SettingsDialog::loadChatLoggingSettings(Account *account)
{
  chatLoggingWidget_->logChatsCheckBox_->setChecked( account->getSaveChats() );
  chatLoggingWidget_->chatSavePathEdit_->setText( account->getSaveChatPath() );

  switch(account->getSavedChatDirectoryStructure())
  {
    case Account::SINGLEDIRECTORY :
    {
      chatLoggingWidget_->singleDirectoryRadioButton_->setChecked( true );
      break;
    }
    case Account::BYYEAR :
    {
      chatLoggingWidget_->yearRadioButton_->setChecked( true );
      break;
    }
    case Account::BYMONTH :
    {
      chatLoggingWidget_->monthRadioButton_->setChecked( true );
      break;
    }
    case Account::BYDAY :
    {
      chatLoggingWidget_->dayRadioButton_->setChecked( true );
      break;
    }
  }
}



// Load account information into the email widget
void SettingsDialog::loadEmailSettings(Account *account)
{
  emailWidget_->useHotmailRadioButton_->setChecked( account->getUseHotmail() );
  emailWidget_->useCommandRadioButton_->setChecked( !account->getUseHotmail() );
  emailWidget_->emailCommandEdit_->setText( account->getEmailCommand() );
  emailWidget_->showEmailInfoCheckBox_->setChecked( account->getShowEmail() );
  emailWidget_->showOtherFoldersCheckBox_->setChecked( account->getShowOtherFolders() );

  // TODO: When account does not use Hotmail, disable Hotmail option box, force to use an e-mail program, or default KDE mail app.
  // emailWidget_->clientButtonGroup_       ->setEnabled(account->getEmailSupported());
  emailWidget_->notificationButtonGroup_ ->setEnabled(account->getEmailSupported());
}



// Load account information into the email widget
void SettingsDialog::loadEmoticonSettings(Account *account)
{
  emoticonWidget_->loadSettings( account );
}



// Update status of dependent options for the notifications
void SettingsDialog::notificationsToggled()
{
  alertsWidget_->showChatMessagesNotificationsCheckBox_->setEnabled( alertsWidget_->showChatStartNotificationsCheckBox_->isChecked() );
}



// Read options from the configuration file.
void SettingsDialog::readOptions()
{
  // Pull in the window size and position
  config_->setGroup("SettingsDialog");
  QSize size = config_->readSizeEntry("Size");
  if(!size.isEmpty())
  {
    resize(size);
  }
  QPoint pos = config_->readPointEntry("Position");
  if(!pos.isNull())
  {
    move(pos);
  }
}



// Save all widget settings
void SettingsDialog::saveAccountSettings()
{
  if ( account_ != 0 )
  {
    QString oldAccountHandle = account_->getHandle();
    QString oldFriendlyName  = account_->getFriendlyName();

    // Save information from the widgets
    saveAccountsSettings( account_ );
    saveAlertsSettings( account_ );
    saveChattingSettings( account_ );
    saveChatLoggingSettings( account_ );
    saveEmailSettings( account_ );
    saveEmoticonSettings( account_ );

    // Commit configuration changes to disk, now
    // Makes sure the settings are not lost if KMess is terminated or crashes.
    KConfig *config = kapp->config();
    account_->saveProperties( config );

    emit changedSettings( account_, oldAccountHandle, oldFriendlyName );
  }
}




// Save account information from the account widget
void SettingsDialog::saveAccountsSettings(Account *account)
{
  accountsWidget_->saveSettings( account );
}



// Save account information from the alerts widget
void SettingsDialog::saveAlertsSettings(Account *account)
{
  account->setAlertsInformation       ( alertsWidget_->useIdleTimerCheckBox_->isChecked(),
                                        alertsWidget_->idleTimeSpinBox_->value(),
                                        alertsWidget_->shakeNudgeCheckBox_->isChecked(),
                                        alertsWidget_->nowListeningCheckBox_->isChecked() );

  account->setNotificationsInformation( alertsWidget_->showOnlineNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->showStatusNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->showChatStartNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->showChatMessagesNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->showOfflineNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->showEmailsNotificationsCheckBox_->isChecked(),
                                        alertsWidget_->notificationDurationSpinBox_->value() );
}



// Save account information from the chatting widget
void SettingsDialog::saveChattingSettings(Account *account)
{
   QFont font, contactFont;

  font = chattingWidget_->fontButton_->font();
  font.setFamily( chattingWidget_->fontButton_->text() );

  contactFont = chattingWidget_->contactFontButton_->font();
  contactFont.setFamily( chattingWidget_->contactFontButton_->text() );

  account->setChatInformation( chattingWidget_->useContactFontCheckBox_->isChecked(),
                               chattingWidget_->useEmoticonsCheckBox_->isChecked(),
                               chattingWidget_->useFontEffectsCheckBox_->isChecked(),
                               chattingWidget_->showTimeCheckbox_->isChecked(),
                               chattingWidget_->groupFollowupCheckbox_->isChecked(),
                               chattingWidget_->chatStyle_->currentText() );


  // After setChatInformation, due the way ChatView changes updates in font and chat.
  // If the ordering is changed, it will re-generate all messages twice;
  // once for the changedFontSettings(), and once for the changedChatStyleSettings() signal.
  account->setFontInformation( font, chattingWidget_->fontColorButton_->color().name(),
                               contactFont, chattingWidget_->contactFontColorButton_->color().name() );
}



// Save account information from the chat logging widget
void SettingsDialog::saveChatLoggingSettings(Account *account)
{
  int directoryStructure = Account::SINGLEDIRECTORY;

  if ( chatLoggingWidget_->singleDirectoryRadioButton_->isChecked() )
  {
    directoryStructure = Account::SINGLEDIRECTORY;
  }
  else if ( chatLoggingWidget_->yearRadioButton_->isChecked() )
  {
    directoryStructure = Account::BYYEAR;
  }
  else if ( chatLoggingWidget_->monthRadioButton_->isChecked() )
  {
    directoryStructure = Account::BYMONTH;
  }
  else if ( chatLoggingWidget_->dayRadioButton_->isChecked() )
  {
    directoryStructure = Account::BYDAY;
  }
  account->setChatLoggingInformation( chatLoggingWidget_->logChatsCheckBox_->isChecked(),
                                      chatLoggingWidget_->chatSavePathEdit_->text(),
                                      directoryStructure );

}



// Save account information from the email widget
void SettingsDialog::saveEmailSettings(Account *account)
{
  account->setEmailInformation( emailWidget_->useHotmailRadioButton_->isChecked(),
                                emailWidget_->emailCommandEdit_->text(),
                                emailWidget_->showEmailInfoCheckBox_->isChecked(),
                                emailWidget_->showOtherFoldersCheckBox_->isChecked() );
}



// Save account information from the emoticon widget
void SettingsDialog::saveEmoticonSettings(Account *account)
{
  emoticonWidget_->saveSettings( account );
}


// Save the options.
void SettingsDialog::saveOptions()
{
  // Save the window geometry
  config_->setGroup("SettingsDialog");
  config_->writeEntry("Size", size());
  config_->writeEntry("Position", pos());
}



// Select the Account page when showing the settings dialog
void SettingsDialog::showEvent( QShowEvent */*event*/ )
{
  showPage( 0 );
}



// The apply button was clicked.  Apply the changes.
void SettingsDialog::slotApply()
{
  saveAccountSettings();
}



// The cancel button was clicked.  Close the dialog.
void SettingsDialog::slotCancel()
{
  reject();
}



// The close button was clicked.  Close the dialog.
void SettingsDialog::slotClose()
{
  reject();
}



// The chat style was changed.
void SettingsDialog::slotChatStyleChanged(const QString &style)
{
  // Update the chat style object
  /* bool styleLoaded = */ chatStyle_->setStyle( style );

  // Update the preview
  updatePreview();
}



// The OK button was pressed.  Apply the changes and close.
void SettingsDialog::slotOk()
{
  saveAccountSettings();
  accept();
}



// The delete button was pressed.
void SettingsDialog::slotUser1()
{
  // Ask the user for confirmation
  if ( ( KMessageBox::warningYesNo( this, i18n("Are you sure you want to delete this account?")) ) == KMessageBox::Yes )
  {
    emit deleteAccount( account_ );
    reject();
  }
}



// Update the chat style preview
void SettingsDialog::updatePreview()
{
  QString htmlBody;

  if( chatStyle_ == 0 || ! chatStyle_->hasStyle() )
  {
    // Don't bother updating the preview for every event fired by the load...Settings() methods.
    return;
  }

  // Update the boolean settings (other settings are updated in their slots)
  chatStyle_->setContactFontColor( chattingWidget_->contactFontColorButton_->color().name()  );
  chatStyle_->setShowTime        ( chattingWidget_->showTimeCheckbox_->isChecked()           );
  chatStyle_->setUseContactFont  ( chattingWidget_->useContactFontCheckBox_->isChecked()     );
  chatStyle_->setUseEmoticons    ( chattingWidget_->useEmoticonsCheckBox_->isChecked()       );
  chatStyle_->setUseFontEffects  ( chattingWidget_->useFontEffectsCheckBox_->isChecked()     );

  // Get the user data
  QString userHandle  = accountsWidget_->handleEdit_->text();
  QString userName    = accountsWidget_->friendlyNameEdit_->text();
  QString userPicture = accountsWidget_->getPictureFileName();

  // Create three fake messages
  ChatMessage fake1( ChatMessage::TYPE_INCOMING, ChatMessage::CONTENT_MESSAGE, true, i18n("Hi, how are you doing? :)"), "test@kmess.org",
                     i18n("Stacy") );
  ChatMessage fake2( ChatMessage::TYPE_OUTGOING, ChatMessage::CONTENT_MESSAGE, false, i18n("Great!"), userHandle,
                     userName, userPicture, chattingWidget_->fontButton_->font(), chattingWidget_->fontColorButton_->color().name() );
  ChatMessage fake3( ChatMessage::TYPE_OUTGOING, ChatMessage::CONTENT_MESSAGE, false, i18n("I /just/ got back from my vacation in Italy!"), userHandle,
                     userName, userPicture, chattingWidget_->fontButton_->font(), chattingWidget_->fontColorButton_->color().name() );

  // Display the sample chat
  htmlBody += "<div class='messageContainer'>" + chatStyle_->convertMessage(fake1) + "</div>\n";

  if( chattingWidget_->groupFollowupCheckbox_->isChecked() )
  {
    QPtrList<ChatMessage> messageGroup;
    messageGroup.append(&fake2);
    messageGroup.append(&fake3);
    htmlBody += "<div class='messageContainer'>" + chatStyle_->convertMessageList(messageGroup) + "</div>\n";
  }
  else
  {
    htmlBody += "<div class='messageContainer'>" + chatStyle_->convertMessage(fake2) + "</div>\n";
    htmlBody += "<div class='messageContainer'>" + chatStyle_->convertMessage(fake3) + "</div>\n";
  }

  QString messageRoot = chatStyle_->convertMessageRoot();
  if( messageRoot.isNull() )
  {
    // Assign standard HTML
    chatMessageView_->setStandardHtml( htmlBody, chatStyle_->getCssFile(), chatStyle_->getBaseFolder() );
  }
  else
  {
    // Assign the custom HTML root
    chatMessageView_->setHtml( messageRoot, htmlBody );
  }
}



// Enable or disable the contact font and color selectors.
void SettingsDialog::useContactFontToggled(bool checked)
{
  // Enable/disable buttons
  chattingWidget_->contactFontButton_->setEnabled( checked );
  chattingWidget_->contactFontColorButton_->setEnabled( checked );

  // Update preview
  chatStyle_->setUseContactFont( checked );
  updatePreview();
}

#include "settingsdialog.moc"
