/***************************************************************************
                          mymoneybanking.cpp
                             -------------------
    begin                : Thu Aug 26 2004
    copyright            : (C) 2004 Martin Preuss
    email                : aquamaniac@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

// ----------------------------------------------------------------------------
// QT Includes

#include <qmessagebox.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <qcheckbox.h>

// ----------------------------------------------------------------------------
// KDE Includes

#include <klocale.h>
#include <kmessagebox.h>
#include <kgenericfactory.h>
#include <kaction.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kguiitem.h>
#include <klineedit.h>
#include <keditlistbox.h>

// ----------------------------------------------------------------------------
// Library Includes

#include <aqbanking/imexporter.h>
#include <aqbanking/jobgettransactions.h>
#include <aqbanking/jobgetbalance.h>
#include <aqbanking/job.h>
// #include <qbanking/qbpickstartdate.h>
#include <qbanking/qbgui.h>
#include <gwenhywfar/logger.h>
#include <gwenhywfar/debug.h>
//#include <kbanking/settings.h>

// ----------------------------------------------------------------------------
// Project Includes

#include "mymoneybanking.h"
#include "kbjobview.h"
#include "kbsettings.h"
#include "kbaccountsettings.h"
#include <kmymoney/mymoneyfile.h>
#include <kmymoney/kmymoneyview.h>
#include <kmymoney/mymoneykeyvaluecontainer.h>
#include <kbpickstartdate.h>

K_EXPORT_COMPONENT_FACTORY( kmm_kbanking,
                            KGenericFactory<KBankingPlugin>( "kmm_kbanking" ) )

KBankingPlugin::KBankingPlugin(QObject *parent, const char* name, const QStringList&) :
  KMyMoneyPlugin::Plugin(parent, name),
  KMyMoneyPlugin::OnlinePlugin(),
  m_accountSettings(0)
{
  m_kbanking=new KMyMoneyBanking(this, "KMyMoney");

  if (m_kbanking) {
    QBGui *gui;

    gui=new QBGui(m_kbanking);
    GWEN_Gui_SetGui(gui->getCInterface());
    GWEN_Logger_SetLevel(0, GWEN_LoggerLevel_Info);
    GWEN_Logger_SetLevel(AQBANKING_LOGDOMAIN, GWEN_LoggerLevel_Debug);
    m_kbanking->setGui(gui);
    if (m_kbanking->init() == 0) {
      // Tell the host application to load my GUI component
      setInstance(KGenericFactory<KBankingPlugin>::instance());
      setXMLFile("kmm_kbanking.rc");

      // create view
      createJobView();

      // create actions
      createActions();

      // load protocol conversion list
      loadProtocolConversion();
    }
    else {
      kdWarning() << "Could not initialize KBanking online banking interface" << endl;
      delete m_kbanking;
      m_kbanking = 0;
    }
  }
}



KBankingPlugin::~KBankingPlugin()
{
  if (m_kbanking) {
    m_kbanking->fini();
    delete m_kbanking;
  }
}


void KBankingPlugin::loadProtocolConversion(void)
{
  m_protocolConversionMap.clear();
  m_protocolConversionMap["aqhbci"] = "HBCI";
  m_protocolConversionMap["aqofxconnect"] = "OFX";
  m_protocolConversionMap["aqyellownet"] = "YellowNet";
  m_protocolConversionMap["aqgeldkarte"] = "Geldkarte";
  m_protocolConversionMap["aqdtaus"] = "DTAUS";
}

void KBankingPlugin::protocols(QStringList& protocolList) const
{
  std::list<std::string> list = m_kbanking->getActiveProviders();
  std::list<std::string>::iterator it;
  for(it = list.begin(); it != list.end(); ++it) {
    // skip the dummy
    if(*it == "aqnone")
      continue;
    QMap<QString, QString>::const_iterator it_m;
    it_m = m_protocolConversionMap.find(*it);
    if(it_m != m_protocolConversionMap.end())
      protocolList << (*it_m);
    else
      protocolList << (*it);
  }
}

QWidget* KBankingPlugin::accountConfigTab(const MyMoneyAccount& acc, QString& name)
{
  const MyMoneyKeyValueContainer& kvp = acc.onlineBankingSettings();
  name = i18n("Online settings");
  m_accountSettings = new KBAccountSettings(acc, 0, 0);
  m_accountSettings->m_usePayeeAsIsButton->setChecked(true);
  m_accountSettings->m_transactionDownload->setChecked(kvp.value("kbanking-txn-download")!="no");
  if(!kvp.value("kbanking-payee-regexp").isEmpty()) {
    m_accountSettings->m_extractPayeeButton->setChecked(true);
    m_accountSettings->m_payeeRegExpEdit->setText(kvp.value("kbanking-payee-regexp"));
    m_accountSettings->m_memoRegExpEdit->setText(kvp.value("kbanking-memo-regexp"));
    m_accountSettings->m_payeeExceptions->clear();
    m_accountSettings->m_payeeExceptions->insertStringList(QStringList::split(";", kvp.value("kbanking-payee-exceptions")));
  }
  return m_accountSettings;
}

MyMoneyKeyValueContainer KBankingPlugin::onlineBankingSettings(const MyMoneyKeyValueContainer& current)
{
  MyMoneyKeyValueContainer kvp(current);
  kvp["provider"] = name();
  if(m_accountSettings) {
    kvp.deletePair("kbanking-payee-regexp");
    kvp.deletePair("kbanking-memo-regexp");
    kvp.deletePair("kbanking-payee-exceptions");
    kvp.deletePair("kbanking-txn-download");
    if(m_accountSettings->m_extractPayeeButton->isChecked()
    && !m_accountSettings->m_payeeRegExpEdit->text().isEmpty()
    && !m_accountSettings->m_memoRegExpEdit->text().isEmpty()) {
      kvp["kbanking-payee-regexp"] = m_accountSettings->m_payeeRegExpEdit->text();
      kvp["kbanking-memo-regexp"] = m_accountSettings->m_memoRegExpEdit->text();
      kvp["kbanking-payee-exceptions"] = m_accountSettings->m_payeeExceptions->items().join(";");
    } else if(m_accountSettings->m_extractPayeeButton->isChecked()) {
      KMessageBox::information(0, i18n("You selected to extract the payee from the memo field but did not supply a regular expression for payee and memo extraction. The option will not be activated."), i18n("Missing information"));
    }
    if(!m_accountSettings->m_transactionDownload->isChecked())
      kvp["kbanking-txn-download"] = "no";
  }
  return kvp;
}

void KBankingPlugin::createJobView(void)
{
  KMyMoneyViewBase* view = viewInterface()->addPage(i18n("Outbox"), "onlinebanking");
  QWidget* frm = dynamic_cast<QWidget*>(view->parent());
  QWidget* w = new KBJobView(m_kbanking, view, "JobView");
  viewInterface()->addWidget(view, w);
  connect(viewInterface(), SIGNAL(viewStateChanged(bool)), frm, SLOT(setEnabled(bool)));
}



void KBankingPlugin::createActions(void)
{
  new KAction(i18n("Configure Aq&Banking..."), "configure", 0, this, SLOT(slotSettings()), actionCollection(), "settings_aqbanking");
  new KAction(i18n("AqBanking importer..."), "", 0, this, SLOT(slotImport()), actionCollection(), "file_import_aqbanking");

  connect(viewInterface(), SIGNAL(viewStateChanged(bool)), action("file_import_aqbanking"), SLOT(setEnabled(bool)));
}

void KBankingPlugin::slotSettings(void)
{
  KBankingSettings bs(m_kbanking);
  if (bs.init())
    kdWarning() << "Error on ini of settings dialog." << endl;
  else {
    bs.exec();
    if (bs.fini())
      kdWarning() << "Error on fini of settings dialog." << endl;
  }
}



bool KBankingPlugin::mapAccount(const MyMoneyAccount& acc, MyMoneyKeyValueContainer& settings)
{
  bool rc = false;
  if (!acc.id().isEmpty()) {
    MyMoneyFile* file = MyMoneyFile::instance();

    QString bankId;
    QString accountId;
    // extract some information about the bank. if we have a sortcode
    // (BLZ) we display it, otherwise the name is enough.
    try {
      const MyMoneyInstitution &bank = file->institution(acc.institutionId());
      bankId = bank.name();
      if(!bank.sortcode().isEmpty())
        bankId = bank.sortcode();
    } catch(MyMoneyException *e) {
      // no bank assigned, we just leave the field emtpy
      delete e;
    }

    // extract account information. if we have an account number
    // we show it, otherwise the name will be displayed
    accountId = acc.number();
    if(accountId.isEmpty())
      accountId = m_account.name();

    // do the mapping. the return value of this method is either
    // true, when the user mapped the account or false, if he
    // decided to quit the dialog. So not really a great thing
    // to present some more information.
    m_kbanking->askMapAccount(acc.id(),
                              bankId.utf8(),
                              accountId.utf8());

    // at this point, the account should be mapped
    // so we search it and setup the account reference in the KMyMoney object
    AB_ACCOUNT* ab_acc;
    ab_acc = AB_Banking_GetAccountByAlias(m_kbanking->getCInterface(), acc.id());
    if(ab_acc) {
      MyMoneyAccount a(acc);
      setupAccountReference(a, ab_acc);
      settings = a.onlineBankingSettings();
      rc = true;
    }
  }
  return rc;
}

QString KBankingPlugin::stripLeadingZeroes(const QString& s) const
{
  QString rc(s);
  QRegExp exp("(0*)(.*)");
  if(exp.match(s) != -1) {
    rc = exp.cap(2);
  }
  return rc;
}

void KBankingPlugin::setupAccountReference(const MyMoneyAccount& acc, AB_ACCOUNT* ab_acc)
{
  MyMoneyKeyValueContainer kvp;

  if(ab_acc) {
    QString accountNumber = stripLeadingZeroes(AB_Account_GetAccountNumber(ab_acc));
    QString routingNumber = stripLeadingZeroes(AB_Account_GetBankCode(ab_acc));

    QString val = QString("%1-%2").arg(routingNumber, accountNumber);
    if(val != acc.onlineBankingSettings().value("kbanking-acc-ref")) {
      MyMoneyKeyValueContainer kvp;

      // make sure to keep our own previous settings
      const QMap<QCString, QString>& vals = acc.onlineBankingSettings().pairs();
      QMap<QCString, QString>::const_iterator it_p;
      for(it_p = vals.begin(); it_p != vals.end(); ++it_p) {
        if(QString(it_p.key()).startsWith("kbanking-")) {
          kvp.setValue(it_p.key(), *it_p);
        }
      }

      kvp.setValue("kbanking-acc-ref", val);
      kvp.setValue("provider", name());
      setAccountOnlineParameters(acc, kvp);
    }
  } else {
    // clear the connection
    setAccountOnlineParameters(acc, kvp);
  }
}

const bool KBankingPlugin::accountIsMapped(const MyMoneyAccount& acc)
{
  AB_ACCOUNT* ab_acc;
  ab_acc = AB_Banking_GetAccountByAlias(m_kbanking->getCInterface(), acc.id());
  return ab_acc != 0;
}



bool KBankingPlugin::updateAccount(const MyMoneyAccount& acc)
{
  bool rc = false;
  if (!acc.id().isEmpty()) {
    AB_ACCOUNT *ba;
    AB_JOB *job;
    int rv;
    int days;
    int year, month, day;
    QDate qd;

    /* get AqBanking account */
    ba=AB_Banking_GetAccountByAlias(m_kbanking->getCInterface(),
                                    acc.id());
    if (!ba) {
      QMessageBox::critical(0,
                            i18n("Account Not Mapped"),
                            i18n("<qt>"
                                 "<p>"
                                 "The given application account "
                                 "has not been mapped to banking "
                                 "accounts."
                                 "</p>"
                                 "</qt>"
                            ),
                            QMessageBox::Ok,QMessageBox::NoButton);
      // clear the connection between the KMyMoney account
      // and the AqBanking equivalent
      setupAccountReference(acc, 0);
      return rc;
    }
    setupAccountReference(acc, ba);

    if(acc.onlineBankingSettings().value("kbanking-txn-download") != "no") {
      /* create getTransactions job */
      job = AB_JobGetTransactions_new(ba);
      rv = AB_Job_CheckAvailability(job, 0);
      if (rv) {
        DBG_ERROR(0, "Job \"GetTransactions\" is not available (%d)", rv);
        QMessageBox::critical(0,
                              i18n("Job not Available"),
                              i18n("<qt>"
                                  "The update job is not supported by the "
                                  "bank/account/backend.\n"
                                  "</qt>"),
                              i18n("Dismiss"), QString::null);
        AB_Job_free(job);
        return rc;
      }

      days = AB_JobGetTransactions_GetMaxStoreDays(job);
      if (days > 0) {
        GWEN_TIME *ti1;
        GWEN_TIME *ti2;

        ti1=GWEN_CurrentTime();
        ti2=GWEN_Time_fromSeconds(GWEN_Time_Seconds(ti1)-(60*60*24*days));
        GWEN_Time_free(ti1);
        ti1=ti2;

        if (GWEN_Time_GetBrokenDownDate(ti1, &day, &month, &year)) {
          DBG_ERROR(0, "Bad date");
          qd=QDate();
        } else
          qd=QDate(year, month+1, day);
        GWEN_Time_free(ti1);
      }

      // get last statement request date from application account object
      // and start from the next day if the date is valid
      QDate lastUpdate = QDate::fromString(acc.value("lastImportedTransactionDate"), Qt::ISODate);
      if(lastUpdate.isValid())
        lastUpdate = lastUpdate.addDays(-3);

      KBPickStartDate psd(m_kbanking, qd, lastUpdate,
                          lastUpdate.isValid() ? 2 : 3, 0,
                          "PickStartDate", true);
      if (psd.exec() != QDialog::Accepted) {
        AB_Job_free(job);
        return rc;
      }

      qd=psd.date();
      if (qd.isValid()) {
        GWEN_TIME *ti1;

        ti1=GWEN_Time_new(qd.year(), qd.month()-1, qd.day(), 0, 0, 0, 0);
        AB_JobGetTransactions_SetFromTime(job, ti1);
        GWEN_Time_free(ti1);
      }

      rv=m_kbanking->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
        DBG_ERROR(0, "Error %d", rv);
        QMessageBox::critical(0,
                              i18n("Error"),
                              i18n("<qt>"
                                  "Could not enqueue the job.\n"
                                  "</qt>"),
                              i18n("Dismiss"), QString::null);
        return rc;
      }
    }

    /* create getBalance job */
    job = AB_JobGetBalance_new(ba);
    rv = AB_Job_CheckAvailability(job, 0);
    if(!rv)
      rv = m_kbanking->enqueueJob(job);
    else
      rv = 0;

    AB_Job_free(job);
    if (rv) {
      DBG_ERROR(0, "Error %d", rv);
      QMessageBox::critical(0,
                            i18n("Error"),
                            i18n("<qt>"
                                "Could not enqueue the job.\n"
                                "</qt>"),
                            i18n("Dismiss"), QString::null);
      return rc;
    }

    // ask if the user want's to execute this job right away or spool it
    // for later execution
    KIconLoader *ic = KGlobal::iconLoader();
    KGuiItem executeButton(i18n("&Execute"),
                          QIconSet(ic->loadIcon("wizard",
                            KIcon::Small, KIcon::SizeSmall)),
                          i18n("Close this window"),
                          i18n("Use this button to close the window"));

    KGuiItem queueButton(i18n("&Queue"),
                          QIconSet(ic->loadIcon("fileexport",
                            KIcon::Small, KIcon::SizeSmall)),
                          i18n("Close this window"),
                          i18n("Use this button to close the window"));

    if(KMessageBox::questionYesNo(0,
        i18n("Do you want to execute or queue this job in the outbox?"),
        i18n("Execution"), executeButton, queueButton) == KMessageBox::Yes) {

      AB_IMEXPORTER_CONTEXT *ctx;

      ctx=AB_ImExporterContext_new();
      rv=m_kbanking->executeQueue(ctx);
      if (!rv)
        m_kbanking->importContext(ctx, 0);
      else {
        DBG_ERROR(0, "Error: %d", rv);
      }
      AB_ImExporterContext_free(ctx);

      // let application emit signals to inform views
      m_kbanking->accountsUpdated();
    }
    rc = true;
  }
  return rc;
}



void KBankingPlugin::slotImport(void)
{
  if (!m_kbanking->interactiveImport())
    kdWarning() << "Error on import dialog" << endl;
}



bool KBankingPlugin::importStatement(const MyMoneyStatement& s)
{
  return statementInterface()->import(s);
}

const MyMoneyAccount& KBankingPlugin::account(const QString& key, const QString& value) const
{
  return statementInterface()->account(key, value);
}

void KBankingPlugin::setAccountOnlineParameters(const MyMoneyAccount& acc, const MyMoneyKeyValueContainer& kvps) const
{
  return statementInterface()->setAccountOnlineParameters(acc, kvps);
}


KMyMoneyBanking::KMyMoneyBanking(KBankingPlugin* parent, const char* appname, const char* fname)
:KBanking(appname, fname)
,m_parent(parent)
{
}



const AB_ACCOUNT_STATUS* KMyMoneyBanking::_getAccountStatus(AB_IMEXPORTER_ACCOUNTINFO *ai)
{
  const AB_ACCOUNT_STATUS *ast;
  const AB_ACCOUNT_STATUS *best;

  best=0;
  ast=AB_ImExporterAccountInfo_GetFirstAccountStatus(ai);
  while(ast) {
    if (!best)
      best=ast;
    else {
      const GWEN_TIME *tiBest;
      const GWEN_TIME *ti;

      tiBest=AB_AccountStatus_GetTime(best);
      ti=AB_AccountStatus_GetTime(ast);

      if (!tiBest) {
        best=ast;
      }
      else {
        if (ti) {
          double d;

          /* we have two times, compare them */
          d=GWEN_Time_Diff(ti, tiBest);
          if (d>0)
            /* newer */
            best=ast;
        }
      }
    }
    ast=AB_ImExporterAccountInfo_GetNextAccountStatus(ai);
  } /* while */
  return best;
}



void KMyMoneyBanking::_xaToStatement(MyMoneyStatement &ks,
                                     const MyMoneyAccount& acc,
                                     const AB_TRANSACTION *t)
{
  const GWEN_STRINGLIST *sl;
  QString s;
  const char *p;
  const AB_VALUE *val;
  const GWEN_TIME *ti;
  const GWEN_TIME *startTime=0;
  MyMoneyStatement::Transaction kt;
  unsigned long h;

  kt.m_fees = MyMoneyMoney();

  // bank's transaction id
  p=AB_Transaction_GetFiId(t);
  if (p)
    kt.m_strBankID=QString("ID ")+QString::fromUtf8(p);

  // payee
  s.truncate(0);
  sl=AB_Transaction_GetRemoteName(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    if (se) {
      p=GWEN_StringListEntry_Data(se);
      assert(p);
      s = QString::fromUtf8(p);
    }
  }
  kt.m_strPayee=s;
  h = MyMoneyTransaction::hash(s.simplifyWhiteSpace());

  // memo
  s.truncate(0);
  sl=AB_Transaction_GetPurpose(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    while (se) {
      p = GWEN_StringListEntry_Data(se);
      assert(p);
      s += QString::fromUtf8(p);
      se = GWEN_StringListEntry_Next(se);
    } // while
  }
  kt.m_strMemo = s;
  h = MyMoneyTransaction::hash(s.simplifyWhiteSpace(), h);

  // see, if we need to extract the payee from the memo field
  const MyMoneyKeyValueContainer& kvp = acc.onlineBankingSettings();
  QString rePayee = kvp.value("kbanking-payee-regexp");
  if(!rePayee.isEmpty() && kt.m_strPayee.isEmpty()) {
    QString reMemo = kvp.value("kbanking-memo-regexp");
    QStringList exceptions = QStringList::split(";", kvp.value("kbanking-payee-exceptions"));

    bool needExtract = true;
    QStringList::const_iterator it_s;
    for(it_s = exceptions.begin(); needExtract && it_s != exceptions.end(); ++it_s) {
      QRegExp exp(*it_s, false);
      if(exp.search(kt.m_strMemo) != -1) {
        needExtract = false;
      }
    }
    if(needExtract) {
      QRegExp expPayee(rePayee, false);
      QRegExp expMemo(reMemo, false);
      if(expPayee.search(kt.m_strMemo) != -1) {
        kt.m_strPayee = expPayee.cap(1);
        if(expMemo.search(kt.m_strMemo) != -1) {
          kt.m_strMemo = expMemo.cap(1);
        }
      }
    }
  }

  // massage whitespaces a bit:
  // - remove leading blanks
  // - remove trailing blanks
  // - reduce multiple blanks to one
  kt.m_strMemo = kt.m_strMemo.simplifyWhiteSpace();
  kt.m_strPayee = kt.m_strPayee.simplifyWhiteSpace();

  // date
  ti=AB_Transaction_GetDate(t);
  if (!ti)
    ti=AB_Transaction_GetValutaDate(t);
  if (ti) {
    int year, month, day;

    if (!startTime)
      startTime=ti;
    else {
      if (GWEN_Time_Diff(ti, startTime)<0)
        startTime=ti;
    }

    if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year)) {
      kt.m_datePosted=QDate(year, month+1, day);
    }
  } else {
    DBG_WARN(0, "No date for transaction");
  }

  // value
  val=AB_Transaction_GetValue(t);
  if (val) {
    if (ks.m_strCurrency.isEmpty()) {
      p=AB_Value_GetCurrency(val);
      if (p)
        ks.m_strCurrency=p;
    }
    else {
      p=AB_Value_GetCurrency(val);
      if (p)
        s=p;
      if (ks.m_strCurrency.lower()!=s.lower()) {
        // TODO: handle currency difference
        DBG_ERROR(0, "Mixed currencies currently not allowed");
      }
    }

    kt.m_amount=MyMoneyMoney(AB_Value_GetValueAsDouble(val));
    h = MyMoneyTransaction::hash(kt.m_amount.toString(), h);
  }
  else {
    DBG_WARN(0, "No value for transaction");
  }

  if (startTime) {
    int year, month, day;

    if (!GWEN_Time_GetBrokenDownDate(startTime, &day, &month, &year)) {
      QDate d(year, month+1, day);

      if (!ks.m_dateBegin.isValid())
        ks.m_dateBegin=d;
      else if (d<ks.m_dateBegin)
        ks.m_dateBegin=d;

      if (!ks.m_dateEnd.isValid())
        ks.m_dateEnd=d;
      else if (d>ks.m_dateEnd)
        ks.m_dateEnd=d;
    }
  }
  else {
    DBG_WARN(0, "No date in current transaction");
  }

  // make hash value unique in case we don't have one already
  if(kt.m_strBankID.isEmpty()) {
    QString hashBase;
    hashBase.sprintf("%s-%07lx", kt.m_datePosted.toString(Qt::ISODate).data(), h);
    int idx = 1;
    QString hash;
    for(;;) {
      hash = QString("%1-%2").arg(hashBase).arg(idx);
      QMap<QString, bool>::const_iterator it;
      it = m_hashMap.find(hash);
      if(it == m_hashMap.end()) {
        m_hashMap[hash] = true;
        break;
      }
      ++idx;
    }
    kt.m_strBankID = QString("%1-%2").arg(acc.id()).arg(hash);
  }

  // store transaction
  ks.m_listTransactions+=kt;
}



bool KMyMoneyBanking::importAccountInfo(AB_IMEXPORTER_ACCOUNTINFO *ai,
                                        uint32_t /*flags*/)
{
  QString s;
  const char *p;
  const AB_TRANSACTION *t;
  MyMoneyStatement ks;
  MyMoneyAccount kacc;
  const AB_ACCOUNT_STATUS *ast;
  const AB_VALUE *val;
  const GWEN_TIME *ti;

  DBG_INFO(0, "Importing account...");

  // account number
  p=AB_ImExporterAccountInfo_GetAccountNumber(ai);
  if (p) {
    ks.m_strAccountNumber = m_parent->stripLeadingZeroes(p);
  }

  p=AB_ImExporterAccountInfo_GetBankCode(ai);
  if(p) {
    ks.m_strRoutingNumber = m_parent->stripLeadingZeroes(p);
  }

  kacc = m_parent->account("kbanking-acc-ref", QString("%1-%2").arg(ks.m_strRoutingNumber, ks.m_strAccountNumber));
  ks.m_accountId = kacc.id();

  // account name
  p=AB_ImExporterAccountInfo_GetAccountName(ai);
  if (p)
    ks.m_strAccountName=p;

  // account type
  switch(AB_ImExporterAccountInfo_GetType(ai)) {
    case AB_AccountType_Bank:
      ks.m_eType=MyMoneyStatement::etSavings;
      break;
    case AB_AccountType_CreditCard:
      ks.m_eType=MyMoneyStatement::etCreditCard;
      break;
    case AB_AccountType_Checking:
      ks.m_eType=MyMoneyStatement::etCheckings;
      break;
    case AB_AccountType_Savings:
      ks.m_eType=MyMoneyStatement::etSavings;
      break;
    case AB_AccountType_Investment:
      ks.m_eType=MyMoneyStatement::etInvestment;
      break;
    case AB_AccountType_Cash:
      ks.m_eType=MyMoneyStatement::etNone;
      break;
    default:
      ks.m_eType=MyMoneyStatement::etNone;
  }

  // account status
  ast=_getAccountStatus(ai);
  if (ast) {
    const AB_BALANCE *bal;

    bal = AB_AccountStatus_GetBookedBalance(ast);
    if (!bal)
      bal = AB_AccountStatus_GetNotedBalance(ast);
    if (bal) {
      val = AB_Balance_GetValue(bal);
      if (val) {
        DBG_INFO(0, "Importing balance");
        ks.m_closingBalance = MyMoneyMoney(AB_Value_GetValueAsDouble(val));
        p = AB_Value_GetCurrency(val);
        if (p)
          ks.m_strCurrency = p;
      }
      ti = AB_Balance_GetTime(bal);
      if (ti) {
        int year, month, day;

        if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year))
          ks.m_dateEnd=QDate(year, month+1, day);
      }
      else {
        DBG_WARN(0, "No time for balance");
      }
    }
    else {
      DBG_WARN(0, "No account balance");
    }
  }
  else {
    DBG_WARN(0, "No account status");
  }

  // clear hash map
  m_hashMap.clear();

  // get all transactions
  t=AB_ImExporterAccountInfo_GetFirstTransaction(ai);
  while(t) {
    _xaToStatement(ks, kacc, t);
    t=AB_ImExporterAccountInfo_GetNextTransaction(ai);
  }

  // import them
  if (!m_parent->importStatement(ks)) {
    if (QMessageBox::critical(0,
                              i18n("Critical Error"),
                              i18n("Error importing statement."),
                              i18n("Continue"),
                              i18n("Abort"), 0, 0)!=0) {
      DBG_ERROR(0, "User aborted");
      return false;
    }
  }
  return true;
}


#include "mymoneybanking.moc"
