/***************************************************************************
                          ksplittransactiondlg.cpp  -  description
                             -------------------
    begin                : Thu Jan 10 2002
    copyright            : (C) 2000-2002 by Michael Edwardes
    email                : mte@users.sourceforge.net
                           Javier Campos Morales <javi_c@users.sourceforge.net>
                           Felix Rodriguez <frodriguez@users.sourceforge.net>
                           John C <thetacoturtle@users.sourceforge.net>
                           Thomas Baumgart <ipwizard@users.sourceforge.net>
                           Kevin Tambascio <ktambascio@users.sourceforge.net>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// ----------------------------------------------------------------------------
// QT Includes

#include <qpushbutton.h>
#include <qlabel.h>
#include <qtable.h>
#include <qtimer.h>
#include <qptrlist.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qcursor.h>

// ----------------------------------------------------------------------------
// KDE Includes

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kguiitem.h>
#include <kpushbutton.h>

// ----------------------------------------------------------------------------
// Project Includes

#include "ksplittransactiondlg.h"
#include "../widgets/kmymoneysplittable.h"
#include "../dialogs/ksplitcorrectiondlg.h"
#include "../widgets/kmymoneycategory.h"
#include "../widgets/kmymoneyedit.h"
#include "../widgets/kmymoneylineedit.h"

KSplitTransactionDlg::KSplitTransactionDlg(const MyMoneyTransaction& t,
                                           const MyMoneyAccount& acc,
                                           const bool amountValid,
                                           const bool deposit,
                                           const MyMoneyMoney& calculatedValue,
                                           QWidget* parent, const char* name)
  : kSplitTransactionDlgDecl(parent, name, true),
  m_transaction(t),
  m_account(acc),
  m_amountValid(amountValid),
  m_isDeposit(deposit),
  m_calculatedValue(calculatedValue)
{
  // for deposits, we invert the sign of all splits.
  // don't forget to revert when we're done ;-)
  if(m_isDeposit) {
    for(unsigned i = 0; i < m_transaction.splits().count(); ++i) {
      MyMoneySplit split = m_transaction.splits()[i];
      split.setValue(-split.value());
      m_transaction.modifySplit(split);
    }
  }

  // add icons to buttons
  KIconLoader *il = KGlobal::iconLoader();
  KGuiItem finishButtenItem( i18n( "&Finish" ),
                    QIconSet(il->loadIcon("button_ok", KIcon::Small, KIcon::SizeSmall)),
                    i18n("Accept splits and return to transaction form"),
                    i18n("Use this to accept all changes to the splits and return to the transaction"));
  finishBtn->setGuiItem(finishButtenItem);

  KGuiItem cancelButtenItem( i18n( "&Cancel" ),
                    QIconSet(il->loadIcon("button_cancel", KIcon::Small, KIcon::SizeSmall)),
                    i18n("Reject all changes to the splits and return to transaction form"),
                    i18n("Use this to reject all changes to the splits and return to the transaction"));
  cancelBtn->setGuiItem(cancelButtenItem);

  KGuiItem clearButtenItem( i18n( "Clear &All" ),
                    QIconSet(il->loadIcon("edittrash", KIcon::Small, KIcon::SizeSmall)),
                    i18n("Clear all splits"),
                    i18n("Use this to clear all splits of this transaction"));
  clearAllBtn->setGuiItem(clearButtenItem);


  // make finish the default
  finishBtn->setDefault(true);

  // setup the focus
  cancelBtn->setFocusPolicy(QWidget::NoFocus);
  finishBtn->setFocusPolicy(QWidget::NoFocus);
  clearAllBtn->setFocusPolicy(QWidget::NoFocus);
  transactionsTable->setFocus();

  // initialize the display
  transactionsTable->setTransaction(m_transaction, m_account);
  updateSums();

  // connect signals with slots
  connect(transactionsTable, SIGNAL(transactionChanged(const MyMoneyTransaction&)),
          this, SLOT(slotSetTransaction(const MyMoneyTransaction&)));
  connect(transactionsTable, SIGNAL(returnPressed()), this, SLOT(accept()));
  connect(transactionsTable, SIGNAL(escapePressed()), this, SLOT(reject()));

  connect(cancelBtn, SIGNAL(clicked()), this, SLOT(reject()));
  connect(finishBtn, SIGNAL(clicked()), this, SLOT(accept()));
  connect(clearAllBtn, SIGNAL(clicked()), this, SLOT(slotClearAllSplits()));

  // Trick: it seems, that the initial sizing of the dialog does
  // not work correctly. At least, the columns do not get displayed
  // correct. Reason: the return value of transactionsTable->visibleWidth()
  // is incorrect. If the widget is visible, resizing works correctly.
  // So, we let the dialog show up and resize it then. It's not really
  // clean, but the only way I got the damned thing working.
  QTimer::singleShot( 10, this, SLOT(initSize()) );
}

KSplitTransactionDlg::~KSplitTransactionDlg()
{
}

void KSplitTransactionDlg::initSize(void)
{
  QDialog::resize(width(), height()+1);
}

void KSplitTransactionDlg::accept()
{
  transactionsTable->slotCancelEdit();
  if(!diffAmount().isZero()) {
    MyMoneySplit split = m_transaction.splitByAccount(m_account.id());
    kSplitCorrectionDlgDecl* dlg = new kSplitCorrectionDlgDecl(0, 0, true);

    // add icons to buttons
    KIconLoader *il = KGlobal::iconLoader();
    KGuiItem okButtenItem( i18n("&Ok" ),
                      QIconSet(il->loadIcon("button_ok", KIcon::Small, KIcon::SizeSmall)),
                      i18n("Accepts the selected action and continues"),
                      i18n("Use this to accept the action and perform it"));
    dlg->okBtn->setGuiItem(okButtenItem);

    KGuiItem cancelButtenItem( i18n( "&Cancel" ),
                      QIconSet(il->loadIcon("button_cancel", KIcon::Small, KIcon::SizeSmall)),
                      i18n("Return to split transaction dialog"),
                      i18n("Use this to continue editing the splits"));
    dlg->cancelBtn->setGuiItem(cancelButtenItem);

    QString total = (-split.value()).formatMoney();
    QString sums = splitsValue().formatMoney();
    QString diff = diffAmount().formatMoney();

    // now modify the text items of the dialog to contain the correct values
    QString q = i18n("The total amount of this transaction is %1 while "
                             "the sum of the splits is %2. The remaining %3 are "
                             "unassigned.")
                .arg(total)
                .arg(sums)
                .arg(diff);
    dlg->explanation->setText(q);

    q = i18n("&Change total amount of transaction to %1.")
        .arg(sums);
    dlg->changeBtn->setText(q);

    q = i18n("&Distribute difference of %1 among all splits.")
        .arg(diff);
    dlg->distributeBtn->setText(q);
    // FIXME remove the following line once distribution among
    //       all splits is implemented
    dlg->distributeBtn->hide();

    q = i18n("&Leave %1 unassigned.")
        .arg(diff);
    dlg->leaveBtn->setText(q);

    connect(dlg->okBtn, SIGNAL(clicked()), dlg, SLOT(accept()));
    connect(dlg->cancelBtn, SIGNAL(clicked()), dlg, SLOT(reject()));

    if(dlg->exec() == QDialog::Accepted) {
      QButton* button = dlg->buttonGroup->selected();
      if(button != NULL) {
        switch(dlg->buttonGroup->id(button)) {
          case 0:       // continue to edit
            break;
          case 1:       // modify total
            split.setValue(-splitsValue());
            m_transaction.modifySplit(split);
            kSplitTransactionDlgDecl::accept();
            break;
          case 2:       // distribute difference
            qDebug("distribution of difference not yet supported in KSplitTransactionDlg::slotFinishClicked()");
            break;
          case 3:       // leave unassigned
            qDebug("leave unassigned should be changed to assign to standard account");
            kSplitTransactionDlgDecl::accept();
            break;
        }
      }
    }
    delete dlg;

  } else
    kSplitTransactionDlgDecl::accept();

  if(result() == Accepted) {
    // for deposits, we inverted the sign of all splits.
    // now we revert it back, so that things are left correct
    if(m_isDeposit) {
      for(unsigned i = 0; i < m_transaction.splits().count(); ++i) {
        MyMoneySplit split = m_transaction.splits()[i];
        split.setValue(-split.value());
        m_transaction.modifySplit(split);
      }
    }
  }
}

void KSplitTransactionDlg::reject()
{
  // cancel any edit activity in the split register
  transactionsTable->slotCancelEdit();
  kSplitTransactionDlgDecl::reject();
}

void KSplitTransactionDlg::slotClearAllSplits()
{
  int answer;
  answer = KMessageBox::warningContinueCancel (NULL,
     i18n("You are about to delete all splits of this transaction. "
          "Do you really want to continue?"),
     i18n("KMyMoney"),
     i18n("Continue")
     );

  if(answer == KMessageBox::Continue) {
    transactionsTable->slotCancelEdit();
    QValueList<MyMoneySplit> list = transactionsTable->getSplits(m_transaction);
    QValueList<MyMoneySplit>::ConstIterator it;

    // clear all but the one referencing the account
    for(it = list.begin(); it != list.end(); ++it) {
      m_transaction.removeSplit(*it);
    }

    transactionsTable->setTransaction(m_transaction, m_account);
    slotSetTransaction(m_transaction);
  }
}

void KSplitTransactionDlg::slotSetTransaction(const MyMoneyTransaction& t)
{
  m_transaction = t;
  updateSums();
}

void KSplitTransactionDlg::updateSums(void)
{
  MyMoneyMoney splits(splitsValue());
  MyMoneySplit split = m_transaction.splitByAccount(m_account.id());

  if(m_amountValid == false) {
    split.setValue(-splits);
    m_transaction.modifySplit(split);
  }

  splitSum->setText("<b>" + splits.formatMoney() + " ");
  splitUnassigned->setText("<b>" + diffAmount().formatMoney() + " ");
  transactionAmount->setText("<b>" + (-split.value()).formatMoney() + " ");
}

MyMoneyMoney KSplitTransactionDlg::splitsValue(void)
{
  MyMoneyMoney splitsValue(m_calculatedValue);
  QValueList<MyMoneySplit> list = transactionsTable->getSplits(m_transaction);
  QValueList<MyMoneySplit>::ConstIterator it;

  // calculate the current sum of all split parts
  for(it = list.begin(); it != list.end(); ++it) {
    if((*it).value() != MyMoneyMoney::autoCalc)
      splitsValue += (*it).value();
  }

  return splitsValue;
}

MyMoneyMoney KSplitTransactionDlg::diffAmount(void)
{
  MyMoneyMoney diff(0);

  // if there is an amount specified in the transaction, we need to calculate the
  // difference, otherwise we display the difference as 0 and display the same sum.
  if(m_amountValid) {
    MyMoneySplit split = m_transaction.splitByAccount(m_account.id());

    diff = -(splitsValue() + split.value());
  }
  return diff;
}

#include "ksplittransactiondlg.moc"
