// -*- Mode: C++; -*-

#ifndef __TFUNCTIONS_H
#define __TFUNCTIONS_H

#include <ctype.h>
#include <algorithm>
#include <string>
#include <sstream>
#include <iostream>
#include <iterator>
#include <vector>
#include <valarray>
#include <list>
#include "stringlist.h"

namespace tfunctions {
    //! test if a list contains empty values
    template<class T> int check_empty (const T&container)
    {
        for (int i = 0; i < container.size();++i) {
            if (container[i].size()==0)
                return i;
        }
        return -1;
    }

    //! convert a string into uppercase chars
    /*!
     * converting strings to uppercase using STL algorithms.<br>
     * Example:
     * \code
     * std::string s = Toupper()("Hello");
     * // s will be "HELLO"
     * \endcode
     * \sa Tolower
     */
    class Toupper
    {
    public:
        //! empty constructor
        Toupper (){}
        //! converts a single char to uppercase
        char operator() (const char& c) { return toupper(c);}
        //! converts a whole c++ string into uppercase
        /*!
         * \param aString string to convert
         * \return the resulting string
         */
        std::string operator()(std::string aString)
        {
            std::string  capital_s = aString;
            std::transform (aString.begin(),aString.end(),
                            capital_s.begin(),
                            Toupper());
            return capital_s;
        }
    private:
    };

    //! convert a string into lowercase chars
    /*!
     * converting strings to lowercase using STL algorithms<br>
     * Example:
     * \code
     * std::string s = Tolower()("HEllo");
     * // s will be "hello"
     * \endcode
     * \sa Toupper
     */
    class Tolower
    {
    public:
        //! empty constructor
        Tolower (){}
        //! convert a single character
        char operator()(char c) { return tolower(c); }
        //! converts a whole c++ string into lowercase
        /*!
         * \param aString string to convert
         * \return the resulting string
         */
        std::string operator()(std::string aString)
        {
            std::string  capital_s = aString;
            std::transform (aString.begin(),aString.end(),
                            capital_s.begin(),
                            Tolower());
            return capital_s;
        }
    private:
    };

    //! Removes spaces and tabs from end and beginning of a string
    class Trim
    {
    public:
        //! empty constructor
        Trim(){}
        //! main operator
        /*!
         * removes all signs being a space from start and end of string.<br>
         * \code
         * string s = "   hello   ";
         * s = Trim()(s);
         * // s will be "hello";
         * \endcode
         * \param s string to be converted
         */
        std::string operator()(const std::string&s)
        {
            std::string c;
            c = s;
            while(c.size()>0 && isspace(c[0]) ) {c.erase(c.begin());}
            while(c.size()>0 && isspace(c[c.size()-1])) {c.resize(c.size()-1);}
            return c;
        }
    };

    //! concatenate a list of values to one value
    /*!
     * This operator class will concat a list of (string-)values to a
     * value of same type. Template T must be a standard list of type
     * S, eg. list of string: T = vector<string>, S = string. T must support
     * numeric iterators and size(), S must support += operator.
     * \code
     * vector<string> s; string del = "*";
     * // setting s to list "a","b","c","d"
     * string res = Implode<vector<string>,string>()(s,del,false)
     * \endcode
     * res in example above will then result in "a*b*c*d"
     */
    template<class T> class Implode
    {
    public:
        //! empty standard constructor
        Implode(){};
        //! The main operator
        /*!
         * This operator will perform the implode<br>
         * but this will only work on string and string-lists, 'cause we can so use
         * realy fast algorithms
         * \param s List to concatenate
         * \param delemiter value to be inserted between parts
         * \param atStart should the delemiter be the first (eg. "*a*b*c" instead of "a*b*c")
         * \return the resulting value of type S
         */
        std::string operator()(const T&s,const std::string&delemiter,bool atStart)
        {
            std::string r;
            std::stringstream _c;
            if (s.size()==0) return "";
            if (atStart)
                _c << delemiter;
            std::ostream_iterator<std::string>
                out(_c,delemiter.c_str());
            copy(s.begin(),s.end(),out);
            r = _c.str();
            r.resize(r.size()-1);
            return r;
        }
    };

    class Explode
    {
    public:
        Explode(){};
        stringlist operator()(const std::string&s,const std::string&delemiter,bool aTrimit)
        {
            unsigned int pos;
            stringlist l;
            std::string iLine = s;
            while ( (pos = iLine.find(delemiter))!=std::string::npos) {
                if (aTrimit) {
                    l.push_back(tfunctions::Trim()(iLine.substr(0,pos)));
                } else {
                    l.push_back(iLine.substr(0,pos));
                }
                iLine = iLine.substr(pos+1);
            }
            if (iLine.size()>0) {
                l.push_back((aTrimit?tfunctions::Trim()(iLine):iLine));
            }
            return l;
        }
    };

    //! simple Sum class (from c++ bible)
    template<class T> class Sum
    {
        T res;
    public:
        Sum(T i = 0):res(i){}
        void operator()(T x){res+=x;}
        T result()const{return res;}
    };
    //! convert valarray to vector
    template<class T> class ArrayToVector
    {
    protected:
        typename std::vector<T> res;
    public:
        ArrayToVector(const typename std::valarray<T>&A)
            :res(A.size())
        {
            for (unsigned int i = 0; i < A.size();++i) {
                res[i]=A[i];
            }
        }
        const typename std::vector<T>&operator()()const{return res;}
    };
}

#endif
