/* This file is part of KNemo
   Copyright (C) 2004, 2005 Percy Leonhardt <percy@eris23.de>
   Copyright (C) 2009 John Stamp <jstamp@users.sourceforge.net>

   KNemo is free software; you can redistribute it and/or modify
   it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   KNemo is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <unistd.h>

#include <kio/global.h>
#include <KHelpMenu>
#include <KIcon>
#include <KAction>
#include <KActionCollection>
#include <KApplication>
#include <KLocale>
#include <KMenu>
#include <KIconLoader>
#include <KProcess>
#include <KNotification>

#include "data.h"
#include "interface.h"
#include "knemodaemon.h"
#include "interfaceicon.h"

const QString InterfaceIcon::ICON_DISCONNECTED = "network_disconnected";
const QString InterfaceIcon::ICON_CONNECTED = "network_connected";
const QString InterfaceIcon::ICON_INCOMING = "network_incoming";
const QString InterfaceIcon::ICON_OUTGOING = "network_outgoing";
const QString InterfaceIcon::ICON_TRAFFIC = "network_traffic";
const QString InterfaceIcon::SUFFIX_PPP = "_ppp";
const QString InterfaceIcon::SUFFIX_LAN = "_lan";
const QString InterfaceIcon::SUFFIX_WLAN = "_wlan";

Q_DECLARE_METATYPE(InterfaceCommand)

InterfaceIcon::InterfaceIcon( Interface* interface )
    : QObject(),
      mInterface( interface ),
      mTray( 0L )
{
    setupToolTipArray();
    commandActions = new KActionCollection( this );
    plotterAction = new KAction( KIcon( "utilities-system-monitor" ),
                       i18n( "&Open Traffic Plotter" ), this );
    statisticsAction = new KAction( KIcon( "view-statistics" ),
                          i18n( "Open &Statistics" ), this );
    configAction = new KAction( KIcon( "configure" ),
                       i18n( "&Configure KNemo..." ), this );
}

InterfaceIcon::~InterfaceIcon()
{
    if ( mTray != 0L )
        delete mTray;
}

void InterfaceIcon::updateStatus( int status )
{
    if ( mTray == 0L )
        return;

    // If the user wants something different than the default icons
    // append the correct suffix to the filename.
    QString suffix;
    if ( mInterface->getSettings().iconSet == Interface::NETWORK )
    {
        suffix = SUFFIX_LAN;
    }
    else if ( mInterface->getSettings().iconSet == Interface::WIRELESS )
    {
        suffix = SUFFIX_WLAN;
    }
    else if ( mInterface->getSettings().iconSet == Interface::MODEM )
    {
        suffix = SUFFIX_PPP;
    }
    else
    {
        suffix = ""; // use standard icons
    }

    // Now set the correct icon depending on the status of the interface.
    if ( status == Interface::NOT_AVAILABLE ||
         status == Interface::NOT_EXISTING )
    {
        mTray->setIcon( UserIcon( ICON_DISCONNECTED + suffix ) );
    }
    else if ( ( status & Interface::RX_TRAFFIC ) &&
              ( status & Interface::TX_TRAFFIC ) )
    {
        mTray->setIcon( UserIcon( ICON_TRAFFIC + suffix ) );
    }
    else if ( status & Interface::RX_TRAFFIC )
    {
        mTray->setIcon( UserIcon( ICON_INCOMING + suffix ) );
    }
    else if ( status & Interface::TX_TRAFFIC )
    {
        mTray->setIcon( UserIcon( ICON_OUTGOING + suffix ) );
    }
    else
    {
        mTray->setIcon( UserIcon( ICON_CONNECTED + suffix ) );
    }
    updateToolTipData();
}

void InterfaceIcon::updateToolTip()
{
    if ( mTray == 0L )
        return;
    mTray->setToolTip( mToolTipTop + mToolTipTime + mToolTipBottom );
}

void InterfaceIcon::updateToolTipTime( QString uptime )
{
    mToolTipTime = "<tr><td>" + mToolTips.value( UPTIME ) + "</td><td>" + uptime + "</td></tr>" ;
    updateToolTip();
}

void InterfaceIcon::updateMenu()
{
    if ( mTray == 0L )
        return;

    // Remove all old entries.
    KMenu* menu = (KMenu*)mTray->contextMenu();
    QList<QAction *> actions = menu->actions();
    foreach ( QAction* action, commandActions->actions() )
        menu->removeAction( action );
    commandActions->clear();
    menu->removeAction( statisticsAction );

    InterfaceSettings& settings = mInterface->getSettings();

    // If the user wants custom commands, add them.
    if ( settings.customCommands )
    {
        int i = 0;
        foreach ( InterfaceCommand command, settings.commands )
        {
            QAction *action = new QAction( command.menuText, this );
            action->setData( QVariant::fromValue( command ) );
            commandActions->addAction( QString( "command%1" ).arg( i ), action );
            ++i;
        }
        QAction* sep = menu->addSeparator();
        commandActions->addAction( "sep", sep );
        menu->insertActions( plotterAction, commandActions->actions() );
    }

    if ( settings.activateStatistics )
        menu->insertAction( configAction, statisticsAction );
}

void InterfaceIcon::updateTrayStatus( int previousState )
{
    bool interfaceExists = mInterface->getData().existing;
    bool interfaceAvailable = mInterface->getData().available;
    bool hideWhenNotExisting = mInterface->getSettings().hideWhenNotExisting;
    bool hideWhenNotAvailable = mInterface->getSettings().hideWhenNotAvailable;

    // notification 'interface not available'
    if ( !interfaceAvailable && mTray != 0L &&
         previousState == Interface::AVAILABLE )
    {
        /* When KNemo is starting we don't show the change in connection
         * status as this would be annoying when KDE starts.
         */
        QString title;
        if ( mInterface->getSettings().alias != QString::null )
            title = mInterface->getSettings().alias;
        else
            title = mInterface->getName();

        KNotification::event( "knemo_disconnected",
                       title + ":\n" + i18n( "Not connected." ) );
    }

    // notification 'interface not existing'
    if ( !interfaceExists && mTray != 0L &&
         previousState != Interface::UNKNOWN_STATE )
    {
        /* When KNemo is starting we don't show the change in connection
         * status as this would be annoying when KDE starts.
         */
        QString title;
        if ( mInterface->getSettings().alias != QString::null )
            title = mInterface->getSettings().alias;
        else
            title = mInterface->getName();

        KNotification::event( "knemo_notexisting",
                              title + ":\n" + i18n( "Not existing." ) );
    }

    /* Remove the icon if
     * - the interface is not available and the option to hide it is selected
     * - the interface does not exist, the option to hide it is selected
     *   and the other option is not selected
     */
    if ( mTray != 0L &&
         ( ( !interfaceAvailable && hideWhenNotAvailable ) ||
           ( !interfaceExists && hideWhenNotExisting && !hideWhenNotAvailable ) ) )
    {
        delete mTray;
        mTray = 0L;
    }
    /* Create the icon if
     * - the interface is available
     * - the interface is not available and the option to hide it is not
     *   selected and the interface does exist
     * - the interface does not exist and the option to hide it is not selected
     *   and the other option is not selected
     */
    else if ( mTray == 0L &&
              ( interfaceAvailable ||
                ( !interfaceAvailable && !hideWhenNotAvailable && interfaceExists ) ||
                ( !interfaceExists && !hideWhenNotExisting && !hideWhenNotAvailable ) ) )
    {
        mTray = new KSystemTrayIcon( mInterface->getName() );
        mTray->setToolTip( mInterface->getName() );
        KMenu* menu = (KMenu *)mTray->contextMenu();

        menu->removeAction( menu->actions().at( 0 ) );
        menu->addTitle( KIcon( "knemo" ), "KNemo - " + mInterface->getName() );
        menu->addAction( plotterAction );
        menu->addAction( statisticsAction );
        menu->addAction( configAction );
        KHelpMenu* helpMenu( new KHelpMenu( menu, KNemoDaemon::aboutData(), false ) );
        menu->addMenu( helpMenu->menu() )->setIcon( KIcon( "help-contents" ) );

        connect( menu, SIGNAL( triggered( QAction * ) ),
                 this, SLOT( menuTriggered( QAction * ) ) );
        connect( mTray, SIGNAL( activated( QSystemTrayIcon::ActivationReason ) ),
                 this, SLOT( iconActivated( QSystemTrayIcon::ActivationReason ) ) );
        connect( plotterAction, SIGNAL( triggered() ),
                 this, SLOT( showGraph() ) );
        connect( statisticsAction, SIGNAL( triggered() ),
                 this, SLOT( showStatistics() ) );
        connect( configAction, SIGNAL( triggered() ),
                 this, SLOT( showConfigDialog() ) );

        updateStatus( mInterface->getState() );
        updateToolTipData();
        updateMenu();
        mTray->show();
    }

    // notification 'interface available'
    if ( interfaceAvailable && mTray != 0L &&
         previousState != Interface::UNKNOWN_STATE )
    {
        /* When KNemo is starting we don't show the change in connection
         * status as this would be annoying when KDE starts.
         */
        QString title;
        if ( mInterface->getSettings().alias != QString::null )
            title = mInterface->getSettings().alias;
        else
            title = mInterface->getName();

        if ( mInterface->getData().wirelessDevice )
        {
            KNotification::event( "knemo_connected",
                                  title + ":\n" + i18n( "Connection established to\n" ) +
                                  mInterface->getWirelessData().essid );
        }
        else
        {
            KNotification::event( "knemo_connected",
                                  title + ":\n" + i18n( "Connection established." ) );
        }
    }
}

void InterfaceIcon::showConfigDialog()
{
    KNemoDaemon::sSelectedInterface = mInterface->getName();

    KProcess process;
    process << "kcmshell4" << "kcm_knemo";
    process.startDetached();
}

void InterfaceIcon::iconActivated( QSystemTrayIcon::ActivationReason reason )
{
    switch (reason)
    {
     case QSystemTrayIcon::Trigger:
         mInterface->showStatusDialog();
         break;
     case QSystemTrayIcon::MiddleClick:
         mInterface->showSignalPlotter( true );
         break;
     default:
         ;
     }
}

void InterfaceIcon::menuTriggered( QAction *action )
{
    if ( !action->data().canConvert<InterfaceCommand>() )
        return;

    InterfaceCommand command = action->data().value<InterfaceCommand>();
    KProcess process;
    if ( command.runAsRoot )
    {
        process << "kdesu";
        process << command.command;
    }
    else
        process << command.command.split( " " );

    process.startDetached();
}

void InterfaceIcon::updateToolTipData()
{
    int toolTipContent = mInterface->getGeneralData().toolTipContent;
    InterfaceData& data = mInterface->getData();

    mToolTipTop = "<table cellspacing=0 cellpadding=0 border=0>";
    mToolTipBottom.clear();

    if ( ( toolTipContent & ALIAS ) &&
         mInterface->getSettings().alias != QString::null )
        mToolTipTop += "<tr><th colspan=2 align=center>" + mInterface->getSettings().alias + "</th></tr>";
    if ( toolTipContent & INTERFACE )
        mToolTipTop += "<tr><td>" + mToolTips.value( INTERFACE ) + "</td><td>" + mInterface->getName() + "</td></tr>";
    if ( data.available )
    {
        if ( toolTipContent & STATUS )
            mToolTipTop += "<tr><td>" + mToolTips.value( STATUS ) + "</td><td>" + i18n( "Connection established." ) + "</td></tr>";
    }
    else if ( data.existing )
    {
        if ( toolTipContent & STATUS )
            mToolTipTop += "<tr><td>" + mToolTips.value( STATUS ) + "</td><td>" + i18n( "Not connected." ) + "</td></tr>";
    }
    else
    {
        if ( toolTipContent & STATUS )
            mToolTipTop += "<tr><td>" + mToolTips.value( STATUS ) + "</td><td>" + i18n( "Not existing." ) + "</td></tr>";
    }

    if ( data.available )
    {
        if ( toolTipContent & IP_ADDRESS )
            mToolTipBottom += "<tr><td>" + mToolTips.value( IP_ADDRESS ) + "</td><td>" + data.ipAddress + "</td></tr>";
        if ( toolTipContent & SUBNET_MASK )
            mToolTipBottom += "<tr><td>" + mToolTips.value( SUBNET_MASK ) + "</td><td>" + data.subnetMask + "</td></tr>";
        if ( mInterface->getType() == Interface::ETHERNET )
        {
            if ( toolTipContent & BCAST_ADDRESS )
                mToolTipBottom += "<tr><td>" + mToolTips.value( BCAST_ADDRESS ) + "</td><td>" + data.broadcastAddress + "</td></tr>";
            if ( toolTipContent & GATEWAY )
                mToolTipBottom += "<tr><td>" + mToolTips.value( GATEWAY ) + "</td><td>" + data.defaultGateway + "</td></tr>";
            if ( toolTipContent & HW_ADDRESS )
                mToolTipBottom += "<tr><td>" + mToolTips.value( HW_ADDRESS ) + "</td><td>" + data.hwAddress + "</td></tr>";
        }
        if ( mInterface->getType() == Interface::PPP )
        {
            if ( toolTipContent & PTP_ADDRESS )
                mToolTipBottom += "<tr><td>" + mToolTips.value( PTP_ADDRESS ) + "</td><td>" + data.ptpAddress + "</td></tr>";
        }
        if ( toolTipContent & RX_PACKETS )
            mToolTipBottom += "<tr><td>" + mToolTips.value( RX_PACKETS ) + "</td><td>" + QString::number( data.rxPackets ) + "</td></tr>";
        if ( toolTipContent & TX_PACKETS )
            mToolTipBottom += "<tr><td>" + mToolTips.value( TX_PACKETS ) + "</td><td>" + QString::number( data.txPackets ) + "</td></tr>";
        if ( toolTipContent & RX_BYTES )
            mToolTipBottom += "<tr><td>" + mToolTips.value( RX_BYTES ) + "</td><td>" + data.rxString + "</td></tr>";
        if ( toolTipContent & TX_BYTES )
            mToolTipBottom += "<tr><td>" + mToolTips.value( TX_BYTES ) + "</td><td>" + data.txString + "</td></tr>";
        if ( toolTipContent & DOWNLOAD_SPEED )
        {
            unsigned long bytesPerSecond = data.incomingBytes / mInterface->getGeneralData().pollInterval;
            mToolTipBottom += "<tr><td>" + mToolTips.value( DOWNLOAD_SPEED ) + "</td><td>" + KIO::convertSize( bytesPerSecond ) + i18n( "/s" ) + "</td></tr>";
        }
        if ( toolTipContent & UPLOAD_SPEED )
        {
            unsigned long bytesPerSecond = data.outgoingBytes / mInterface->getGeneralData().pollInterval;
            mToolTipBottom += "<tr><td>" + mToolTips.value( UPLOAD_SPEED ) + "</td><td>" + KIO::convertSize( bytesPerSecond ) + i18n( "/s" ) + "</td></tr>";
        }
    }

    if ( data.available && data.wirelessDevice )
    {
        WirelessData& wdata = mInterface->getWirelessData();
        if ( toolTipContent & ESSID )
            mToolTipBottom += "<tr><td>" + mToolTips.value( ESSID ) + "</td><td>" + wdata.essid + "</td></tr>";
        if ( toolTipContent & MODE )
            mToolTipBottom += "<tr><td>" + mToolTips.value( MODE ) + "</td><td>" + wdata.mode + "</td></tr>";
        if ( toolTipContent & FREQUENCY )
            mToolTipBottom += "<tr><td>" + mToolTips.value( FREQUENCY ) + "</td><td>" + wdata.frequency + "</td></tr>";
        if ( toolTipContent & BIT_RATE )
            mToolTipBottom += "<tr><td>" + mToolTips.value( BIT_RATE ) + "</td><td>" + wdata.bitRate + "</td></tr>";
        if ( toolTipContent & ACCESS_POINT )
            mToolTipBottom += "<tr><td>" + mToolTips.value( ACCESS_POINT ) + "</td><td>" + wdata.accessPoint + "</td></tr>";
        if ( toolTipContent & LINK_QUALITY )
            mToolTipBottom += "<tr><td>" + mToolTips.value( LINK_QUALITY ) + "</td><td>" + wdata.linkQuality + "</td></tr>";
        if ( toolTipContent & NICK_NAME )
            mToolTipBottom += "<tr><td>" + mToolTips.value( NICK_NAME ) + "</td><td>" + wdata.nickName + "</td></tr>";
        if ( toolTipContent & ENCRYPTION )
        {
            if ( wdata.encryption == true )
            {
                mToolTipBottom += "<tr><td>" + mToolTips.value( ENCRYPTION ) + "</td><td>" + i18n( "active" ) + "</td></tr>";
            }
            else
            {
                mToolTipBottom += "<tr><td>" + mToolTips.value( ENCRYPTION ) + "</td><td>" + i18n( "off" ) + "</td></tr>";
            }
        }
    }
    mToolTipBottom += "</table>";
    updateToolTip();
}

void InterfaceIcon::setupToolTipArray()
{
    // Cannot make this data static as the i18n macro doesn't seem
    // to work when called to early i.e. before setting the catalogue.
    mToolTips.insert( INTERFACE, i18n( "Interface" ) );
    mToolTips.insert( ALIAS, i18n( "Alias" ) );
    mToolTips.insert( STATUS, i18n( "Status" ) );
    mToolTips.insert( UPTIME, i18n( "Uptime" ) );
    mToolTips.insert( IP_ADDRESS, i18n( "IP-Address" ) );
    mToolTips.insert( SUBNET_MASK, i18n( "Subnet Mask" ) );
    mToolTips.insert( HW_ADDRESS, i18n( "HW-Address" ) );
    mToolTips.insert( PTP_ADDRESS, i18n( "PtP-Address" ) );
    mToolTips.insert( RX_PACKETS, i18n( "Packets Received" ) );
    mToolTips.insert( TX_PACKETS, i18n( "Packets Sent" ) );
    mToolTips.insert( RX_BYTES, i18n( "Bytes Received" ) );
    mToolTips.insert( TX_BYTES, i18n( "Bytes Sent" ) );
    mToolTips.insert( ESSID, i18n( "ESSID" ) );
    mToolTips.insert( MODE, i18n( "Mode" ) );
    mToolTips.insert( FREQUENCY, i18n( "Frequency" ) );
    mToolTips.insert( BIT_RATE, i18n( "Bit Rate" ) );
    mToolTips.insert( ACCESS_POINT, i18n( "Access Point" ) );
    mToolTips.insert( LINK_QUALITY, i18n( "Link Quality" ) );
    mToolTips.insert( BCAST_ADDRESS, i18n( "Broadcast Address" ) );
    mToolTips.insert( GATEWAY, i18n( "Default Gateway" ) );
    mToolTips.insert( DOWNLOAD_SPEED, i18n( "Download Speed" ) );
    mToolTips.insert( UPLOAD_SPEED, i18n( "Upload Speed" ) );
    mToolTips.insert( NICK_NAME, i18n( "Nickname" ) );
    mToolTips.insert( ENCRYPTION, i18n( "Encryption" ) );
}

void InterfaceIcon::showStatistics()
{
    emit statisticsSelected();
}

void InterfaceIcon::showGraph()
{
    mInterface->showSignalPlotter( false );
}

#include "interfaceicon.moc"
