/***************************************************************************
 *   Copyright (C) 2005 by yodor   *
 *   yodor@developer.bg   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "chartframe.h"
#include <qtooltip.h>


ChartFrame::ChartFrame(int step, QWidget *parent, const char *name) : QWidget(parent,name), step_val(step)
{
	setBackgroundMode(Qt::NoBackground);
	//setPaletteBackgroundColor(Qt::black);
	setWFlags(Qt::WNoAutoErase);
	
	vin = 0;
	vout=0;
	w=0;
	maxRate=1;
	buf=0;
	buf.resize(width(),height());
	chartstyle=1;
	QToolTip::add(this,"Click to change style");
}
const QPixmap& ChartFrame::getBuffer()
{
	
	return buf;
}
ChartFrame::~ChartFrame()
{
	if (vin) delete vin;
	if (vout) delete vout;
	vin=0;
	vout=0;
}
void ChartFrame::resizeEvent(QResizeEvent *res)
{
	QSize new_size = res->size();
	QSize old_size = res->oldSize();


	int old_w = w;

	w = new_size.width() / step_val;
	
	buf.resize(new_size.width(),new_size.height());
	
	if (!vin){
		vin = new int[w];
		vout = new int[w];
		clearArrays(vin,vout,w);
	}
	else {
		
		int n_in[w];
		int n_out[w];
		clearArrays(n_in,n_out,w);

		//copy old contents to new contents
		if (old_w>w) {
			vin+=(old_w-w);
			vout+=(old_w-w);
			memcpy(n_in,vin,w);
			memcpy(n_out,vout,w);
		}
		else if (old_w<w) {
			memcpy(n_in,vin,old_w);
			memcpy(n_out,vout,old_w);
		}
		delete vin;
		delete vout;
		vin = new int[w];
		vout = new int[w];
		memcpy(vin,n_in,w);
		memcpy(vout,n_out,w);
	}

	drawChart();
	
}

void ChartFrame::clearChart()
{
	clearArrays(vin,vout,w);
	drawChart();

}
void ChartFrame::paintEvent(QPaintEvent *)
{
	
	QPainter paint(this);
	paint.drawPixmap(0,0,buf);

}
void ChartFrame::drawChart()
{
	
	buf.fill(QColor(0,0,0));

	if (!vin)return;
	if (!vout)return;

	QPainter paint(&buf);
	QPen pen;
	int hh = height();
	int ww = width();

	pen.setColor(QColor(100,0,0));
	pen.setWidth(1);	
	paint.setPen(pen);
	
	for (int a=0;a<ww;a+=step_val)
	{
		paint.drawLine(a,0,a,hh);	
	}
	for (int a=0;a<hh;a+=step_val)
	{
		paint.drawLine(0,a,ww,a);	
	}

//	int x = 0;
	int y = 0;

	QBrush b;
	b.setStyle(QBrush::SolidPattern);

	int num_blocks = height() / step_val;
	if (chartstyle==1) num_blocks=height();
	//double scaled =  maxRate / num_blocks;
	int yIN = height();
	int yOUT = height();
	y=height();

	pen.setWidth(step_val/2);
	pen.setJoinStyle(RoundJoin);
	pen.setCapStyle(RoundCap);

	for (int a=0;a<w;a++) {
		double inval = ((double)num_blocks/(double)maxRate) * vin[a];
		double outval = ((double)num_blocks/(double)maxRate) * vout[a] ;
		
		int x = a * step_val ;
		
		int vblock=0;
		
		
		if (chartstyle==0) {
			y = height();
			b.setColor(Qt::red);
			
			while (vblock<inval) {
				y-=step_val;
				paint.fillRect(x+1,y+1,step_val-1,step_val-1,b);
				vblock++;
			}
		}
		else {
			pen.setColor(Qt::red);
			paint.setPen(pen);			
			paint.drawLine(x,yIN,x+step_val,(int)(y-inval));
			yIN=(int)(y-inval);
			
		}
		
		if (chartstyle==0) {
			b.setColor(Qt::green);
			while (vblock<(inval+outval)) {
				y-=step_val;
				paint.fillRect(x+1,y+1,step_val-1,step_val-1,b);
				vblock++;
			}
		}
		else {
			pen.setColor(Qt::green);
			paint.setPen(pen);
			paint.drawLine(x,yOUT,x+step_val,(int)(y-outval));
			yOUT=(int)(y-outval);
			
		}
	}
}
void ChartFrame::clearArrays(int *in, int *out, int size)
{
	QMutexLocker lock(&mutex);

	for (int a=0;a<size;a++)
	{
		in[a]=0;
		out[a]=0;
	}
}

void ChartFrame::appendValue(Q_ULLONG in, Q_ULLONG out)
{
	QMutexLocker lock(&mutex);

	if (w<1)return;

	
	
	Q_ULLONG total = 0;
	Q_ULLONG gmax = 0;
	for (int a=1;a<w;a++)
	{
		vin[a-1]=vin[a];
		vout[a-1]=vout[a];
		total = vin[a]+vout[a];
		if (total>gmax)gmax=total;
		
	}
	vin[w-1]=in;
	vout[w-1]=out;
	total = (in+out);
	if (total>gmax)gmax=total;
	maxRate=gmax;
	if (maxRate==0)maxRate=1;
	
	drawChart();
	
}

void ChartFrame::mouseReleaseEvent(QMouseEvent *)
{
	chartstyle++;
	if (chartstyle>1)chartstyle=0;
	emit chartStyleChange(chartstyle);
}

void ChartFrame::setChartStyle(int s)
{
	if (s<0)s=0;
	if (s>1)s=1;
	chartstyle=s;
}

int ChartFrame::getChartStyle()
{
	return chartstyle;
}


