/* This file is part of the KDE project
  Copyright (C) 2007 Florian Piquemal <flotueur@yahoo.fr>
  Copyright (C) 2007 Alexis Ménard <darktears31@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
* Boston, MA 02110-1301, USA.
*/

#include "kptperteditor.h"
#include "kptproject.h"

#include <KoDocument.h>

namespace KPlato
{

void PertEditor::setProject( Project *project )
{
    if ( m_project ) {
        disconnect( m_project, SIGNAL( relationRemoved(Relation *) ), this, SLOT( dispAvailableTasks(Relation *) ) );
        disconnect( m_project, SIGNAL( relationAdded(Relation *) ), this, SLOT( dispAvailableTasks(Relation *) ) );
        disconnect( m_project, SIGNAL( nodeChanged( Node* ) ), this, SLOT( slotUpdate() ) );
    }
    m_project = project;
    m_node = project; // TODO: review
    if ( m_project ) {
        connect( m_project, SIGNAL( relationRemoved(Relation *) ), this, SLOT( dispAvailableTasks(Relation *) ) );
        connect( m_project, SIGNAL( relationAdded(Relation *) ), this, SLOT( dispAvailableTasks(Relation *) ) );
        connect( m_project, SIGNAL( nodeChanged( Node* ) ), this, SLOT( slotUpdate() ) );
    }
    draw();
}

void PertEditor::draw( Project &project)
{
    setProject( &project );
    draw();
}

void PertEditor::draw()
{
    m_tasktree->clear();
    if ( m_project == 0 ) {
        return;
    }
    
    foreach(Node * currentNode, m_project->projectNode()->childNodeIterator()){
        kDebug() << currentNode->type();
        if (currentNode->type()!=4){
            QTreeWidgetItem * item = new QTreeWidgetItem( m_tasktree );
            item->setText( 0, currentNode->name());
            drawSubTasksName(item,currentNode);
        }
        //kDebug() <<"[void KPlato::PertEditor::draw( Project &project )] TASK FOUNDED";
    }
    

}

void PertEditor::drawSubTasksName( QTreeWidgetItem *parent, Node * currentNode)
{
    if (currentNode->numChildren() > 0){
        foreach(Node * currentChild, currentNode->childNodeIterator()){
            if (currentChild->type()!=4){
                QTreeWidgetItem * item = new QTreeWidgetItem( parent );
                item->setText( 0, currentChild->name());
                drawSubTasksName( item, currentChild);
            }
            //kDebug() <<"[void KPlato::PertEditor::draw( Project &project )] SUBTASK FOUNDED";
        }
    }
}


//-----------------------------------
PertEditor::PertEditor( KoDocument *part, QWidget *parent ) 
    : ViewBase( part, parent ),
    m_node( 0 ),
    m_project( 0 )
{
    kDebug() <<" ---------------- KPlato: Creating PertEditor ----------------";
    widget.setupUi(this);
    widget.assignList->setSelectedLabel(i18n("Required Tasks :"));
    widget.assignList->setAvailableLabel(i18n("Available Tasks :"));
    widget.assignList->setShowUpDownButtons(false);
    widget.assignList->layout()->setMargin(0);
    widget.assignList->setMoveOnDoubleClick(false);

    m_tasktree = widget.tableTaskWidget;
    m_assignList = widget.assignList;
    m_part = part;
    
    for (int i=0;i< list_nodeNotView.size();i++)
        list_nodeNotView.removeFirst();

    connect( m_tasktree, SIGNAL( itemSelectionChanged() ), SLOT( dispAvailableTasks() ) );
    connect( this, SIGNAL( refreshAvailableTaskList() ), SLOT( dispAvailableTasks() ) );
    connect( m_assignList, SIGNAL(added(QListWidgetItem *)), this, SLOT(addTaskInRequiredList(QListWidgetItem * )) );
    connect( m_assignList, SIGNAL(removed(QListWidgetItem *)), this, SLOT(removeTaskFromRequiredList(QListWidgetItem * )) );

    // connects used to refresh the kactionselector after an undo/redo
}

void PertEditor::updateReadWrite( bool rw )
{
    //TODO
}

void PertEditor::dispAvailableTasks( Relation *rel ){
    dispAvailableTasks();
}


void PertEditor::dispAvailableTasks(){

    list_nodeNotView.clear();

    list_nodeNotView.begin();

    QList<QTreeWidgetItem*> selectedItemList=m_tasktree->selectedItems();
    
    if(!(selectedItemList.count()<1))
    {
        QString selectedTaskName = m_tasktree->selectedItems().first()->text(0);
        m_assignList->availableListWidget()->clear();
        m_assignList->selectedListWidget()->clear();
        
        loadRequiredTasksList(itemToNode(selectedTaskName, m_node));
        
        listNodeNotView(itemToNode(selectedTaskName, m_node));
        
        foreach(Node * currentNode, m_node->childNodeIterator() )
        {
            // Checks if the curent node is not a milestone
            // and if it isn't the same as the selected task in the m_tasktree
            if ( currentNode->type() != Node::Type_Milestone 
		&& currentNode->type() != Node::Type_Summarytask 
		&& currentNode->name() != selectedTaskName
                &&  !list_nodeNotView.contains(currentNode)
                && (m_assignList->selectedListWidget()->findItems(currentNode->name(),0)).empty())
            {
                m_assignList->availableListWidget()->addItem(currentNode->name());
            }
            else
            {
                QFont* fakeItemFont = new QFont();
                QBrush* fakeItemBrush = new QBrush();
                    fakeItemFont->setItalic(true);
                    fakeItemBrush->setColor(QColor ( Qt::lightGray));
                QListWidgetItem* fakeItem = new QListWidgetItem();
                    fakeItem->setText(currentNode->name());
                    fakeItem->setFont(*fakeItemFont);
                    fakeItem->setForeground(*fakeItemBrush);
                    fakeItem->setFlags(Qt::ItemIsEnabled);
                m_assignList->availableListWidget()->addItem(fakeItem);
            }
        }

        list_nodeNotView.clear();
    }
    else
    {
        m_assignList->availableListWidget()->clear();
        m_assignList->selectedListWidget()->clear();
    }
}

//return parents of the node
QList<Node*> PertEditor::listNodeNotView(Node * node)
{
    list_nodeNotView = node->getParentNodes();
    foreach(Node* currentNode,m_node->childNodeIterator())
    {
        if (currentNode->isDependChildOf(node))
        {
            list_nodeNotView.append(currentNode);
        }
    }
    return list_nodeNotView;
}


Node * PertEditor::itemToNode(QString itemName, Node * startNode){
    Node * result = 0;
    if (startNode->numChildren() > 0){
        foreach(Node * currentNode, startNode->childNodeIterator() ){
            if (currentNode->name() == itemName) {
                return currentNode;
            } else {
                result=itemToNode(itemName, currentNode);
            }
        }
    }
    return result;
}

void PertEditor::addTaskInRequiredList(QListWidgetItem * currentItem){
    // add the relation between the selected task and the current task
    QString selectedTaskName = m_tasktree->selectedItems().first()->text(0);

    Relation* m_rel=new Relation (itemToNode(currentItem->text(), m_node),itemToNode(selectedTaskName, m_node));
    AddRelationCmd * addCmd= new AddRelationCmd(*m_project,m_rel,currentItem->text());
    m_part->addCommand( addCmd );

}

void PertEditor::removeTaskFromRequiredList(QListWidgetItem * currentItem){
    // remove the relation between the selected task and the current task
    QString selectedTaskName = m_tasktree->selectedItems().first()->text(0);

    // remove the relation
    Relation* m_rel = itemToNode(selectedTaskName, m_node)->findParentRelation(itemToNode(currentItem->text(), m_node));
    DeleteRelationCmd * delCmd= new DeleteRelationCmd(*m_project,m_rel,currentItem->text());
    m_part->addCommand( delCmd );

    emit refreshAvailableTaskList();
}

void PertEditor::loadRequiredTasksList(Node * taskNode){
    // Display the required task list into the rigth side of m_assignList
    m_assignList->selectedListWidget()->clear();
    foreach(Relation * currentRelation, taskNode->dependParentNodes()){
            m_assignList->selectedListWidget()->addItem(currentRelation->parent()->name());
        }
}

void PertEditor::slotUpdate(){

 draw();
}

} // namespace KPlato

#include "kptperteditor.moc"
