/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QComboBox>
#include <QDialog>
#include <QLabel>
#include <QLineEdit>
#include <QMessageBox>
#include <QPushButton>
#include <QToolButton>

#include "connection.h"
#include "itemview.h"
#include "ldapsearchdialog.h"

#include "aclselectionfield.h"

using namespace Form;

/**
 * An acl can have the following form where xxx is the uid and yyy the right:
 *
 * xxx yyy
 * xxx yyy yyy
 * xxx xxx yyy
 * xxx xxx yyy yyy
 * group:xxx yyy
 * group:xxx yyy yyy
 * group:xxx xxx yyy yyy
 *
 * so a simple split( ' ' ) doesn't work here...
 */
static QStringList splitAcl( const QString &acl )
{
  QStringList retval;

  QStringList aclParts = acl.split( " " );
  if ( aclParts.count() == 2 ) {
    retval.append( aclParts[ 0 ] );
    retval.append( aclParts[ 1 ] );
  } else if ( aclParts.count() >= 3 ) {
    if ( aclParts[ aclParts.count() - 2 ] == "read" ) { // yyy yyy
      QStringList uidParts;
      for ( int i = 0; i < (aclParts.count() - 2); ++i )
        uidParts.append( aclParts[ i ] );

      retval.append( uidParts.join( " " ) );
      retval.append( QString( "%1 %2" ).arg( aclParts[ aclParts.count() - 2 ], aclParts[ aclParts.count() - 1 ] ) );
    } else {
      QStringList uidParts;
      for ( int i = 0; i < (aclParts.count() - 1); ++i )
        uidParts.append( aclParts[ i ] );

      retval.append( uidParts.join( " " ) );
      retval.append( aclParts[ aclParts.count() - 1 ] );
    }
  }

  return retval;
}

class AclModel : public QAbstractListModel
{
  public:
    AclModel( QObject *parent = 0 )
      : QAbstractListModel( parent )
    {
    }

    virtual int columnCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return 1;
    }

    virtual int rowCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return mList.count();
    }

    virtual QVariant data( const QModelIndex &index, int role = Qt::DisplayRole ) const
    {
      if ( !index.isValid() )
        return QVariant();

      if ( role == Qt::DisplayRole )
        return mList[ index.row() ].displayName;
      else if ( role == Qt::UserRole )
        return mList[ index.row() ].acl;
      else
        return QVariant();
    }

    virtual bool setData( const QModelIndex &index, const QVariant &value, int role = Qt::EditRole )
    {
      if ( !index.isValid() ) {
        return false;
      }

      if ( !value.isValid() )
        return false;

      if ( role == Qt::UserRole ) {
        QStringList acl = value.toStringList();

        mList[ index.row() ].acl = acl;

        QString displayName;
        if ( acl[ 0 ].startsWith( "group:" ) )
          displayName = tr( "Group: %1 (%2)" ).arg( acl[ 0 ].mid( 6 ), acl[ 1 ] );
        else
          displayName = tr( "Person: %1 (%2)" ).arg( acl[ 0 ], acl[ 1 ] );

        mList[ index.row() ].displayName = displayName;
      }

      return true;
    }

    virtual bool insertRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginInsertRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.insert( row, Entry() );

      endInsertRows();

      return true;
    }

    virtual bool removeRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginRemoveRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.removeAt( row );

      endRemoveRows();

      return true;
    }

  private:
    class Entry
    {
      public:
        QString displayName;
        QStringList acl;
    };

    QList<Entry> mList;
};

class AclItemView : public ItemView
{
  public:
    AclItemView( QWidget *parent )
      : ItemView( parent )
    {
    }

  protected:
    virtual void doAdd()
    {
      AclDialog dlg( this );

      if ( dlg.exec() ) {
        QStringList acl = dlg.acl();

        QAbstractItemModel *model = mListView->model();
        int count = model->rowCount();

        bool found = false;
        for ( int i = 0; i < count; ++i ) {
          QStringList modelAcl = model->data( model->index( i, 0 ), Qt::UserRole ).toStringList();
          if ( acl[ 0 ] == modelAcl[ 0 ] && acl[ 1 ] == modelAcl[ 1 ] ) {
            found = true;
            break;
          }
        }

        if ( !found ) {
          model->insertRow( count );
          model->setData( model->index( count, 0 ), acl, Qt::UserRole );
        }
      }
    }
};

AclSelectionField::AclSelectionField( const QString &name, QWidget *parent )
  : Field( name, parent )
{
  mView = new AclItemView( this );

  mModel = new AclModel( this );
  mView->setModel( mModel );

  fieldLayout()->addWidget( mView, 0, 0 );
}

void AclSelectionField::loadEntry( const Entry &entry )
{
  QStringList acls = entry.values( "acl" );
  for ( int i = 0; i < acls.count(); ++i ) {
    QStringList modelAcl = splitAcl( acls[ i ] );

    mModel->insertRow( i );
    mModel->setData( mModel->index( i, 0 ), modelAcl, Qt::UserRole );
  }
}

void AclSelectionField::saveEntry( Entry &entry ) const
{
  entry.clearValue( "acl" );

  for ( int i = 0; i < mModel->rowCount(); ++i ) {
    QStringList acl = mModel->data( mModel->index( i, 0 ), Qt::UserRole ).toStringList();
    entry.addValue( "acl", QString( "%1 %2" ).arg( acl[ 0 ], acl[ 1 ] )  );
  }
}

bool AclSelectionField::isValid( QString&, PagePolicy::State ) const
{
  return true;
}

void AclSelectionField::setEditable( bool editable )
{
  mView->setEnabled( editable );
}


AclDialog::AclDialog( QWidget *parent )
 : QDialog( parent )
{
  QGridLayout *layout = new QGridLayout( this );

  QLabel *label = new QLabel( tr( "User/Group:" ), this );
  layout->addWidget( label, 0, 0 );

  mUidLabel = new QLineEdit( this );
  mUidLabel->setReadOnly( true );
  layout->addWidget( mUidLabel, 0, 1, 1, 2 );

  QToolButton *toolButton = new QToolButton( this );
  toolButton->setText( "..." );
  connect( toolButton, SIGNAL( clicked() ), this, SLOT( selectUid() ) );
  layout->addWidget( toolButton, 0, 3 );

  label = new QLabel( tr( "Rights:" ), this );
  layout->addWidget( label, 1, 0 );

  mRights = new QComboBox( this );
  layout->addWidget( mRights, 1, 1, 1, 3 );

  mRights->addItem( tr( "none" ) );
  mRights->addItem( tr( "post" ) );
  mRights->addItem( tr( "read" ) );
  mRights->addItem( tr( "read/post" ) );
  mRights->addItem( tr( "append" ) );
  mRights->addItem( tr( "write" ) );
  mRights->addItem( tr( "read anon" ) );
  mRights->addItem( tr( "read anon/post" ) );
  mRights->addItem( tr( "read hidden" ) );
  mRights->addItem( tr( "read hidden/post" ) );
  mRights->addItem( tr( "all" ) );

  QPushButton *button = new QPushButton( tr( "&Ok" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
  layout->addWidget( button, 2, 2 );

  button = new QPushButton( tr( "&Cancel" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  layout->addWidget( button, 2, 3 );
}

QStringList AclDialog::acl() const
{
  QStringList rights;
  rights.append( "none" );
  rights.append( "post" );
  rights.append( "read" );
  rights.append( "read/post" );
  rights.append( "append" );
  rights.append( "write" );
  rights.append( "read anon" );
  rights.append( "read anon/post" );
  rights.append( "read hidden" );
  rights.append( "read hidden/post" );
  rights.append( "all" );

  QStringList acl;
  acl.append( mUid );
  acl.append( rights[ mRights->currentIndex() ] );

  return acl;
}

void AclDialog::selectUid()
{
  QStringList blacklist;
  blacklist << QString( "cn=manager,cn=internal,%1" ).arg( Connection::self()->baseDn() );
  blacklist << QString( "cn=calendar,cn=internal,%1" ).arg( Connection::self()->baseDn() );
  blacklist << QString( "cn=nobody,cn=internal,%1" ).arg( Connection::self()->baseDn() );

  LdapSearchDialog dlg( Connection::self(), this );
  dlg.setSearchFilter( "(|(&(objectClass=kolabInetOrgPerson)(|(uid=*%i*)(alias=*%i*)(mail=*%i*)))(&(objectClass=kolabGroupOfNames)(cn=*%i*)))" );
  dlg.setDisplayFormat( "%cn% (%mail%)" );
  dlg.setBlackList( blacklist );

  if ( dlg.exec() ) {
    const QLdapEntry::List entries = dlg.selectedEntries();
    if ( entries.isEmpty() )
      return;

    const QLdapEntry entry = entries.first();
    mUidLabel->setText( entry.value( "cn" ) );

    if ( entry.values( "objectClass" ).contains( "kolabInetOrgPerson" ) ) {
      mUid = entry.value( "mail" );
    } else {
      mUid = QString( "group:%1" ).arg( entry.value( "cn" ) );
    }
  }
}

void AclDialog::accept()
{
  if ( mUid.isEmpty() ) {
    QMessageBox::warning( this, tr( "Invalid Input" ), tr( "You have to select an user or group!" ) );
    return;
  }

  done( Accepted );
}

