/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QComboBox>
#include <QDialog>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QMessageBox>
#include <QPushButton>

#include "itemview.h"

#include "invitationpolicyfield.h"

using namespace Form;

class InvitationPolicyModel : public QAbstractListModel
{
  public:
    InvitationPolicyModel( QObject *parent = 0 )
      : QAbstractListModel( parent )
    {
    }

    virtual int columnCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return 1;
    }

    virtual int rowCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return mList.count();
    }

    virtual QVariant data( const QModelIndex &index, int role = Qt::DisplayRole ) const
    {
      if ( !index.isValid() )
        return QVariant();

      if ( role == Qt::DisplayRole )
        return mList[ index.row() ].displayName;
      else if ( role == Qt::UserRole )
        return mList[ index.row() ].pair;
      else
        return QVariant();
    }

    virtual bool setData( const QModelIndex &index, const QVariant &value, int role = Qt::EditRole )
    {
      if ( !index.isValid() ) {
        return false;
      }

      if ( !value.isValid() )
        return false;

      if ( role == Qt::UserRole ) {

        static QMap<QString, QString> policyMap;
        if ( policyMap.isEmpty() ) {
          policyMap.insert( "ACT_ALWAYS_ACCEPT", tr( "Always accept" ) );
          policyMap.insert( "ACT_ALWAYS_REJECT", tr( "Always reject" ) );
          policyMap.insert( "ACT_REJECT_IF_CONFLICTS", tr( "Reject if conflicts" ) );
          policyMap.insert( "ACT_MANUAL_IF_CONFLICTS", tr( "Manual if conflicts" ) );
          policyMap.insert( "ACT_MANUAL", tr( "Manual" ) );
        }

        QStringList pair = value.toStringList();

        mList[ index.row() ].pair = pair;
        mList[ index.row() ].displayName = QString( "%1 (%2)" ).arg( pair[ 0 ], policyMap[ pair[ 1 ] ] );
      }

      return true;
    }

    virtual bool insertRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginInsertRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.insert( row, Entry() );

      endInsertRows();

      return true;
    }

    virtual bool removeRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginRemoveRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.removeAt( row );

      endRemoveRows();

      return true;
    }

  private:
    class Entry
    {
      public:
        QString displayName;
        QStringList pair;
    };

    QList<Entry> mList;
};

class InvitationPolicyDialog : public QDialog
{
  public:
    InvitationPolicyDialog( QWidget *parent )
      : QDialog( parent )
    {
      QGridLayout *layout = new QGridLayout( this );

      QLabel *label = new QLabel( tr( "Email Address:" ), this );
      layout->addWidget( label, 0, 0 );

      mMail = new QLineEdit( this );
      label->setBuddy( mMail );
      layout->addWidget( mMail, 0, 1, 1, 3 );

      label = new QLabel( tr( "Policy:" ), this );
      layout->addWidget( label, 1, 0 );

      mPolicy = new QComboBox( this );
      mPolicy->addItem( tr( "Always accept" ) );
      mPolicy->addItem( tr( "Always reject" ) );
      mPolicy->addItem( tr( "Reject if conflicts" ) );
      mPolicy->addItem( tr( "Manual if conflicts" ) );
      mPolicy->addItem( tr( "Manual" ) );
      label->setBuddy( mPolicy );
      layout->addWidget( mPolicy, 1, 1, 1, 3 );

      QPushButton *button = new QPushButton( tr( "&Ok" ), this );
      connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
      layout->addWidget( button, 2, 2 );

      button = new QPushButton( tr( "&Cancel" ), this );
      connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
      layout->addWidget( button, 2, 3 );
    }

    virtual void accept()
    {
      if ( mMail->text().isEmpty() ) {
        QMessageBox::warning( this, QString(), tr( "No email address set!" ) );
        return;
      }

      done( Accepted );
    }

    QStringList pair() const
    {
      QStringList pair;
      pair.append( mMail->text() );

      switch ( mPolicy->currentIndex() ) {
        case 0:
          pair.append( "ACT_ALWAYS_ACCEPT" );
          break;
        case 1:
          pair.append( "ACT_ALWAYS_REJECT" );
          break;
        case 2:
          pair.append( "ACT_REJECT_IF_CONFLICTS" );
          break;
        case 3:
          pair.append( "ACT_MANUAL_IF_CONFLICTS" );
          break;
        case 4:
          pair.append( "ACT_MANUAL" );
          break;
      }

      return pair;
    }

  private:
    QLineEdit* mMail;
    QComboBox* mPolicy;
};

class InvitationPolicyItemView : public ItemView
{
  public:
    InvitationPolicyItemView( QWidget *parent )
      : ItemView( parent )
    {
    }

  protected:
    virtual void doAdd()
    {
      InvitationPolicyDialog dlg( this );
      if ( dlg.exec() ) {
        QStringList pair = dlg.pair();

        QAbstractItemModel *model = mListView->model();

        // check whether rule is already part of the model
        bool found = false;
        for ( int k = 0; k < model->rowCount(); ++k ) {
          QStringList modelPair = model->data( model->index( k, 0 ), Qt::UserRole ).toStringList();
          if ( pair[ 0 ] == modelPair[ 0 ] && pair[ 1 ] == modelPair[ 1 ] ) {
            found = true;
            break;
          }
        }

        if ( !found ) {
          int row = model->rowCount();
          model->insertRow( row );
          model->setData( model->index( row, 0 ), pair, Qt::UserRole );
        }
      }
    }
};

InvitationPolicyField::InvitationPolicyField( const QString &name, QWidget *parent )
  : Field( name, parent )
{
  mView = new InvitationPolicyItemView( this );

  mModel = new InvitationPolicyModel( this );
  mView->setModel( mModel );

  fieldLayout()->addWidget( mView, 0, 0 );
}

void InvitationPolicyField::loadEntry( const Entry &entry )
{
  QStringList invitationPolicies = entry.values( "kolabInvitationPolicy" );
  for ( int i = 0; i < invitationPolicies.count(); ++i ) {
    QString policy = invitationPolicies[ i ];

    QStringList pair;
    if ( policy.contains( ':' ) ) {
      pair = policy.split( ":" );
    } else {
      pair.append( "anyone" );
      pair.append( policy );
    }

    mModel->insertRow( i );
    mModel->setData( mModel->index( i, 0 ), pair, Qt::UserRole );
  }
}

void InvitationPolicyField::saveEntry( Entry &entry ) const
{
  entry.clearValue( "kolabInvitationPolicy" );

  for ( int i = 0; i < mModel->rowCount(); ++i ) {
    QStringList pair = mModel->data( mModel->index( i, 0 ), Qt::UserRole ).toStringList();
    if ( pair[ 0 ] == "anyone" )
      entry.addValue( "kolabInvitationPolicy", pair[ 1 ] );
    else
      entry.addValue( "kolabInvitationPolicy", QString( "%1:%2" ).arg( pair[ 0 ], pair[ 1 ] ) );
  }
}

bool InvitationPolicyField::isValid( QString &message, PagePolicy::State ) const
{
  return true;
}

void InvitationPolicyField::setEditable( bool editable )
{
  mView->setEnabled( editable );
}
