/***************************************************************************
 *   Copyright (C) 2005-2010 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QBuffer>
#include <QGroupBox>
#include <QCheckBox>
#include <QClipboard>
#include <QDir>
#include <QDropEvent>
#include <QEventLoop>
#include <QLabel>
#include <QObject>
#include <QRadioButton>
#include <QStyle>
#include <QToolTip>
#include <QMovie>
#include <QPixmap>
#include <QMouseEvent>
#include <QKeyEvent>
#include <QCustomEvent>
#include <QProgressBar>
#include <QPainter>
#include <QWhatsThis>

#include <iostream>

#include <kio/jobuidelegate.h>
#include <KCmdLineArgs>
#include <KLocale>
#include <KAboutData>
#include <kio/copyjob.h>
#include <kio/deletejob.h>
#include <KAction>
#include <KDebug>
#include <KHelpMenu>
#include <KIconLoader>
#include <kio/netaccess.h>
#include <KLed>
#include <KLineEdit>
#include <KMessageBox>
#include <KNumInput>
#include <KPassivePopup>
#include <KPropertiesDialog>
#include <KPushButton>
#include <KRun>
#include <KSystemTrayIcon>
#include <KTabBar>
#include <KTabWidget>
#include <KTemporaryFile>
#include <KUrlRequester>
#include <KActionCollection>
#include <KMenu>

#include "komparator.h"
#include "kdatecombo.h"
#include "kquery.h"
#include "komparatorfilefilter.h"
#include "komparedialog.h"
#include "komparatorcomparejob.h"
#include "komparatoremptydirectoriesjob.h"
#include "komparatorurllist.h"
#include "kfileitemext.h"
#include "klistviewitemsingle.h"
#include "klistviewitemdups.h"
#include "komparatordynamicerrordialog.h"

#include "komparatorwidget.h"

#include <version.h>

KomparatorWidget::KomparatorWidget( const KUrl &url1, const KUrl &url2, QWidget* parent )
{
	setupUi( this );

	m_query = new KQuery( this );
	compare_job = new KomparatorCompareJob( this );

	m_empty_directory_job = new KomparatorEmptyDirectoryJob( this );

	m_error_dialog = new KomparatorDynamicErrorDialog( this );
	connect( this, SIGNAL( signalAddErrorMessage( QString ) ), m_error_dialog, SLOT( slotAddErrorMessage( QString ) ) );

	m_files = NULL;
	m_dirs_dupes = NULL;

	m_popup_menu = NULL;

	m_systray = NULL;

	m_requested_download = NULL;
	m_requested_download_job = NULL;

	current_list_view = NULL;

	m_status = CLEAN;
	m_file_remove_action = IDLE;

	m_properties_renamed = false;

	m_unreadable_files.clear();

	m_url1 = KUrl( "" );
	m_url2 = KUrl( "" );
	m_virtual_subdirectory_current = KUrl( "" );

	m_red = QColor( 200, 30, 30 );   // #e61e1e
	m_green = QColor( 5, 155, 10 );  // #059b0a
	m_blue = QColor( 15, 25, 200 );  // #0f19c8
	m_purple = QColor( 145, 40, 190 ); // #9128be

	// QComboBoxes not populated after KDE 4 port
	size_QComboBox->insertItem( 999, i18n( "(none)" ) );
	size_QComboBox->insertItem( 999, i18n( "At Least" ) );
	size_QComboBox->insertItem( 999, i18n( "At Most" ) );
	size_QComboBox->insertItem( 999, i18n( "Equal To" ) );

	bytes_QComboBox->insertItem( 999, i18n( "Byte(s)" ) );
	bytes_QComboBox->insertItem( 999, i18n( "KB" ) );
	bytes_QComboBox->insertItem( 999, i18n( "MB" ) );

	modifiedLastTime_QComboBox->insertItem( 999, i18n( "Minute(s)" ) );
	modifiedLastTime_QComboBox->insertItem( 999, i18n( "Hour(s)" ) );
	modifiedLastTime_QComboBox->insertItem( 999, i18n( "Day(s)" ) );
	modifiedLastTime_QComboBox->insertItem( 999, i18n( "Month(s)" ) );
	modifiedLastTime_QComboBox->insertItem( 999, i18n( "Year(s)" ) );

	connect( size_QComboBox, SIGNAL( activated( int ) ), this, SLOT( sizeComboActivated( int ) ) );

	KHelpMenu *help_menu = new KHelpMenu( this, KCmdLineArgs::aboutData() );
	connect( help_KPushButton, SIGNAL( clicked() ), help_menu, SLOT( appHelpActivated() ) );
	connect( about_KPushButton, SIGNAL( clicked() ), help_menu, SLOT( aboutApplication() ) );

	connect( m_query, SIGNAL( addFile( const KFileItem*, const QString& ) ),
					this, SLOT( slotAddFile( const KFileItem*, const QString& ) ) );

	connect( m_query, SIGNAL( result( int, QString ) ), this, SLOT( slotResult( int, QString ) ) );

	connect( duplicates1_KListView, SIGNAL( itemSelectionChanged() ), this, SLOT( slotDuplicateSelected() ) );

	connect( this, SIGNAL( signalProgress( QString, int, QString ) ), this, SLOT( slotProgress( QString, int, QString ) ) );

	slotCount( DUPLICATES, 1, -1 );
	slotCount( DUPLICATES, 2, -1 );
	slotCount( MISSING, 1, -1 );
	slotCount( MISSING, 2, -1 );
	slotCount( NEWER, 1, -1 );
	slotCount( NEWER, 2, -1 );
	connect( this, SIGNAL( signalCount( int, int, int ) ), this, SLOT( slotCount( int, int, int ) ) );

	duplicates1_KomparatorFileFilter->setFilterType( DUPLICATES_TYPE );
	duplicates2_KomparatorFileFilter->setFilterType( MISSING_TYPE ); // This is really "MISSING_TYPE", as it's a list view
	missing1_KomparatorFileFilter->setFilterType( MISSING_TYPE ); // with KListViewItemSingle with no special feats.
	missing2_KomparatorFileFilter->setFilterType( MISSING_TYPE );
	newer1_KomparatorFileFilter->setFilterType( NEWER_TYPE );
	newer2_KomparatorFileFilter->setFilterType( NEWER_TYPE );

	connect( duplicates1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*duplicates_normal*/, const bool &/*duplicates_multiple*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*duplicates_normal*/, const bool &/*duplicates_multiple*/ ) ) );
	connect( duplicates2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );
	connect( missing1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );
	connect( missing2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );
	connect( newer1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ) );
	connect( newer2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ) );

	connect( missing1_KListView, SIGNAL( signalDropped( QWidget *, QWidget *, QPoint ) ),
		 this, SLOT( slotKomparatorTreeWidgetDropped( QWidget *, QWidget *, QPoint ) ) );
	connect( missing2_KListView, SIGNAL( signalDropped( QWidget *, QWidget *, QPoint ) ),
		this, SLOT( slotKomparatorTreeWidgetDropped( QWidget *, QWidget *, QPoint ) ) );
	connect( newer1_KListView, SIGNAL( signalDropped( QWidget *, QWidget *, QPoint ) ),
		this, SLOT( slotKomparatorTreeWidgetDropped( QWidget *, QWidget *, QPoint ) ) );
	connect( newer2_KListView, SIGNAL( signalDropped( QWidget *, QWidget *, QPoint ) ),
		this, SLOT( slotKomparatorTreeWidgetDropped( QWidget *, QWidget *, QPoint ) ) );

	duplicatesTrash_KPushButton->setIcon( KIcon( DesktopIcon( "user-trash" ) ) );
	duplicatesTrash_KPushButton->setText( "" );

	duplicatesDelete_KPushButton->setIcon( KIcon( DesktopIcon( "edit-delete" )  ) );
	duplicatesDelete_KPushButton->setText( "" );

	search_KPushButton->setIcon( KIcon( SmallIcon( "komparator4" ) ) );

	about_KPushButton->setIcon( KIcon( SmallIcon( "help-about" ) ) );
	help_KPushButton->setIcon( KIcon( SmallIcon( "help-contents" ) ) );
	whatsThis_KPushButton->setIcon( KIcon( SmallIcon( "help-contextual" ) ) );

	addPreset_KPushButton->setIcon( KIcon( SmallIcon( "list-add" ) ) );
	removePreset_KPushButton->setIcon( KIcon( SmallIcon( "list-remove" ) ) );
	updatePreset_KPushButton->setIcon( KIcon( SmallIcon( "document-save" ) ) );

	QSize small_size = 1.5 * SmallIcon( "edit-copy" ).size(); // save original icon set size, because we'll change them.

	copyMissing1To2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	copyMissing1To2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "edit-copy" ), SmallIcon( "arrow-right-double" ) ) ) );
	copyMissing1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyMissing1To2_KPushButton->setText( "" );
	copyMissing2To1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	copyMissing2To1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "edit-copy" ) ) ) );
	copyMissing2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyMissing2To1_KPushButton->setText( "" );
	moveMissing1To2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	moveMissing1To2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "go-jump" ), SmallIcon( "arrow-right-double" ) ) ) );
	moveMissing1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveMissing1To2_KPushButton->setText( "" );
	moveMissing2To1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	moveMissing2To1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "go-jump" ) ) ) );
	moveMissing2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveMissing2To1_KPushButton->setText( "" );
	removeMissing1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	removeMissing1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-right" ), SmallIcon( "edit-clear" ) ) ) );
	removeMissing1_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeMissing1_KPushButton->setText( "" );
	removeMissing2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	removeMissing2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "edit-clear" ), SmallIcon( "arrow-left" ) ) ) );
	removeMissing2_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeMissing2_KPushButton->setText( "" );

	copyNewer1To2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	copyNewer1To2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "edit-copy" ), SmallIcon( "arrow-right-double" ) ) ) );
	copyNewer1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyNewer1To2_KPushButton->setText( "" );
	copyNewer2To1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	copyNewer2To1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "edit-copy" ) ) ) );
	copyNewer2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyNewer2To1_KPushButton->setText( "" );
	moveNewer1To2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	moveNewer1To2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "go-jump" ), SmallIcon( "arrow-right-double" ) ) ) );
	moveNewer1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveNewer1To2_KPushButton->setText( "" );
	moveNewer2To1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	moveNewer2To1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "go-jump" ) ) ) );
	moveNewer2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveNewer2To1_KPushButton->setText( "" );
	removeNewer1_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	removeNewer1_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "arrow-right" ), SmallIcon( "edit-clear" ) ) ) );
	removeNewer1_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeNewer1_KPushButton->setText( "" );
	removeNewer2_KPushButton->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	removeNewer2_KPushButton->setIcon( KIcon( broadIcon( SmallIcon( "edit-clear" ), SmallIcon( "arrow-left" ) ) ) );
	removeNewer2_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeNewer2_KPushButton->setText( "" );

	QPalette pal = duplicatesReference1_RadioButton->palette();
	pal.setColor( QPalette::ButtonText, m_green );
	duplicatesReference1_RadioButton->setPalette( pal );
	pal.setColor( QPalette::ButtonText, m_blue );
	duplicatesReference2_RadioButton->setPalette( pal );

	duplicates1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	duplicates1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	duplicates1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	duplicates1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	duplicates2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 7, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );

	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );

	bool warning_trash = cfg.readEntry( "komparator_message_trash", true );
	bool warning_delete = cfg.readEntry( "komparator_message_delete", true );
	bool warning_overwrite_copy = cfg.readEntry( "komparator_message_overwrite_copy", true );
	bool warning_overwrite_move = cfg.readEntry( "komparator_message_overwrite_move", true );
	bool warning_reload = cfg.readEntry( "komparator_message_reload", true );
	bool warning_remove_preset = cfg.readEntry( "komparator_message_remove_preset", true );

	warningTrash_CheckBox->setChecked( warning_trash );
	warningDelete_CheckBox->setChecked( warning_delete );
	warningOverwriteCopy_CheckBox->setChecked( warning_overwrite_copy );
	warningOverwriteMove_CheckBox->setChecked( warning_overwrite_move );
	warningReload_CheckBox->setChecked( warning_reload );
	warningRemovePreset_CheckBox->setChecked( warning_remove_preset );

	connect( warningTrash_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningTrash() ) );
	connect( warningDelete_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningDelete() ) );
	connect( warningOverwriteCopy_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningOverwriteCopy() ) );
	connect( warningOverwriteMove_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningOverwriteMove() ) );
	connect( warningReload_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningReload() ) );
	connect( warningRemovePreset_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningRemovePreset() ) );


	m_config = KSharedConfig::openConfig( "komparator4", KSharedConfig::FullConfig, "config" );
	m_configgroup = KConfigGroup( m_config, "Global" );
	bool systray = m_configgroup.readEntry( "systray", true );
	bool notification_ready = m_configgroup.readEntry( "notification_ready", false );
	bool text_color = m_configgroup.readEntry( "text_color", true );

	systray_CheckBox->setChecked( systray );
	notification_CheckBox->setChecked( notification_ready );
	textColor_CheckBox->setChecked( text_color );

	QSize total_size = m_configgroup.readEntry( "total_size", QSize( 800, 600 ) );
	parent->resize( total_size );

	// fill preset_QComboBox.
	KomparatorPreset *new_preset = new KomparatorPreset( this, m_config );
	m_preset_list.append( new_preset );
	preset_QComboBox->insertItem( 99999, new_preset->title() );

	int i;
	for ( i=1; i<50; i++ )
	{
		new_preset = new KomparatorPreset( this, m_config );
		if ( !new_preset->readConfig( i ) )
		{
			delete new_preset;
			break;
		}
		else
		{
			m_preset_list.append( new_preset );
			preset_QComboBox->insertItem( 99999, new_preset->title() );
		}
	}

	preset_QComboBox->setCurrentIndex( m_configgroup.readEntry( "preset", 0 ) );
	slotPresetSelected( m_configgroup.readEntry( "preset", 0 ) ); // select previously used preset; default if not available.

	if ( !url1.isEmpty() ) url1_KUrlRequester->setUrl( url1 ); // override preset settings with command line arguments, if there are any.
	if ( !url2.isEmpty() ) url2_KUrlRequester->setUrl( url2 );

	url1_KUrlRequester->setFocus();

	url1_KUrlRequester->setMode( KFile::File | KFile::Directory );
	url1_KUrlRequester->setFilter( "application/x-tgz application/x-tbz application/x-tar application/x-zip" );

	url2_KUrlRequester->setMode( KFile::File | KFile::Directory );
	url2_KUrlRequester->setFilter( "application/x-tgz application/x-tbz application/x-tar application/x-zip" );

	// Still "Global", read sizes of splitter
	QList<int> list;
	for ( i=0; i<2; i++ )
	{
		list.push_back( m_configgroup.readEntry( QString( "splitter%1" ).arg( i ), 100 ) );
	}
	duplicates_Splitter->setSizes( list );

	for( i=0; i<main_TabWidget->count(); i++ )
	{
		main_TabWidget->setTabIcon( i, KIcon( "opinion-okay" ) );
	}

	m_systray = new KSystemTrayIcon( KIcon( "komparator4" ), parent );

//	BUG in KDE: doesn't work
// 	m_systray_movie = new QMovie(
// 		KIconLoader::global()->loadMovie( QString/*::fromLatin1*/( "komparator4_busy" ), KIconLoader::MainToolbar, 32 ) );

	m_systray_movie = new QMovie( KIconLoader::global()->moviePath( "komparator4_busy", KIconLoader::MainToolbar, 32 ) );

	m_systray->setMovie( m_systray_movie );
	m_systray->setToolTip( KCmdLineArgs::aboutData()->shortDescription() );

	connect( systray_CheckBox, SIGNAL( clicked() ), this, SLOT( slotSystray() ) );
	if ( systray_CheckBox->isChecked() )
	{
		m_systray->show();
	}
	connect( this, SIGNAL( signalStartSystrayMovie() ), this, SLOT( slotStartSystrayMovie() ) );
	connect( this, SIGNAL( signalStopSystrayMovie() ), this, SLOT( slotStopSystrayMovie() ) );

	connect( textColor_CheckBox, SIGNAL( clicked() ), this, SLOT( slotTextColor() ) );
	KStandardAction::quit( (Komparator *)parent, SLOT( slotQuit() ), m_systray->actionCollection() );

	addWhatsThisHelp();

	m_progress_timer = new QTimer( this );
	m_progress_timer->setSingleShot( false );
	connect( m_progress_timer, SIGNAL( timeout() ), this, SLOT( slotProgressUndefined() ) );
	emit( signalProgress( i18n( "Ready." ), -1, "" ) );
}

void KomparatorWidget::addWhatsThisHelp()
{
	const QString regexContainsWhatsThis // taken from kfind
		= i18n( "<qt>Enter the filename you are looking for. <br>Alternatives may be separated by a semicolon \";\".<br><br>"
			"The filename may contain the following special characters:"
			"<ul>"
			"<li><b>?</b> matches any single character</li>"
			"<li><b>*</b> matches zero or more of any characters</li>"
			"<li><b>[...]</b> matches any of the characters in braces</li>"
			"</ul>"
			"<br>"
			"Example searches:"
			"<ul>"
			"<li><b>*.kwd;*.txt</b> finds all files ending with .kwd or .txt</li>"
			"<li><b>go[dt]</b> finds god and got</li>"
			"<li><b>Hel?o</b> finds all files that start with \"Hel\" and end with \"o\", "
			"having one character in between</li>"
			"<li><b>My Document.kwd</b> finds a file of exactly that name</li>"
			"</ul></qt>" );
	regex1_TextLabel->setWhatsThis( regexContainsWhatsThis );
	regexContains_KLineEdit->setWhatsThis( regexContainsWhatsThis );

	const QString regexContainsNotWhatsThis // taken from kfind
		= i18n( "<qt>Enter the filename you want to exclude from your search. <br>Alternatives may be separated by a semicolon \";\".<br><br>"
			"The filename may not contain the following special characters:"
			"<ul>"
			"<li><b>?</b> matches any single character</li>"
			"<li><b>*</b> matches zero or more of any characters</li>"
			"<li><b>[...]</b> matches any of the characters in braces</li>"
			"</ul>"
			"<br>"
			"Example searches:"
			"<ul>"
			"<li><b>*.kwd;*.txt</b> excludes all files ending with .kwd or .txt</li>"
			"<li><b>go[dt]</b> excludes god and got</li>"
			"<li><b>Hel?o</b> excludes all files that start with \"Hel\" and end with \"o\", "
			"having one character in between</li>"
			"<li><b>My Document.kwd</b> excludes all files with exactly that name</li>"
			"</ul></qt>" );
	regex2_TextLabel->setWhatsThis( regexContainsNotWhatsThis );
	regexContainsNot_KLineEdit->setWhatsThis( regexContainsNotWhatsThis );

	const QString findSizeWhatsThis
		= i18n( "<qt>Restrict the search result by file size. Only files matching these settings are found "
			"(and considered in duplicates, missing and newer search).<br>"
			"<ul>"
			"<li><i>(none)</i>: No restriction, all files are found. (default)</li>"
			"<li><i>At Least</i>: Files equal to or bigger than the specified size are found.</li>"
			"<li><i>At Most</i>:Files equal to or smaller than the specified size are found.</li>"
			"<li><i>Equal To</i>: Files equal to the specified size are found.</li>"
			"</ul>"
			"The file size selected in the combo box (bytes, KB, MB) are not rounded off.<br>"
			"Therefore a selected size of <i>42 KB</i> will only match files, "
			"that have exactly the size <i>42*1024 = 43008 bytes</i>."
			"</qt>" );
	size_TextLabel->setWhatsThis( findSizeWhatsThis );
	size_QComboBox->setWhatsThis( findSizeWhatsThis );
	bytes_KIntNumInput->setWhatsThis( findSizeWhatsThis );
	bytes_QComboBox->setWhatsThis( findSizeWhatsThis );

	const QString modificationTimeWhatsThis
		= i18n( "<qt>Restrict the search result by creation or modification time. On remote / not *nix "
			"file systems, this might not be displayed correctly, as copying can change the modification "
			"time on these systems.<br>"
			"<ul>"
			"<li><i>any</i>: No restriction. (default)</li>"
			"<li><i>between date x and date y</i>: Files that were created or modified between "
			"date x and date y are found (including date x and y).</li>"
			"<li><i>during the last z ...</i>: Files that were created or modified during the last "
			"z minutes, hours, days, months or years are found, including the current minute, hour, day, month, year. "
			"This means, if you are searching on 10th of July for files created or modified "
			"during the last 5 days, files from 6th of July are found but not from 5th.</li>"
			"</ul>"
			"</qt>" );
	modified_ButtonGroup->setWhatsThis( modificationTimeWhatsThis );
	modifiedNoRestriction_RadioButton->setWhatsThis( modificationTimeWhatsThis );
	modifiedBetween_RadioButton->setWhatsThis( modificationTimeWhatsThis );
	modifiedFrom_KDateCombo->setWhatsThis( modificationTimeWhatsThis );
	and_TextLabel->setWhatsThis( modificationTimeWhatsThis );
	modifiedTo_KDateCombo->setWhatsThis( modificationTimeWhatsThis );
	modifiedLastTime_RadioButton->setWhatsThis( modificationTimeWhatsThis );
	modifiedLastTime_KIntNumInput->setWhatsThis( modificationTimeWhatsThis );
	modifiedLastTime_QComboBox->setWhatsThis( modificationTimeWhatsThis );

	const QString presetWhatsThis
		= i18n( "<qt>Presets can simplify your life, if you perform a specific "
			"kind of search regularly. Choosing a preset will set the gui to the preset's settings. "
			"If the preset has empty urls, the current urls aren't changed, if it is selected.<br>"
			"You can"
			"<br><br><ul>"
			"<img src=\"%1\">&nbsp;<i>Add a preset</i>: All current settings will be saved as they are. You can choose "
			"a name for the preset (that will be displayed in the combo box), and the urls. "
			"If urls are given, choosing the preset later will set the url fields to these urls. "
			"If the given urls are empty, choosing the preset won't change the currently selected urls."
			"</ul><br><ul>"
			"<img src=\"%2\">&nbsp;<i>Save to the current preset</i>: Overwrite the currently selected preset by the current "
			"settings of the GUI. You cannot change the name but the urls."
			"</ul><br><ul>"
			"<img src=\"%3\">&nbsp;<i>Remove the current preset</i>: Remove the current preset selecting the preset above."
			"</ul><br>"
			"Please note that no changes can be done to the <i>default</i> preset."
			"</qt>" ).arg( KIconLoader::global()->iconPath( "list-add", KIconLoader::Small ) ).arg( KIconLoader::global()->iconPath( "document-save", KIconLoader::Small ) ).arg( KIconLoader::global()->iconPath( "list-remove", KIconLoader::Small ) );

	preset_GroupBox->setWhatsThis( presetWhatsThis );

	const QString ignoreEmptyWhatsThis
		= i18n( "<qt>When comparing files by file size, don't find empty files, "
			"because they are equal by file size, checksum and binary comparison anyway."
			"</qt>" );
	ignoreEmptyFiles_CheckBox->setWhatsThis( ignoreEmptyWhatsThis );

	const QString sizeWhatsThis
		= i18n( "<qt>Disable search for duplicate files by file size. "
			"In this case, duplicate files are only found by sub-path "
			"in their respective directory."
			"</qt>" );
	size_CheckBox->setWhatsThis( sizeWhatsThis );

	const QString checksumWhatsThis
		= i18n( "<qt>Confirm duplicate files matches (found by <i>size comparison</i>) by MD5 checksum. "
			"Remote files must be downloaded for MD5 checksum. Calculating the checksum is slow on large files."
			"</qt>" );
	calculateChecksum_CheckBox->setWhatsThis( checksumWhatsThis );

	const QString binaryWhatsThis
		= i18n( "<qt>Confirm duplicate files matches (found by <i>size comparison</i> and <i>MD5 checksum</i>) "
			"binary comparison. "
			"Remote files must be downloaded for binary comparison. Comparing the files is very slow on large files. "
			"Therefore, you can restrict binary comparison to small files."
			"</qt>" );
	binaryComparison_CheckBox->setWhatsThis( binaryWhatsThis );
	binaryComparison_KIntSpinBox->setWhatsThis( binaryWhatsThis );
	MB_TextLabel->setWhatsThis( binaryWhatsThis );
	filesSmaller_TextLabel->setWhatsThis( binaryWhatsThis );

	const QString duplicatesOnlyWhatsThis
		= i18n( "<qt>Only find duplicates inside the right URL, disabling the left URL and therefore also "
			"the missing and newer files search."
			"</qt>" );
	duplicatesOnly_CheckBox->setWhatsThis( duplicatesOnlyWhatsThis );

	const QString recursiveWhatsThis
		= i18n( "<qt>Search directories including all subdirectories.<br>"
			"Links to directories are only searched, if <i>Follow soft links</i> is enabled."
			"</qt>" );
	includeSubdirectories_CheckBox->setWhatsThis( recursiveWhatsThis );

	const QString includeHiddenItemsWhatsThis
		= i18n( "<qt>Find hidden files and search hidden directories, too."
			"</qt>" );
	includeHiddenItems_CheckBox->setWhatsThis( includeHiddenItemsWhatsThis );

	const QString fileIndexWhatsThis // also from kfind
		= i18n( "<qt>This lets you use the files' index created by the <i>slocate</i> "
			"package to speed-up the search; remember to update the index from time to time "
			"(using <i>updatedb</i>)."
			"</qt>" );
	useFilesIndex_CheckBox->setWhatsThis( fileIndexWhatsThis );

	const QString caseSensitiveWhatsThis
		= i18n( "<qt>This lets you decide whether the regular expression in the <i>Regex</i> "
			"line edit are interpreted strictly or loosely regarding the letters' case.<br>"
			"This also affects the comparison of relative paths and file names, and therefore "
			"the newer / missing files search result."
			"</qt>" );
	caseSensitiveSearch_CheckBox->setWhatsThis( caseSensitiveWhatsThis );

	const QString findEmptyWhatsThis
		= i18n( "<qt>Any directories found during the search are checked whether they are empty, "
			"and if they are, insert them into the <i>Duplicate files</i> list."
			"</qt>" );
	findEmptyDirectories_CheckBox->setWhatsThis( findEmptyWhatsThis );

	const QString softLinksWhatsThis
		= i18n( "<qt>Treat soft-links as the files that they are linked to.<br>"
			"If the soft-link is a directory, search it, too."
			"</qt>" );
	followSoftLinks_CheckBox->setWhatsThis( softLinksWhatsThis );

	const QString referenceWhatsThis
		= i18n( "<qt>Specify reference directory. This changes sorting of "
			"the lower duplicate files list and selects all files from the other directory, "
			"that you can easily delete the files from the other directory."
			"</qt>" );
	duplicatesReference_ButtonGroup->setWhatsThis( referenceWhatsThis );

	const QString duplicates1WhatsThis
		= i18n( "<qt>All files that have duplicates are shown in this list, and, if you "
			"want to, also the first empty directory.<br><br>"
			"<i>To view the duplicates</i> of a file and the file itself (or the empty directories), "
			"you need to <i>click on the respective file</i>, and the list of the files / directories "
			"will appear in the list below.<br><br>"
			"It rather by chance which file of all duplicates is shown here, "
			"it depends on the order in which the files were found.<br><br>"
			"If you want to find duplicates by file size (and MD5 checksum and binary "
			"comparison) these files might have different paths but will have the same size "
			"(or are completely equal) and therefore appear in this list.<br><br>"
			"<font color=black><b>Black</b></font> colored items have only one duplicate (that is "
			"in the other searched url and has the same relative path), <font color=blue><b>blue</b></font> "
			"colored ones have duplicates that are no duplicates by relative path."
			"</qt>" );
	duplicates1_KListView->setWhatsThis( duplicates1WhatsThis );

	const QString duplicates2WhatsThis
		= i18n( "<qt>All duplicates of one file size (and MD5 checksum and binary comparison) "
			"are shown here (or all empty directories), depending on the selected item in the "
			"list above.<br><br>"
			"They are sorted by reference directory (selected on the right). "
			"Files that do not belong to the reference directory come first. "
			"Files from the reference directory are not selected initially, so "
			"you can easily delete the other ones.<br><br>"
			"For easier identification, files within the left url are colored "
			"<font color=#059b0a><b>green</b></font>, the other ones "
			"<font color=#0f19c8><b>blue</b></font>."
			"</qt>" );
	duplicates2_KListView->setWhatsThis( duplicates2WhatsThis );

	const QString missing1WhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) + ".<br><br>" +
			i18n( "All files, that exist in the left url but not in the right one, appear in this list.</qt>" );
	missing1_KListView->setWhatsThis( missing1WhatsThis );

	const QString missing2WhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) + ".<br><br>" +
			i18n( "All files that exist in the right url but not in the left one appear in this list.</qt>" );
	missing2_KListView->setWhatsThis( missing2WhatsThis );

	const QString newerWhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) +
			i18n( ", combined with a comparison of creation / modification time and file size (and MD5 "
				"checksum and binary comparison), if you want to.<br><br>"
				"There are three different kinds of pairs of files:"
				"<ul>"
				"<li><i>Two different files (by file size etc.), one newer than the other</i>: "
				"The newer file is colored <font color=black><b>black</b></font>, "
				"the older one <font color=#0f19c8><b>blue</b></font>.</li>"
				"<li><i>Two equal files (by file size etc.), one newer than the other</i>: "
				"The newer file is colored <font color=black><b>black</b></font>, "
				"the older one <font color=#9128be><b>purple</b></font>.</li>"
				"<li><i>Two different files with the same time stamp</i>: "
				"Both files are colored <font color=#e61e1e><b>red</b></font>, because "
				"Komparator cannot decide, which file is the desired one.</li>"
				"</ul>"
				"</qt>" );
	newer1_KListView->setWhatsThis( newerWhatsThis );
	newer2_KListView->setWhatsThis( newerWhatsThis );

	const QString warningTrashWhatsThis
		= i18n( "<qt>Show a warning before trashing files.</qt>" );
	warningTrash_CheckBox->setWhatsThis( warningTrashWhatsThis );
	const QString warningDeleteWhatsThis
		= i18n( "<qt>Show a warning before deleting files.</qt>" );
	warningDelete_CheckBox->setWhatsThis( warningDeleteWhatsThis );
	const QString warningOverwriteCopyWhatsThis
		= i18n( "<qt>Show a warning before overwriting files while copying files.</qt>" );
	warningOverwriteCopy_CheckBox->setWhatsThis( warningOverwriteCopyWhatsThis );
	const QString warningOverwriteMoveWhatsThis
		= i18n( "<qt>Show a warning before overwriting files while moving files.</qt>" );
	warningOverwriteMove_CheckBox->setWhatsThis( warningOverwriteMoveWhatsThis );
	const QString warningReloadWhatsThis
		= i18n( "<qt>Show a warning before reloading the search result.</qt>" );
	warningReload_CheckBox->setWhatsThis( warningReloadWhatsThis );
	const QString warningRemovePresetWhatsThis
		= i18n( "<qt>Show a warning before removing a preset.</qt>" );
	warningRemovePreset_CheckBox->setWhatsThis( warningRemovePresetWhatsThis );
	const QString systrayIconWhatsThis
		= i18n( "<qt>Show a system tray icon.</qt>" );
	systray_CheckBox->setWhatsThis( systrayIconWhatsThis );
	const QString notificationWhatsThis
		= i18n( "<qt>Show a passive notification when a user interaction is finished and komparator isn't the active window.</qt>" );
	notification_CheckBox->setWhatsThis( notificationWhatsThis );
	const QString textColorWhatsThis
		= i18n( "<qt>Use the komparator text color theme as described in the documentation.</qt>" );
	textColor_CheckBox->setWhatsThis( textColorWhatsThis );

	const QString tabWhatsThis
		= i18n( "<qt>Enable or disable tabs by clicking on the icons."
			"<br><br><ul>"
			"<img source=\"%1\">&nbsp;&nbsp;&nbsp;Display search results."
			"</ul><br><ul>"
			"<img source=\"%2\">&nbsp;&nbsp;&nbsp;Don't display search result."
			"</ul><br>"
			"If you don't need the result of one tab, you can disable it "
			"to have a slight search speed improvement. Disabling both <i>missing</i> "
			"and <i>newer tab</i> (with enabled <i>Size comparison</i> check box) disables "
			"the relative path comparison, which will lead to a significant search speed improvement."
			"</qt>" ).arg( KIconLoader::global()->iconPath( "opinion-okay", KIconLoader::Small ) ).arg( KIconLoader::global()->iconPath( "opinion-no", KIconLoader::Small ) );

	main_TabWidget->setWhatsThis( tabWhatsThis );

	const QString url1WhatsThis
		= i18n( "<qt>Enter an URL to be searched. You must use an absolute path as "
			"<i>/path/to/directory</i>, <b>not</b> a relative one as <i>../directory</i>.<br>"
			"You can use any KIO-slave protocol, like <i>ftp://</i>, <i>fish://</i> "
			"or <i>media:/</i>." ) + "<br><br>" +
			i18n( "This will be the parent URL of the files displayed on the left side."
				"</qt>" );
	url1_TextLabel->setWhatsThis( url1WhatsThis );
	url1_KUrlRequester->setWhatsThis( url1WhatsThis );

	const QString url2WhatsThis
		= i18n( "<qt>Enter an URL to be searched. You must use an absolute path as "
			"<i>/path/to/directory</i>, <b>not</b> a relative one as <i>../directory</i>.<br>"
			"You can use any KIO-slave protocol, like <i>ftp://</i>, <i>fish://</i> "
				"or <i>media:/</i>." ) + "<br><br>" +
			i18n( "This will be the parent URL of the files displayed on the right side."
					"</qt>" );
	url2_TextLabel->setWhatsThis( url2WhatsThis );
	url2_KUrlRequester->setWhatsThis( url2WhatsThis );
	const QString virtualSubdirectoriesWhatsThis
		= i18n( "<qt>Here you can specify \"virtual subdirectories\" to the right side's URL. "
			"Virtual subdirectories essentially behave as if these directories were linked into the "
			"right side's URL (parent directory).<br>"
			"This might be useful if you want to backup several directories into a single one.<br><br>"
			"You should not use virtual subdirectories, that have the same name as a directory, that "
			"really exists in your parent directory, neither is it a good idea to use "
			"real subdirectories of the parent directory as virtual subdirectories."
			"</qt>" );
	subdirectories_KomparatorUrlList->setWhatsThis( virtualSubdirectoriesWhatsThis );

	const QString filterWhatsThis
		= i18n( "<qt>Filter the search results, and hide all entries that don't match "
			"your specifications. <font color=blue><b>Blue</b></font> background color of the "
			"filter button and the counter shows, that some entries are hidden."
			"</qt>" );
	duplicates1_KomparatorFileFilter->setWhatsThis( filterWhatsThis );
	duplicates2_KomparatorFileFilter->setWhatsThis( filterWhatsThis );
	missing1_KomparatorFileFilter->setWhatsThis( filterWhatsThis );
	missing2_KomparatorFileFilter->setWhatsThis( filterWhatsThis );
	newer1_KomparatorFileFilter->setWhatsThis( filterWhatsThis );
	newer2_KomparatorFileFilter->setWhatsThis( filterWhatsThis );

	const QString counterWhatsThis
		= i18n( "<qt>Number of items that are currently shown. <font color=blue><b>Blue</b></font> background color "
		"shows, that some entries are hidden."
		"</qt>" );
	duplicates1ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
	duplicates2ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
	missing1ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
	missing2ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
	newer1ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
	newer2ItemCount_TextLabel->setWhatsThis( counterWhatsThis );
}

KomparatorWidget::~KomparatorWidget()
{
	m_query->kill();

	compare_job->stop();

	m_empty_directory_job->stop();

	m_progress_timer->stop();

	if ( m_requested_download ) // Killing current download job and md5sum / byte array operation. Temporary file will be deleted.
	{
		m_requested_download->setDownloadError( KIO::ERR_USER_CANCELED );
	}
	m_requested_download = NULL;

	if ( m_requested_download_job )
	{
		QString del_str = ((KIO::FileCopyJob*)m_requested_download_job)->destUrl().url();

		m_requested_download_job->kill();

		KIO::NetAccess::del( del_str, this );
		KIO::NetAccess::del( del_str + ".part", this ); // FIXME: ugly.
	}
	m_requested_download_job = NULL;

	compare_job->wait(); // compare_job uses m_files etc., so we need to wait until kompare has really finished.

	m_empty_directory_job->wait();

	m_symlinked_directories.clear();
	m_virtual_subdirectories.clear();

	duplicates1_KListView->clear();
	duplicates2_KListView->clear();

	missing1_KListView->clear();
	missing2_KListView->clear();

	newer1_KListView->clear();
	newer2_KListView->clear();

	// write global config
	m_configgroup.writeEntry( "systray", systray_CheckBox->isChecked() );
	m_configgroup.writeEntry( "notification_ready", notification_CheckBox->isChecked() );
	m_configgroup.writeEntry( "text_color", textColor_CheckBox->isChecked() );

	// write widget size
	m_configgroup.writeEntry( "total_size", size() );

	// write current preset
	m_configgroup.writeEntry( "preset", preset_QComboBox->currentIndex() );

	// write sizes of splitter
	int i = 0;
	QListIterator<int> it( duplicates_Splitter->sizes() );
	while( it.hasNext() )
	{
        m_configgroup.writeEntry( QString( "splitter%1" ).arg( i ), it.next() );
		i++;
	}

	m_configgroup.sync();

	// write presets to config file
	for ( i=1; i<50; i++ )
	{
		KConfigGroup cfg( m_config, QString( "Preset%1" ).arg( i ) );
		if ( !cfg.exists() ) break;
		cfg.deleteGroup();
	}

	KomparatorPreset *current_preset;
	for ( i=1; i<m_preset_list.size(); i++ )
	{
		current_preset = m_preset_list[i];
		current_preset->saveConfig( i );
	}

	delete m_files;
	m_files = NULL;

	KFileItemExt *to_delete = m_dirs_dupes;
	while ( to_delete )
	{
		to_delete = to_delete->duplicates_size;
		delete m_dirs_dupes;
		m_dirs_dupes = to_delete;
	}

	delete m_popup_menu;
	m_popup_menu = NULL;

	delete m_query;
	m_query = NULL;

	delete compare_job;
	compare_job = NULL;

	delete m_empty_directory_job;
	m_empty_directory_job = NULL;
}

void KTabBar::mousePressEvent( QMouseEvent * e ) // This function is responsible to enable / disable icons. Taken from ktabbar.cpp.
{
	QTabBar::mousePressEvent( e );

	QSize iconsize = tabIcon( currentIndex() ).actualSize( iconSize() );

	int xoff = style()->pixelMetric( QStyle::PM_TabBarTabShiftHorizontal, 0, this ) - 9;
	int yoff = style()->pixelMetric( QStyle::PM_TabBarTabShiftVertical, 0, this ) - 10;

	QRect iconrect = QRect( mapToGlobal( QPoint( tabRect( currentIndex() ).x() - xoff, tabRect( currentIndex() ).y() - yoff ) ), iconsize );

	if ( iconrect.contains( mapToGlobal( e->pos() ) ) )
	{
 		if ( tabIcon( currentIndex() ).pixmap( iconSize() ).toImage() == KIcon( "opinion-no" ).pixmap( iconSize() ).toImage() )
 		{
			setTabIcon( currentIndex(), KIcon( "opinion-okay" ) );
 		}
		else
		{
			setTabIcon( currentIndex(), KIcon( "opinion-no" ) );
		}
	}
}

/****************************************
* Drag & Drop Functions                *
****************************************/

void KomparatorWidget::slotKomparatorTreeWidgetDropped( QWidget *source, QWidget *destination, const QPoint &pos )
{
	if ( source == missing2_KListView && destination == missing1_KListView ) current_list_view = missing2_KListView; // set origin of drag.
	else if ( source == missing1_KListView && destination == missing2_KListView ) current_list_view = missing1_KListView;
	else if ( source == newer2_KListView && destination == newer1_KListView ) current_list_view = newer2_KListView;
	else if ( source == newer1_KListView && destination == newer2_KListView ) current_list_view = newer1_KListView;
	else return;

	if ( KApplication::keyboardModifiers() == Qt::ShiftModifier ) slotMoveToOtherSide();
	else if ( KApplication::keyboardModifiers() == Qt::ControlModifier ) slotCopyToOtherSide();
	else
	{
		KMenu *moveOrCopyMenu = new KMenu( this );

		QString move_here = i18n( "Move here" ) + "\t" + tr( "Shift" );
		QString copy_here = i18n( "Copy here" ) + "\t" + tr( "Ctrl" );
		QString cancel = i18n( "Cancel" ) + "\t" + tr( "Esc" );

		moveOrCopyMenu->addAction( SmallIcon( "go-jump" ), move_here, this, SLOT( slotMoveToOtherSide() ) );
		moveOrCopyMenu->addAction( SmallIcon( "edit-copy" ), copy_here, this, SLOT( slotCopyToOtherSide() ) );
		moveOrCopyMenu->addSeparator();
		moveOrCopyMenu->addAction( SmallIcon( "process-stop" ), cancel );

		moveOrCopyMenu->popup( destination->mapToGlobal( pos ) );
	}
}



/****************************************
* Keyboard Functions                   *
****************************************/

void KomparatorWidget::keyPressEvent( QKeyEvent *event )
{
	if ( duplicates1_KListView->hasFocus() ) current_list_view = duplicates1_KListView;
	else if ( duplicates2_KListView->hasFocus() ) current_list_view = duplicates2_KListView;
	else if ( missing1_KListView->hasFocus() ) current_list_view = missing1_KListView;
	else if ( missing2_KListView->hasFocus() ) current_list_view = missing2_KListView;
	else if ( newer1_KListView->hasFocus() ) current_list_view = newer1_KListView;
	else if ( newer2_KListView->hasFocus() ) current_list_view = newer2_KListView;
	else
	{
		event->ignore();
		return;
	}

	switch( event->key() ) // ignoring events passes them to the parent (=KomparatorWidget).
	{
		case Qt::Key_Delete:
			if ( event->modifiers() & Qt::ShiftModifier )
			{
				slotDeleteFiles();
			}
			else
			{
				slotTrashFiles();
			}
			event->accept();
			return;
		case Qt::Key_C:
			if ( event->modifiers() == Qt::ControlModifier )
			{
				event->accept();
				slotCopySelection();
				return;
			}
			break;
		case Qt::Key_Enter:  // Enter and return are equal events for us.
		case Qt::Key_Return:
			event->ignore(); // handled automatically
			return;
			break;
		default:
			break;
	}

	event->ignore();
}



/****************************************
* Context Menu Functions               *
****************************************/

void KomparatorWidget::slotContextMenu( KomparatorTreeWidget *list_view, const QPoint &p )
{
	if ( !list_view ) return;

	QTreeWidgetItem *item = list_view->itemAt( p );
	if ( !item ) return;

	int count = list_view->selectedItems().count();

	current_list_view = list_view;

	if ( count == 0 ) return;

	if ( !m_popup_menu )
	{
		m_popup_menu = new KMenu( this );
	}
	else
	{
		delete m_popup_menu;
		m_popup_menu = new KMenu( this );
	}

	if ( count == 1 ) // The accelerator keys actually don't do anything but show the shortcut in the menu
	{                 // reason: 1) if no popup menu has been created, accelerators wouldn't work
		QString title;  //         2) the accelerator keys would affect the item, that was clicked with popupmenu last time, not the currently selected item.
		title = ( current_list_view == duplicates1_KListView ) ? item->text( 1 ) : item->text( 0 );
		m_popup_menu->addTitle( KIcon( ((KListViewItemSingle *)item)->item->pixmap( 16 ) ), title );
	}
	else
	{
		m_popup_menu->addTitle( i18n( "Selected Files" ) );
	}

	m_popup_menu->addAction( KIcon( SmallIcon( "edit-clear" ) ), i18n( "Remove from list" ), this, SLOT( slotRemoveFromList() ) );
	if ( list_view == missing1_KListView || list_view == newer1_KListView )
	{
		m_popup_menu->addAction( SmallIcon( "edit-copy" ), i18n( "Copy to right side" ), this, SLOT( slotCopyToOtherSide() ) );
		m_popup_menu->addAction( SmallIcon( "go-jump" ), i18n( "Move to right side" ), this, SLOT( slotMoveToOtherSide() ) );
	}
	else if ( list_view == missing2_KListView || list_view == newer2_KListView )
	{
		m_popup_menu->addAction( SmallIcon( "edit-copy" ), i18n( "Copy to left side" ), this, SLOT( slotCopyToOtherSide() ) );
		m_popup_menu->addAction( SmallIcon( "go-jump" ), i18n( "Move to left side" ), this, SLOT( slotMoveToOtherSide() ) );
	}
	else if ( list_view == duplicates1_KListView )
	{
		m_popup_menu->addAction( SmallIcon( "user-trash" ), i18n( "Trash non-reference files" ), this, SLOT( slotTrashNonreferenceFiles() ) );
		m_popup_menu->addAction( SmallIcon( "edit-delete" ), i18n( "Delete non-reference files" ), this, SLOT( slotDeleteNonreferenceFiles() ) );
	}
	m_popup_menu->addSeparator();

	if ( ( current_list_view == missing1_KListView && missing1_KListView->selectedItems().count() == 1 && missing2_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == missing2_KListView && missing2_KListView->selectedItems().count() == 1 && missing1_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == newer1_KListView && newer1_KListView->selectedItems().count() == 1 && newer2_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == newer2_KListView && newer2_KListView->selectedItems().count() == 1 && newer1_KListView->selectedItems().count() == 1 ) )
	{
		m_popup_menu->addAction( SmallIcon( "document-preview" ), i18n( "Compare selected files" ), this, SLOT( slotCompareSelected() ) );
	}

	if ( ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) && current_list_view->selectedItems().count() == 1 )
	{
		m_popup_menu->addAction( SmallIcon( "document-preview" ), i18n( "Compare newer files pair" ), this, SLOT( slotCompareEqual() ) );
	}

	if ( count == 1 )
	{
		m_popup_menu->addAction( SmallIcon( "quickopen" ), i18n( "Open" ), this, SLOT( slotOpenBinding() ) );
		m_popup_menu->addAction( SmallIcon( "inode-directory" ), i18n( "Open Directory" ), this, SLOT( slotOpenDirectory() ) );
		m_popup_menu->addSeparator();
	}

	m_popup_menu->addAction( SmallIcon( "edit-copy" ), i18n( "Copy" ), this, SLOT( slotCopySelection() ) );
	m_popup_menu->addAction( SmallIcon( "user-trash" ), i18n( "Trash" ), this, SLOT( slotTrashFiles() ) );
	m_popup_menu->addAction( SmallIcon( "edit-delete" ), i18n( "Delete" ), this, SLOT( slotDeleteFiles() ) );

	if ( count == 1 )
	{
		m_popup_menu->addSeparator();
		m_popup_menu->addAction( i18n( "Open With..." ), this, SLOT( slotOpenWith() ) );
	}

	m_popup_menu->addSeparator();
	m_popup_menu->addAction( i18n( "Properties" ), this, SLOT( slotFileProperties() ) );

	m_popup_menu->popup( list_view->mapToGlobal( p ) );
}



/****************************************
* Interface Functions                  *
****************************************/

void KomparatorWidget::slotSearch()
{
	switch ( m_status )
	{
		case CLEAN:
			m_interaction_cancel = false;

			search1_KLed->setColor( QColor( 255, 0, 0 ) );  // Red
			search2_KLed->setColor( QColor( 255, 0, 0 ) );
			search3_KLed->setColor( QColor( 255, 0, 0 ) );

			enable( false );

			url1_KUrlRequester->button()->setEnabled( false );
			url2_KUrlRequester->button()->setEnabled( false );
			url1_KUrlRequester->lineEdit()->setReadOnly( true );
			url2_KUrlRequester->lineEdit()->setReadOnly( true );
			duplicateOptions_GroupBox->setEnabled( false );
			findEmptyDirectories_CheckBox->setEnabled( false );
			preset_GroupBox->setEnabled( false );
			subdirectories_KomparatorUrlList->setReadOnly( true );

			if ( !initializeQuery() ) break;

			m_unreadable_files.clear();
			m_number_of_files = 0;

			m_interaction_empty_dirs.clear();
			m_interaction_empty_dirs_file_urls.clear();

			m_status = SEARCHING_URL_1;
			search_KPushButton->setText( i18n( "Cancel" ) );
			search_KPushButton->setIcon( KIcon( SmallIcon( "process-stop" ) ) );

			emit( signalProgress( i18n( "Searching left url for files..." ), -1, "" ) );
			m_progress_timer->start( 100 );
			emit( signalStartSystrayMovie() );

			m_symlinked_directories.clear();

			if ( !duplicatesOnly_CheckBox->isChecked() ) m_query->start();
			else slotResult( 0, "" );
			return;
			break;

		case SEARCHING_URL_1:
		case SEARCHING_URL_2:
			m_interaction_cancel = true;
			m_symlinked_directories.clear();
			m_virtual_subdirectories.clear();
			m_query->kill();
			break;

		case SEARCHING_EMPTY_DIRECTORIES:
			m_empty_directory_job->stop();
			m_empty_directory_job->wait();

			m_interaction_cancel = true;
			break;

		case COMPARING:
			compare_job->stop();

			if ( m_requested_download ) // Killing current download job and md5sum /
			{                         // byte array operation. Temporary file will be deleted.
				m_requested_download->setDownloadError( KIO::ERR_USER_CANCELED );
			}
			m_requested_download = NULL;

			if ( m_requested_download_job )
			{
				QString del_str = ((KIO::FileCopyJob*)m_requested_download_job)->destUrl().url();

				m_requested_download_job->kill();
			}
			m_requested_download_job = NULL;

			compare_job->wait(); // kompare uses files_dupes etc., so we need to wait until kompare has really finished.

			break;

		case READY:
			break;

		case INTERACTING:
			m_interaction_cancel = true;
			return;

		default:
			break;
	}

	duplicates1_KListView->clear();
	duplicates2_KListView->clear();

	missing1_KListView->clear();
	missing2_KListView->clear();

	newer1_KListView->clear();
	newer2_KListView->clear();

	delete m_files;
	m_files = NULL;

	KFileItemExt *to_delete = m_dirs_dupes;
	while ( to_delete )
	{
		to_delete = to_delete->duplicates_size;
		delete m_dirs_dupes;
		m_dirs_dupes = to_delete;
	}

	m_symlinked_directories.clear();
	m_virtual_subdirectories.clear();

	search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
	search2_KLed->setColor( QColor( 0, 255, 0 ) );
	search3_KLed->setColor( QColor( 0, 255, 0 ) );

	enable( true );

	url1_KUrlRequester->button()->setEnabled( true );
	url2_KUrlRequester->button()->setEnabled( true );
	url1_KUrlRequester->lineEdit()->setReadOnly( false );
	url2_KUrlRequester->lineEdit()->setReadOnly( false );
	duplicateOptions_GroupBox->setEnabled( true );
	findEmptyDirectories_CheckBox->setEnabled( true );
	preset_GroupBox->setEnabled( true );
	subdirectories_KomparatorUrlList->setReadOnly( false );

	m_unreadable_files.clear();
	m_number_of_files = 0;

	emit( signalCount( DUPLICATES, 1, 0 ) );
	emit( signalCount( DUPLICATES, 2, 0 ) );
	emit( signalCount( MISSING, 1, 0 ) );
	emit( signalCount( MISSING, 2, 0 ) );
	emit( signalCount( NEWER, 1, 0 ) );
	emit( signalCount( NEWER, 2, 0 ) );

	m_status = CLEAN;
	search_KPushButton->setText( i18n( "Search" ) );
	search_KPushButton->setIcon( KIcon( SmallIcon( "komparator4" ) ) );

	resizeListViewColumns();

	m_progress_timer->stop();
	// process signalProgress events, so the next event will display the current state.
	KApplication::kApplication()->processEvents( QEventLoop::ExcludeSocketNotifiers, 50 /* Process for maximum of x milliseconds, then continue */ );

	m_url1 = KUrl( "" );
	m_url2 = KUrl( "" );

	emit( signalProgress( i18n( "Ready." ), -1, "" ) );
	emit( signalStopSystrayMovie() );
}

void KomparatorWidget::slotWhatsThis()
{
	QWhatsThis::enterWhatsThisMode();
}

void KomparatorWidget::slotDuplicateSelected()
{
	duplicates2_KListView->clear();

	if ( !duplicates1_KListView->currentItem() ) return;

	KFileItemExt *tmpfile;

	KListViewItemSingle *tmpitem = new KListViewItemSingle( duplicates2_KListView,
		(((KListViewItemDups*)duplicates1_KListView->currentItem())->item), NULL );
	emit( signalCount( DUPLICATES, 2, -1 ) );
	tmpitem->setText( 1, prettyUrl( tmpitem->item->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ) );

	if ( (((KListViewItemDups*)duplicates1_KListView->currentItem())->item)->dir == 0 )
	{
		tmpitem->setText( 7, i18n( "Left" ) );
		tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_green );

	}
	else
	{
		tmpitem->setText( 7, i18n( "Right" ) );
		tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_blue );
	}

	tmpfile = ((KListViewItemDups*)duplicates1_KListView->currentItem())->item->duplicates_size;

	while ( tmpfile )
	{
		tmpitem = new KListViewItemSingle( duplicates2_KListView, tmpfile, NULL );
		emit( signalCount( DUPLICATES, 2, -1 ) );
		tmpitem->setText( 1, prettyUrl( tmpitem->item->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ) );

		if ( tmpfile->dir == 0 )
		{
			tmpitem->setText( 7, i18n( "Left" ) );
			tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_green );
		}
		else
		{
			tmpitem->setText( 7, i18n( "Right" ) );
			tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_blue );
		}

		tmpfile = tmpfile->duplicates_size;
	}

	slotReferenceLeft( duplicatesReference1_RadioButton->isChecked() );
	updateFilters();
}

void KomparatorWidget::slotReferenceLeft( bool is_ref )
{
	duplicates2_KListView->sortByColumn( 7, is_ref ? Qt::DescendingOrder : Qt::AscendingOrder );

	// evil workaround to make grey lines disappear which come from CTRL-unselect style.
	duplicates2_KListView->setSelectionMode( QAbstractItemView::SingleSelection );
	duplicates2_KListView->clearSelection();
	QTreeWidgetItemIterator it_tmp( duplicates2_KListView );
	duplicates2_KListView->setCurrentItem( (*it_tmp) ); // select, so all other items are unselected.
	duplicates2_KListView->setSelectionMode( QAbstractItemView::ExtendedSelection );

	QTreeWidgetItemIterator it( duplicates2_KListView );
	while ( (*it) )
	{
		(*it)->setSelected( ( ( !is_ref && ((KListViewItemSingle*)*it)->item->dir == 0 ) ||
			( is_ref && ((KListViewItemSingle*)*it)->item->dir == 1 ) ) ? true : false );
		it++;
	}

//	Don't do that any more; this is unexpected behavior.
// 	duplicates2_KListView->setFocus();
}

void KomparatorWidget::slotRemoveFromList()
{
	if ( !current_list_view ) return;

	emit( signalProgress( ( current_list_view->selectedItems().count() == 1 ) ?
		i18n( "Removing 1 file from list..." ) :
		i18n( "Removing %1 files from list..." ).arg( current_list_view->selectedItems().count() ), -1, "" ) );
	emit( signalStartSystrayMovie() );

	if ( current_list_view == duplicates1_KListView ||
		current_list_view == missing1_KListView ||
		current_list_view == missing2_KListView ||
		current_list_view == newer1_KListView ||
		current_list_view == newer2_KListView )
	{
		QListIterator<QTreeWidgetItem*> myit( current_list_view->selectedItems() );
		while ( myit.hasNext() ) delete myit.next();
	}
	else
	{
		QListIterator<QTreeWidgetItem*> myit( duplicates2_KListView->selectedItems() );
		while ( myit.hasNext() )
		{
			removeDeletedFromLists( ((KListViewItemSingle*)myit.next())->item, true, false );
		}
	}

	int visible_items = 0;
	QTreeWidgetItemIterator it( current_list_view );
	while( (*it) )
	{
		if ( !(*it)->isHidden() ) visible_items++;

		++it;
	}

	if ( current_list_view == duplicates1_KListView ) emit( signalCount( DUPLICATES, 1, visible_items ) );
	else if ( current_list_view == duplicates2_KListView ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( current_list_view == missing1_KListView ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( current_list_view == missing2_KListView ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( current_list_view == newer1_KListView ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( current_list_view == newer2_KListView ) emit( signalCount( NEWER, 2, visible_items ) );

	current_list_view = NULL;

	emit( signalProgress( "", -1, "" ) );
	emit( signalStopSystrayMovie() );
}

void KomparatorWidget::slotCopyToOtherSide()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIcon( KIcon( SmallIcon( "process-stop" ) ) );

	m_interaction_copy_remaining_items = current_list_view->selectedItems();
	m_interaction_total = m_interaction_copy_remaining_items.count();
	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_interaction_total == 1 ) ?
		i18n( "Copying 1 file from one side to the other..." ) :
		i18n( "Copying %1 files from one side to the other..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
		m_url2.url() : m_url1.url();

	bool overwrite = false; // doesn't make sense to overwrite if missing (then the target file shouldn't exist), but should overwrite if newer.

	KListViewItemSingle *i;
	if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		QStringList copy;

		QListIterator<QTreeWidgetItem*> myit( m_interaction_copy_remaining_items );
		while ( myit.hasNext() )
		{
			i = (KListViewItemSingle *)myit.next();
			copy.push_back( prettyUrl( i->item->url(), URL_DIR_OR_URL, KUrl( "" ) ) + " " + i18n( "to" )
				+ " " + prettyUrl( i->brother->item->url(), URL_DIR_OR_URL, KUrl( "" ) ) );
		}

		QString msg;
		msg = ( m_interaction_total == 1 ) ?
			i18n( "Do you really want to copy the selected file overwriting the existing one?" ) :
			i18n( "Do you really want to copy the %1 selected files overwriting the existing ones?" ).arg( m_interaction_total );

		KGlobal::config()->reparseConfiguration();
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );

		if ( KMessageBox::warningContinueCancelList( (this)->parentWidget(), msg, copy, i18n( "Copy files" ),
					KGuiItem( i18n( "Overwrite" ), "editoverwrite"), KStandardGuiItem::cancel(), "komparator_message_overwrite_copy" ) == KMessageBox::Cancel )
		{
			QApplication::restoreOverrideCursor();
			m_interaction_copy_remaining_items.clear();
			slotMoveNextStat( NULL );
			return;
		}
		QApplication::restoreOverrideCursor();

		KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
		KConfigGroup cfg( config_global, "Notification Messages" );
		cfg.sync();
		bool warning_overwrite_copy = cfg.readEntry( "komparator_message_overwrite_copy", true );
		warningOverwriteCopy_CheckBox->setChecked( warning_overwrite_copy );

		overwrite = true;
	}

	bool success;
	KUrl dest;
	int last_items_dir = ((KListViewItemSingle *)m_interaction_copy_remaining_items.last())->item->dir;
	QListIterator<QTreeWidgetItem*> myit( m_interaction_copy_remaining_items );
	while ( myit.hasNext() )
	{
		i = (KListViewItemSingle *)myit.next();
		m_virtual_subdirectory_current = KUrl( "" );
		if ( last_items_dir == 0 )
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );

			KUrl::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KUrl::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upUrl().prettyUrl() + "/" + prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upUrl();
					dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
		}
		dest.cleanPath();

		KUrl dir = dest.upUrl();

		while ( !KIO::NetAccess::mkdir( dir, this ) ) // create all destination subdirs if not present
		{
			KUrl mkdir = dir.upUrl();
			success = true;

			while ( !KIO::NetAccess::mkdir( mkdir, this ) )
			{
				if ( mkdir == mkdir.upUrl() )
				{
					success = false;
					break;
				}

				mkdir = mkdir.upUrl();
			}

			if ( !success ) break;
		}
	}

	KUrl src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
	src.cleanPath();

	KIO::FileCopyJob *job = KIO::file_copy( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
		KIO::HideProgressInfo | KIO::Overwrite );
	connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotCopyNext( KJob * ) ) );
}

void KomparatorWidget::slotCopyNext( KJob *finished_job )
{
	if ( finished_job->error() == 0 )
	{
		KIO::StatJob *job = KIO::stat( ((KIO::FileCopyJob *)finished_job)->destUrl(), KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotCopyNextStat( KJob * ) ) );
	}
	else
	{
		emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 		((KIO::Job *)finished_job)->ui()->setWindow( this );
// 		((KIO::Job *)finished_job)->ui()->showErrorMessage();
		slotCopyNextStat( NULL ); // failed to copy item; call stat with NULL means: remove from list but don't do postprocessing
	}
}

void KomparatorWidget::slotCopyNextStat( KJob *finished_job )
{
	if ( finished_job )
	{
		KIO::UDSEntry entry;
		if ( finished_job->error() == 0 )
		{
			// create a new item that holds the file that was just copied.
			entry = ((KIO::StatJob *)finished_job)->statResult();
		}
		else
		{
			emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 			((KIO::Job *)finished_job)->ui()->setWindow( this );
// 			((KIO::Job *)finished_job)->ui()->showErrorMessage();
		}
		KFileItemExt *newitem = new KFileItemExt( entry, ((KIO::StatJob *)finished_job)->url() );
		newitem->dir = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? 1 : 0;
		newitem->virtual_parent_path = newitem->dir ? m_url2 : m_url1;
		newitem->parent_path = m_virtual_subdirectory_current.isEmpty() ? newitem->virtual_parent_path : m_virtual_subdirectory_current.upUrl();

		newitem->next = m_files; // Add it to m_files to delete if necessary
		m_files = newitem;

		newitem->isdupe_size = 1;
		newitem->isdupe_path = 1;

		if ( !((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->isdupe_size &&
				!((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->hasdupes_size &&
				( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->size() > 0 ||
					!ignoreEmptyFiles_CheckBox->isChecked() ) ) // no duplicate, copying created a new duplicate
		{
			newitem->duplicates_size = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size;
			if ( newitem->duplicates_size ) newitem->hasdupes_size = 1;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size = newitem;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->hasdupes_size = 1;
			newitem->dup_size_parent = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item;
			if ( isEnabled( DUPLICATES ) )
			{
				(new KListViewItemDups( duplicates1_KListView, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item ))->setUseColor( textColor_CheckBox->isChecked() );
				emit( signalCount( DUPLICATES, 1, -1 ) );
			}
			// We don't need to check whether this is a "real duplicate" as copying created a duplicate that has the same sub-path.
		}
		else // is already a duplicate, append copied file. must delete overwritten file from duplicate chain.
		{
			KFileItemExt *curfile = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size = newitem;
			newitem->dup_size_parent = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size->duplicates_size = curfile;
			if ( curfile ) curfile->dup_size_parent = newitem;
		}

		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother ) // This means, the brother file was overwritten.
		{
			if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent ) // Not the first in duplicate chain.
			{
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->isdupe_size = 0;
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->hasdupes_size = 0;
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent->duplicates_size =
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size;
				if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size )
				{
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size->dup_size_parent =
						((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent;
				}
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size = NULL;
			}
			else                                                                                    // First in duplicate chain.
			{
				if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size &&
						( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->size() > 0 ||
							!ignoreEmptyFiles_CheckBox->isChecked() ) )
				{
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size->dup_size_parent = NULL;
					if ( isEnabled( DUPLICATES ) )
					{
						(new KListViewItemDups( duplicates1_KListView, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
						emit( signalCount( DUPLICATES, 1, -1 ) );
					}
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size = NULL;
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->hasdupes_size = 0;
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->isdupe_size = 0;
				}
			}

			QTreeWidgetItemIterator it2( duplicates1_KListView );
			while( (*it2) )
			{
				if ( !((KListViewItemDups*)(*it2))->item->duplicates_size )
				{
					delete ((KListViewItemDups*)(*it2));
				}

				((KListViewItemDups*)(*it2))->updateColor();

				it2++;
			}
		}

		if ( findEmptyDirectories_CheckBox->isChecked() ) // search previously empty directories, whether they are still empty.
		{
			// The most simple approach: check, whether any empty directory is a parent directory of our copied file. Delete this directory.
			// There can only be ONE empty directory that's a parent of our file.
			KFileItemExt *tmp_dir = m_dirs_dupes;
			while ( tmp_dir )
			{
				if ( tmp_dir->url().isParentOf( newitem->url() ) )
				{
					if ( tmp_dir->dup_size_parent ) // not first in duplicate chain, simply skip it.
					{
						tmp_dir->dup_size_parent->duplicates_size = tmp_dir->duplicates_size;
						tmp_dir->hasdupes_size = 0;

						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->dup_size_parent->hasdupes_size = 1;
							tmp_dir->duplicates_size->dup_size_parent = tmp_dir->dup_size_parent;
						}

						delete tmp_dir;
					}
					else // first in duplicate chain
					{
						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->duplicates_size->dup_size_parent = NULL;
							if ( isEnabled( DUPLICATES ) )
							{
								(new KListViewItemDups( duplicates1_KListView, tmp_dir->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
								emit( signalCount( DUPLICATES, 1, -1 ) );
							}
						}

						QTreeWidgetItemIterator it( duplicates1_KListView );
						while( (*it) )
						{
							if ( ((KListViewItemDups*)(*it))->item->url() == tmp_dir->url() )
							{
								delete ((KListViewItemDups*)(*it));
								break;
							}

							it++;
						}
						m_dirs_dupes = tmp_dir->duplicates_size;

						delete tmp_dir;
					}

					break;
				}

				tmp_dir = tmp_dir->duplicates_size;
			}
		}

		delete ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother; // remove the brother from the other list as it was overwritten.

		slotDuplicateSelected(); // We eventually changed something here. Reselect to make sure, new duplicates appear on duplicates2 immediately.

		delete m_interaction_copy_remaining_items.takeLast();
	}
	else
	{
		m_interaction_copy_remaining_items.removeLast(); // don't delete as item was not copied
	}

	if ( m_interaction_copy_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		bool overwrite = ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) ? true : false;

		KUrl src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
		KUrl dest;

		m_virtual_subdirectory_current = KUrl( "" );
		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 )
		{
			dest = m_url2;
			dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );

			KUrl::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KUrl::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upUrl().prettyUrl() + "/" + prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upUrl();
					dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) + "/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = m_url1;
			dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
		}

		src.cleanPath();
		dest.cleanPath();

		KIO::FileCopyJob *job = KIO::file_copy( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
			overwrite ? ( KIO::HideProgressInfo | KIO::Overwrite ) : KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotCopyNext( KJob * ) ) );
	}
	else
	{
		m_interaction_copy_remaining_items.clear();
		updateFilters();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Copying files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );
		resizeListViewColumns();

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
	}
}

void KomparatorWidget::slotMoveToOtherSide()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIcon( KIcon( SmallIcon( "process-stop" ) ) );

	m_interaction_copy_remaining_items = current_list_view->selectedItems();
	m_interaction_total = m_interaction_copy_remaining_items.count();
	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_interaction_total == 1 ) ?
		i18n( "Moving 1 file from one side to the other..." ) :
		i18n( "Moving %1 files from one side to the other..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
		m_url2.url() : m_url1.url();

	bool overwrite = false; // doesn't make sense to overwrite if missing (then the target file shouldn't exist), but should overwrite if newer.
	KListViewItemSingle *i;
	if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		QStringList move;

		QListIterator<QTreeWidgetItem*> myit( m_interaction_copy_remaining_items );
		while ( myit.hasNext() )
		{
			i = (KListViewItemSingle *)myit.next();
			move.push_back( prettyUrl( i->item->url(), URL_DIR_OR_URL, KUrl( "" ) ) + " " + i18n( "to" ) + " " + prettyUrl( i->brother->item->url(), URL_DIR_OR_URL, KUrl( "" ) ) );
		}

		QString msg;
		msg = ( m_interaction_total == 1 ) ?
			i18n( "Do you really want to move the selected file overwriting the existing one?" ) :
			i18n( "Do you really want to move the %1 selected files overwriting the existing ones?" ).arg( m_interaction_total );

		KGlobal::config()->reparseConfiguration();
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );

		if ( KMessageBox::warningContinueCancelList( parentWidget(), msg, move, i18n( "Move files" ),
			KGuiItem( i18n( "Overwrite" ), "editoverwrite"), KStandardGuiItem::cancel(), "komparator_message_overwrite_move" ) == KMessageBox::Cancel )
		{
			QApplication::restoreOverrideCursor();
			m_interaction_copy_remaining_items.clear();
			slotMoveNextStat( NULL );
			return;
		}
		QApplication::restoreOverrideCursor();

		KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
		KConfigGroup cfg( config_global, "Notification Messages" );
		cfg.sync();
		bool warning_overwrite_move = cfg.readEntry( "komparator_message_overwrite_move", true );
		warningOverwriteMove_CheckBox->setChecked( warning_overwrite_move );

		overwrite = true;
	}

	bool success;
	KUrl dest;
	int last_items_dir = ((KListViewItemSingle *)m_interaction_copy_remaining_items.last())->item->dir;

	QListIterator<QTreeWidgetItem*> myit( m_interaction_copy_remaining_items );
	while ( myit.hasNext() )
	{
		i = (KListViewItemSingle *)myit.next();
		m_virtual_subdirectory_current = KUrl( "" );
		if ( last_items_dir == 0 )
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );

			KUrl::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KUrl::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upUrl().prettyUrl() + "/" + prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upUrl();
					dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
		}
		dest.cleanPath();

		KUrl dir = dest.upUrl();

		while ( !KIO::NetAccess::mkdir( dir, this ) ) // create all destination subdirs if not present
		{
			KUrl mkdir = dir.upUrl();
			success = true;

			while ( !KIO::NetAccess::mkdir( mkdir, this ) )
			{
				if ( mkdir == mkdir.upUrl() )
				{
					success = false;
					break;
				}

				mkdir = mkdir.upUrl();
			}

			if ( !success ) break;
		}
	}

	KUrl src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
	src.cleanPath();

	KIO::FileCopyJob *job = KIO::file_move( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
		overwrite ? ( KIO::HideProgressInfo | KIO::Overwrite ) : KIO::HideProgressInfo );
	connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotMoveNext( KJob * ) ) );
}

void KomparatorWidget::slotMoveNext( KJob *finished_job )
{
	if ( finished_job->error() == 0 )
	{
		KIO::StatJob *job = KIO::stat( ((KIO::FileCopyJob *)finished_job)->destUrl(), KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotMoveNextStat( KJob * ) ) );
	}
	else
	{
		emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 		((KIO::Job *)finished_job)->ui()->setWindow( this );
// 		((KIO::Job *)finished_job)->ui()->showErrorMessage();
		slotMoveNextStat( NULL ); // failed to copy item; call stat with NULL means: remove from list but don't do postprocessing
	}
}

void KomparatorWidget::slotMoveNextStat( KJob *finished_job )
{
	if ( finished_job )
	{
		KIO::UDSEntry entry;
		if ( finished_job->error() == 0 )
		{
			// create a new item that holds the file that was just copied.
			entry = ((KIO::StatJob *)finished_job)->statResult();
		}
		else
		{
			emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 			((KIO::Job *)finished_job)->ui()->setWindow( this );
// 			((KIO::Job *)finished_job)->ui()->showErrorMessage();
		}

		if ( findEmptyDirectories_CheckBox->isChecked() ) // search previously empty directories, whether they are still empty.
		{
			// The most simple approach: check, whether any empty directory is a parent directory of our copied file. Delete this directory.
			// There can only be ONE empty directory that's a parent of our file.
			KFileItemExt *tmp_dir = m_dirs_dupes;
			while ( tmp_dir )
			{
				if ( tmp_dir->url().isParentOf( ((KIO::StatJob *)finished_job)->url() ) )
				{
					if ( tmp_dir->dup_size_parent ) // not first in duplicate chain, simply skip it.
					{
						tmp_dir->dup_size_parent->duplicates_size = tmp_dir->duplicates_size;
						tmp_dir->hasdupes_size = 0;

						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->dup_size_parent->hasdupes_size = 1;
							tmp_dir->duplicates_size->dup_size_parent = tmp_dir->dup_size_parent;
						}

						delete tmp_dir;
					}
					else // first in duplicate chain
					{
						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->duplicates_size->dup_size_parent = NULL;
							if ( isEnabled( DUPLICATES ) )
							{
								(new KListViewItemDups( duplicates1_KListView, tmp_dir->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
								emit( signalCount( DUPLICATES, 1, -1 ) );
							}
						}

						QTreeWidgetItemIterator it( duplicates1_KListView );
						while( (*it) )
						{
							if ( ((KListViewItemDups*)(*it))->item->url() == tmp_dir->url() )
							{
								delete ((KListViewItemDups*)(*it));
								break;
							}
							it++;
						}

						m_dirs_dupes = tmp_dir->duplicates_size;

						delete tmp_dir;
					}
				}

				tmp_dir = tmp_dir->duplicates_size;
			}

			m_interaction_empty_dirs.push_back( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url() );
		}

		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother ) // Removes the overwritten from all lists, especially duplicates.
		{
			removeDeletedFromLists( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item, true /*duplicates_only*/, false /* add brother to missing tab */ );
			delete ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother = NULL;
		}

		// Update the moved file to have the new path (and new entry, and new dir).
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->setUDSEntry( entry, ((KIO::StatJob *)finished_job)->url() );

		// The dir entry is not yet ok, it represents the old directory (before move).
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir =
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir ? 0 : 1;
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->virtual_parent_path =
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir ? m_url2 : m_url1;
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path =
			m_virtual_subdirectory_current.isEmpty() ? ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->virtual_parent_path : m_virtual_subdirectory_current.upUrl();

		if ( isEnabled( MISSING ) ) // no file was overwritten, the only change must be done to missing list views.
		{
			KListViewItemSingle *new_item_single = new KListViewItemSingle(
				( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 ) ?
				missing1_KListView : missing2_KListView,
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item, NULL );
			new_item_single->setText( 1, prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(),
				URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) );
			emit( signalCount( MISSING, ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 ) ? 1 : 2, -1 ) );
		}

		QTreeWidgetItemIterator it2( duplicates1_KListView ); // replace path of duplicate1 items by the new path.
		while( (*it2) )
		{
			KUrl tmp_url = ((KListViewItemDups*)(*it2))->item->url(); // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			((KListViewItemDups*)(*it2))->setText( 0, tmp_url.pathOrUrl() + ((KListViewItemDups*)(*it2))->item->url().upUrl().path( KUrl::RemoveTrailingSlash ) );

			it2++;
		}

		slotDuplicateSelected(); // We eventually changed something here. Reselect to make sure, new duplicates appear on duplicates2 immediately.

		delete m_interaction_copy_remaining_items.takeLast();
	}
	else
	{
		m_interaction_copy_remaining_items.removeLast(); // don't delete as item was not copied
	}

	if ( m_interaction_copy_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
			m_url2.url() : m_url1.url();

		bool overwrite = ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) ? true : false;

		KUrl src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
		KUrl dest;

		m_virtual_subdirectory_current = KUrl( "" );
		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 )
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );

			KUrl::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KUrl::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upUrl().prettyUrl() + "/" + prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upUrl();
					dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) + "/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KUrl( destination );
			dest.addPath( prettyUrl( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
		}

		src.cleanPath();
		dest.cleanPath();

		KIO::FileCopyJob *job = KIO::file_move( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
			overwrite ? ( KIO::HideProgressInfo | KIO::Overwrite ) : KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotMoveNext( KJob * ) ) );
	}
	else
	{
		KUrl::List empty_list;
		checkEmptyDirs( m_interaction_empty_dirs, empty_list );

		m_interaction_empty_dirs.clear();
		m_interaction_empty_dirs_file_urls.clear();
		m_interaction_copy_remaining_items.clear();
		updateFilters();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Moving files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );
		resizeListViewColumns();

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
	}
}

void KomparatorWidget::slotOpenBinding()
{
	if ( current_list_view )
	{
		((KListViewItemSingle*) current_list_view->currentItem())->item->run();
	}

	current_list_view = NULL;
}

void KomparatorWidget::slotExecute( QTreeWidgetItem *item, int )
{
	((KListViewItemSingle*) item)->item->run();
}

void KomparatorWidget::slotOpenDirectory()
{
	if ( !current_list_view ) return;

	if ( ((KListViewItemSingle*) current_list_view->currentItem())->item->isDir() )
	{
		new KRun( ((KListViewItemSingle*) current_list_view->currentItem())->item->url(), this );
		return;
	}

	KUrl url = ((KListViewItemSingle*) current_list_view->currentItem())->item->url();
	url.setFileName( QString::null );

	new KRun( url, this );

	current_list_view = NULL;
}

void KomparatorWidget::slotCopySelection()
{
	if ( !current_list_view ) return;

	// Here we do the same as in QDragObject *KomparatorTreeWidget::dragObject().
	// this function is protected and therefore not usable.
	QList< QUrl > urls;

	QPixmap *pm = NULL;

	QTreeWidgetItemIterator it( current_list_view );
	while ( (*it) )
	{
		if ( (*it)->isSelected() )
		{
			urls.append( ((KListViewItemSingle*)*it)->item->url().url() );
			if ( !pm )
			{
				pm = new QPixmap( ((KListViewItemSingle*)*it)->item->pixmap(32) );
			}
		}
		it++;
	}

	QMimeData *ret = new QMimeData;

	ret->setUrls( urls );

	QClipboard *cb = QApplication::clipboard();
	cb->setMimeData( ret );
	if ( pm && !pm->isNull() ) cb->setPixmap( *pm );

	current_list_view = NULL;
}

void KomparatorWidget::slotTrashFiles()
{
	if ( !current_list_view ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = TRASH;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();
	m_interaction_remove_remaining_items.clear();

	QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
	QTreeWidgetItem *i;
	QListIterator<QTreeWidgetItem*> myit( tmp_selected );
	while ( myit.hasNext() )
	{
		i = myit.next();
		m_interaction_remove_remaining_items.append( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups *) i)->item :
			((KListViewItemSingle *) i)->item );
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotDeleteFiles()
{
	if ( !current_list_view ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = DELETE;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();
	m_interaction_remove_remaining_items.clear();

	QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
	QTreeWidgetItem *i;
	QListIterator<QTreeWidgetItem*> myit( tmp_selected );
	while ( myit.hasNext() )
	{
		i = myit.next();
		m_interaction_remove_remaining_items.append( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups *) i)->item :
			((KListViewItemSingle *) i)->item );
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotTrashNonreferenceFiles()
{
	if ( !current_list_view || current_list_view != duplicates1_KListView ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = TRASH_COREF;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();
	m_interaction_remove_remaining_items.clear();

	KFileItemExt *tmp_item;
	QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
	QTreeWidgetItem *i;
	QListIterator<QTreeWidgetItem*> myit( tmp_selected );
	while ( myit.hasNext() )
	{
		i = myit.next();
		tmp_item = ((KListViewItemDups*) i)->item;
		if ( tmp_item->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
		{
			m_interaction_remove_remaining_items.append( tmp_item );
		}
		while ( tmp_item->duplicates_size )
		{
			if ( tmp_item->duplicates_size->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
			{
				m_interaction_remove_remaining_items.append( tmp_item->duplicates_size );
			}
			tmp_item = tmp_item->duplicates_size;
		}
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotDeleteNonreferenceFiles()
{
	if ( !current_list_view || current_list_view != duplicates1_KListView ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = DELETE_COREF;

	m_interaction_empty_dirs.clear();
	m_interaction_empty_dirs_file_urls.clear();
	m_interaction_remove_remaining_items.clear();

	KFileItemExt *tmp_item;
	QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
	QTreeWidgetItem *i;
	QListIterator<QTreeWidgetItem*> myit( tmp_selected );
	while ( myit.hasNext() )
	{
		i = myit.next();
		tmp_item = ((KListViewItemDups*) i)->item;
		if ( tmp_item->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
		{
			m_interaction_remove_remaining_items.append( tmp_item );
		}
		while ( tmp_item->duplicates_size )
		{
			if ( tmp_item->duplicates_size->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
			{
				m_interaction_remove_remaining_items.append( tmp_item->duplicates_size );
			}
			tmp_item = tmp_item->duplicates_size;
		}
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::removeFirst()
{
	if ( m_interaction_total == 0 ) return;

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIcon( KIcon( SmallIcon( "process-stop" ) ) );

	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
		( m_interaction_total == 1 ) ? i18n( "Trashing 1 file..." ) : i18n( "Trashing %1 files..." ).arg( m_interaction_total ) :
		( m_interaction_total == 1 ) ? i18n( "Deleting 1 file..." ) : i18n( "Deleting %1 files..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QStringList del;
	KFileItemExt *i;
	QListIterator<KFileItemExt*> myit( m_interaction_remove_remaining_items );
	while ( myit.hasNext() )
	{
		i = myit.next();
		del.push_back( prettyUrl( i->url(), URL_DIR_OR_URL, KUrl( "" ) ) );
	}

	QString msg = ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
		( m_interaction_total == 1 ) ?
			i18n( "Do you really want to throw the selected file away?" ) :
			i18n( "Do you really want to throw the %1 selected files away?" ).arg( m_interaction_total ) :
		( m_interaction_total == 1 ) ?
			i18n( "Do you really want to delete the selected file?" ) :
			i18n( "Do you really want to delete the %1 selected files?" ).arg( m_interaction_total );

	KGlobal::config()->reparseConfiguration();
	QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );

	if ( KMessageBox::warningContinueCancelList( (this)->parentWidget(), msg, del,
			( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ? i18n( "Trash files" ) : i18n( "Delete files" ),
			( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
				KGuiItem( i18n( "Trash" ), "user-trash") : KGuiItem( i18n( "Delete" ), "edit-delete" ), KStandardGuiItem::cancel(),
				( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ? "komparator_message_trash" : "komparator_message_delete" )
					== KMessageBox::Cancel )
	{
		QApplication::restoreOverrideCursor();
		m_interaction_remove_remaining_items.clear();
		slotRemoveNext( NULL );
		return;
	}
	QApplication::restoreOverrideCursor();

	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.sync();
	if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
	{
		bool warning_trash = cfg.readEntry( "komparator_message_trash", true );
		warningTrash_CheckBox->setChecked( warning_trash );
	}
	else
	{
		bool warning_delete = cfg.readEntry( "komparator_message_delete", true );
		warningDelete_CheckBox->setChecked( warning_delete );
	}

	if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
	{
		KIO::CopyJob *job = KIO::trash( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotRemoveNext( KJob * ) ) );
	}
	else if ( m_file_remove_action == DELETE || m_file_remove_action == DELETE_COREF )
	{
		KIO::DeleteJob *job = KIO::del( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotRemoveNext( KJob * ) ) );
	}
	else
	{
		kDebug() << "Invalid file remove action. This point should never be reached. Please report this to the author.";
	}
}

void KomparatorWidget::slotRemoveNext( KJob *finished_job )
{
	if ( finished_job )
	{
		if ( finished_job->error() == 0 )
		{
			if ( findEmptyDirectories_CheckBox->isChecked() )
			{
				m_interaction_empty_dirs.push_back( ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
					((KIO::CopyJob *)finished_job)->srcUrls().first() : // append only first item, as we delete ONE
					((KIO::DeleteJob *)finished_job)->urls().first() ); //   item at a time.
			}

			removeDeletedFromLists( m_interaction_remove_remaining_items.last() );
		}
		else
		{
			emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 			((KIO::Job *)finished_job)->ui()->setWindow( this );
// 			((KIO::Job *)finished_job)->ui()->showErrorMessage();
		}
	}

	// remove item from list; "removeDeletedFromLists" has already deleted the listviewitem.
	if ( m_interaction_remove_remaining_items.count() > 0 ) m_interaction_remove_remaining_items.removeLast();

	if ( m_interaction_remove_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
		{
			KIO::CopyJob *job = KIO::trash( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
			connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotRemoveNext( KJob * ) ) );
		}
		else if ( m_file_remove_action == DELETE || m_file_remove_action == DELETE_COREF )
		{
			KIO::DeleteJob *job = KIO::del( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
			connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotRemoveNext( KJob * ) ) );
		}
		else
		{
			kDebug() << "Invalid file remove action. This point should never be reached. Please report this to the author.";
		}
	}
	else // canceled or no items remaining
	{
		KUrl::List empty_list;
		checkEmptyDirs( m_interaction_empty_dirs, empty_list );
		m_interaction_empty_dirs.clear();
		m_interaction_empty_dirs_file_urls.clear();
		updateFilters();

		m_interaction_remove_remaining_items.clear();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1,
			m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ?
			i18n( "Trashing files finished." ) :
			i18n( "Deleting files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );
		resizeListViewColumns();

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
	}
}

void KomparatorWidget::removeDeletedFromLists( KFileItemExt *item, bool duplicates_only, bool add_brother_missing )
{
	bool created_missing = false;
	KListViewItemSingle *new_item_single;

	QList< KListViewItemSingle* > to_delete_single;
	if ( !duplicates_only )
	{
		if ( !item->isDir() )
		{
			// remove from missing list.
			if ( item->dir == 0 ) // left
			{
				QTreeWidgetItemIterator it( missing1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle*)*it)->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)(*it) );
					}
					it++;
				}
			}
			else                  // right
			{
				QTreeWidgetItemIterator it( missing2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle*)*it)->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)(*it) );
					}
					it++;
				}
			}

			// remove from newer list.
			if ( item->dir == 0 ) // left
			{
				QTreeWidgetItemIterator it2( newer1_KListView );
				while( (*it2) )
				{
					if ( ((KListViewItemSingle*)(*it2))->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)(*it2) );    // Remove the file from the one side

						if ( isEnabled( MISSING ) )                                        // Append the other file to the other missing side if user wants to.
						{
							new_item_single = new KListViewItemSingle( missing2_KListView, ((KListViewItemSingle*)(*it2))->brother->item, NULL );
							new_item_single->setText( 1, prettyUrl( ((KListViewItemSingle*)(*it2))->brother->item->url(), URL_REPLACE, ((KListViewItemSingle*)(*it2))->brother->item->parent_path ) );
							emit( signalCount( MISSING, 2, -1 ) );
						}

						created_missing = true;
						to_delete_single.append( ((KListViewItemSingle*)(*it2))->brother ); // Remove the other file (newer or older).
					}
					it2++;
				}
			}
			else                  // right
			{
				QTreeWidgetItemIterator it2( newer2_KListView );
				while( (*it2) )
				{
					if ( ((KListViewItemSingle*)(*it2))->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)(*it2) );    // Remove the file from the one side

						if ( isEnabled( MISSING ) )                                        // Append the other file to the other missing side if user wants to.
						{
							new_item_single = new KListViewItemSingle( missing1_KListView, ((KListViewItemSingle*)(*it2))->brother->item, NULL );
							new_item_single->setText( 1, prettyUrl( ((KListViewItemSingle*)(*it2))->brother->item->url(), URL_REPLACE, ((KListViewItemSingle*)(*it2))->brother->item->parent_path ) );
							emit( signalCount( MISSING, 1, -1 ) );
						}

						created_missing = true;
						to_delete_single.append( ((KListViewItemSingle*)(*it2))->brother ); // Remove the other file (newer or older).
					}
					it2++;
				}
			}
		}
	}

	item->isdupe_size = 0;  // Make sure the item is removed from duplicates1_KListView, if necessary.
	item->hasdupes_size = 0;

	item->isdupe_path = 0;  // Make sure the item's brother is removed from duplicates1_KListView, if necessary.
	item->hasdupes_path = 0;

	if ( !created_missing )
	{
		QString item_str, cmp_str;
		item_str = prettyUrl( item->url(), URL_REPLACE, item->parent_path ) + item->url().fileName();

		KFileItemExt *curfile = item;

		while( curfile->dup_size_parent ) curfile = curfile->dup_size_parent; // Go to the first in duplicate chain.

		while ( curfile )
		{
			if ( curfile->url() != item->url() ) // We don't want to add the file that we want to delete.
			{
				cmp_str = prettyUrl( curfile->url(), URL_REPLACE, curfile->parent_path ) + curfile->url().fileName();

				if ( !QString::compare( item_str, cmp_str ) )
				{
					if ( curfile->dir == 0 )
					{
						if ( isEnabled( MISSING ) && add_brother_missing )
						{
							new_item_single = new KListViewItemSingle( missing1_KListView, curfile, NULL );
							new_item_single->setText( 1, prettyUrl( curfile->url(), URL_REPLACE, curfile->parent_path ) );
							emit( signalCount( MISSING, 1, -1 ) );
						}
					}
					else
					{
						if ( isEnabled( MISSING ) && add_brother_missing )
						{
							new_item_single = new KListViewItemSingle( missing2_KListView, curfile, NULL );
							new_item_single->setText( 1, prettyUrl( curfile->url(), URL_REPLACE, curfile->parent_path ) );
							emit( signalCount( MISSING, 2, -1 ) );
						}
					}
				}
			}

			curfile = curfile->duplicates_size;
		}
	}

	QTreeWidgetItemIterator it3( duplicates2_KListView );
	while( (*it3) )
	{
		if ( ((KListViewItemSingle*)(*it3))->item == item )
		{
			to_delete_single.append( (KListViewItemSingle*)(*it3) );
		}
		it3++;
	}

	// skip this file in duplicate chain
	if ( item->dup_size_parent ) // not the first in duplicate chain.
	{
		KFileItemExt *curfile = item->dup_size_parent;
		curfile->duplicates_size = item->duplicates_size;

		if ( !curfile->dup_size_parent && !curfile->duplicates_size ) // a single file. no duplicate.
		{
			curfile->hasdupes_size = 0;
			curfile->isdupe_size = 0;
		}
	}
	else // first in duplicate chain
	{
		if ( item->duplicates_size )
		{
			item->duplicates_size->dup_size_parent = NULL;
			if ( item->duplicates_size->duplicates_size )
			{
				item->duplicates_size->hasdupes_size = 1;
				item->duplicates_size->isdupe_size = 1;
			}
			else
			{
				item->duplicates_size->hasdupes_size = 0;
				item->duplicates_size->isdupe_size = 0;
			}
			if ( isEnabled( DUPLICATES ) )
			{
				(new KListViewItemDups( duplicates1_KListView, item->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() ); // We might have the same file in two items!
				emit( signalCount( DUPLICATES, 1, -1 ) );
			}
		}
		else // Last item in duplicate chain. only for directories.
		{
			if ( item == m_dirs_dupes )
			{
				delete m_dirs_dupes;
				m_dirs_dupes = NULL;
			}
		}
	}

	while ( !to_delete_single.isEmpty() ) delete to_delete_single.takeFirst(); // really remove missing / newer / duplicates2

	QList< KListViewItemDups* > to_delete_dups; // Delete items that are invalid.
	QTreeWidgetItemIterator it5( duplicates1_KListView );
	while( (*it5) )
	{
		if ( ((KListViewItemDups*)(*it5))->item->isdupe_size == 0 && ((KListViewItemDups*)(*it5))->item->hasdupes_size == 0 && !((KListViewItemDups*)(*it5))->item->isDir() )
		{
			to_delete_dups.append( ((KListViewItemDups*)(*it5)) );
		}
		else if ( !((KListViewItemDups*)(*it5))->item->duplicates_size && !((KListViewItemDups*)(*it5))->item->isDir() )
		{
			to_delete_dups.append( ((KListViewItemDups*)(*it5)) );
		}
		else if ( ((KListViewItemDups*)(*it5))->item->isDir() && ((KListViewItemDups*)(*it5))->item == item )
		{
			to_delete_dups.append( ((KListViewItemDups*)(*it5)) );
		}
		else // we don't want to delete this duplicate list, but check, whether it's a duplicate of two identical sub-paths or not.
		{
			((KListViewItemDups*)(*it5))->updateColor();
		}

		it5++;
	}

	QListIterator< KListViewItemDups* > it6( to_delete_dups ); // really remove duplicates1
	while( it6.hasNext() ) delete it6.next();

	slotDuplicateSelected();
}

void KomparatorWidget::checkEmptyDirs( KUrl::List &urls, KUrl::List &file_urls ) // if a file was deleted, the directory might be empty now.
{
	m_empty_directory_job->initialize( &urls, &file_urls );
	m_empty_directory_job->start();
}

void KomparatorWidget::slotOpenWith()
{
	if ( current_list_view )
	{
		KRun::displayOpenWithDialog( KUrl::split( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups*) current_list_view->currentItem())->item->url() :
			((KListViewItemSingle*) current_list_view->currentItem())->item->url() ), this );
	}

	current_list_view = NULL;
}

void KomparatorWidget::slotFileProperties()
{
	if ( current_list_view )
	{
		KFileItemList tmp_list;
		QTreeWidgetItem *i;
		QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
		QListIterator<QTreeWidgetItem*> myit( tmp_selected );
		while ( myit.hasNext() )
		{
			i = myit.next();
			tmp_list.append( (KFileItem)( ( current_list_view == duplicates1_KListView ) ?
				*((KListViewItemDups*) i)->item :
				*((KListViewItemSingle*) i)->item ) );
		}

		KPropertiesDialog *prop_dlg = new KPropertiesDialog( tmp_list, this );
		connect( prop_dlg, SIGNAL( applied() ), this, SLOT( slotFilePropertiesChanged() ) );
		connect( prop_dlg, SIGNAL( saveAs( const KUrl &, KUrl & ) ), this, SLOT( slotFilePropertiesRenamed( const KUrl &, KUrl & ) ) );
		prop_dlg->show();
	}
}

void KomparatorWidget::slotFilePropertiesRenamed( const KUrl &old_url, KUrl &new_url )
{
	if ( current_list_view )
	{
		if ( old_url != new_url ) m_properties_renamed = true;
		if ( current_list_view == duplicates1_KListView )
		{
			((KListViewItemDups *) current_list_view->selectedItems().first())->item->setUrl( new_url );
		}
		else
		{
			((KListViewItemSingle *) current_list_view->selectedItems().first())->item->setUrl( new_url );
		}
	}
}

void KomparatorWidget::slotFilePropertiesChanged()
{
	if ( current_list_view )
	{
		QList< KFileItemExt* > tmp_list;
		QTreeWidgetItem *i;
		QList< QTreeWidgetItem* > tmp_selected = current_list_view->selectedItems();
		QListIterator<QTreeWidgetItem*> myit( tmp_selected );
		while ( myit.hasNext() )
		{
			i = myit.next();
			tmp_list.append( ( current_list_view == duplicates1_KListView ) ?
				((KListViewItemDups *) i)->item :
				((KListViewItemSingle *) i)->item );
		}

		if ( m_properties_renamed == true )
		{
			m_properties_renamed = false;

			QStringList list;
			list.append( prettyUrl( tmp_list.first()->url(), URL_DIR_OR_URL, KUrl( "" ) ) );

			KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
			KConfigGroup cfg( config_global, "Notification Messages" );
			cfg.sync();
			if ( KMessageBox::warningContinueCancelList( this,
				i18n( "You renamed a file. Komparator needs to re-start the search on the whole tree to display correct results. "
							"If you don't want that, keep in mind that all search results concerning this file are invalid." ),
				list, i18n( "Re-start search" ), KGuiItem( i18n( "Re-start" ), "reload"), KStandardGuiItem::cancel(), "komparator_message_reload" ) != KMessageBox::Cancel )
			{
				cfg.sync();
				bool warning_reload = cfg.readEntry( "komparator_message_reload", true );
				warningReload_CheckBox->setChecked( warning_reload );

				slotSearch(); // Reset.
				slotSearch(); // Start search.
			}
		}
		else
		{
			m_interaction_cancel = false;
			m_status = INTERACTING;
			search_KPushButton->setText( i18n( "Cancel" ) );
			search_KPushButton->setIcon( KIcon( SmallIcon( "process-stop" ) ) );

			m_interaction_remove_remaining_items = tmp_list;
			m_interaction_total = m_interaction_remove_remaining_items.count();
			m_interaction_progress = 0;
			m_interaction_progress_str = ( m_interaction_total == 1 ) ?
			i18n( "Getting file properties of 1 file..." ) :
			i18n( "Getting file properties of %1 files..." ).arg( m_interaction_total );

			enable( false );

			emit( signalProgress( m_interaction_progress_str, -1, "" ) );
			emit( signalStartSystrayMovie() );

			KIO::StatJob *job = KIO::stat( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
			connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotFilePropertiesNextStat( KJob * ) ) );
		}
	}
}

void KomparatorWidget::slotFilePropertiesNextStat( KJob *finished_job )
{
	if ( finished_job )
	{
		if ( finished_job->error() == 0 )
		{
			m_interaction_remove_remaining_items.last()->setUDSEntry( ((KIO::StatJob *)finished_job)->statResult(),
				m_interaction_remove_remaining_items.last()->url() );

			QTreeWidgetItemIterator it( duplicates1_KListView );
			while( (*it) )
			{
				if ( ((KListViewItemDups *)(*it))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemDups *)(*it))->setText( 4, ((KListViewItemDups *)(*it))->item->permissionsString() );
					((KListViewItemDups *)(*it))->setText( 5, ((KListViewItemDups *)(*it))->item->user() );
					((KListViewItemDups *)(*it))->setText( 6, ((KListViewItemDups *)(*it))->item->group() );
				}
				it++;
			}

			QTreeWidgetItemIterator it2( duplicates2_KListView );
			while( (*it2) )
			{
				if ( ((KListViewItemSingle *)(*it2))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)(*it2))->setText( 4, ((KListViewItemSingle *)(*it2))->item->permissionsString() );
					((KListViewItemSingle *)(*it2))->setText( 5, ((KListViewItemSingle *)(*it2))->item->user() );
					((KListViewItemSingle *)(*it2))->setText( 6, ((KListViewItemSingle *)(*it2))->item->group() );
				}
				it2++;
			}

			QTreeWidgetItemIterator it3( missing1_KListView );
			while( (*it3) )
			{
				if ( ((KListViewItemSingle *)(*it3))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)(*it3))->setText( 4, ((KListViewItemSingle *)(*it3))->item->permissionsString() );
					((KListViewItemSingle *)(*it3))->setText( 5, ((KListViewItemSingle *)(*it3))->item->user() );
					((KListViewItemSingle *)(*it3))->setText( 6, ((KListViewItemSingle *)(*it3))->item->group() );
				}
				it3++;
			}

			QTreeWidgetItemIterator it4( missing2_KListView );
			while( (*it4) )
			{
				if ( ((KListViewItemSingle *)(*it4))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)(*it4))->setText( 4, ((KListViewItemSingle *)(*it4))->item->permissionsString() );
					((KListViewItemSingle *)(*it4))->setText( 5, ((KListViewItemSingle *)(*it4))->item->user() );
					((KListViewItemSingle *)(*it4))->setText( 6, ((KListViewItemSingle *)(*it4))->item->group() );
				}
				it4++;
			}

			QTreeWidgetItemIterator it5( newer1_KListView );
			while( (*it5) )
			{
				if ( ((KListViewItemSingle *)(*it5))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)(*it5))->setText( 4, ((KListViewItemSingle *)(*it5))->item->permissionsString() );
					((KListViewItemSingle *)(*it5))->setText( 5, ((KListViewItemSingle *)(*it5))->item->user() );
					((KListViewItemSingle *)(*it5))->setText( 6, ((KListViewItemSingle *)(*it5))->item->group() );
				}
				it5++;
			}

			QTreeWidgetItemIterator it6( newer2_KListView );
			while( (*it6) )
			{
				if ( ((KListViewItemSingle *)(*it6))->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)(*it6))->setText( 4, ((KListViewItemSingle *)(*it6))->item->permissionsString()  );
					((KListViewItemSingle *)(*it6))->setText( 5, ((KListViewItemSingle *)(*it6))->item->user() );
					((KListViewItemSingle *)(*it6))->setText( 6, ((KListViewItemSingle *)(*it6))->item->group() );
				}
				it6++;
			}
		}
		else
		{
			emit signalAddErrorMessage( ((KIO::Job *)finished_job)->detailedErrorStrings().at( 0 ) );
// 			((KIO::Job *)finished_job)->ui()->setWindow( this );
// 			((KIO::Job *)finished_job)->ui()->showErrorMessage();
		}
	}

	m_interaction_remove_remaining_items.removeLast();

	if ( m_interaction_remove_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		KIO::StatJob *job = KIO::stat( m_interaction_remove_remaining_items.last()->url(), KIO::HideProgressInfo );
		connect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotFilePropertiesNextStat( KJob * ) ) );
	}
	else // canceled or no items remaining
	{
		m_interaction_remove_remaining_items.clear();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Getting file properties finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );
		resizeListViewColumns();

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
	}
}

void KomparatorWidget::slotCompareEqual()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() != 1 ) return;

	KFileItemExt *item1;
	KFileItemExt *item2;

	if ( current_list_view == newer1_KListView )
	{
		item1 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->brother->item;
	}
	else if ( current_list_view == newer2_KListView )
	{
		item2 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->item;
		item1 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->brother->item;
	}
	else return;

	KompareDialog *dlg = new KompareDialog( this, item1, item2, prettyUrl( item1->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ),
		prettyUrl( item2->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ), calculateChecksum_CheckBox->isChecked() );
	int close_result = dlg->exec();
	int result = dlg->getResult();

	if ( close_result == 0 )
	{
		if ( result & COPY_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			slotMoveToOtherSide();
		}
		if ( result & COPY_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			slotMoveToOtherSide();
		}

		m_interaction_empty_dirs.clear();
		m_interaction_empty_dirs_file_urls.clear();
		m_interaction_remove_remaining_items.clear();

		if ( result & TRASH_LEFT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & DELETE_LEFT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & TRASH_RIGHT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item2 );
		}
		if ( result & DELETE_RIGHT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item2 );
		}

		m_interaction_total = m_interaction_remove_remaining_items.count();
		if ( m_interaction_total ) removeFirst();

		if ( result & DIRTY )
		{
			KMessageBox::information( this, i18n( "You modified a file during comparison. "
				"You need to restart the search to make these changes take effect." ),
				i18n( "Re-start search" ) );
		}
	}
}

void KomparatorWidget::slotCompareSelected()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() != 1 ) return;

	KFileItemExt *item1;
	KFileItemExt *item2;

	if ( current_list_view == missing1_KListView || current_list_view == missing2_KListView )
	{
		item1 = ((KListViewItemSingle *)missing1_KListView->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)missing2_KListView->selectedItems().first())->item;
	}
	else if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		item1 = ((KListViewItemSingle *)newer1_KListView->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)newer2_KListView->selectedItems().first())->item;
	}
	else return;

	KompareDialog *dlg = new KompareDialog( this, item1, item2, prettyUrl( item1->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ),
		prettyUrl( item2->url(), URL_DIR_OR_URL_ONLY, KUrl( "" ) ), calculateChecksum_CheckBox->isChecked() );
	int close_result = dlg->exec();
	int result = dlg->getResult();

	if ( close_result == 0 )
	{
		if ( result & COPY_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			else if ( current_list_view == missing2_KListView )
			{
				missing1_KListView->clearSelection();
				QTreeWidgetItemIterator it( missing1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing1_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			else if ( current_list_view == missing2_KListView )
			{
				missing1_KListView->clearSelection();
				QTreeWidgetItemIterator it( missing1_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item1 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing1_KListView;
			}
			slotMoveToOtherSide();
		}
		if ( result & COPY_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			else if ( current_list_view == missing1_KListView )
			{
				missing2_KListView->clearSelection();
				QTreeWidgetItemIterator it( missing2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing2_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QTreeWidgetItemIterator it( newer2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			else if ( current_list_view == missing1_KListView )
			{
				missing2_KListView->clearSelection();
				QTreeWidgetItemIterator it( missing2_KListView );
				while( (*it) )
				{
					if ( ((KListViewItemSingle *)(*it))->item == item2 )
					{
						(*it)->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing2_KListView;
			}
			slotMoveToOtherSide();
		}

		m_interaction_empty_dirs.clear();
		m_interaction_empty_dirs_file_urls.clear();
		m_interaction_remove_remaining_items.clear();

		if ( result & TRASH_LEFT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & DELETE_LEFT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & TRASH_RIGHT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item2 );
		}
		if ( result & DELETE_RIGHT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item2 );
		}

		m_interaction_total = m_interaction_remove_remaining_items.count();
		if ( m_interaction_total ) removeFirst();

		if ( result & DIRTY )
		{
			KMessageBox::information( this, i18n( "You modified a file during comparison. "
				"You need to restart the search to make these changes take effect." ),
				i18n( "Re-start search" ) );
		}
	}
}

void KomparatorWidget::slotCopyMissingL2R()
{
	current_list_view = missing1_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyMissingR2L()
{
	current_list_view = missing2_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyNewerL2R()
{
	current_list_view = newer1_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyNewerR2L()
{
	current_list_view = newer2_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotMoveMissingL2R()
{
	current_list_view = missing1_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveMissingR2L()
{
	current_list_view = missing2_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveNewerL2R()
{
	current_list_view = newer1_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveNewerR2L()
{
	current_list_view = newer2_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotRemoveSelectionMissing1()
{
	current_list_view = missing1_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionMissing2()
{
	current_list_view = missing2_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionNewer1()
{
	current_list_view = newer1_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionNewer2()
{
	current_list_view = newer2_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotTrashDuplicates()
{
	current_list_view = duplicates2_KListView;
	slotTrashFiles();
}

void KomparatorWidget::slotDeleteDuplicates()
{
	current_list_view = duplicates2_KListView;
	slotDeleteFiles();
}

void KomparatorWidget::slotPresetSelected( int sel )
{
	if ( m_status == READY ) slotSearch(); // cancel

	KomparatorPreset *current_preset = m_preset_list[sel];
	if ( current_preset )
	{
		current_preset->setToInterface();
	}
}

void KomparatorWidget::slotAddPreset()
{
	KomparatorPreset *new_preset = new KomparatorPreset( this, m_config );
	if ( new_preset->readFromInterface() )
	{
		m_preset_list.append( new_preset );
		preset_QComboBox->insertItem( 999999, new_preset->title() );
		preset_QComboBox->setCurrentIndex( preset_QComboBox->count()-1 );
		new_preset->saveConfig( preset_QComboBox->count()-1 );
		slotPresetSelected( preset_QComboBox->count()-1 );
	}
	else // user canceled
	{
		delete new_preset;
		new_preset = NULL;
	}
}

void KomparatorWidget::slotUpdatePreset()
{
	KomparatorPreset *current_preset = m_preset_list[preset_QComboBox->currentIndex()];
	if ( preset_QComboBox->currentIndex() != 0 && current_preset ) // don't save to default item
	{
		current_preset->readFromInterface( true ); // reread
	}
}

void KomparatorWidget::slotRemovePreset()
{
	KGlobal::config()->reparseConfiguration();
	QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );

	if ( KMessageBox::warningContinueCancel( this, i18n( "Do you really want to remove the preset \"%1\"?" ).arg(
				preset_QComboBox->currentText() ), i18n( "Remove preset" ),
			KGuiItem( i18n( "Remove" ), "edit-delete" ), KStandardGuiItem::cancel(), "komparator_message_remove_preset" ) == KMessageBox::Cancel )
	{
		QApplication::restoreOverrideCursor();
		return;
	}
	QApplication::restoreOverrideCursor();

	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.sync();
	bool warning_remove_preset = cfg.readEntry( "komparator_message_remove_preset", true );
	warningRemovePreset_CheckBox->setChecked( warning_remove_preset );

	int current_item = preset_QComboBox->currentIndex();

	if ( current_item == 0 ) return; // don't delete default

	if ( current_item != -1 ) delete m_preset_list.takeAt( current_item );

	preset_QComboBox->removeItem( current_item );

	preset_QComboBox->setCurrentIndex( current_item-1 );
	slotPresetSelected( current_item-1 );
}

void KomparatorWidget::slotSystray()
{
	if ( systray_CheckBox->isChecked() )
		m_systray->show();
	else
		m_systray->hide();
}

void KomparatorWidget::slotTextColor()
{
	QTreeWidgetItemIterator it_dups1( duplicates1_KListView );
	QTreeWidgetItemIterator it_dups2( duplicates2_KListView );
	QTreeWidgetItemIterator it_missing1( missing1_KListView );
	QTreeWidgetItemIterator it_missing2( missing2_KListView );
	QTreeWidgetItemIterator it_newer1( newer1_KListView );
	QTreeWidgetItemIterator it_newer2( newer2_KListView );

	while ( (*it_dups1) )
	{
		((KListViewItemDups*)(*it_dups1))->setUseColor( textColor_CheckBox->isChecked() );
		++it_dups1;
	}
	while ( (*it_dups2) )
	{
		((KListViewItemSingle*)(*it_dups2))->setKind( (FILTER)((KListViewItemSingle*)(*it_dups2))->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)(*it_dups2))->color() );
		++it_dups2;
	}
	while ( (*it_missing1) )
	{
		((KListViewItemSingle*)(*it_missing1))->setKind( (FILTER)((KListViewItemSingle*)(*it_missing1))->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)(*it_missing1))->color() );
		++it_missing1;
	}
	while ( (*it_missing2) )
	{
		((KListViewItemSingle*)(*it_missing2))->setKind( (FILTER)((KListViewItemSingle*)(*it_missing2))->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)(*it_missing2))->color() );
		++it_missing2;
	}
	while ( (*it_newer1) )
	{
		((KListViewItemSingle*)(*it_newer1))->setKind( (FILTER)((KListViewItemSingle*)(*it_newer1))->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)(*it_newer1))->color() );
		++it_newer1;
	}
	while ( (*it_newer2) )
	{
		((KListViewItemSingle*)(*it_newer2))->setKind( (FILTER)((KListViewItemSingle*)(*it_newer2))->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)(*it_newer2))->color() );
		++it_newer2;
	}
}

void KomparatorWidget::slotWarningTrash()
{
	bool warning_trash = warningTrash_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_trash", warning_trash );
	cfg.sync();
}

void KomparatorWidget::slotWarningDelete()
{
	bool warning_delete = warningDelete_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_delete", warning_delete );
	cfg.sync();
}

void KomparatorWidget::slotWarningOverwriteCopy()
{
	bool warning_overwrite_copy = warningOverwriteCopy_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_overwrite_copy", warning_overwrite_copy );
	cfg.sync();
}

void KomparatorWidget::slotWarningOverwriteMove()
{
	bool warning_overwrite_move = warningOverwriteMove_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_overwrite_move", warning_overwrite_move );
	cfg.sync();
}

void KomparatorWidget::slotWarningReload()
{
	bool warning_reload = warningReload_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_reload", warning_reload );
	cfg.sync();
}

void KomparatorWidget::slotWarningRemovePreset()
{
	bool warning_remove_preset = warningRemovePreset_CheckBox->isChecked();
	KSharedConfigPtr config_global = KSharedConfig::openConfig( "komparator4rc" );
	KConfigGroup cfg( config_global, "Notification Messages" );
	cfg.writeEntry( "komparator_message_remove_preset", warning_remove_preset );
	cfg.sync();
}


void KomparatorWidget::sizeComboActivated( int act )
{
	bytes_KIntNumInput->setEnabled( (bool)act );
	bytes_QComboBox->setEnabled( (bool)act );
}


/****************************************
* File Filter Functions                 *
****************************************/

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
	const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
	const bool &consider_date, const QDate &date, const bool &newer_real, const bool &newer_equal, const bool &newer_same_time )
{
	KomparatorTreeWidget *tmp_list_view;
	if ( filter == duplicates2_KomparatorFileFilter ) tmp_list_view = duplicates2_KListView; // the next three won't be processed here usually
	else if ( filter == missing1_KomparatorFileFilter ) tmp_list_view = missing1_KListView;
	else if ( filter == missing2_KomparatorFileFilter ) tmp_list_view = missing2_KListView;
	else if ( filter == newer1_KomparatorFileFilter ) tmp_list_view = newer1_KListView;
	else if ( filter == newer2_KomparatorFileFilter ) tmp_list_view = newer2_KListView;
	else
	{
		kDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!";
		return;
	}

	int visible_items = 0;
	QTreeWidgetItemIterator it( tmp_list_view );
	while( (*it) )
	{
		if ( (*it)->text( 0 ).indexOf( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemSingle*)*it)->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( NAME );
		}

		if ( (*it)->text( 1 ).indexOf( path ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( PATH );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( PATH );
		}

		if ( (*it)->text( 5 ).indexOf( owner ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( OWNER );
		}

		if ( (*it)->text( 6 ).indexOf( group ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( GROUP );
		}

		if ( ((KListViewItemSingle*)*it)->item->size() >= min_size &&
			((KListViewItemSingle*)*it)->item->size() <= max_size )
		{
			((KListViewItemSingle*)*it)->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemSingle*)*it)->item->time( KFileItem::ModificationTime ).toTime_t() );
			if ( dt.date() == date )
			{
				((KListViewItemSingle*)*it)->removeFilter( DATE );
			}
			else
			{
				((KListViewItemSingle*)*it)->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemSingle*)*it)->removeFilter( DATE );
		}

		if ( newer_real )
		{
			((KListViewItemSingle*)*it)->removeFilter( NEWER_REAL );
		}
		else
		{
			if ( ((KListViewItemSingle*)*it)->kind() == NEWER_REAL )
			{
				((KListViewItemSingle*)*it)->addFilter( NEWER_REAL );
			}
		}

		if ( newer_equal )
		{
			((KListViewItemSingle*)*it)->removeFilter( NEWER_EQUAL );
		}
		else
		{
			if ( ((KListViewItemSingle*)*it)->kind() == NEWER_EQUAL )
			{
				((KListViewItemSingle*)*it)->addFilter( NEWER_EQUAL );
			}
		}

		if ( newer_same_time )
		{
			((KListViewItemSingle*)*it)->removeFilter( NEWER_SAME_TIME );
		}
		else
		{
			if ( ((KListViewItemSingle*)*it)->kind() == NEWER_SAME_TIME )
			{
				((KListViewItemSingle*)*it)->addFilter( NEWER_SAME_TIME );
			}
		}

		((KListViewItemSingle*)*it)->setVisibility();

		if ( !(*it)->isHidden() ) visible_items++;

		++it;
	}

	if ( filter == duplicates2_KomparatorFileFilter ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( filter == missing1_KomparatorFileFilter ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( filter == missing2_KomparatorFileFilter ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( filter == newer1_KomparatorFileFilter ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( filter == newer2_KomparatorFileFilter ) emit( signalCount( NEWER, 2, visible_items ) );

	tmp_list_view = NULL;
}

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
			const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
			const bool &consider_date, const QDate &date, const bool &duplicates_normal, const bool &duplicates_multiple )
{
	KomparatorTreeWidget *tmp_list_view;
	if ( filter == duplicates1_KomparatorFileFilter ) tmp_list_view = duplicates1_KListView;
	else
	{
		kDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!";
		return;
	}

	int visible_items = 0;
	QTreeWidgetItemIterator it( tmp_list_view );
	while( (*it) )
	{
		if ( (*it)->text( 1 ).indexOf( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemDups*)(*it))->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemDups*)(*it))->addFilter( NAME );
		}

		if ( (*it)->text( 0 ).indexOf( path ) != -1 )
		{
			((KListViewItemDups*)(*it))->removeFilter( PATH );
		}
		else
		{
			((KListViewItemDups*)(*it))->addFilter( PATH );
		}

		if ( (*it)->text( 5 ).indexOf( owner ) != -1 )
		{
			((KListViewItemDups*)(*it))->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemDups*)(*it))->addFilter( OWNER );
		}

		if ( (*it)->text( 6 ).indexOf( group ) != -1 )
		{
			((KListViewItemDups*)(*it))->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemDups*)(*it))->addFilter( GROUP );
		}

		if ( ((KListViewItemDups*)(*it))->item->size() >= min_size &&
			((KListViewItemDups*)(*it))->item->size() <= max_size )
		{
			((KListViewItemDups*)(*it))->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemDups*)(*it))->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemDups*)(*it))->item->time( KFileItem::ModificationTime ).toTime_t() );
			if ( dt.date() == date )
			{
				((KListViewItemDups*)(*it))->removeFilter( DATE );
			}
			else
			{
				((KListViewItemDups*)(*it))->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemDups*)(*it))->removeFilter( DATE );
		}

		if ( ((KListViewItemDups*)(*it))->isDuplicatesPair() )
		{
			if ( duplicates_normal )
			{
				((KListViewItemDups*)(*it))->removeFilter( DUPLICATES_NORMAL );
			}
			else
			{
				((KListViewItemDups*)(*it))->addFilter( DUPLICATES_NORMAL );
			}
		}
		else
		{
			if ( duplicates_multiple )
			{
				((KListViewItemDups*)(*it))->removeFilter( DUPLICATES_MULTIPLE );
			}
			else
			{
				((KListViewItemDups*)(*it))->addFilter( DUPLICATES_MULTIPLE );
			}
		}

		((KListViewItemDups*)(*it))->setVisibility();

		if ( !(*it)->isHidden() ) visible_items++;

		++it;
	}

	emit( signalCount( DUPLICATES, 1, visible_items ) );

	tmp_list_view = NULL;
}

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
			const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
			const bool &consider_date, const QDate &date )
{
	KomparatorTreeWidget *tmp_list_view;
	if ( filter == duplicates2_KomparatorFileFilter ) tmp_list_view = duplicates2_KListView;
	else if ( filter == missing1_KomparatorFileFilter ) tmp_list_view = missing1_KListView;
	else if ( filter == missing2_KomparatorFileFilter ) tmp_list_view = missing2_KListView;
	else if ( filter == newer1_KomparatorFileFilter ) tmp_list_view = newer1_KListView; // these won't be processed here usually; however it would be legal.
	else if ( filter == newer2_KomparatorFileFilter ) tmp_list_view = newer2_KListView;
	else
	{
		kDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!";
		return;
	}

	int visible_items = 0;
	QTreeWidgetItemIterator it( tmp_list_view );
	while( (*it) )
	{
		if ( (*it)->text( 0 ).indexOf( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemSingle*)*it)->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( NAME );
		}

		if ( (*it)->text( 1 ).indexOf( path ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( PATH );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( PATH );
		}

		if ( (*it)->text( 5 ).indexOf( owner ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( OWNER );
		}

		if ( (*it)->text( 6 ).indexOf( group ) != -1 )
		{
			((KListViewItemSingle*)*it)->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( GROUP );
		}

		if ( ((KListViewItemSingle*)*it)->item->size() >= min_size &&
			((KListViewItemSingle*)*it)->item->size() <= max_size )
		{
			((KListViewItemSingle*)*it)->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemSingle*)*it)->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemSingle*)*it)->item->time( KFileItem::ModificationTime ).toTime_t() );
			if ( dt.date() == date )
			{
				((KListViewItemSingle*)*it)->removeFilter( DATE );
			}
			else
			{
				((KListViewItemSingle*)*it)->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemSingle*)*it)->removeFilter( DATE );
		}

		((KListViewItemSingle*)*it)->setVisibility();

		if ( !(*it)->isHidden() ) visible_items++;

		++it;
	}

	if ( filter == duplicates2_KomparatorFileFilter ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( filter == missing1_KomparatorFileFilter ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( filter == missing2_KomparatorFileFilter ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( filter == newer1_KomparatorFileFilter ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( filter == newer2_KomparatorFileFilter ) emit( signalCount( NEWER, 2, visible_items ) );

	tmp_list_view = NULL;
}


/****************************************
* Internal Functions                   *
****************************************/

void KomparatorWidget::slotSizeTypeChanged( int i )
{
	bytes_KIntNumInput->setEnabled( i );
	bytes_QComboBox->setEnabled( i );
}

void KomparatorWidget::slotSizeEnabled( bool e )
{
	calculateChecksum_CheckBox->setEnabled( e );
	if ( !e )
	{
		calculateChecksum_CheckBox->setChecked( false );
	}
}

void KomparatorWidget::slotMD5ChecksumEnabled( bool e )
{
	binaryComparison_CheckBox->setEnabled( e );
	if ( e )
	{
		binaryComparison_KIntSpinBox->setEnabled( ( binaryComparison_CheckBox->isChecked() ) ? true : false );
	}
	else
	{
		binaryComparison_CheckBox->setChecked( false );
		binaryComparison_KIntSpinBox->setEnabled( false );
	}
}

void KomparatorWidget::slotDuplicatesOnlyEnabled( bool e )
{
	url1_KUrlRequester->setEnabled( !e );
}

void KomparatorWidget::slotProgress( QString str, int percent, QString notification )
{
	status_Label->setText( str );
	main_KProgress->setTextVisible( ( percent == -1 ) ? false : true );
	main_KProgress->setMaximum( 100 );
	main_KProgress->setValue( ( percent == -1 ) ? 0 : percent );

	if ( notification != "" && notification_CheckBox->isChecked() && !isActiveWindow() )
	{
		KPassivePopup::message( KPassivePopup::Boxed, i18n( "Komparator" ), notification,
			SmallIcon( "komparator4" ), this, 2500 );
	}
}

void KomparatorWidget::slotProgressUndefined()
{
	main_KProgress->setTextVisible( false );
	if ( main_KProgress->maximum() != 0 ) main_KProgress->setMaximum( 0 );
}

void KomparatorWidget::enable( bool e )
{
	settings_TabWidget->setEnabled( e );
	main_TabWidget->setEnabled( e );

	if ( e ) QApplication::restoreOverrideCursor();
	else QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );

	duplicates1_KListView->setUpdatesEnabled( e );
	duplicates2_KListView->setUpdatesEnabled( e );

	missing1_KListView->setUpdatesEnabled( e );
	missing2_KListView->setUpdatesEnabled( e );

	newer1_KListView->setUpdatesEnabled( e );
	newer2_KListView->setUpdatesEnabled( e );

	if ( !e )
	{
		duplicates1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		duplicates2_KListView->header()->setResizeMode( QHeaderView::Interactive );
		missing1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		missing2_KListView->header()->setResizeMode( QHeaderView::Interactive );
		newer1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		newer2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	}
	else
	{
		duplicates1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		duplicates1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		duplicates1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		duplicates1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
		duplicates2_KListView->header()->setResizeMode( QHeaderView::Interactive );
		duplicates2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		duplicates2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		duplicates2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
		duplicates2_KListView->header()->setResizeMode( 7, QHeaderView::ResizeToContents );
		missing1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		missing1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		missing1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		missing1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
		missing2_KListView->header()->setResizeMode( QHeaderView::Interactive );
		missing2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		missing2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		missing2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
		newer1_KListView->header()->setResizeMode( QHeaderView::Interactive );
		newer1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		newer1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		newer1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
		newer2_KListView->header()->setResizeMode( QHeaderView::Interactive );
		newer2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
		newer2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
		newer2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	}
}

void KomparatorWidget::resizeListViewColumns()
{
	duplicates1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	duplicates2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer2_KListView->header()->setResizeMode( QHeaderView::Interactive );

	int i;
	for ( i=0; i<7; i++ )
	{
		duplicates1_KListView->resizeColumnToContents( i );
		duplicates2_KListView->resizeColumnToContents( i );
		missing1_KListView->resizeColumnToContents( i );
		missing2_KListView->resizeColumnToContents( i );
		newer1_KListView->resizeColumnToContents( i );
		newer2_KListView->resizeColumnToContents( i );
	}
	duplicates2_KListView->resizeColumnToContents( 7 );

	duplicates1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	duplicates1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	duplicates1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	duplicates1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	duplicates2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	duplicates2_KListView->header()->setResizeMode( 7, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	missing1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	missing2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	missing2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer1_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	newer1_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( QHeaderView::Interactive );
	newer2_KListView->header()->setResizeMode( 4, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( 5, QHeaderView::ResizeToContents );
	newer2_KListView->header()->setResizeMode( 6, QHeaderView::ResizeToContents );

	duplicates1_KListView->sortByColumn( 0, Qt::AscendingOrder );
	duplicates2_KListView->sortByColumn( 0, Qt::AscendingOrder );
	missing1_KListView->sortByColumn( 0, Qt::AscendingOrder );
	missing2_KListView->sortByColumn( 0, Qt::AscendingOrder );
	newer1_KListView->sortByColumn( 0, Qt::AscendingOrder );
	newer2_KListView->sortByColumn( 0, Qt::AscendingOrder );
}

void KomparatorWidget::updateFilters()
{
	duplicates1_KomparatorFileFilter->slotEmitFilterChanged();
	duplicates2_KomparatorFileFilter->slotEmitFilterChanged();
	missing1_KomparatorFileFilter->slotEmitFilterChanged();
	missing2_KomparatorFileFilter->slotEmitFilterChanged();
	newer1_KomparatorFileFilter->slotEmitFilterChanged();
	newer2_KomparatorFileFilter->slotEmitFilterChanged();
}

void KomparatorWidget::slotStartSystrayMovie()
{
	if ( m_systray_movie )
	{
		m_systray_movie->start();
	}
}

void KomparatorWidget::slotStopSystrayMovie()
{
	if ( m_systray_movie )
	{
		m_systray_movie->stop();
		m_systray_movie->jumpToFrame( 0 );
		m_systray_movie->stop();
	}
}

QString KomparatorWidget::prettyUrl( const KUrl &url, URL_MODE mode, const KUrl &rep  )
{
	KUrl tmp_url;

	QString ret = "";
	switch( mode )
	{
		case URL_DIR_OR_URL:
			tmp_url = url; // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			ret = tmp_url.pathOrUrl() + url.path( KUrl::RemoveTrailingSlash );
			break;
		case URL_DIR_OR_URL_ONLY:
			tmp_url = url; // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			ret = tmp_url.pathOrUrl() + url.upUrl().path( KUrl::RemoveTrailingSlash );
			break;
		case URL_FILE_NAME:
			ret = KIO::decodeFileName( url.fileName() );
			break;
		case URL_REPLACE:
			ret = KUrl::relativePath( rep.path( KUrl::RemoveTrailingSlash ), url.upUrl().path( KUrl::RemoveTrailingSlash ) );
			break;
		case URL_URL:
				ret = url.prettyUrl();
			break;
		default:
			ret = url.url();
			break;
	}

	return ret;
}

bool KomparatorWidget::isEnabled( LIST_VIEW list_view )
{
	if ( main_TabWidget->tabIcon( (int)list_view ).pixmap( main_TabWidget->iconSize() ).toImage() ==
		KIcon( "opinion-no" ).pixmap( main_TabWidget->iconSize() ).toImage() )
	{
		return false;
	}

	return true;
}

KIcon KomparatorWidget::broadIcon( const QPixmap pm1, const QPixmap pm2 )
{
	if ( pm1.size() != pm2.size() ) return KIcon();
	QPixmap ret_pm( 3 * pm1.width(), pm1.height() );
	ret_pm.fill( Qt::transparent );

	QPainter p( &ret_pm );

	p.drawPixmap( QRect( 0, 0, pm1.width(), pm1.height() ), pm1 );
	p.drawPixmap( QRect( 2 * pm1.width(), 0, pm1.width(), pm1.height() ), pm2 );

	p.end();

	return KIcon( ret_pm );
}

bool KomparatorWidget::isSystrayVisible()
{
	if ( !m_systray ) return false;
	if ( !m_systray->isVisible() ) return false;
	return true;
}

void KomparatorWidget::slotCount( int list_view, int which, int visible_items )
{
	switch( list_view )
	{
		case DUPLICATES:
			if ( which == 1 )
			{
				duplicates1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? duplicates1_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == duplicates1_KListView->topLevelItemCount() )
				{
					QPalette pal;
					duplicates1_KomparatorFileFilter->setPalette( pal );
					duplicates1ItemCount_TextLabel->setPalette( pal );
					duplicates1ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = duplicates1_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					duplicates1_KomparatorFileFilter->setPalette( pal );

					pal = duplicates1ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					duplicates1ItemCount_TextLabel->setPalette( pal );
					duplicates1ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			else if ( which == 2 )
			{
				duplicates2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? duplicates2_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == duplicates2_KListView->topLevelItemCount() )
				{
					QPalette pal;
					duplicates2_KomparatorFileFilter->setPalette( pal );
					duplicates2ItemCount_TextLabel->setPalette( pal );
					duplicates2ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = duplicates2_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					duplicates2_KomparatorFileFilter->setPalette( pal );

					pal = duplicates2ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					duplicates2ItemCount_TextLabel->setPalette( pal );
					duplicates2ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			break;
		case MISSING:
			if ( which == 1 )
			{
				missing1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? missing1_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == missing1_KListView->topLevelItemCount() )
				{
					QPalette pal;
					missing1_KomparatorFileFilter->setPalette( pal );
					missing1ItemCount_TextLabel->setPalette( pal );
					missing1ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = missing1_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					missing1_KomparatorFileFilter->setPalette( pal );

					pal = missing1ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					missing1ItemCount_TextLabel->setPalette( pal );
					missing1ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			else if ( which == 2 )
			{
				missing2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? missing2_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == missing2_KListView->topLevelItemCount() )
				{
					QPalette pal;
					missing2_KomparatorFileFilter->setPalette( pal );
					missing2ItemCount_TextLabel->setPalette( pal );
					missing2ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = missing2_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					missing2_KomparatorFileFilter->setPalette( pal );

					pal = missing2ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					missing2ItemCount_TextLabel->setPalette( pal );
					missing2ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			break;
		case NEWER:
			if ( which == 1 )
			{
				newer1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? newer1_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == newer1_KListView->topLevelItemCount() )
				{
					QPalette pal;
					newer1_KomparatorFileFilter->setPalette( pal );
					newer1ItemCount_TextLabel->setPalette( pal );
					newer1ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = newer1_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					newer1_KomparatorFileFilter->setPalette( pal );

					pal = newer1ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					newer1ItemCount_TextLabel->setPalette( pal );
					newer1ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			else
			{
				newer2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? newer2_KListView->topLevelItemCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == newer2_KListView->topLevelItemCount() )
				{
					QPalette pal;
					newer2_KomparatorFileFilter->setPalette( pal );
					newer2ItemCount_TextLabel->setPalette( pal );
					newer2ItemCount_TextLabel->setAutoFillBackground( false );
				}
				else
				{
					QPalette pal = newer2_KomparatorFileFilter->palette();
					pal.setColor( QPalette::Button, QColor( 110, 145, 200 ) );
					newer2_KomparatorFileFilter->setPalette( pal );

					pal = newer2ItemCount_TextLabel->palette();
					pal.setColor( QPalette::Window, QColor( 110, 145, 200 ) );
					newer2ItemCount_TextLabel->setPalette( pal );
					newer2ItemCount_TextLabel->setAutoFillBackground( true );
				}
			}
			break;
		default:
			break;
	}
}



/****************************************
* KQuery Functions                     *
****************************************/

void KomparatorWidget::slotAddFile( const KFileItem* item, const QString & )
{
	KFileItemExt *new_item;
	KIO::UDSEntry entry;
	KUrl parent_url = ( m_status == SEARCHING_URL_1 ) ? m_url1 : m_url2;

	if ( item->isFile() && ( followSoftLinks_CheckBox->isChecked() || !item->isLink() ) )
	{ // Real m_files are added always, (soft)links only if user wants to.
		if ( !item->isReadable() )
		{
			// The last string is the filename as it might contain "%2" for example, which makes "arg()" insert its string at the wrong position
			m_unreadable_files.append( QString( "%3  %2  %1" ).arg( item->permissionsString() ).arg(
				item->user() + ":" + item->group() ).arg( item->url().prettyUrl() ) );
		}
		else
		{
			m_number_of_files++;
			new_item = new KFileItemExt( *item );

			if ( m_status == SEARCHING_URL_1 )
			{
				new_item->dir = 0;
				new_item->virtual_parent_path = parent_url;
				new_item->parent_path = parent_url;
			}
			else
			{
				new_item->dir = 1;
				if ( !m_virtual_subdirectory_current.isEmpty() )
				{
					new_item->parent_path = m_virtual_subdirectory_current.upUrl();
				}
				else
				{
					new_item->parent_path = parent_url;
				}
				new_item->virtual_parent_path = parent_url;
			}

			new_item->next = m_files;

			m_files = new_item;
		}
		if ( findEmptyDirectories_CheckBox->isChecked() )
		{
			KUrl tmp_url = item->url();
			m_interaction_empty_dirs_file_urls.push_back( tmp_url );
		}
	}
	else if ( item->isDir() && ( followSoftLinks_CheckBox->isChecked() || !item->isLink() )  )
	{
		if ( findEmptyDirectories_CheckBox->isChecked() )
		{
			KUrl tmp_url = item->url();
			tmp_url.setPath( item->url().path( KUrl::AddTrailingSlash ) );
			tmp_url.setFileName( "a" ); // The filename doesn't matter, as "checkEmptyDirs" uses "upUrl". But it must be present.
			m_interaction_empty_dirs.push_back( tmp_url );
		}

		if ( item->isLink() && followSoftLinks_CheckBox->isChecked() )
		{
			bool link_is_parent = false;

			KUrl recursive_parent( item->url().url() );
			recursive_parent.setFileName( "" );
			recursive_parent.addPath( item->linkDest() );
			recursive_parent.cleanPath();

			KIO::UDSEntry entry;
			KIO::NetAccess::stat( recursive_parent, entry, this );
			KFileItem recursive_parent_item( entry, recursive_parent );

			if ( recursive_parent.isParentOf( item->url() ) ) link_is_parent = true;

			while ( recursive_parent_item.isLink() && !link_is_parent )
			{
				recursive_parent = recursive_parent_item.url().url();
				recursive_parent.setFileName( "" );
				recursive_parent.addPath( recursive_parent_item.linkDest() );
				recursive_parent.cleanPath();

				KIO::NetAccess::stat( recursive_parent, entry, this );
				recursive_parent_item = KFileItem( entry, recursive_parent );

				if ( recursive_parent.isParentOf( item->url() ) ) link_is_parent = true;
			}

			if ( link_is_parent )
			{
				kDebug() << "Recursive links detected. " << item->url() << " is a link to "
					<< recursive_parent.url() << ", which is a parent of the item.";
				QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
				KMessageBox::information( this, i18n(
					"Not recursing into symlinked directory %1 because it is a link to its parent directory %2." ).arg(
					item->url().prettyUrl() ).arg( recursive_parent.prettyUrl() ), i18n( "Recursive links detected" ) );
				QApplication::restoreOverrideCursor();
			}
			else
			{
				m_symlinked_directories.append( KUrl( item->url() ) );
			}
		}
	}
}

void KomparatorWidget::slotResult( int error_code, QString error_string )
{
	if ( error_code == 0 )
	{
		switch( m_status )
		{
			case SEARCHING_URL_1:
			{
				if ( !m_symlinked_directories.isEmpty() && includeSubdirectories_CheckBox->isChecked() ) // Search symlinked directories before going over to url 2.
				{
					m_query->setPath( m_symlinked_directories.front() );
					m_symlinked_directories.pop_front();
					m_query->start();
					return;
				}

				search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
				search2_KLed->setColor( QColor( 255, 0, 0 ) );  // Red
				search3_KLed->setColor( QColor( 255, 0, 0 ) );

				m_status = SEARCHING_URL_2;

				m_query->setPath( m_url2 );

				emit( signalProgress( i18n( "Searching right url for files..." ), -1, "" ) );

				m_symlinked_directories.clear();

				m_query->start();
				break;
			}
			case SEARCHING_URL_2:
			{
				if ( !m_symlinked_directories.isEmpty() && includeSubdirectories_CheckBox->isChecked() )
				{
					m_query->setPath( m_symlinked_directories.front() );
					m_symlinked_directories.pop_front();
					m_query->start();
					return;
				}
				if ( !m_virtual_subdirectories.isEmpty() && includeSubdirectories_CheckBox->isChecked() ) // Virtual subdirectories only exist for the right Url.
				{
					m_query->setPath( m_virtual_subdirectories.front() );
					m_virtual_subdirectory_current = m_virtual_subdirectories.front();
					m_virtual_subdirectories.pop_front();
					m_query->start();
					return;
				}
				m_virtual_subdirectory_current = KUrl( "" );

				m_progress_timer->stop();

				m_status = SEARCHING_EMPTY_DIRECTORIES;

				emit( signalProgress( i18n( "Searching for empty directories..." ), -1, "" ) );

				if ( m_interaction_empty_dirs.count() > 0 )
				{
					checkEmptyDirs( m_interaction_empty_dirs, m_interaction_empty_dirs_file_urls );
					break;
				}
			}

			case SEARCHING_EMPTY_DIRECTORIES:
			{
				m_interaction_empty_dirs.clear();
				m_interaction_empty_dirs_file_urls.clear();
				m_virtual_subdirectories.clear();

				search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
				search2_KLed->setColor( QColor( 0, 255, 0 ) );
				search3_KLed->setColor( QColor( 255, 0, 0 ) );  // Red

				emit( signalProgress( "", -1, "" ) );

				compare_job->initialize( m_files, ignoreEmptyFiles_CheckBox->isChecked(), size_CheckBox->isChecked(),
					calculateChecksum_CheckBox->isChecked(), binaryComparison_CheckBox->isChecked(),
					binaryComparison_KIntSpinBox->value(), m_number_of_files, isEnabled( DUPLICATES ),
					isEnabled( MISSING ) && !duplicatesOnly_CheckBox->isChecked(), isEnabled( NEWER ) && !duplicatesOnly_CheckBox->isChecked(),
					caseSensitiveSearch_CheckBox->isChecked() );

				m_status = COMPARING;

				compare_job->start();
				break;
			}

			default:
				break;
		}
	}
	else if ( error_code == KIO::ERR_USER_CANCELED )
	{
		if ( m_status != CLEAN ) slotSearch();
		return; // cancel is handled in slotSearch()
	}
	else if ( error_code == KIO::ERR_MALFORMED_URL )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		if ( m_status == SEARCHING_URL_1 ) KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		else if ( m_status == SEARCHING_URL_2 ) KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		slotSearch(); // Like clicking cancel button.
		return;
	}
	else
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, error_string );
		QApplication::restoreOverrideCursor();
		slotSearch(); // Like cancel
		return;
	}
}

bool KomparatorWidget::initializeQuery()
{
  KIO::filesize_t size;
  KIO::filesize_t sizeunit;

	// only start if we have valid dates
	if ( !isDateValid() ) return false;

	if ( url1_KUrlRequester->url().isEmpty() && !duplicatesOnly_CheckBox->isChecked() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}
	else if ( url2_KUrlRequester->url().isEmpty() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	KUrl url1 = url1_KUrlRequester->url();
	KUrl url2 = url2_KUrlRequester->url();
	url1.adjustPath( KUrl::RemoveTrailingSlash );
	url2.adjustPath( KUrl::RemoveTrailingSlash );

	m_virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();

	KIO::UDSEntry entry;
	if ( !duplicatesOnly_CheckBox->isChecked() )
	{
		KIO::NetAccess::stat( url1, entry, this );
	}
	KFileItem url1_item( entry, url1 );

	KIO::NetAccess::stat( url2, entry, this );
	KFileItem url2_item( entry, url2 );

	if ( url1_item.isLocalFile() && url1_item.isReadable() )
	{
		KMimeType::Ptr type1 = url1_item.determineMimeType();

		if      ( type1->is( "application/x-tgz" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-tbz" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-tar" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-zip" ) ) url1.setProtocol( "zip" );
		// else we don't know the correct protocol. this doesn't mean, it cannot be searched by kquery.
		// therefore we don't give an error message here, but let kquery fail if it doesn't work.
	}

	if ( url2_item.isLocalFile() && url2_item.isReadable() )
	{
		KMimeType::Ptr type2 = url2_item.determineMimeType();

		if      ( type2->is( "application/x-tgz" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-tbz" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-tar" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-zip" ) ) url2.setProtocol( "zip" );
		// else we don't know the correct protocol. this doesn't mean, it cannot be searched by kquery.
		// therefore we don't give an error message here, but let kquery fail if it doesn't work.
	}

	url1.cleanPath();
	url2.cleanPath();

	if ( url1_item.localPath() != "" ) url1_KUrlRequester->setUrl( url1_item.localPath() );
	else url1_KUrlRequester->setUrl( url1.pathOrUrl() );
	if ( url2_item.localPath() != "" ) url2_KUrlRequester->setUrl( url2_item.localPath() );
	else url2_KUrlRequester->setUrl( url2.pathOrUrl() );

	if ( !url1_item.isReadable() ) // it can be also not a dir to be a valid search path. e.g. media:/sd**/file.zip
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}
	else if ( !url2_item.isReadable() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	m_url1 = url1;
	m_url2 = url2;

	m_query->setPath( m_url1 );

	QString regexContains = regexContains_KLineEdit->text().isEmpty() ? "*" : regexContains_KLineEdit->text();
	QString regexContainsNot = regexContainsNot_KLineEdit->text();
	m_query->setRegExp( regexContains, regexContainsNot, caseSensitiveSearch_CheckBox->isChecked() );

	m_query->setRecursive( includeSubdirectories_CheckBox->isChecked() );

	switch ( bytes_QComboBox->currentIndex() )
	{
		case 0:
			sizeunit = 1; // one byte
			break;
		case 2:
			sizeunit = 1048576; // 1M
			break;
		case 1:
		default:
			sizeunit = 1024; // 1k
			break;
	}
	size = bytes_KIntNumInput->value() * sizeunit;

	m_query->setSizeRange( size_QComboBox->currentIndex(), size, 0 );

	// dates
	QDateTime epoch;
	epoch.setTime_t( 0 );

	// Add date predicate
	if ( !modifiedNoRestriction_RadioButton->isChecked() ) // Modified
	{
		if ( modifiedBetween_RadioButton->isChecked() ) // Between dates
		{
			QDate q1, q2;
			modifiedFrom_KDateCombo->getDate( &q1 );
			modifiedTo_KDateCombo->getDate( &q2 );

			// do not generate negative numbers .. find doesn't handle that
			time_t time1 = epoch.secsTo( QDateTime( q1 ) );
			time_t time2 = epoch.secsTo( QDateTime( q2 ).addDays( 1 ) ) - 1; // Include the last day

			m_query->setTimeRange( time1, time2 );
		}
		else
		{
			time_t cur = time( NULL );
			time_t minutes = cur;

			switch ( modifiedLastTime_QComboBox->currentIndex() )
			{
					case 0: // minutes
						minutes = modifiedLastTime_KIntNumInput->value();
						break;
					case 1: // hours
						minutes = 60 * modifiedLastTime_KIntNumInput->value();
						break;
					case 2: // days
						minutes = 60 * 24 * modifiedLastTime_KIntNumInput->value();
						break;
					case 3: // months
						minutes = 60 * 24 * (time_t)( modifiedLastTime_KIntNumInput->value() * 30.41667 );
						break;
					case 4: // years
						minutes = 12 * 60 * 24 * (time_t)( modifiedLastTime_KIntNumInput->value() * 30.41667 );
						break;
					default:
						break;
			}

			m_query->setTimeRange( cur - minutes * 60, 0 );
		}
	}
	else
		m_query->setTimeRange( 0, 0 );

	m_query->setUsername( "" );
	m_query->setGroupname( "" );

	m_query->setFileType( 0 );

	m_query->setMimeType( QStringList() );

	// Use locate to speed-up search ?
	m_query->setUseFileIndex( useFilesIndex_CheckBox->isChecked() );

	m_query->setShowHiddenFiles( includeHiddenItems_CheckBox->isChecked() );

	return true;
}

bool KomparatorWidget::isDateValid()
{
	// All m_files
	if ( modifiedNoRestriction_RadioButton->isChecked() ) return true;

	if ( modifiedLastTime_RadioButton->isChecked() )
	{
		if (modifiedLastTime_KIntNumInput->value() > 0 ) return true;

		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( this, i18n( "Unable to search within a period which is less than a minute." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	// If we can not parse either of the dates or
	// "from" date is bigger than "to" date return false.
	QDate hi1, hi2;

	QString str;
	if ( ! modifiedFrom_KDateCombo->getDate(&hi1).isValid() ||
			! modifiedTo_KDateCombo->getDate(&hi2).isValid() )
		str = i18n( "The date is not valid." );
	else if ( hi1 > hi2 )
		str = i18n( "Invalid date range." );
	else if ( QDate::currentDate() < hi1 )
		str = i18n( "Unable to search dates in the future." );

	if ( !str.isNull() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::sorry( 0, str );
		QApplication::restoreOverrideCursor();
		return false;
	}

	return true;
}



/****************************************
* Interaction Slots                     *
****************************************/

void KomparatorWidget::slotInteractionDuplicate( KFileItemExt *file )
{
	(new KListViewItemDups( duplicates1_KListView, file ))->setUseColor( textColor_CheckBox->isChecked() );
	emit( signalCount( DUPLICATES, 1, -1 ) );
}

void KomparatorWidget::slotInteractionCompMissing( KFileItemExt *file )
{
	int dir = file->dir;
	KListViewItemSingle *new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? missing1_KListView : missing2_KListView, file, NULL );
	new_item_single1->setText( 1, prettyUrl( file->url(), URL_REPLACE, file->parent_path ) );
	emit( signalCount( MISSING, ( dir == 0 ) ? 1 : 2, -1 ) );
}

void KomparatorWidget::slotInteractionCompNewer( KFileItemExt *file1, KFileItemExt *file2 )
{
	int dir = file1->dir;

	KListViewItemSingle *new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? newer1_KListView : newer2_KListView, file1, NULL );
	new_item_single1->setText( 1, prettyUrl( file1->url(), URL_REPLACE, file1->parent_path ) );

	KListViewItemSingle *new_item_single2 = new KListViewItemSingle( ( dir == 0 ) ? newer2_KListView : newer1_KListView, file2, new_item_single1 );
	new_item_single2->setText( 1, prettyUrl( file2->url(), URL_REPLACE, file2->parent_path ) );

	new_item_single1->brother = new_item_single2;
	emit( signalCount( NEWER, 1, -1 ) );
	emit( signalCount( NEWER, 2, -1 ) );

	new_item_single1->setKind( NEWER_REAL, textColor_CheckBox->isChecked() );
	new_item_single2->setKind( NEWER_REAL, textColor_CheckBox->isChecked(), m_blue );
}

void KomparatorWidget::slotInteractionCompNewerEqual( KFileItemExt *file1, KFileItemExt *file2 )
{
	int dir = file1->dir;

	KListViewItemSingle *new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? newer1_KListView : newer2_KListView, file1, NULL );
	new_item_single1->setText( 1, prettyUrl( file1->url(), URL_REPLACE, file1->parent_path ) );

	KListViewItemSingle *new_item_single2 = new KListViewItemSingle( ( dir == 0 ) ? newer2_KListView : newer1_KListView, file2, new_item_single1 );
	new_item_single2->setText( 1, prettyUrl( file2->url(), URL_REPLACE, file2->parent_path ) );

	new_item_single1->brother = new_item_single2;
	emit( signalCount( NEWER, 1, -1 ) );
	emit( signalCount( NEWER, 2, -1 ) );

	new_item_single1->setKind( NEWER_EQUAL, textColor_CheckBox->isChecked() );
	new_item_single2->setKind( NEWER_EQUAL, textColor_CheckBox->isChecked(), m_purple );
}

void KomparatorWidget::slotInteractionCompNewerSameTime( KFileItemExt *file1, KFileItemExt *file2 )
{
	int dir = file1->dir;

	KListViewItemSingle *new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? newer1_KListView : newer2_KListView, file1, NULL );
	new_item_single1->setText( 1, prettyUrl( file1->url(), URL_REPLACE, file1->parent_path ) );

	KListViewItemSingle *new_item_single2 = new KListViewItemSingle( ( dir == 0 ) ? newer2_KListView : newer1_KListView, file2, new_item_single1 );
	new_item_single2->setText( 1, prettyUrl( file2->url(), URL_REPLACE, file2->parent_path ) );

	new_item_single1->brother = new_item_single2;
	emit( signalCount( NEWER, 1, -1 ) );
	emit( signalCount( NEWER, 2, -1 ) );

	new_item_single1->setKind( NEWER_SAME_TIME, textColor_CheckBox->isChecked(), m_red );
	new_item_single2->setKind( NEWER_SAME_TIME, textColor_CheckBox->isChecked(), m_red );
}

void KomparatorWidget::slotInteractionCompFinished()
{
	compare_job->wait();
	updateFilters();

	search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
	search2_KLed->setColor( QColor( 0, 255, 0 ) );
	search3_KLed->setColor( QColor( 0, 255, 0 ) );

	enable( true );
	resizeListViewColumns();

	if ( m_unreadable_files.count() > 0 )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ) );
		KMessageBox::informationList( this, ( m_unreadable_files.count() == 1 ) ?
			i18n( "1 file was unreadable, and will not be considered any futher." ) :
			i18n( "%1 files were unreadable, and will not be considered any futher." ).arg( m_unreadable_files.count() )
			+ " " + i18n( "Change the owner and/or permissions to fix that problem." ),
			m_unreadable_files ); // sorryList message box doesn't exist; errorList is probably too much.
		QApplication::restoreOverrideCursor();
	}
	m_unreadable_files = QStringList();

	m_status = READY;

	emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Search and comparison finished." ) ) );
	emit( signalStopSystrayMovie() );

	search_KPushButton->setText( i18n( "Clear" ) );
	search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
}

void KomparatorWidget::slotInteractionEmptyFinished()
{
	KIO::UDSEntry entry;
//	KIO::NetAccess::stat( ((KIO::SimpleJob*)job)->url(), entry, this );
	KUrl::List empty_directory_list = m_empty_directory_job->getEmptyDirectoryList();

	KUrl::List::Iterator it;
	for ( it = empty_directory_list.begin(); it != empty_directory_list.end(); ++it )
	{
		KFileItemExt *new_item_dupes = new KFileItemExt( entry, (*it) );
		new_item_dupes->dir = 0;
		if ( m_url2.isParentOf( new_item_dupes->url() ) ) new_item_dupes->dir = 1;

		KUrl::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
		KUrl::List::Iterator it2;
		for ( it2 = virtual_subdirectories.begin(); it2 != virtual_subdirectories.end(); ++it2 )
		{
			if ( (*it2).isParentOf( new_item_dupes->url() ) )
			{
				new_item_dupes->dir = 1;
				new_item_dupes->parent_path = new_item_dupes->virtual_parent_path;
				break;
			}
			else
			{
				new_item_dupes->parent_path = new_item_dupes->dir ? m_url2 : m_url1;
			}
		}
		new_item_dupes->virtual_parent_path = new_item_dupes->dir ? m_url2 : m_url1;

		new_item_dupes->isdupe_size = 1;

		if ( m_dirs_dupes )
		{
			m_dirs_dupes->hasdupes_size = 1;
			KFileItemExt *tmp_dupes = m_dirs_dupes->duplicates_size;
			m_dirs_dupes->duplicates_size = new_item_dupes;
			new_item_dupes->dup_size_parent = m_dirs_dupes;
			m_dirs_dupes->duplicates_size->duplicates_size = tmp_dupes;
			if ( m_dirs_dupes->duplicates_size->duplicates_size )
			{
				m_dirs_dupes->duplicates_size->hasdupes_size = 1;
				m_dirs_dupes->duplicates_size->duplicates_size->dup_size_parent = m_dirs_dupes->duplicates_size;
			}
		}
		else
		{
			m_dirs_dupes = new_item_dupes;
			if ( isEnabled( DUPLICATES ) )
			{
				(new KListViewItemDups( duplicates1_KListView, m_dirs_dupes ))->setUseColor( textColor_CheckBox->isChecked() );
				emit( signalCount( DUPLICATES, 1, -1 ) );
			}
		}
	}

	slotDuplicateSelected();
	if ( m_status == SEARCHING_EMPTY_DIRECTORIES ) slotResult( 0, "" );
	else
	{
		updateFilters();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Searching for empty directories finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );
		resizeListViewColumns();

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIcon( KIcon( SmallIcon( "edit-clear" ) ) );
		return;
	}
}

void KomparatorWidget::slotInteractionDownloadFile( KFileItemExt *file )
{
	KTemporaryFile *tmp_file = new KTemporaryFile;
	tmp_file->open();

	m_requested_download = file;
	file->setTmpFile( tmp_file );

	m_requested_download_job = KIO::file_copy( file->url(), tmp_file->fileName(), -1, KIO::HideProgressInfo | KIO::Overwrite );
	connect( m_requested_download_job, SIGNAL( result( KJob * ) ), this, SLOT( slotFileDownloaded( KJob * ) ) );
	m_requested_download_job->start();
}

void KomparatorWidget::slotInteractionProgressProgress( const QString &message, const int &progress )
{
	emit( signalProgress( message, progress, "" ) );
}

void KomparatorWidget::slotInteractionMessageError( const QString &message, const QString &title )
{
	KMessageBox::error( this, message, title );
}

void KomparatorWidget::slotFileDownloaded( KJob *job )
{
	if ( m_requested_download )
	{
		if ( job->error() != 0 )
		{
			emit signalAddErrorMessage( ((KIO::Job *)job)->detailedErrorStrings().at( 0 ) );
// 			((KIO::Job *)job)->ui()->setWindow( this );
// 			((KIO::Job *)job)->ui()->showErrorMessage();
		}

		m_requested_download->setDownloadError( job->error() );
	}
	else // no m_requested_download ??? that means, something is terribly wrong. might happen on "cancel". trying to delete temp file anyway.
	{
		KIO::NetAccess::del( ((KIO::FileCopyJob*)job)->destUrl().url(), this );
		KIO::NetAccess::del( ((KIO::FileCopyJob*)job)->destUrl().url() + ".part", this ); // FIXME: ugly.
	}

	m_requested_download = NULL;
	m_requested_download_job = NULL;
}

#include "komparatorwidget.moc"
