/***************************************************************************
 *   Copyright (C) 2006 by Conrad Hoffmann                                 *
 *   conrausch@gmx.de                                                      *
 *   http://conrausch.doesntexist.org                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* DESKLIST PLUGIN FOR THE KOPETE KDE INSTANT MESSENGER
 *
 * base class implementation
 *
 */

#include <qlayout.h>
#include <qtimer.h>
#include <qptrlist.h>

#include <kdebug.h>
#include <kapplication.h>
#include <kaboutdata.h>
#include <kaction.h>
#include <kgenericfactory.h>
#include <krootpixmap.h>
#include <kwin.h>
#include <klocale.h>
#include <kmainwindow.h>
#include <kpopupmenu.h>

#include <kopeteuiglobal.h>
#include <kopetemetacontact.h>
#include <kopetecontact.h>
#include <kopetecontactlist.h>
#include <kopeteonlinestatus.h>
#include <kopetegroup.h>

#include "kopetedesklistplugin.h"
#include "kopetedesklistitem.h"
#include "kopetedesklistgroup.h"
#include "kopete_desklist.h"
#include "kopetedesklistrootwidget.h"

typedef KGenericFactory<KopeteDesklistPlugin> KopeteDesklistPluginFactory;

static const KAboutData aboutdata( "kopete_desklist", I18N_NOOP( "KopeteDesklist" ) , "0.3.0" );
K_EXPORT_COMPONENT_FACTORY( kopete_desklist, KopeteDesklistPluginFactory( &aboutdata ) )


KopeteDesklistPlugin::KopeteDesklistPlugin( QObject* parent, const char* name, const QStringList& )
		: Kopete::Plugin( KopeteDesklistPluginFactory::instance(), parent, name ),
		m_mainWidget( 0L ),
		m_bufferedWidget( 0L )
{
	kdDebug() << k_funcinfo << "Loading DeskList Plugin..." << endl;
	
	if ( !pluginStatic_ )
	{
		pluginStatic_ = this;
	}

	KopeteDesklistKcfg::self()->readConfig();

	// add item to (contact) menu
	KActionMenu* desklistMenu = new KActionMenu( i18n( "Desklist Plugin" ), QString::fromLatin1( "desktop" ), actionCollection(), "desklistMenu" );
	m_toggleItemVisibility = new KToggleAction( i18n( "&Exclude this item from desklist" ), 0, this, SLOT( slotToggleItemVisibility() ), actionCollection() , "toggleItemVisibility" );
	desklistMenu->insert( m_toggleItemVisibility );
	QObject::connect ( Kopete::ContactList::self() , SIGNAL( selectionChanged() ) , this, SLOT( slotPrepareContextMenu() ) );
	setXMLFile( "desklistui.rc" );

	QTimer::singleShot( 1000, this, SLOT( init() ) );
}


KopeteDesklistPlugin::~KopeteDesklistPlugin()
{
	pluginStatic_ = 0L;
}


KopeteDesklistPlugin* KopeteDesklistPlugin::pluginStatic_ = 0L;


KopeteDesklistPlugin* KopeteDesklistPlugin::self()
{
	return pluginStatic_ ;
}


void KopeteDesklistPlugin::init()
{
	// init timer
	m_redrawTimer = new QTimer( this );
	QObject::connect( m_redrawTimer, SIGNAL( timeout() ), this, SLOT( slotRedraw() ) );
	// listen for changes
	QObject::connect( Kopete::ContactList::self(), SIGNAL( metaContactAdded( Kopete::MetaContact* ) ), this, SLOT( slotMetaContactAdded( Kopete::MetaContact* ) ) );
	QObject::connect( Kopete::ContactList::self(), SIGNAL( metaContactRemoved( Kopete::MetaContact* ) ), this, SLOT( slotMetaContactRemoved( Kopete::MetaContact* ) ) );
	QObject::connect( Kopete::ContactList::self(), SIGNAL( metaContactAddedToGroup( Kopete::MetaContact*, Kopete::Group* ) ), this, SLOT( slotGroupMembersChanged( Kopete::MetaContact*, Kopete::Group* ) ) );
	QObject::connect( Kopete::ContactList::self(), SIGNAL( metaContactRemovedFromGroup( Kopete::MetaContact*, Kopete::Group* ) ), this, SLOT( slotGroupMembersChanged( Kopete::MetaContact*, Kopete::Group* ) ) );
	// listen for config changes
	QObject::connect( this, SIGNAL( settingsChanged() ), this, SLOT( slotSettingsChanged() ) );

	// initialize internal contact list
	generateList();
}


void KopeteDesklistPlugin::aboutToUnload()
{
	kdDebug() << k_funcinfo << "Unloading DeskList..." << endl;
	if ( m_mainWidget != 0L )
	{
		// this should suffice, as the desklist is created w/ the Qt::WDestructiveClose flag
		m_mainWidget->close();
	}
	if ( m_bufferedWidget != 0L )
	{
		m_bufferedWidget->close();
	}
	emit readyForUnload();
}


void KopeteDesklistPlugin::generateList()
{
	kdDebug() << k_funcinfo << "Generating internal contact list..." << endl;

	// gather all contacts
	QPtrList<Kopete::MetaContact> metaList = Kopete::ContactList::self()->metaContacts();
	for ( Kopete::MetaContact* mc = metaList.first(); mc != 0L; mc = metaList.next() )
	{
		slotMetaContactAdded( mc );
	}
	kdDebug() << k_funcinfo << "Done adding contacts!" << endl;
	slotTriggerDelayedRedraw();
}


void KopeteDesklistPlugin::requestRedraw()
{
	slotTriggerDelayedRedraw();
}


void KopeteDesklistPlugin::requestGroupRefresh( Kopete::MetaContact* cause )
{
	QPtrList<Kopete::Group> groupList = cause->groups();
	for ( Kopete::Group* group = groupList.first(); group != 0L; group = groupList.next() )
	{
		refreshGroup( group );
	}
	slotTriggerDelayedRedraw();
}


void KopeteDesklistPlugin::refreshGroup( Kopete::Group* group )
{
	LinkedList* groupList = m_activeGroups.find( group->displayName() );
	if ( groupList != 0L )
	{
		groupList->clear();
		QPtrList<Kopete::MetaContact> metaList = group->members();
		for ( Kopete::MetaContact* mc = metaList.first(); mc != 0L; mc = metaList.next() )
		{
			groupList->insert( mc );
		}
	}
}


void KopeteDesklistPlugin::showContextMenu( Kopete::MetaContact* mc, const QPoint& position )
{
	kdDebug() << k_funcinfo << "Trying to show context menu..." << endl;
	QPtrList<Kopete::MetaContact> selection;
	QPtrList<Kopete::Group> groups;
	selection.append( mc );
	Kopete::ContactList::self()->setSelectedItems( selection, groups );

	KMainWindow *window = dynamic_cast<KMainWindow*>( Kopete::UI::Global::mainWidget() );
	if ( ! window )
	{
		kdError() << k_funcinfo << "Main window not found, unable to display context-menu; "
			<< "Kopete::UI::Global::mainWidget() = " << Kopete::UI::Global::mainWidget() << endl;
		return;
	}

	KPopupMenu *popup = dynamic_cast<KPopupMenu *>(	window->factory()->container( "contact_popup", window ) );
	if ( popup )
	{
		for ( uint i = 0; i < popup->count(); i++ )
		{
			int id = popup->idAt( i );
			QString s = popup->text( id ).remove( '&' ).stripWhiteSpace();
			if ( s == i18n( "Properties" ) || s == i18n( "Groups" ) )
			{
				popup->setItemEnabled( id, false );
			}
		}
		m_popup = popup;

		QString title = i18n( "Translators: format: '<nickname> (<online status>)'", "%1 (%2)" ).
			arg( mc->displayName(), mc->statusString() );

		if ( title.length() > 43 )
			title = title.left( 40 ) + QString::fromLatin1( "..." );

		if ( popup->title( 0 ).isNull() )
			popup->insertTitle ( title, 0, 0 );
		else
			popup->changeTitle ( 0, title );

		// Submenus for separate contact actions
		bool sep = false;  //FIXME: find if there is already a separator in the end - Olivier
		QPtrList<Kopete::Contact> it = mc->contacts();
		for( Kopete::Contact *c = it.first(); c; c = it.next() )
		{
			if( sep )
			{
				popup->insertSeparator();
				sep = false;
			}

			KPopupMenu *contactMenu = it.current()->popupMenu();
			connect( popup, SIGNAL( aboutToHide() ), contactMenu, SLOT( deleteLater() ) );
			QString nick = c->property(Kopete::Global::Properties::self()->nickName()).value().toString();
			QString text = nick.isEmpty() ?  c->contactId() : i18n( "Translators: format: '<displayName> (<id>)'", "%2 <%1>" ). arg( c->contactId(), nick );
			text = text.replace( "&","&&" ); // cf BUG 115449

			if ( text.length() > 41 )
				text = text.left( 38 ) + QString::fromLatin1( "..." );

			popup->insertItem( c->onlineStatus().iconFor( c, 16 ), text , contactMenu );
		}

		connect( popup, SIGNAL( aboutToHide() ), this, SLOT( slotRestoreContextMenu() ) );
		popup->popup( position );
	}
}


void KopeteDesklistPlugin::slotPrepareContextMenu()
{
	// for now, only deal with single selections
	if ( Kopete::ContactList::self()->selectedMetaContacts().count() == 1 )
	{
		Kopete::MetaContact* mc = Kopete::ContactList::self()->selectedMetaContacts().first();
		if ( mc )
		{
			m_toggleItemVisibility->setChecked( string2bool( mc->pluginData( this, "excludeFromDesklist" ) ) );
		}
	}
	else if ( Kopete::ContactList::self()->selectedGroups().count() == 1 )
	{
		Kopete::Group* group = Kopete::ContactList::self()->selectedGroups().first();
		if ( group )
		{
			m_toggleItemVisibility->setChecked( string2bool( group->pluginData( this, "excludeFromDesklist" ) ) );
		}
	}
}


void KopeteDesklistPlugin::slotRestoreContextMenu()
{
	if ( m_popup )
	{
		for ( uint i = 0; i < m_popup->count(); i++ )
		{
			int id = m_popup->idAt( i );
			QString s = m_popup->text( id ).remove( '&' ).stripWhiteSpace();
			if ( s == i18n( "Properties" ) || s == i18n( "Groups" ) )
			{
				m_popup->setItemEnabled( id, true );
			}
		}
	}
}


void KopeteDesklistPlugin::slotToggleItemVisibility()
{
	// for now, only deal with single selections
	if ( Kopete::ContactList::self()->selectedMetaContacts().count() == 1 )
	{
		Kopete::MetaContact* mc = Kopete::ContactList::self()->selectedMetaContacts().first();
		if ( mc )
		{
			mc->setPluginData( this, "excludeFromDesklist", bool2string( m_toggleItemVisibility->isChecked() ) );
			slotTriggerDelayedRedraw();
		}
	}
	else if ( Kopete::ContactList::self()->selectedGroups().count() == 1 )
	{
		Kopete::Group* group = Kopete::ContactList::self()->selectedGroups().first();
		if ( group )
		{
			group->setPluginData( this, "excludeFromDesklist", bool2string( m_toggleItemVisibility->isChecked() ) );
			slotTriggerDelayedRedraw();
		}
	}
}


void KopeteDesklistPlugin::slotRedraw()
{
	kdDebug() << k_funcinfo << "Redrawing..." << endl;

	// now create a new widget in the buffer
	m_bufferedWidget = new KopeteDesklistRootWidget();

	// clear the current item list
	m_activeItems.clear();

	QDictIterator<LinkedList> it( m_activeGroups );
	for( ; it.current(); ++it )
	{
		if ( it.current()->isVisible() && ( ! string2bool( it.current()->group()->pluginData( this, "excludeFromDesklist" ) ) ) )
		{
			bool showGroupMembers = true;
			if ( KopeteDesklistKcfg::showGroupLabels() )
			{
				KopeteDesklistGroup* groupItem = new KopeteDesklistGroup( it.current()->group(), m_bufferedWidget );
				m_bufferedWidget->rootLayout()->add( groupItem );
				groupItem->show();
				QObject::connect( groupItem, SIGNAL( stateChanged() ), this, SLOT( slotTriggerImmediateRedraw() ) );
				showGroupMembers = groupItem->isExpanded();
			}
			if ( showGroupMembers )
			{
				for ( Kopete::MetaContact* mc = it.current()->first(); mc; mc = it.current()->next() )
				{
					if ( mc->status() != Kopete::OnlineStatus::Offline && mc->status() != Kopete::OnlineStatus::Unknown )
					{
						if ( ! string2bool( mc->pluginData( this, "excludeFromDesklist" ) ) )
						{
							KopeteDesklistItem* item = new KopeteDesklistItem( mc, m_forceSingleLine, m_bufferedWidget );
							m_bufferedWidget->rootLayout()->add( item );
							m_activeItems.insert( mc->metaContactId(), item );
						}
					}
				}
			}
		}
	}
	m_bufferedWidget->reposition();

	// once we have a new widget in the buffer, switch to that one...
	if ( m_bufferedWidget != 0L )
	{
		if ( m_mainWidget != 0L )
		{
			m_bufferedWidget->stackUnder( m_mainWidget );
		}
		m_bufferedWidget->show();
		if ( m_mainWidget != 0L )
		{
			m_mainWidget->close( true );
		}
		m_mainWidget = m_bufferedWidget;
		m_bufferedWidget = 0L;
	}
}


void KopeteDesklistPlugin::slotSettingsChanged()
{
	kdDebug() << k_funcinfo << "Reloading settings..." << endl;
	KopeteDesklistKcfg::self()->readConfig();
	slotTriggerDelayedRedraw();
}


void KopeteDesklistPlugin::slotMetaContactAdded( Kopete::MetaContact* mc )
{
	if ( mc == Kopete::ContactList::self()->myself() )
	{
		kdDebug() << k_funcinfo << "Rejecting 'myself' meta contact." << endl;
		return;
	}

	QPtrList<Kopete::Group> groupList = mc->groups();
	for ( Kopete::Group* group = groupList.first(); group != 0L; group = groupList.next() )
	{
		LinkedList* list = m_activeGroups.find( group->displayName() );
		if ( list != 0L )
		{
			list->insert( mc );
		}
		else
		{
			list = new LinkedList( group );
			list->insert( mc );
			m_activeGroups.insert( group->displayName(), list );
		}
	}

	QObject::connect( mc, SIGNAL( onlineStatusChanged( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ), this, SLOT( slotEvaluateStatusChange( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ) );
	QObject::connect( mc, SIGNAL( contactAdded( Kopete::Contact* ) ), this, SLOT( slotTriggerDelayedRedraw() ) );
	QObject::connect( mc, SIGNAL( contactRemoved( Kopete::Contact* ) ), this, SLOT( slotTriggerDelayedRedraw() ) );
}


void KopeteDesklistPlugin::slotMetaContactRemoved( Kopete::MetaContact* mc )
{
	kdDebug() << k_funcinfo << "Meta contact removal detected" << endl;
	m_redrawTimer->stop();
	requestGroupRefresh( mc );
}


void KopeteDesklistPlugin::slotGroupMembersChanged( Kopete::MetaContact* mc, Kopete::Group* fromGroup )
{
	kdDebug() << k_funcinfo << "Group membership change detected" << endl;
	refreshGroup( fromGroup );
	slotTriggerImmediateRedraw();
}


void KopeteDesklistPlugin::slotTriggerImmediateRedraw()
{
	kdDebug() << k_funcinfo << "Immmediate redraw requested" << endl;
	if ( ! m_redrawTimer->isActive() )
	{
		kdDebug() << k_funcinfo << "Doing redraw." << endl;
		slotRedraw();
	}
}


void KopeteDesklistPlugin::slotTriggerDelayedRedraw()
{
	kdDebug() << k_funcinfo << "Delayed redraw requested" << endl;
	if ( ! m_redrawTimer->isActive() )
	{
		kdDebug() << k_funcinfo << "Initiating delayed redraw" << endl;
		// schedule a redraw
		m_redrawTimer->start( 1000, true );

		// and use the time to calculate group sizes
		int onlineContacts = 0;
		QDictIterator<LinkedList> it( m_activeGroups );
		for( ; it.current(); ++it )
		{
			int onlineGroupContacts = 0;
			for ( Kopete::MetaContact* mc = it.current()->first(); mc; mc = it.current()->next() )
			{
				if ( mc->status() != Kopete::OnlineStatus::Offline && mc->status() != Kopete::OnlineStatus::Unknown )
				{
					onlineGroupContacts++;
				}
			}
			it.current()->setOnlineCount( onlineGroupContacts );
			onlineContacts += onlineGroupContacts;
		}
		m_forceSingleLine = KopeteDesklistKcfg::enableForceSingleLine() && ( onlineContacts >= KopeteDesklistKcfg::forceSingleLineThreshold() );
		// TODO get screen height here and check for auto resizing
	}
}

void KopeteDesklistPlugin::slotEvaluateStatusChange( Kopete::MetaContact* mc, Kopete::OnlineStatus::StatusType status )
{
	if ( status != Kopete::OnlineStatus::Offline && status != Kopete::OnlineStatus::Unknown )
	{
		KopeteDesklistItem* item = m_activeItems.find( mc->metaContactId() );
		if ( item && item->isShown() )
		{
			item->updateMetaStatus();
		}
		else
		{
			slotTriggerDelayedRedraw();
		}
	}
	else
	{
		KopeteDesklistItem* item = m_activeItems.find( mc->metaContactId() );
		if ( item )
		{
			slotTriggerDelayedRedraw();
		}
	}
}


QString KopeteDesklistPlugin::bool2string( bool b )
{
	return ( b ? "true" : "false" );
}


bool KopeteDesklistPlugin::string2bool( QString s )
{
	return ( s == "true" ? true : false );
}


#include "kopetedesklistplugin.moc"

