//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KOPETESILC_SILCCHANNELCONTACT_H
#define KOPETESILC_SILCCHANNELCONTACT_H

//#include "silcaccount.h"
#include "silccontact.h"

class SilcAccount;
class SilcBuddyContact;

/**
 * @brief Reimplementation of Kopete::Contact class, used for SILC Channels
 *
 * Every channel of the connected SILC network will have it's very own
 * SilcChannelContact class associated as soon as it is known to
 * kopete_silc.  This is as soon as the users tries to either add a specific
 * channel to the contact list or otherwise joins a channel.
 *
 * The name of the represented channel is stored as the nickName() of this
 * contact.
 * 
 * @author Stefan Siegl <ssiegl@gmx.de>
 */
class SilcChannelContact : public SilcContact {
  Q_OBJECT;

public:
  SilcChannelContact(SilcAccount *, const QString &channelname, 
		     Kopete::MetaContact *meta,
		     const QString &icon = QString::null);
  virtual ~SilcChannelContact();
  
  /**
   * @brief return the primary message manager of this contact
   */
  virtual Kopete::ChatSession *manager(Kopete::Contact::CanCreateFlags =
				       Kopete::Contact::CannotCreate);

  /**
   * @brief join the channel
   */
  virtual void join(void);
  virtual void join(bool founder, bool auth, const QString &password = "");

  /**
   * @brief leave the channel (if joined)
   */
  virtual void leave(void);

  /** 
   * @brief invite buddy into this channel
   */
  void invite(const SilcBuddyContact *buddy);

  /**
   * @brief kick user from channel
   */
  void kick(SilcBuddyContact *buddy);

  /** 
   * @brief callback for libsilc to pass us an incoming channel message
   */
  static void silc_channel_message(SilcTK::SilcClient client, 
				   SilcTK::SilcClientConnection conn,
				   SilcTK::SilcClientEntry sender, 
				   SilcTK::SilcChannelEntry channel,
				   SilcTK::SilcMessagePayload payload,
				   SilcTK::SilcMessageFlags flags, 
				   const unsigned char *message,
				   SilcTK::SilcUInt32 message_len);

  static void silc_channel_message(SilcTK::SilcClient client,
				   SilcTK::SilcClientConnection conn,
				   SilcTK::SilcClientEntry sender,
				   SilcTK::SilcChannelEntry channel,
				   SilcTK::SilcMessagePayload payload,
				   SilcTK::SilcChannelPrivateKey key,
				   SilcTK::SilcMessageFlags flags, 
				   const unsigned char *message,
				   SilcTK::SilcUInt32 message_len);

  inline SilcTK::SilcChannelEntry channelEntry(void) 
  {
    return _channelEntry;
  }

  /**
   * @brief return the channel's topic
   */
  inline const QString topic(void) const 
  {
    return (_channelEntry && _channelEntry->topic) ? 
      QString::fromUtf8(_channelEntry->topic) : QString::null;
  }

  /**
   * @brief set channel topic
   */
  void setTopic(QString &);

  void setChannelEntry(SilcTK::SilcChannelEntry e);

  bool isOp(SilcBuddyContact *buddy);
  bool isFounder(SilcBuddyContact *buddy);

  inline bool isPrivate(void) const
  {
    return channelMode() & SILC_CHANNEL_MODE_PRIVATE;
  }
  inline bool isSecret(void) const
  {
    return channelMode() & SILC_CHANNEL_MODE_SECRET;
  }
  inline bool isInviteOnly(void) const
  {
    return channelMode() & SILC_CHANNEL_MODE_INVITE;
  }
  inline bool isSilenceUser(void) const
  {
    return channelMode() & SILC_CHANNEL_MODE_SILENCE_USERS;
  }
  inline bool isSilenceOperator(void) const
  {
    return channelMode() & SILC_CHANNEL_MODE_SILENCE_OPERS;
  }

  void setPrivate(bool);
  void setSecret(bool);
  void setInviteOnly(bool);
  void setSilenceUser(bool);
  void setSilenceOperator(bool);

  void setOp(SilcBuddyContact *buddy, bool);

  void setNickNameForMode(int mode);

  inline bool allowRichText(void) const { return _allowRichText; };
  inline void setAllowRichText(bool v) { _allowRichText = v; };

  virtual void serialize(QMap<QString, QString>&, QMap<QString, QString>&);

  /** 
   * @brief check whether the given buddy is joined into this channel
   */
  bool isJoined(SilcBuddyContact *buddy);

  /**
   * @brief display channel configuration dialog
   */
  virtual void slotUserInfo(void);
  
  /**
   * @brief send file as mime to contact
   */
  virtual void sendFileAsMime(const QString &fileName);
  
  virtual void sendFile(const KURL &sourceURL = KURL(),
			const QString &fileName = QString::null,
			uint fileSize = 0L);
private slots:
  /**
   * @brief send (outgoing) message to the SILC network
   */
  virtual void slotSendMessage(Kopete::Message &, Kopete::ChatSession *);

  /**
   * @brief autojoin channel when SilcAccount is going online
   */
  void slotOnlineStatusChanged(Kopete::Contact *contact,
			       const Kopete::OnlineStatus &status,
			       const Kopete::OnlineStatus &oldStatus);

public slots:
  void updateBuddyOnlineStatus(SilcBuddyContact *buddy);

private:
  /**
   * @brief buddies to be invited after join 
   */
  QPtrList<SilcBuddyContact> toInvite;

  /**
   * @brief mode to be set after join
   */
  SilcTK::SilcUInt32 modeToBeSet;

  const Kopete::OnlineStatus &getBuddyOnlineStatus(SilcBuddyContact *buddy);

  SilcTK::SilcChannelEntry _channelEntry;

  /**
   * @brief return the CU-Mode for buddy, -1 on error.
   */
  int channelUserMode(SilcBuddyContact *buddy);

  void commitModeChange(void);

  inline int channelMode(void) const
  {
    return _channelEntry ? _channelEntry->mode : 0;
  }

  bool _allowRichText;
};

#endif // KOPETESILC_SILCCHANNELCONTACT_H
