/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifndef koverartist_cddbquery_h
#define koverartist_cddbquery_h

#include "disc.h"

#include <qobject.h>


class KURL;
class CdInfo;
class QWidget;
class QStringList;
namespace KIO { class Job; }



/** Class for querying a cddb server.
 */
class CddbQuery: public QObject
{
   Q_OBJECT
   typedef QObject Inherited;

public:
   /** Create a query object with an optional parent window. */
   CddbQuery(QWidget* parent=0);
   virtual ~CddbQuery();

   /** Asynchronously start a cddb query for an inserted audio-cd.
    * Emits @ref finished when done.
    */
   virtual void cddbQueryAudioCd();

   /** Allows access to the internal disc object. This object is filled by
    * cddb queries, and you need to copy the results if needed.
    */
   KoverArtist::Disc& disc() { return mDisc; }
   const KoverArtist::Disc& disc() const { return mDisc; }

   /** Scan the audio-cd. Creates the disc-info object, @see discInfo. */
   bool readCd();

   /** Decode cddb-server site url. */
   bool fromUrl(const KURL& url);

   /** Allows access to the internal disc-info object. Can be NULL. */
   CdInfo* discInfo() { return mDiscInfo; }
   const CdInfo* discInfo() const { return mDiscInfo; }

   /** Test if the given url could be used for a cddb request. */
   static bool canDecode(const KURL& url);

signals:
   /** Emitted to inform the user what is going on. */
   void status(const QString&);

   /** Emitted when an asynchronous job ended. */
   void finished();

protected slots:
   virtual void cddbDataArrived(KIO::Job* aJob, const QByteArray& aData);
   virtual void cddbResult(KIO::Job* aJob);

protected:
   virtual void startCddbQuery(const QString& cmd);
   virtual void parseCddbEntry(const QStringList& lines);
   virtual void cddbFetchDetails(const QString& genre, const QString& cddbId,
                                 const QString& title);
   virtual QString handleInexactMatch(const QStringList& matches) const;

   virtual bool testBusy() const;
   virtual void splitString(QStringList& result, const QString& str,
                            const QString& sep=" ", int elems=-1) const;

private:
   CddbQuery(const CddbQuery&);
   CddbQuery& operator=(const CddbQuery&);

protected:
   QWidget *mParent;
   KIO::Job *mJob;
   CdInfo *mDiscInfo;
   KoverArtist::Disc mDisc;
   QString mHello, mData;
   bool mActive;
};


#endif /*koverartist_cddbquery_h*/
