;;;; -*- mode: lisp -*-
;;;;
;;;; $Id: multipart-form-encoding.lisp,v 1.2 2004/12/16 15:08:36 sven Exp $
;;;;
;;;; Some utilities to deal with multipart form encoded data
;;;;
;;;; Copyright (C) 2004 Sven Van Caekenberghe, Beta Nine BVBA. All Rights Reserved.
;;;;
;;;; You are granted the rights to distribute and use this software
;;;; as governed by the terms of the Lisp Lesser GNU Public License
;;;; (http://opensource.franz.com/preamble.html), also known as the LLGPL.
;;;;

(in-package :kpax)

(export '(parse-multipart-header 
          extract-multipart-parts
          find-multipart-header-attribute
          find-multipart-header-named
          find-multipart-named))

(defun parse-multipart-header (string)
  "Parse a line of the form header-name: header-value; att1-name='att1-value'; att2-name='att2-value' into (header-name header-value ((att1-name . att1-value) (att2-name . att2-value)))"
  (let ((colon-position (position #\: string))
        header-name header-value attributes)
    (setf header-name (string-trim " " (subseq string 0 colon-position)))
    (let ((tokens (s-utils:tokens string :start (1+ colon-position) :separators ";")))
      (setf header-value (string-trim " " (pop tokens)))
      (dolist (token tokens)
        (let ((subtokens (s-utils:tokens token :separators "=")))
          (push (cons (string-trim " " (first subtokens))
                      (string-trim "\"'" (second subtokens)))
                attributes)))
      (append (list header-name header-value) attributes))))

(defun extract-multipart-parts (body-string)
  "Extract all parts of a multipart/form-data encoded body-string into ((part1-headers part1-data) ..)"
  (with-input-from-string (in body-string)
    (let ((boundary (string-right-trim '(#\Return #\Newline) (read-line in nil nil)))
          parts headers data line)
      (loop
       ;; parse and collect all header lines
       (setf headers nil)
       (loop
        (setf line (read-line in nil nil))
        (when line (setf line (string-right-trim '(#\Return #\Newline) line)))
        (cond ((null line) (return-from extract-multipart-parts parts))
              ((equal line "") (return))
              (t (push (parse-multipart-header line) headers))))
       ;; collect all data lines
       (setf data
             (string-right-trim '(#\Return #\Newline)
                                (with-output-to-string (out)
                                  (loop
                                   (setf line (read-line in nil nil))
                                   (cond ((or (null line) 
                                              (string= line boundary :end1 (min (length line) (length boundary))))
                                          (return))
                                         (t 
                                          (write-string line out) (terpri out)))))))
       ;; finish the part
       (push (list headers data) parts)))))

(defun find-multipart-header-attribute (attribute-name multipart-header)
  "Find the value of a named attribute in a multipart/form-data decoded header"
  (let ((attributes (rest (rest multipart-header))))
    (cdr (find attribute-name attributes :test #'string-equal :key #'first))))

(defun find-multipart-header-named (header-name multipart-headers)
  "Find a named header in a list of multipart/form-data decoded headers"
  (find header-name multipart-headers :test #'string-equal :key #'first))

(defun find-multipart-named (part-name parts)
  "Find a named part in a list of multipart/form-data decoded parts"
  (loop 
   :for (header data) :in parts 
   :do (let ((content-disposition-header (find-multipart-header-named "Content-Disposition" header)))
         (when content-disposition-header
           (let ((name (find-multipart-header-attribute "name" content-disposition-header)))
             (when (string-equal name part-name)
               (return-from find-multipart-named (list header data))))))))

;;;; eof