;;;; -*- mode: lisp -*-
;;;;
;;;; $Id: request-response.lisp,v 1.31 2005/04/08 09:13:00 sven Exp $
;;;;
;;;; Superclass of all request-response objects, defining part of the server interface
;;;;
;;;; Copyright (C) 2004 Sven Van Caekenberghe, Beta Nine BVBA. All Rights Reserved.
;;;;
;;;; You are granted the rights to distribute and use this software
;;;; as governed by the terms of the Lisp Lesser GNU Public License
;;;; (http://opensource.franz.com/preamble.html), also known as the LLGPL.
;;;;

(in-package :kpax)

(export
 '(request-response
   get-server
   get-application
   get-session
   get-uri
   get-uri-string
   get-request-sub-path
   get-request-header-value
   get-request-headers
   get-request-parameter-value
   get-request-parameter-values
   get-request-parameters
   get-request-body
   get-request-method
   get-request-ip-address
   get-content-stream
   get-response-header-value
   get-response-status
   get-response-mime-type
   get-cookie
   set-cookie
   static-url
   dynamic-url))

(defclass request-response (attributes-mixin)
  ((uri-string :initform nil)
   (uri :initform nil)
   (server :accessor get-server :initarg :server :initform nil)
   (application :accessor get-application :initform nil)
   (session :accessor get-session :initform nil)
   (sub-path :accessor get-request-sub-path :initform nil)
   (response-status :accessor get-response-status :initform :ok)
   (response-mime-type :accessor get-response-mime-type :initform "text/html")
   (response-headers :accessor get-response-headers :initform nil)
   (content-stream :reader get-content-stream :initform (make-string-output-stream)))
  (:documentation "I represent a request-response from a server"))

(defgeneric get-application (request-response)
  (:documentation "Get the web application to which this request-response was directed"))

(defgeneric get-server (request-response)
  (:documentation "Get the web application server handling this request-response"))

(defgeneric get-session (request-response)
  (:documentation "Get the session bound to this request-response"))

(defgeneric get-content-stream (request-response)
  (:documentation "Access the stream to write content to for this request-response's reply"))

(defgeneric get-response-mime-type (request-response)
  (:documentation "Get the mime-type for this request-response's reply"))

(defgeneric get-response-status (request-response)
  (:documentation "Get the statuc for this request-response's reply (:ok, :created, :not-found, :error or :moved)"))

(defgeneric get-uri-string (request-response)
  (:documentation "Get the URI string underlying this request-response"))

(defgeneric get-uri (request-response)
  (:documentation "Get the URI underlying this request-response")
  (:method ((request-response request-response))
   (with-slots (uri) request-response
     (or uri
         (setf uri (puri:parse-uri (get-uri-string request-response)))))))

(defgeneric get-request-body (request-response)
  (:documentation "Get the request data, 'body', from request-response as a (possibly empty) string"))

(defgeneric get-request-method (request-response)
  (:documentation "Get the request HTTP method, a keyword symbol most likely :get or :post (but extendible)"))

(defgeneric get-request-header-value (request-response header-name)
  (:documentation "Get the request header value for header-name in request-response"))

(defgeneric get-request-headers (request-response)
  (:documentation "Get the list of all header names in request-response"))

(defgeneric get-request-parameter-value (request-response parameter-name)
  (:documentation "Get the first request parameter value for parameter-name in request-response"))

(defgeneric get-request-parameter-values (request-response parameter-name)
  (:documentation "Get the list of request parameter values for parameter-name in request-response"))

(defgeneric get-request-parameters (request-response)
  (:documentation "Get the list of all request parameter names in request-response"))

(defgeneric get-request-ip-address (request-response)
  (:documentation "Get the ip address of the client of request-response as a dotted string"))

(defgeneric get-request-sub-path (request-response)
  (:documentation "Get the sub path list for this request (the uri components minus the server and web app prefixes)"))

(defgeneric get-response-header-value (request-response header-name)
  (:documentation "Get the (first) response header value for header-name in request-response")
  (:method ((request-response request-response) header-name)
   (cdr (assoc header-name (get-response-headers request-response) :test #'string-equal))))

(defgeneric get-response-header-values (request-response header-name)
  (:documentation "Get a list of all the response header values for header-name in request-response")
  (:method ((request-response request-response) header-name)
   (mapcar #'cdr (remove-if-not #'(lambda (header-name-value) (string-equal (car header-name-value) header-name))
                                (get-response-headers request-response)))))
 
(defgeneric (setf get-response-header-value) (value request-response header-name)
  (:documentation "Set the response header value for header-name in request-response to value")
  (:method (value (request-response request-response) header-name)
   (let ((pair (assoc header-name (get-response-headers request-response) :test #'string-equal)))
     (if pair
         (setf (cdr pair) value)
       (push (cons header-name value) (get-response-headers request-response)))
     value)))

(defgeneric add-response-header-value (request-response header-name value)
  (:documentation "Add a response header named header-name equal to value to request-response")
  (:method ((request-response request-response) header-name value)
   (push (cons header-name value) (get-response-headers request-response))))

(defgeneric commit (request-response)
  (:documentation "Commit the request-response"))

(defmethod print-object ((request-response request-response) stream)
  (print-unreadable-object (request-response stream :type t :identity t)
    (format stream "~s" (get-uri request-response))))

(defmethod reset-response ((request-response request-response))
  (setf (get-response-status request-response) :ok
        (get-response-mime-type request-response) "text/html"
        (get-response-headers request-response) nil)
  ;; read the output from the string stream to reset it
  (get-output-stream-string (get-content-stream request-response)))

(defgeneric static-url (request-response scope relative-url &rest args)
  (:documentation "Generate a static URL in the scope (:server :webapp) of request-response using relative-url and args")
  (:method ((request-response request-response) scope relative-url &rest args)
   (let ((prefix (get-complete-static-prefix scope
                                             (get-application request-response)
                                             (get-server request-response))))
     (format nil "~a~?" prefix relative-url args))))

(defun write-keyword-value-list (keyword-value-list out)
  (when keyword-value-list
    (let ((key (car keyword-value-list))
          (value (cadr keyword-value-list))
          (tail (cddr keyword-value-list)))
      (write key :stream out :case :downcase :escape nil)
      (write-char #\= out)
      (write-string (uri-encode-for-query (princ-to-string value)) out)
      (when tail
        (write-string "&amp;" out)
        (write-keyword-value-list tail out)))))

(defgeneric dynamic-url (request-response relative-url &rest args)
  (:documentation "Generate a dynamic URL in the context of request-response based on relative-url and args")
  (:method ((request-response request-response) relative-url &rest args)
   (let ((prefix (get-complete-dynamic-prefix (get-application request-response)
                                              (get-server request-response))))
     (with-output-to-string (out)
       (write-string prefix out)
       (append-session-id request-response out)
       (cond ((null relative-url))
             ((stringp relative-url) (format out "~?" relative-url args))
             ((symbolp relative-url) 
              (write relative-url :stream out :case :downcase :escape nil)
              (when args
                (write-char #\? out)
                (write-keyword-value-list args out)))
             (t (error "unknown relative-url type ~s" relative-url)))))))

(defmethod logm ((request-response request-response) level format-string &rest args)
  (apply #'logm (get-server request-response) level format-string args))

(defgeneric get-cookie (request-response name)
  (:documentation "Return the most specific string value of the cookie with name in request-response"))

(defgeneric set-cookie (request-response name value path &optional expires)
  (:documentation "Set the code with name and value for path in request-response (expires is :session or :never)"))

;;;; eof