;;;; -*- mode: lisp -*-
;;;;
;;;; $Id: kpax.lisp,v 1.13 2004/09/01 08:06:50 sven Exp $
;;;;
;;;; The main KPAX file, managing web-apps and kpax as a whole
;;;;
;;;; Copyright (C) 2004 Sven Van Caekenberghe, Beta Nine BVBA. All Rights Reserved.
;;;;
;;;; You are granted the rights to distribute and use this software
;;;; as governed by the terms of the Lisp Lesser GNU Public License
;;;; (http://opensource.franz.com/preamble.html), also known as the LLGPL.
;;;;

(in-package :kpax)

(export 
 '(init-kpax 
   destroy-kpax
   run-kpax 
   stop-kpax
   restart-kpax
   pause-kpax
   defwebapp 
   get-web-app
   map-web-apps
   start-web-app 
   stop-web-app 
   restart-web-app))

(defun get-web-app (name)
  "Get the web application known by name (return nil when not found)"
  (gethash name *web-apps*))

(defun map-web-apps (function)
  "Apply function on each defined web application and return the result"
  (loop :for web-app :being :the :hash-values :of *web-apps*
        :collect (funcall function web-app)))

(defun ensure-web-app (name options)
  "Either create a new web-app or use an existing one, resetting its options"
  (let ((web-app (get-web-app name)))
    (unless web-app
      (setf web-app (make-instance 'web-app :name name)
            (gethash name *web-apps*) web-app))
    (setf (get-option-list web-app) options)
    (process-option-list web-app)
    (ignore-errors 
      (stop-web-app name)
      (start-web-app name))
    web-app))

(defmacro defwebapp (name &rest options)
  "Define a web application by name with the options listed"
  `(ensure-web-app ,name 
                   ,(cons 'list (append (list :load-truename (or *load-truename* *compile-file-truename*))
                                        (list :load-package *package*)
                                        (reduce #'append options)))))

(defun start-web-app (name &rest options)
  "Start the web application known by name, installing and publishing it"
  (start (get-web-app name) options))

(defun stop-web-app (name &rest options)
  "Stop the web application known by name, making it no longer accessible"
  (stop (get-web-app name) options))

(defun restart-web-app (name &rest options)
  "Stop and start the web application known by name"
  (apply #'stop-web-app name options)
  (apply #'start-web-app name options))

(defun init-kpax (server-class &rest options)
  "Initialize the KPAX web application framework"
  (when (not (null *web-app-server*))
    (stop-kpax :force t))
  (destructuring-bind (&key debug &allow-other-keys)
      options
    (declare (ignore debug))
    (setf *web-app-server*
          (make-instance server-class))
    (init *web-app-server* options)))

(defun run-kpax (&rest options)
  "Run the KPAX web application server"
  (when (null *web-app-server*)
    (error "The KPAX web application server doesn't seem to be initialized (use INIT-KPAX)"))
  (startup *web-app-server* options))

(defun stop-kpax (&rest options)
  "Stop the KPAX web application server"
  (when (null *web-app-server*)
    (error "The KPAX web application server doesn't seem to be initialized (use INIT-KPAX)"))
  (shutdown *web-app-server* options))

(defun restart-kpax (&rest options)
  "Restart the KPAX web application server"
  (apply #'stop-kpax options)
  (apply #'run-kpax options))

(defun destroy-kpax (&rest options)
  "Stop and remove the the KPAX web application server"
  (when *web-app-server*
    (shutdown *web-app-server* options)
    (setf *web-app-server* nil)
    :destroyed))

(defun pause-kpax (&rest options)
  "Take all KPAX web applications temporarily off-line, showing a 'mainteance in progress' page"
  (when (null *web-app-server*)
    (error "The KPAX web application server doesn't seem to be initialized (use INIT-KPAX)"))
  (pause *web-app-server* options))

;;;; eof