/***************************************************************************
 *   Copyright (C) 2005 by Danny Kukawka                                   *
 *                         <dkukawka@suse.de>, <danny.kukawka@web.de>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/
 
/*! \file configuredialog.cpp
 * All here displayed file members of configureDialog.cpp are related to operations with the 
 * configure dialog for kpowersave
 * \brief 	In this file can be found all configure dialog related code. 
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \version 	0.6.0
 * \date    	2005
 * \todo	fix dpms spinbox handling. be sure that e.g. value for poweroff
 *		could not be lower than the calue for suspend.
 */

 // KDE - Headers
 #include <kaudioplayer.h>
 #include <kconfig.h>
 #include <kiconloader.h>
 #include <klocale.h>
 #include <kmessagebox.h>

 // QT - Headers
 #include <qcheckbox.h>
 #include <qcombobox.h>
 #include <qdialog.h>
 #include <qgroupbox.h>
 #include <qlabel.h>
 #include <qlineedit.h>
 #include <qlistbox.h>
 #include <qpushbutton.h>
 #include <qslider.h>
 #include <qspinbox.h>
 #include <qstringlist.h>
 #include <qtabwidget.h>
 #include <qtoolbox.h>
 #include <qtooltip.h>
 #include <qstring.h>

 #include "configuredialog.h"

/*! This is the default constructor of the class ConfigureDialog. */
ConfigureDialog::ConfigureDialog( KConfig *config, pDaemon *pd, QStringList schemes, QStringList _actions, 
			          QString _currentScheme, QWidget *parent, const char *name)
				 :configure_Dialog(parent, name, false, WDestructiveClose )
{
	settings = config;
	powersave_schemes = schemes;
	actions = _actions;
	pdaemon = pd;	

	QString session = getenv("DESKTOP_SESSION");
	if(session.startsWith("gnome")) gnome_session = true;
        else gnome_session = false;

	initalised = false; 
	general_changed = false;
	scheme_changed = false;
	displayed_WARN_autosuspend = false;

	// check if brightness is supported
	if((brightnessLevels = pdaemon->getBrightnessLevels()) > 0) {
		brightness_supported = true;
		brightness_last = pdaemon->getBrightness(true);
	}
	else { 
		brightness_supported = false;
		brightness_last = -1;
	}
	brightness_changed = false;

	currentScheme = -1;


	setIcons();
	setTooltips();
	setGeneralSettings();
	setSchemeList();
	setInactivityBox();
	
	// select the current scheme in the listbox
	if(!_currentScheme.isNull()) {
		int pos = schemes.findIndex(_currentScheme);
		if(pos > -1) {
			listBox_schemes->setCurrentItem(pos);
			currentScheme = pos;
		}
		else listBox_schemes->setCurrentItem(0);
	}
	else listBox_schemes->setCurrentItem(0);
	
	tL_valueBrightness->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	tL_valueBrightness->setBackgroundMode(Qt::PaletteBase);
	tL_valueBrightness->setAlignment(Qt::AlignCenter);
	tL_valueBrightness->setFocusPolicy(NoFocus);

	// hide Lid item if this is not a laptop
	if (!pd->isLaptop()) cB_lockLid->hide();

	initalised = true;

	QString dummy;
	// set brightness dummys 
	dummy = i18n("If the current desktop user is inactive, dim the display to:");
	dummy = i18n("Enable dim display on inactivity");
	dummy = i18n("Blacklist");
	dummy = i18n("Here you can add programs which should, if running, prevent the dimming of the display.");
	dummy = i18n("Would you like to import a predefined blacklist?");
	// CPU shutdown strings:
	dummy = i18n("Disable CPUs/Cores");
	dummy = i18n("Max. running CPUs:");
	dummy = i18n("Max. running CPUs/Cores:");
	dummy = i18n("Min. running CPUs:");
	dummy = i18n("Min. running CPUs/Cores:");
	dummy = i18n("Enable to switch off CPUs/cores");
	dummy = i18n("You have a multiprocessor/multicore machine.");
	dummy = i18n("You can disable CPUs/cores to reduce power consumption and save battery power.");
	// Advanced Powersave dummys
	// simple words
	dummy = i18n("Device");
	dummy = i18n("Devices");
	dummy = i18n("Device class");
	dummy = i18n("activate");
	dummy = i18n("Activate");
	dummy = i18n("deactivate");
	dummy = i18n("Deactivate");
	dummy = i18n("activated");
	dummy = i18n("deactivated");
	dummy = i18n("do nothing");
	// complexer
	dummy = i18n("Deactivate following devices:");
	dummy = i18n("Activate following devices");
	dummy = i18n("Reactivate following devices");
	dummy = i18n("Deactivate following device classes:");
	dummy = i18n("Activate following devices classes");
	dummy = i18n("Reactivate following device classes");
	dummy = i18n("If the scheme switched all devices are again activated.");
	dummy = i18n("This is a experimental feature.");
	dummy = i18n("If you have problems with this feature, please report them.");
	dummy = i18n("Select one of the available devices and click on ");
	dummy = i18n("Select one of the available device classes and click on ");
	dummy = i18n("Select one or more of the available devices and click on ");
	dummy = i18n("Select one or more of the available device classes and click on ");
	dummy = i18n("Please note: If you e.g. deactivate a network device you may lose your internet connection.");
}

/*! This is the default destructor of the class ConfigureDialog. */
ConfigureDialog::~ConfigureDialog(){
	 // no need to delete child widgets, Qt does it all for us
}

/* ---- START General setup SECTION ---- */

/*! 
 * This used to fill the QListBox listBox_schemes with the existing
 * schemes.
 */
 void ConfigureDialog::setSchemeList(){
	
	for ( QStringList::Iterator it = powersave_schemes.begin(); it != powersave_schemes.end(); ++it ) {
		QString _tmp = *it;
		if(_tmp == "Performance" || _tmp == i18n("Performance"))
			listBox_schemes->insertItem(SmallIcon("scheme_power", QIconSet::Automatic), i18n(_tmp));
		else if(_tmp == "Powersave" || _tmp == i18n("Powersave"))	
			listBox_schemes->insertItem(SmallIcon("scheme_powersave", QIconSet::Automatic), i18n(_tmp));
		else if(_tmp == "Presentation" || _tmp == i18n("Presentation"))	
			listBox_schemes->insertItem(SmallIcon("scheme_presentation", QIconSet::Automatic), i18n(_tmp));
		else if(_tmp == "Acoustic" || _tmp == i18n("Acoustic"))	
			listBox_schemes->insertItem(SmallIcon("scheme_acoustic", QIconSet::Automatic), i18n(_tmp));
		else if(_tmp == "AdvancedPowersave" || _tmp == i18n("Advanced Powersave"))	
			listBox_schemes->insertItem(SmallIcon("scheme_advanced_powersave", QIconSet::Automatic), i18n("Advanced Powersave"));
		else {
			listBox_schemes->insertItem(i18n(_tmp));
		}
    	}
}


/*! 
 * This used to set all needed Icons for the dialog.
 */
void ConfigureDialog::setIcons(){

	/* set all Icons */
	this->setIcon(SmallIcon("kpowersave", QIconSet::Automatic));
	buttonApply->setIconSet(SmallIconSet("apply", QIconSet::Automatic));
	buttonCancel->setIconSet(SmallIconSet("cancel", QIconSet::Automatic));
	buttonOk->setIconSet(SmallIconSet("ok", QIconSet::Automatic));
	buttonHelp->setIconSet(SmallIconSet("help", QIconSet::Automatic));

	pB_editBlacklist->setIconSet(SmallIconSet("configure", QIconSet::Automatic));
	pB_editGBlacklist->setIconSet(SmallIconSet("configure", QIconSet::Automatic));
	
	toolBox2->setItemIconSet( 0 ,SmallIcon("kscreensaver", QIconSet::Automatic));
	toolBox2->setItemIconSet( 1 ,SmallIcon("display", QIconSet::Automatic));
	if(actions[0] == "Suspend to Disk") 
		toolBox2->setItemIconSet( 2 ,SmallIcon("suspend_to_disk", QIconSet::Automatic));
	else if(actions[0] == "Suspend to RAM") 
		toolBox2->setItemIconSet( 2 ,SmallIcon("suspend_to_ram", QIconSet::Automatic));
	else if(actions[0] == "Standby") 
		toolBox2->setItemIconSet( 2 ,SmallIcon("stand_by", QIconSet::Automatic));
	toolBox2->setItemIconSet( 3 ,SmallIcon("misc", QIconSet::Automatic));
	pB_configNotify->setIconSet(SmallIconSet("knotify", QIconSet::Automatic));
}

/*! 
 * This used to set all needed Tooltips for the dialog.
 */
void ConfigureDialog::setTooltips(){
		
	// QToolTip::setWakeUpDelay ( 1000 );
	QToolTip::add(cB_specificSettings, i18n("This enables specific screen saver settings. \n"
						"Note: If selected, the global screen saver settings are \n"
						"overwritten while kpowersave runs."));
	QToolTip::add(cB_disable_Ss, i18n("This disables the screen saver. \n"
					  "Note: If selected, the global screen saver settings are \n"
					  "overwritten while kpowersave runs."));
	QToolTip::add(cB_blankScreen, i18n("This blanks the screen instead of using a specific screen saver. \n"
					   "Note: This may work only with KScreensaver."));
	QToolTip::add(cB_SpecificPM, i18n("This enables specific DPMS settings. \n"
					  "Note: If selected, the global DPMS settings are \n"
					  "overwritten while kpowersave runs."));
	QToolTip::add(cB_disablePM, i18n("This disables DPMS support."));
	QToolTip::add(cB_lockSuspend, i18n("If selected, the screen is locked on suspend or standby."));
	QToolTip::add(cB_lockLid, i18n("If selected, the screen is locked if the lid close event is triggered."));
	
	QToolTip::add(cB_autoSuspend, i18n("Check this box to enable or disable automatic suspension of the computer."));
	QToolTip::add(cB_autoInactivity, i18n("Activate this action if the user was inactive for the defined time \n"
					      "(greater than 0 minutes). If empty, nothing happens."));
	
	toolBox2->setItemToolTip ( 0, i18n("All scheme-related screen saver and DPMS settings."));
	toolBox2->setItemToolTip ( 1, i18n("All scheme-related display brightness settings."));
	toolBox2->setItemToolTip ( 2, i18n("All scheme-related automatic suspend settings."));

	QToolTip::add(brightnessSlider, i18n("Here you can change the brightness of your display. \n"
					     "Use the slider to change the brightness directly for \n"
					     "testing. To reset back to previous level, please press \n"
					     "the 'Reset' button. "));
	QToolTip::add(pB_resetBrightness, i18n("Use this button to set back the slider and the display brightness."));
}

/*!
 * This function is used to get the real name of the predefined schemes.
 * This is a workaround for the case if Yast translated the name of the 
 * scheme In this case is a new config section created and the old settings
 * would be lost.
 * \param s_scheme QString with the 'current' name of the scheme
 * \return QString with the english name of the scheme 
 */
QString ConfigureDialog::getSchemeRealName(QString s_scheme)
{
	if( s_scheme == "Performance" || s_scheme == i18n("Performance"))
		return "Performance";
	else if( s_scheme == "Powersave" || s_scheme == i18n("Powersave"))
		return "Powersave";
	else if( s_scheme == "Presentation" || s_scheme == i18n("Presentation"))
		return "Presentation";
	else if( s_scheme == "Acoustic" || s_scheme == i18n("Acoustic"))
		return "Acoustic";
	else return s_scheme;
}

/*!
 * This function is used to read the settings from configfile and
 * set the values to the related dialog elements.
 * \param schemeID Integer value, represent the position of the 
 *		   scheme in the QListBox and within the pdaemon-list.
 */
void ConfigureDialog::setConfigToDialog( int schemeID ){
	initalised = false;
	
	QString s_scheme = getSchemeRealName(powersave_schemes[schemeID]);
	if (settings->hasGroup(s_scheme)){
		settings->setGroup(s_scheme);
	}
	// no configuration found, set to default values ?!
	else { 
		if(settings->hasGroup("default-scheme"))
			settings->setGroup("default-scheme");
		else return;
	}
	
	cB_disable_Ss_toggled(settings->readBoolEntry("disableSs",false));
	cB_blankScreen->setChecked(settings->readBoolEntry("blankSs",false));
	cB_specificSettings_toggled(settings->readBoolEntry("specSsSettings",false));
	
	int i_standby = settings->readNumEntry("standbyAfter", -1);
	if(i_standby >= 0) {
		sB_standby->setValue(i_standby);
	}
	else {
		// load value from default-scheme
		if(settings->hasGroup("default-scheme")){
			settings->setGroup("default-scheme");
			i_standby = settings->readNumEntry("standbyAfter", -1);
			if(i_standby >= 0) {
				sB_standby->setValue(i_standby);
			}
			else sB_standby->setValue(0);
		}
		else{
			sB_standby->setValue(0);
		}
		// reset group to selected scheme
		if (settings->hasGroup(s_scheme)) settings->setGroup(s_scheme);
		else settings->setGroup("default-scheme");
	}
	
	int i_suspend = settings->readNumEntry("suspendAfter", -1);
	if(i_suspend >= 0) {
		sB_suspend->setValue(i_suspend);
	}
	else {
		// load value from default-scheme
		if(settings->hasGroup("default-scheme")){
			settings->setGroup("default-scheme");
			i_standby = settings->readNumEntry("suspendAfter", -1);
			if(i_standby >= 0) {
				sB_standby->setValue(i_standby);
			}
			else sB_standby->setValue(0);
		}
		else{
			sB_standby->setValue(0);
		}
		// reset group to selected scheme
		if (settings->hasGroup(s_scheme)) settings->setGroup(s_scheme);
		else settings->setGroup("default-scheme");
	}
	
	int i_poweroff = settings->readNumEntry("powerOffAfter", -1);
	if(i_poweroff >= 0) {
		sB_powerOff->setValue(i_poweroff);
	}
	else {
		// load value from default-scheme
		if(settings->hasGroup("default-scheme")){
			settings->setGroup("default-scheme");
			i_standby = settings->readNumEntry("powerOffAfter", -1);
			if(i_poweroff >= 0) {
				sB_powerOff->setValue(i_poweroff);
			}
			else sB_powerOff->setValue(0);
		}
		else{
			sB_powerOff->setValue(0);
		}
		// reset group to selected scheme
		if (settings->hasGroup(s_scheme)) settings->setGroup(s_scheme);
		else settings->setGroup("default-scheme");
	}
	
	cB_SpecificPM_toggled(settings->readBoolEntry("specPMSettings",false));
	cB_disablePM_toggled(settings->readBoolEntry("disableDPMS",false));
	
	// set autosuspend related settings
	QString _action = settings->readEntry("autoInactiveAction", "NULL");
	if( _action != "NULL") {
		int _index = actions.findIndex(_action);
		if( _index != -1) { 
			cB_autoInactivity->setCurrentItem( _index );
			cB_autoInactivity_activated( _index );
			cB_Blacklist->setEnabled(true);
		}
		else {
			// set to empty element if not supported by current machine or
			// if the value is "_NONE_"
			cB_autoInactivity->setCurrentItem( 0 );
			cB_autoInactivity_activated( 0 );
			cB_Blacklist->setEnabled(false);
		}
	}
	else {
		// set to disabled (to reduce code), if a entry found set to enabled !
		cB_Blacklist->setEnabled(false);
		
		if(settings->hasGroup("default-scheme")){
			settings->setGroup("default-scheme");
			 _action = settings->readEntry("autoInactiveAction", "NULL");
			if(_action != "NULL") {
				int _index = actions.findIndex(_action);
				if( _index != -1) { 
					cB_autoInactivity->setCurrentItem( _index );
					tL_autoInactivity_After->setEnabled(true);
					cB_Blacklist->setEnabled(true);
				}
				else {
					cB_autoInactivity->setCurrentItem( 0 );
				}
			}
			else {
				cB_autoInactivity->setCurrentItem( 0 );
			}
			// reset group to selected scheme
			if (settings->hasGroup(s_scheme)) settings->setGroup(s_scheme);
			else settings->setGroup("default-scheme");
		}
		else{
			cB_autoInactivity->setCurrentItem( 0 );
		}
	}
	if( cB_autoInactivity->currentItem() == 0 ) {
		sB_autoInactivity->setEnabled(false);
		tL_autoInactivity_After->setEnabled(false);
	}
	
	int i_autoInactivityAfter = settings->readNumEntry("autoInactiveActionAfter", -1);
	if(i_autoInactivityAfter >= 0) {
		sB_autoInactivity->setValue(i_autoInactivityAfter);
	}
	else {
		// load value from default-scheme
		if(settings->hasGroup("default-scheme")){
			settings->setGroup("default-scheme");
			i_autoInactivityAfter = settings->readNumEntry("autoInactiveActionAfter", -1);
			if(i_standby >= 0) {
				sB_autoInactivity->setValue(i_autoInactivityAfter);
			}
			else sB_autoInactivity->setValue(0);
		}
		else{
			sB_autoInactivity->setValue(0);
		}
		// reset group to selected scheme
		if (settings->hasGroup(s_scheme)) settings->setGroup(s_scheme);
		else settings->setGroup("default-scheme");
	}

	if (brightness_supported) {
		// enable the widgets
		cB_Brightness->setEnabled(true);
		cB_Brightness_toggled(settings->readBoolEntry("enableBrightness",false));

		if (brightnessLevels > 33) 
			brightnessSlider->setLineStep(3);
		else 
			brightnessSlider->setLineStep(100/brightnessLevels);

		brightnessSlider->setPageStep(10);

		int i_brightnessPercent = settings->readNumEntry("brightnessPercent", -1);
		if(i_brightnessPercent >= 0) {
			brightnessSlider->setValue(i_brightnessPercent);
			tL_valueBrightness->setText(QString::number(i_brightnessPercent) + " %");
		}
		else {
			brightnessSlider->setValue(100);
			tL_valueBrightness->setText(QString::number(100)+ " %");	
		}
		
		tL_brightness->setText(i18n("Your hardware supports to change the brightness. The "
					    "values of the slider are in percent and mapped "
					    "to the available brightness levels of your hardware."));
	}
	else {
		cB_Brightness->setEnabled(false);
		gB_Brightness->setEnabled(false);
		tL_brightness->setText(i18n("Your Hardware currently not support changing the brightness "
					    "of your display."));
	}

	cB_disableNotifications->setChecked(settings->readBoolEntry("disableNotifications",false));
	
	cB_Blacklist_toggled(settings->readBoolEntry("autoInactiveSchemeBlacklistEnabled",false));
	cB_autoSuspend_toggled(settings->readBoolEntry("autoSuspend",false));
		
	initalised = true;
	scheme_changed = false;
	if(!general_changed) buttonApply->setEnabled(false);
	currentScheme = schemeID;
}

/*!
 * This is used to set the values from the section general in configfile to the 
 * related items in the dialog.
 */
void ConfigureDialog::setGeneralSettings() {
	
	settings->setGroup("General");
	
	cB_lockSuspend->setChecked(settings->readBoolEntry("lockOnSuspend",false));
	cB_lockLid->setChecked(settings->readBoolEntry("lockOnLidClose",false));
	cB_autostart->setChecked(settings->readBoolEntry("Autostart",false));
	cB_autostart_neverAsk->setChecked(settings->readBoolEntry("AutostartNeverAsk",false));
	
	QString lockmethod = settings->readEntry("lockMethod", "NULL");
	if(comboB_lock->count() == 0 ){
		comboB_lock->insertItem(i18n("Select Automatically"),0); 
		comboB_lock->insertItem(i18n("KScreensaver"),1);
		comboB_lock->insertItem(i18n("XScreensaver"),2);
		comboB_lock->insertItem(i18n("xlock"),3);
		if (gnome_session) comboB_lock->insertItem(i18n("GNOME Screensaver"),4);
	}
	
	if (lockmethod == "automatic") comboB_lock->setCurrentItem(0);
	else if (lockmethod == "kscreensaver")  comboB_lock->setCurrentItem(1);
	else if (lockmethod == "xscreensaver") comboB_lock->setCurrentItem(2);
	else if (lockmethod == "xlock") comboB_lock->setCurrentItem(3);
	else if (gnome_session && (lockmethod == "gnomescreensaver")) comboB_lock->setCurrentItem(4);
	else comboB_lock->setCurrentItem(0);
	
	if(cB_lockSuspend->isOn() || cB_lockLid->isOn() ) {
		tL_lockWith->setEnabled(true);
		comboB_lock->setEnabled(true);
	}
	else {
		tL_lockWith->setEnabled(false);
		comboB_lock->setEnabled(false);
	}
}

/* ---- END General setup SECTION ---- */

/* ---- START store settings SECTION ---- */

/*! 
 * This used to save changes in settings of the current scheme.
 */
void ConfigureDialog::saveSchemeSettings() {

	QString s_scheme = getSchemeRealName(powersave_schemes[currentScheme]);
	settings->setGroup(s_scheme);

	settings->writeEntry("specSsSettings",cB_specificSettings->isOn());
	settings->writeEntry("disableSs",cB_disable_Ss->isOn());
	settings->writeEntry("blankSs",cB_blankScreen->isOn());
	settings->writeEntry("specPMSettings",cB_SpecificPM->isOn());
	settings->writeEntry("disableDPMS",cB_disablePM->isOn());
	
	settings->writeEntry("standbyAfter",sB_standby->value());
	settings->writeEntry("suspendAfter",sB_suspend->value());
	settings->writeEntry("powerOffAfter",sB_powerOff->value());

	settings->writeEntry("disableNotifications",cB_disableNotifications->isOn());
		
	if(cB_autoInactivity->currentText() == " "){
		settings->writeEntry("autoInactiveAction", "_NONE_");
	}
	else{
		int _index = cB_autoInactivity->currentItem();
		if(_index > 0) {
			/*if(_index == (cB_autoInactivity->count()-1)) {
				settings->writeEntry("autoInactiveAction","shutdown");	
			}
			else {
				settings->writeEntry("autoInactiveAction",actions[(_index)]);
			}*/
			settings->writeEntry("autoInactiveAction",actions[(_index)]);
		}
		settings->writeEntry("autoInactiveActionAfter",sB_autoInactivity->value());
	}
	settings->writeEntry("autoSuspend",cB_autoSuspend->isOn());
	settings->writeEntry("autoInactiveSchemeBlacklistEnabled",cB_Blacklist->isOn());
	
	settings->writeEntry("enableBrightness",cB_Brightness->isOn());
	if(brightness_changed)
		settings->writeEntry("brightnessPercent",brightnessSlider->value());

	settings->sync();
	scheme_changed = false;
	if(!general_changed) buttonApply->setEnabled(false);
}

/*! 
 * This used to save changes in settings of the current scheme.
 */
void ConfigureDialog::saveGeneralSettings() {
	
	settings->setGroup("General");
	
	settings->writeEntry("lockOnSuspend",cB_lockSuspend->isOn());
	settings->writeEntry("lockOnLidClose",cB_lockLid->isOn());
	settings->writeEntry("Autostart",cB_autostart->isOn());
	settings->writeEntry("AutostartNeverAsk",cB_autostart_neverAsk->isOn());
	
	QString selected_method = "";
	int _selected = comboB_lock->currentItem();
	if(_selected == 0) selected_method = "automatic";
	else if(_selected == 1) selected_method = "kscreensaver";
	else if(_selected == 2) selected_method = "xscreensaver";
	else if(_selected == 3) selected_method = "xlock";
	else if(gnome_session && (_selected == 4)) selected_method = "gnomescreensaver";
	settings->writeEntry( "lockMethod", selected_method );
	
		
	settings->sync();
	general_changed = false;
	if(!scheme_changed) buttonApply->setEnabled(false);
}

/* ---- END store settings SECTION ---- */

/* ---- START monitor changes SECTION ---- */

/*!
 * SLOT: Called if a value within the Tab 'General Settings' is changed.
 */
void ConfigureDialog::general_valueChanged(){
	if(initalised) {
		general_changed = true;
		buttonApply->setEnabled(true);
		// enable/disable the comboB_lock and tL_lockWith
		if(cB_lockSuspend->isOn() || cB_lockLid->isOn() ) {
			tL_lockWith->setEnabled(true);
			comboB_lock->setEnabled(true);
		}
		else {
			tL_lockWith->setEnabled(false);
			comboB_lock->setEnabled(false);
		}
	}
}

/*!
 * SLOT: Called if a value within the Tab 'Scheme Settings' is changed.
 */
void ConfigureDialog::scheme_valueChanged(){
	if(initalised) {
		scheme_changed = true;
		buttonApply->setEnabled(true);
	}
}

/*!
 * SLOT: called if the current scheme in the QListBox listBox_schemes is
 * changed/ a other scheme was selected.
 */
void ConfigureDialog::listBox_schemes_currentChanged(){
	
	// store current settings if something changed
	if(initalised && scheme_changed ) {
		int res = KMessageBox::warningYesNo(this,
						    i18n("There are unsaved changes in the active scheme.\n"
							 "Apply the changes before jumping to the next scheme "
							 "or discard the changes?"), 
						    i18n("Unsaved Changes"), KStdGuiItem::apply(), KStdGuiItem::discard());
		
		if (res == KMessageBox::Yes) {
			// Save changes
			saveSchemeSettings();
		}
		if (res == KMessageBox::No) {
			// discard changes and reset trigger
			scheme_changed = false;
		}
	}
	// set to new Item
	setConfigToDialog(listBox_schemes->currentItem());
}

/* ---- END monitor changes SECTION ---- */

/* ---- START BUTTON SECTION ---- */

/*!
 * SLOT: called if the 'Apply' button is clicked.
 */
void ConfigureDialog::buttonApply_clicked(){
	if(initalised && scheme_changed ) {
		saveSchemeSettings();
		scheme_changed = false;
	}
	if(initalised && general_changed ) {
		saveGeneralSettings();
		general_changed = false;
	}
}

/*!
 * SLOT: called if the 'Chancel' button is clicked.
 */
void ConfigureDialog::buttonCancel_clicked(){
	if (scheme_changed || general_changed) {
		int res = KMessageBox::warningYesNoCancel(this,
             						  i18n("There are unsaved changes.\n" "Apply the changes "
							       "before cancel or discard the changes?"), 
						          i18n("Unsaved Changes"), KStdGuiItem::apply(), KStdGuiItem::discard());
		if (res == KMessageBox::Yes) {
			buttonApply_clicked();
		}
		else if (res == KMessageBox::Cancel) return;
	}
	close();
}

/*!
 * SLOT: called if the 'OK' button is clicked.
 */
void ConfigureDialog::buttonOk_clicked(){
	
	buttonApply_clicked();
	buttonApply->setEnabled(false);
	close();
}

/*!
 * SLOT: called if the 'Help' button is clicked.
 */
void ConfigureDialog::buttonHelp_clicked(){
	emit openHelp();
}

/* ----  END BUTTON SECTION ---- */

/* ---- START SCREENSAVER SECTION ---- */

/*!
 * SLOT: called if QCheckBox cB_specificSettings is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled off
 */
void ConfigureDialog::cB_specificSettings_toggled(bool state){
	
	if(!initalised) cB_specificSettings->setChecked(state);
	
	/* set widgets visible/disabled */
	if(state){
		cB_disable_Ss->setEnabled(true);
		if(cB_disable_Ss->isOn()){
			cB_blankScreen->setEnabled(false);
		}
		else {
			cB_blankScreen->setEnabled(true);
		}
	}
	else {
		cB_disable_Ss->setEnabled(false);
		cB_blankScreen->setEnabled(false);
	}
}

/*!
 * SLOT: called if QCheckBox cB_disable_Ss is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled off
 */
void ConfigureDialog::cB_disable_Ss_toggled(bool state){

	/* set widgets visible/disabled */
	if(!initalised) cB_disable_Ss->setChecked(state);
	cB_blankScreen->setEnabled(!state);
}


/* ---- END SCREENSAVER SECTION ---- */

/* ---- START DPMS SECTION ---- */

/*!
 * SLOT: called if QCheckBox cB_SpecificPM is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled offtrue
 */
void ConfigureDialog::cB_SpecificPM_toggled(bool state){
	
	if(!initalised) cB_SpecificPM->setChecked(state);
	
	/* set widgets visible/disabled */
	cB_disablePM->setEnabled(state);
	if(cB_disablePM->isOn()) {
		state = false;
	}
	tL_standbyAfter->setEnabled(state);
	sB_standby->setEnabled(state);
	tL_suspendAfter->setEnabled(state);
	sB_suspend->setEnabled(state);
	tL_powerOffAfter->setEnabled(state);
	sB_powerOff->setEnabled(state);
}

/*!
 * SLOT: called if QCheckBox cB_disablePM is toggled.
 * \param state boolean, true if toggled on
 *			 false if toggled off
 */
void ConfigureDialog::cB_disablePM_toggled(bool state){
	
	if(!initalised) cB_disablePM->setChecked(state);

	/* set widgets visible/disabled */
	tL_standbyAfter->setEnabled(!state);
	sB_standby->setEnabled(!state);
	tL_suspendAfter->setEnabled(!state);
	sB_suspend->setEnabled(!state);
	tL_powerOffAfter->setEnabled(!state);
	sB_powerOff->setEnabled(!state);
}

/* ----  END DPMS SECTION ---- */

/* ---- START Inactivity SECTION ---- */

/*! 
 * This used to set the inactivity related combobox cB_autoInactivity .
 */
void ConfigureDialog::setInactivityBox(){

	cB_autoInactivity->clear();
	// add "nothing" at start of the list
	actions.push_front(" ");
	// add "Turn Off Computer" at end of the list
	// QString _to_i18n = i18n("Turn Off Computer");
	// actions.append("Turn Off Computer");
	
	for ( QStringList::Iterator it = actions.begin(); it != actions.end(); ++it ) {
		cB_autoInactivity->insertItem( i18n( *it ) );
	}
}

/*! 
 * This used to set the autosuspend related widgets.
 */
void ConfigureDialog::cB_autoSuspend_toggled( bool toggled ) {

	if(cB_autoSuspend->isOn() != toggled)
		cB_autoSuspend->setChecked(toggled);	

	tL_autoInactivity_explain->setEnabled(toggled);
	cB_autoInactivity->setEnabled(toggled);
	if(cB_autoInactivity->currentItem() > 0) {
		cB_autoInactivity->setEnabled(true);
		tL_autoInactivity_After->setEnabled(true);
		sB_autoInactivity->setEnabled(true);
		cB_Blacklist->setEnabled(true);
		cB_Blacklist_toggled(cB_Blacklist->isOn());
	}
	
	if(!toggled) {
		cB_autoSuspend->setChecked(false);
		tL_autoInactivity_After->setEnabled(false);
		cB_autoInactivity->setEnabled(false);
		sB_autoInactivity->setEnabled(false);
		cB_Blacklist->setEnabled(false);
		pB_editBlacklist->setEnabled(false);
	}
	
	scheme_changed = true;
	buttonApply->setEnabled(true);
}

/*!
 * SLOT: called if a new item in cB_autoInactivity is selected.
 * \param selectedItem Integer, contains the ID of the new item in the
 *				comboBox.
 */
void ConfigureDialog::cB_autoInactivity_activated( int selectedItem ) {
	if( actions[selectedItem] != " " ) {
		if(!displayed_WARN_autosuspend && initalised) {
			QString _msg = i18n("<b>Note:</b> If you select this option, the computer will suspend or standby "
					    "if the current user is inactive for the defined time even if somebody is "
					    "logged in remotely to the X server.<br><br> This feature can also produce "
					    "problems with some programs, such as video players or cd burner. These "
					    "programs can be blacklisted by checking <b>Enable scheme-specific blacklist"
					    "</b> and click <b>Edit Blacklist...</b>. If this does not help, report the "
					    "problem or deactivate autosuspend.<br><br> Really use this option?");
			QString _tmp = i18n("<b>Note:</b> If you select this option, the computer will suspend or standby "
					    "if the current user is inactive for the defined time. <br><br> This feature "
					    "can also produce problems with some programs, such as video players or "
					    "cd burner. These programs can be blacklisted by checking <b>Enable "
					    "scheme-specific blacklist</b> and click <b>Edit Blacklist...</b>. If this does "
					    "not help, report the problem or deactivate autosuspend.<br><br> Really use "
					    "this option?");
			int tmp = KMessageBox::warningContinueCancel(this, _msg);
			if (tmp ==  KMessageBox::Cancel) {
				selectedItem = 0;
			}
			displayed_WARN_autosuspend = true;
		}
		sB_autoInactivity->setEnabled(true);
		tL_autoInactivity_After->setEnabled(true);
		cB_Blacklist->setEnabled(true);
		if(cB_Blacklist->isChecked()) pB_editBlacklist->setEnabled(true);
	
	}
	else {
		sB_autoInactivity->setEnabled(false);
		tL_autoInactivity_After->setEnabled(false);
		cB_Blacklist->setEnabled(false);
		pB_editBlacklist->setEnabled(false);
	}
	
	cB_autoInactivity->setCurrentItem( selectedItem );
	
	if(initalised) {
		scheme_changed = true;
		buttonApply->setEnabled(true);
	}
}

/*!
 * SLOT: Called if the 'Enable scheme specific blacklist' is toggled.
 * \param toggled boolean, true if toggled on
 *			   false if toggled off
 */
void ConfigureDialog::cB_Blacklist_toggled( bool toggled ){
	
	pB_editBlacklist->setEnabled(toggled);
	
	if(initalised) {
		buttonApply->setEnabled(true);
		scheme_changed = true;
	}
	else {
		cB_Blacklist->setChecked(toggled);
	}
}

/*!
 * SLOT: Called if the 'edit blacklist' button is toggled.
 */
void ConfigureDialog::pB_editBlacklist_clicked(){

	QString _top_text = "";
	bool initialiseImport = false;

	if(tabWidget->currentPageIndex() == 0 ) {
		QString s_scheme = getSchemeRealName(powersave_schemes[listBox_schemes->currentItem()]);
		_top_text = listBox_schemes->currentText();
		if (settings->hasGroup(s_scheme)){
			settings->setGroup(s_scheme);
		}
		blacklist = settings->readListEntry("autoInactiveSchemeBlacklist", ',');
		if( blacklist.empty()) {
			QString _msg = i18n("The blacklist of the selected scheme is empty. "
					    "Import the general blacklist?");
			int tmp = KMessageBox::questionYesNo(this, _msg, QString(), i18n("Import"), i18n("Do Not Import"));
			if (tmp ==  KMessageBox::Yes) {
				initialiseImport = true;
				if(settings->hasGroup("General")){
					settings->setGroup("General");
					blacklist = settings->readListEntry("autoInactiveBlacklist", ',');
				}
			}
		}
	}
	else {
		if(settings->hasGroup("General")){
			_top_text = i18n("General Blacklist");
			settings->setGroup("General");
			blacklist = settings->readListEntry("autoInactiveBlacklist", ',');
		}
	}
	blacklistEDlg = new blacklistEditDialog(blacklist, _top_text, initialiseImport);
	
	connect( blacklistEDlg, SIGNAL(config_finished(QStringList)), this, 
		 SLOT(saveSchemeBlacklist(QStringList)));
	blacklistEDlg->exec();
}

/*!
 * SLOT: Called if the signal config_finished(QStringList) recieved
 * and the autosuspend blacklist is edited.
 * \param new_blacklist QStringlist with the edited blacklisted processes
 */

void ConfigureDialog::saveSchemeBlacklist( QStringList new_blacklist){

	if(tabWidget->currentPageIndex() == 0 ) {
		QString s_scheme = getSchemeRealName(powersave_schemes[currentScheme]);
		settings->setGroup(s_scheme);
		settings->writeEntry("autoInactiveSchemeBlacklist", new_blacklist, ',');
	}
	else {
		settings->setGroup("General");
		settings->writeEntry("autoInactiveBlacklist", new_blacklist, ',');
	}
	settings->sync();
}

/* ---- END Inactivity SECTION ---- */

/* ---- START Brightness SECTION ---- */
/*! \b SLOT: to enable the brigthness related widgets */
void ConfigureDialog::cB_Brightness_toggled( bool toggled ) {
	gB_Brightness->setEnabled(toggled);
	cB_Brightness->setChecked(toggled);
	connect(brightnessSlider, SIGNAL(valueChanged (int)), this, SLOT(brightnessSlider_sliderMoved(int)));
}

/*! \b SLOT: to change the brightness if the slider is changed */
void ConfigureDialog::brightnessSlider_sliderMoved( int new_value ) {
	scheme_valueChanged();
	tL_valueBrightness->setText(QString::number(new_value) + " %");
	pdaemon->setBrightness(new_value, true);
	pB_resetBrightness->setEnabled(true);
	brightness_changed = true;
}

/*! \b SLOT: to reset the brightness if the reset button clicked */
void ConfigureDialog::pB_resetBrightness_clicked( ) {
	pdaemon->setBrightness(brightness_last, true);
	brightnessSlider->setValue(brightness_last);
	pB_resetBrightness->setEnabled(false);
	brightness_changed = false;
}

/* ---- END Brightness SECTION ---- */

/*! \b SLOT: to open the KNotify config dialog */
void ConfigureDialog::pB_configNotify_released( ) {
	emit openKNotify();
}

#include "configuredialog.moc"
