 /**************************************************************************
 *   Copyright (C) 2004 by  Thomas Renninger                               *
 *                            <trenn@suse.de> and                          *
 *                          Danny Kukawka                                  *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/
 
#include "kpowersave.h"
#include "infodialog.h"
#include <powersave_dbus.h>

// KDE headers:
#include <kaboutapplication.h>
#include <kapplication.h>
#include <kaudioplayer.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <knotifydialog.h>
#include <kpassivepopup.h>
#include <kpopupmenu.h>

// other Qt headers:
#include <qcursor.h>
#include <qevent.h>
#include <qfile.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qtooltip.h>

/*! 
*  \file 	kpowersave.cpp
*  \brief 	This file contains the main functionality of the kpowersave-applet.
*/

/*! 
 * This is the default constructor of the class kpowersave. 
 */
kpowersave::kpowersave( bool force_acpi_check ) : KSystemTray(0, "kpowersave"), DCOPObject("KPowersaveIface") {
	myDebug ("kpowersave::kpowersave");

	new_screen = new screen();
	pdaemon = new pDaemon();
	current_settings = new settings();
	autoSuspend = new autosuspend();
	
	yast2 = NULL;
	xscreensaver = NULL;

	config = KGlobal::config();
	config->setGroup("General");
	if(!config->readBoolEntry("AlreadyStarted", false) || force_acpi_check){
		config->writeEntry("AlreadyStarted", true);
		// check whether APM or ACPI supported, otherwise end up
		// and don't start kpowersave ever again until force_acpi_check == true.
		if (checkACPI() == NOT_SUPPORTED){
			config->writeEntry("Autostart", false);
			qDebug("This machine does not support ACPI, close KPowersave now.");
			exit(-1);
		}
	}

	// connect to signals for changes
	connect(pdaemon, SIGNAL(generalDataChanged()), this, SLOT(update()));
	connect(pdaemon, SIGNAL(schemeDataChanged()), this, SLOT(updateSchemeMenu()));
	connect(pdaemon, SIGNAL(setSuspend(QString)), this, SLOT(set_suspend_type(QString)));
	
	// connect to signals for messages/popups
	connect(pdaemon, SIGNAL(forwardClientMessage(QString)), this, SLOT(showClientMessage(QString)));	
	connect(pdaemon, SIGNAL(powersaveNotRunningDialog()), this, SLOT(showPowersaveErrorMsg()));
	connect(pdaemon, SIGNAL(progressDialogRequest(QString)), this, SLOT(showProgressDialog(QString)));
	//connect(pdaemon, SIGNAL(progressDialogRequest(QString, int)), this, SLOT(showProgressDialog(QString, int)));
	connect(autoSuspend, SIGNAL(displayErrorMsg(QString)), this, SLOT(showClientMessage(QString)));	

	// connect to events
	connect(pdaemon, SIGNAL(lidclosed()), this, SLOT(lockscreen_on_lidcloseEvent()));
	connect(pdaemon, SIGNAL(lidopened()), this, SLOT(handleLidOpenEvent()));
	connect(pdaemon, SIGNAL(lockscreen()), this, SLOT(lockscreen_on_lockEvent()));
	connect(pdaemon, SIGNAL(resumeFromSuspend(bool)), this, SLOT(setAutoSuspend(bool)));
	connect(autoSuspend, SIGNAL(inactivityTimeExpired()), this, SLOT(do_autosuspend()));

	// connects for emit KNotify events
	connect(pdaemon, SIGNAL(ACStatus(bool)), this, SLOT(notifyACStatusChange (bool)));
	connect(pdaemon, SIGNAL(batteryWARNState(QString)), this, SLOT(notifyBatteryStatusChange (QString)));
	connect(pdaemon, SIGNAL(resumedFrom(QString)), this, SLOT(notifyResume(QString)));
	connect(pdaemon, SIGNAL(schemeDataChanged()), this, SLOT(notifySchemeSwitch()));
	connect(pdaemon, SIGNAL(setSuspend(QString)), this, SLOT(notifySuspend(QString)));
	connect(autoSuspend, SIGNAL(inactivityTimeExpired()), this, SLOT(notifyAutosuspend()));

	config->sync();
	
	config_dialog_shown = false;
	suspend_dialog_shown = false;
	detailedIsShown = false;
	brightness_supported = false;
	hal_error_shown = false;
	icon_set_colored = false;
	icon_BG_is_colored = false;
	
	countWhiteIconPixel = 0;

	pixmap_name = "NONE";
        suspendType = "NULL";

	BAT_WARN_ICON_Timer = new QTimer(this);
	connect(BAT_WARN_ICON_Timer, SIGNAL(timeout()), this, SLOT(do_setIconBG()));
	
	DISPLAY_HAL_ERROR_Timer = new QTimer(this);
	connect(DISPLAY_HAL_ERROR_Timer, SIGNAL(timeout()), this, SLOT(showHalErrorMsg()));

	if(pdaemon->getBrightnessLevels() > 0) brightness_supported = true;

	init_menu();
	update();
	
	QString dummy_trans;
	dummy_trans = i18n("Inactivity detected.\n Machine autosuspend in %1 seconds.").arg(30);
	dummy_trans = i18n("Inactivity detected.\n Dim display in %1 seconds to %2%.").arg(30).arg(30);
	dummy_trans = i18n("Cannot connect to D-BUS. The D-BUS daemon may not be running.");
	dummy_trans = i18n("Scheme switched to %1. \n Deactivate following devices: %2").arg("").arg("");
	dummy_trans = i18n("Scheme switched to %1. \n Activate following devices: %2").arg("").arg("");
	dummy_trans = i18n("Report ...");
	dummy_trans = i18n("Suspend to RAM is not supported on your machine.");
	dummy_trans = i18n("If you know that it should work, you might override this detection with setting "
			   "SUSPEND2RAM_FORCE=\"yes\" in /etc/powersave/sleep.");
	dummy_trans = i18n("Additionally, please mail the output of %1 to %2 . Thanks!").arg("").arg("");
	dummy_trans = i18n("Power consumption");
	
}
	

/*! This is the default destructor of class kpowersave. */
kpowersave::~kpowersave(){
	myDebug ("kpowersave::~kpowersave");

	if (pdaemon)
		delete pdaemon;
	if (new_screen)
		delete new_screen;
}

/*!
 * use this function to initalise the main kicker menu
 */
void kpowersave::init_menu()
{
	KPOWERSAVE_CONFIGURE_ID = this->contextMenu()->insertItem(SmallIcon("configure", QIconSet::Automatic),
								  i18n("Configure KPowersave..."), 
								  this, SLOT(show_configure_dialog()));
	KPOWERSAVE_CONFIGURE_EVENTS_ID = this->contextMenu()->insertItem(SmallIcon("knotify", QIconSet::Automatic),
									 i18n("Configure Notifications..."), 
									 this, SLOT(show_configure_notifications_dialog()));
#ifdef ENABLE_YAST_ENTRY
	YAST_MODULE_MENU_ID = this->contextMenu()->insertItem(SmallIcon("yast", QIconSet::Automatic),
							      i18n("Start YaST2 Power Management Module..."), 
							      this, SLOT(do_config()));
#endif
	
	SLEEP_SEPARATOR_MENU_ID = this->contextMenu()->insertSeparator();

	if (pdaemon->suspend2disk_supported){ 
		SUSPEND2DISK_MENU_ID = this->contextMenu()->insertItem( SmallIconSet("suspend_to_disk", 
									QIconSet::Automatic),
									i18n("Suspend to Disk"), this,
									SLOT(do_suspend2disk()));
	}
	if (pdaemon->suspend2ram_supported){
		SUSPEND2RAM_MENU_ID = this->contextMenu()->insertItem( SmallIconSet("suspend_to_ram", 
								       QIconSet::Automatic),
								       i18n("Suspend to RAM"), this,
								       SLOT(do_suspend2ram()));
	}
	if (pdaemon->standby_supported){
		STANDBY_MENU_ID = this->contextMenu()->insertItem( SmallIconSet("stand_by", QIconSet::Automatic),
								   i18n("Standby"), this, SLOT(do_standby()));
	}
	
	speed_menu = new QPopupMenu(this, i18n("Set CPU Frequency Policy"));
	speed_menu->insertItem(i18n("Performance"), CPU_HIGH);
	speed_menu->insertItem(i18n("Dynamic"), CPU_AUTO);
	speed_menu->insertItem(i18n("Powersave"), CPU_LOW);
	
	CPUFREQ_SEPARATOR_MENU_ID = contextMenu()->insertSeparator();
	
	CPUFREQ_MENU_ID = contextMenu()->insertItem(i18n("Set CPU Frequency Policy"), speed_menu);
	connect(speed_menu, SIGNAL(activated(int)), this, SLOT(do_setSpeedPolicy(int)));


	SCHEME_SEPARATOR_MENU_ID = contextMenu()->insertSeparator();
	
	scheme_menu = new QPopupMenu(this, i18n("Set Active Scheme"));
	SCHEME_MENU_ID = contextMenu()->insertItem(i18n("Set Active Scheme"), scheme_menu);
	connect(scheme_menu, SIGNAL(activated(int)), this, SLOT(do_setActiveScheme(int)));
	
	// menu entry for the autosuspend disable checkbox, disabled by default, only 
	// displayed if autosuspend for the current scheme is activated
	AUTOSUSPEND_SEPARATOR_MENU_ID = contextMenu()->insertSeparator();
	AUTOSUSPEND_MENU_ID = this->contextMenu()->insertItem( i18n("Disable Actions on Inactivity"), 
							       this,SLOT(do_setAutosuspend()));
	this->contextMenu()->setItemVisible(AUTOSUSPEND_SEPARATOR_MENU_ID, FALSE);
	this->contextMenu()->setItemVisible(AUTOSUSPEND_MENU_ID, FALSE);

	HELP_SEPARATOR_MENU_ID = contextMenu()->insertSeparator();

	help_menu = new QPopupMenu(this, i18n("&Help"));
	
	help_menu->insertItem( SmallIcon("help", QIconSet::Automatic), i18n("&KPowersave Handbook"), 
			       this, SLOT(slotHelp()));
	help_menu->insertSeparator();
	help_menu->insertItem( i18n("&Report a bug ..."), this, SLOT(slotReportBug()));
	help_menu->insertItem( SmallIcon("kpowersave", QIconSet::Automatic), 
			       i18n("&About KPowersave"), this, SLOT(slotAbout()));

	HELP_MENU = contextMenu()->insertItem(SmallIcon("help", QIconSet::Automatic),i18n("&Help"), help_menu);
	
	connect(this, SIGNAL(quitSelected()), this, SLOT(_quit()));
}


/*!
 * This funtion load and manipulate the icons for the kickerapplet-section. 
 * The redraw interval depends on \ref icon_set_colored and \ref BAT_icon_BG_intervall.
 */
void kpowersave::redrawPixmap(){
	myDebug ("kpowersave::redrawPixmap");

	// if colored icon_background: normal redraw intervall is set off.
	// Icon (only) redrawed every BAT_icon_BG_intervall
	if (icon_set_colored) {
		if (icon_state_changed) {
			loadIcon();
			drawIcon();
		}
	}
	else {
		loadIcon();
		drawIcon();
	}
}

/*!
 * Starts the configure dialog of kpowersave.
 */
void kpowersave::show_configure_dialog() {
	myDebug ("kpowersave::show_configure_dialog");

	if(!config_dialog_shown) {
		QStringList _schemeList;
		QStringList _actions;
		
		if (pdaemon->schemes.count() > 0){
			if (!(pdaemon->daemon_running <= 0)){
				if (pdaemon->suspend2disk_allowed == 1) _actions.append("Suspend to Disk");
				if (pdaemon->suspend2ram_allowed == 1) _actions.append("Suspend to RAM");
				if (pdaemon->standby_allowed == 1) _actions.append("Standby");
			}
			
			_schemeList = pdaemon->schemes.schemeList();
			
			configDlg = new ConfigureDialog(config, pdaemon, _schemeList, _actions, 
							pdaemon->schemes.currentScheme());
			configDlg->show();
			config_dialog_shown = true;
			connect(configDlg, SIGNAL(destroyed()), this, SLOT(observe_configDlg()));
			connect(configDlg, SIGNAL(openHelp()), this, SLOT(slotHelp()));
			connect(configDlg, SIGNAL(openKNotify()), this, SLOT(show_configure_notifications_dialog()));
		}
		else {
			KPassivePopup::message(i18n("WARNING"), 
					i18n("Cannot find any schemes. Maybe the powersave daemon is not running.\n"
					"Starting it with '/etc/init.d/powersaved start' should help."), 
					SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
		}
	}
}


/*!
 * Starts the notification configure dialog of kpowersave.
 */
void kpowersave::show_configure_notifications_dialog() {
	myDebug ("kpowersave::show_configure_notifications_dialog");

	KNotifyDialog::configure(this);
}

/*!
 * Load the icons (from filesystem) for the kicker applet to \ref pixmap .
 * To reduce the systemload the icons are only reloaded if \ref pixmap_name
 * is changed.
 */
void kpowersave::loadIcon(){
	myDebug ("kpowersave::loadIcon");

	QString pixmap_name_tmp = "NONE";

	if (pdaemon->on_AC_power == AC_ONLINE || pdaemon->battery_state == BAT_NONE) {
		icon_set_colored = false;
		
		if (pdaemon->battery_state == BAT_NONE || (pdaemon->perc < 0 || pdaemon->perc >= 99))
			pixmap_name_tmp = QString("laptoppower");
		else
			pixmap_name_tmp = QString("laptopcharge");
	
		hal_error_shown = false;
		DISPLAY_HAL_ERROR_Timer->stop();
	}
	else if (pdaemon->on_AC_power == HAL_ERROR || pdaemon->on_AC_power == UNKNOWN ) {
		if(!hal_error_shown && pdaemon->on_AC_power == HAL_ERROR) {
			if(!DISPLAY_HAL_ERROR_Timer->isActive() && !hal_error_shown) {
				hal_error_shown = true;
				DISPLAY_HAL_ERROR_Timer->start(HAL_ERROR_MSG_intervall, true);
			}
		}
		pixmap_name_tmp = QString("ERROR");
	}
	else {
		switch(pdaemon->battery_state) {
			case BAT_CRIT:
			case BAT_LOW:
				if (icon_BG_is_colored) pixmap_name_tmp = QString("laptopbattery");
				else pixmap_name_tmp = QString("laptopbatteryRED");
				icon_BG_is_colored = !icon_BG_is_colored;
				icon_set_colored = true;
				break;
			case BAT_WARN:
				if (icon_BG_is_colored) pixmap_name_tmp = QString("laptopbattery");
				else pixmap_name_tmp = QString("laptopbatteryORANGE");
				icon_BG_is_colored = !icon_BG_is_colored;
				icon_set_colored = true;
				break;
			default:
				// if battery is discharging and not in warning, low or critical state
				pixmap_name_tmp = QString("laptopbattery");
				icon_set_colored = false;
				icon_BG_is_colored = false;
		}
		
		if (icon_set_colored){
			icon_state_changed = false;
			BAT_WARN_ICON_Timer->start(BAT_icon_BG_intervall, true);
		}
		hal_error_shown = false;
	}

	// reload icon only if new icon selected
	if(pixmap_name_tmp != pixmap_name) {
		pixmap_name = pixmap_name_tmp;
		if (pixmap_name.startsWith("ERROR")) {
			pixmap = SmallIcon("laptoppower", 22, KIcon::DisabledState);
		}
		else 
			pixmap = SmallIcon(pixmap_name, 22);
	}
}


/*!
 * This function draw the battery-capacity (colored field) to the icon.
 * Here also counted the white pixel in the icon-files. Since the icons are
 * the same and white pixel only in the retangel of the icon, the white pixel
 * stored in \ref countWhiteIconPixel only one time.
 */
void kpowersave::drawIcon(){
	myDebug ("kpowersave::drawIcon");

	QImage image = pixmap.convertToImage();
	int w = image.width();
	int h = image.height();
	int x, y;
	
	if((pixmap_name.contains("laptopbattery") || pixmap_name.contains("charge")) && countWhiteIconPixel == 0) {
		for (x = 0; x < w; x++)
			for (y = 0; y < h; y++)
				if(QColor(image.pixel(x, y)) == Qt::white) countWhiteIconPixel++;
	}
	
	int c = (countWhiteIconPixel * pdaemon->perc) / 100;
	
	if (c > 0) {
		uint ui;
		QRgb Rgb_set;
		
		if (pdaemon->on_AC_power == AC_ONLINE) {
			Rgb_set = qRgb(0x00, 0xff, 0x00); //green
		}
		else {
			switch(pdaemon->battery_state) {
				case BAT_CRIT:
				case BAT_LOW:
					Rgb_set = qRgb(0xff, 0x00, 0x00);//red
					break;
				case BAT_WARN:
					Rgb_set = qRgb(0xff, 0x55, 0x00); //orange
					break;
				default:
					Rgb_set = qRgb(0x00, 0xff, 0x00); //green
			}
		}
		if (image.depth() <= 8) {
			ui = image.numColors();
			image.setNumColors(ui + 1);
			image.setColor(ui, Rgb_set);
		}
		ui = 0xff000000 | Rgb_set;
		
		for (y = h - 1; y >= 0; y--) {
			for (x = 0; x < w ; x++) {
				if(QColor(image.pixel(x, y)) == Qt::white) {
					image.setPixel(x, y, ui);
					c--;
					if (c <= 0) goto quit;
				}
			}
		}
	}
quit:
	fullIcon.convertFromImage(image);
	setPixmap(fullIcon);
}

/*!
* By this function we fill and update the Tooltip for the icon in the kicker applet.
* The content of the tooltip alway updated, if something change.
* \todo Check if the tooltip also updated if mouse \b is over the icon, if not we maybe 
*	should implement this.\n If it is possible we should update the tooltip permanently 
*	while the mouse cursor is over the widget
*/
void kpowersave::updateTooltip(){
	myDebug ("kpowersave::updateTooltip");

	QString tmp, num3;
	num3.setNum(pdaemon->left % 60);
	num3 = num3.rightJustify(2, '0');

	if (pdaemon->on_AC_power == AC_ONLINE) {
		if (pdaemon->perc == 100) tmp = i18n("Plugged in -- fully charged");
		// assume that no battery is there
		else {
			if ((pdaemon->perc < 0 && pdaemon->left < 0) 
			     || pdaemon->battery_state == BAT_NONE) {
				tmp = i18n("Plugged in");
			}
			else{
				// if both null, no info, this may change to -1 again later ...
				if (pdaemon->left > 0){
					if (pdaemon->charge_state == CHARG_STATE_CHARGING)
						tmp = i18n("Plugged in -- %1% charged (%2:%3 h until full charged)")
						           .arg(pdaemon->perc).arg(pdaemon->left / 60).arg(num3);
					else
						tmp = i18n("Plugged in -- %1% charged (%2:%3 remaining hours)")
						           .arg(pdaemon->perc).arg(pdaemon->left / 60).arg(num3);
				}
				else if (pdaemon->charge_state == CHARG_STATE_CHARGING && checkACPI() == APM) {
					tmp = i18n("Plugged in -- %1% charged").arg(pdaemon->perc);
				}
				else{
					if (pdaemon->perc == -1) tmp = i18n("Plugged in -- no battery");
					else tmp = i18n("Plugged in -- %1% charged").arg(pdaemon->perc);
				}
			}
		}
	}
	else if (pdaemon->on_AC_power == HAL_ERROR || pdaemon->battery_state == BAT_HAL_ERROR){
		tmp = i18n("No information about battery and AC status available");
	}
	else{
		if (pdaemon->left >= 0){
			tmp = i18n("Running on batteries -- %1% charged (%2:%3 hours remaining)")
			      .arg(pdaemon->perc).arg(pdaemon->left / 60).arg(num3);
		}
		else {
			tmp = i18n("Running on batteries -- %1% charged").arg(pdaemon->perc);
		}
	}
	// add string whether battery is charging, but only if < 100% to avoid
	// stupid tooltip message on machines which always with 100% and on AC 
	// are charging, as e.g. Sony Vaio FS vgn-fs115b
	if (pdaemon->charge_state == CHARG_STATE_CHARGING && pdaemon->perc < 100) 
		tmp += i18n(" -- battery is charging");

	QToolTip::add(this, tmp);
}

/*!
 * \b SLOT to starts the Yast2-power-management module. This called by the menuentry 
 * with ID \ref YAST_MODULE_MENU_ID, named "Start YaST2 Power Management Module".
 * It create a new KProcess and execute "/sbin/yast2 power-management" with kdesu.
 */
void kpowersave::do_config(){
	myDebug ("kpowersave::do_config");
	
	#ifdef ENABLE_YAST_ENTRY
	if (yast2 != NULL) 
		delete yast2;
	
	yast2 = new KProcess;
	*yast2 << "kdesu" << "--nonewdcop" << "/sbin/yast2" << "power-management";

	connect(yast2, SIGNAL(processExited(KProcess *)),
	         SLOT(slotConfigProcessExited(KProcess *)));
	yast2->start(KProcess::NotifyOnExit);
	#else
	return;
	#endif
}

/*!
 * \b SLOT to open the KPowersave help
 */
void kpowersave::slotHelp()
{
    kapp->invokeHelp( "", "kpowersave" );
}

/*!
 * \b SLOT to open the KPowersave About dialog
 */
void kpowersave::slotAbout()
{
 	KAboutApplication a( this );
        a.exec();
}

/*!
 * \b SLOT to open the website to report bugs
 */
void kpowersave::slotReportBug()
{
#ifdef DISTRO_IS_SUSE
	kapp->invokeBrowser("http://opensuse.org/Report_a_Bug");
#else
	kapp->invokeBrowser("http://sourceforge.net/tracker/?group_id=124576&atid=700009");
#endif
}

/*!
 * \b SLOT to set the icon background on/off if battery is in critical, low or warning-state. Within 
 * this function we set \ref icon_state_changed to true and call \ref redrawPixmap() to redraw the 
 * kickerapplet icon and create a icon with blinking background. \n \n
 * The slot called by the QTimer \ref BAT_WARN_ICON_Timer . The interval of the timer is defined 
 * trough \ref BAT_icon_BG_intervall and starts here: \ref loadIcon() .  
 */
void kpowersave::do_setIconBG(){
	myDebug ("kpowersave::do_setIconBG");
	
	if (icon_set_colored) icon_state_changed = true;
	redrawPixmap();
}
/*!
 * \b SLOT to enable/disable the autosuspend.
 */
void kpowersave::do_setAutosuspend(){
	myDebug ("kpowersave::do_setAutosuspend");	

	if(!contextMenu()->isItemChecked(AUTOSUSPEND_MENU_ID)) {
		autoSuspend->stop();
		contextMenu()->setItemChecked(AUTOSUSPEND_MENU_ID, true);	
	}
        else {
		if(current_settings->autoSuspend) {
			contextMenu()->setItemChecked(AUTOSUSPEND_MENU_ID, FALSE);
			setAutoSuspend(false);
		}
		else {
			contextMenu()->setItemVisible(AUTOSUSPEND_MENU_ID, FALSE);
			contextMenu()->setItemChecked(AUTOSUSPEND_MENU_ID, FALSE);
			contextMenu()->setItemVisible(AUTOSUSPEND_SEPARATOR_MENU_ID, FALSE);
		}
        }
}

/*!
 * \b SLOT which called if the \ref configDlg is destroyed. We set within this SLOT 
 * \ref config_dialog_shown to false.
 */
void kpowersave::observe_configDlg(){
	myDebug ("kpowersave::observe_configDlg");
	
	current_settings->load_general_settings();
	setSchemeSettings();
	
	config_dialog_shown=false;
}

/*!
 * \b SLOT which called from \ref do_config() if the 'kdesu yast2' KProcess exited.
 * This function control the return value and display if needed a errormessage on failure.
 */
void kpowersave::slotConfigProcessExited(KProcess *proc){
	myDebug ("kpowersave::slotConfigProcessExited");

	#ifdef ENABLE_YAST_ENTRY
	if (proc->normalExit()){
		if (proc->exitStatus() != 0 && proc->exitStatus() != 16){
			KPassivePopup::message(i18n("WARNING"),
			                        i18n("Could not start YaST Power Management Module. "
			                              "Check if it is installed."),
			                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
		}
	}
	else{
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("Could not start YaST Power Management Module. "
		                              "Check if it is installed."),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
	}
	#else
	return;
	#endif
}

/*!
 * \b SLOT to send the command for "suspend to disk" to the powersave daemon.
 * If there is a error while "suspend to disk" the user get e messagebox.
 * This function need a running powersave daemon for "suspend to disk".
 * \return boolean with the result of the operation
 * \retval true if successful 
 * \retval false if command not supported or if powersaved not running
 */
bool kpowersave::do_suspend2disk(){
	myDebug ("kpowersave::do_suspend2disk");

	int res;
	// sent to admin interface if we are run as root
	if (getuid() == 0)
		res = dbusSendSimpleMessage(ADMIN_MESSAGE,
					    "SuspendToDisk");
	else
		res = dbusSendSimpleMessage(ACTION_MESSAGE,
					    "SuspendToDisk");
		
        suspendType = "suspend2disk";
        switch (res) {
        case REPLY_SUCCESS:
                return true;
        case REPLY_DISABLED:
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("Suspend to disk disabled by administrator."),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
		this->contextMenu()->setItemEnabled(SUSPEND2DISK_MENU_ID, FALSE);
                break;
        default:
                KPassivePopup::message(i18n("WARNING"),
                                i18n("The powersave daemon must be running in the background for a suspend to disk."),
                                SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
                break;
	}
	return false;
}

/*!
 * \b SLOT to send the command for "suspend to RAM" to the powersave daemon.
 * If there is a error while "suspend to RAM" the user get e messagebox.
 * This function need a running powersave daemon for "suspend to RAM".
 * \return boolean with the result of the operation
 * \retval true if successful 
 * \retval false if command not supported or if powersaved not running
 */
bool kpowersave::do_suspend2ram(){
	myDebug ("kpowersave::do_suspend2ram");

	int res;
	if (getuid() == 0)
		res = dbusSendSimpleMessage(ADMIN_MESSAGE,
					    "SuspendToRam");
	else
		res = dbusSendSimpleMessage(ACTION_MESSAGE,
					    "SuspendToRam");
	
	suspendType = "suspend2ram";
        switch (res) {
        case REPLY_SUCCESS:
                return true;
        case REPLY_DISABLED:
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("Suspend to RAM disabled by administrator."),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
		this->contextMenu()->setItemEnabled(SUSPEND2RAM_MENU_ID, FALSE);
                break;
        default:
                KPassivePopup::message(i18n("WARNING"),
                                i18n("The powersave daemon must be running in the background for a suspend to RAM."),
                                SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
                break;
	}
	return false;
}

/*!
 * \b SLOT to send the command for "stand-by" to the powersave daemon.
 * If there is a error while "stand-by" the user get e messagebox.
 * This function need a running powersave daemon for "stand-by".
 * \return boolean with the result of the operation
 * \retval true if successful 
 * \retval false if command not supported or if powersaved not running
 */
bool kpowersave::do_standby(){
	myDebug ("kpowersave::do_standby");

	int res;
	if (getuid() == 0)
		res = dbusSendSimpleMessage(ADMIN_MESSAGE,
					    "Standby");
	else
		res = dbusSendSimpleMessage(ACTION_MESSAGE,
					    "Standby");
		
        suspendType = "standby";
        switch (res) {
        case REPLY_SUCCESS:
                return true;
        case REPLY_DISABLED:
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("Standby disabled by administrator."),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
		this->contextMenu()->setItemEnabled(STANDBY_MENU_ID, FALSE);
                break;
        default:
                KPassivePopup::message(i18n("WARNING"),
                                i18n("The powersave daemon must be running in the background for standby."),
                                SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
                break;
	}
	return false;
}

/*!
 * \b SLOT to send the related suspend command for autosuspend if the signal
 * \ref autosuspend::autosuspendTimeExpired() was recieved. 
 * \return boolean with the result of the operation
 * \retval true if successful 
 * \retval false if command not supported or if powersaved not running
 * \todo	add check if the requested command is supported befor send and 
 *		add message for this case to tell that maybe changed config!
 */
bool kpowersave::do_autosuspend() {
	myDebug ("kpowersave::do_autosuspend");
	
	autoSuspend->stop();
	if(current_settings->autoSuspend) {
		if(current_settings->autoInactiveAction == "Suspend to Disk") {
			return do_suspend2disk();
		} else if (current_settings->autoInactiveAction == "Suspend to RAM") {
			return do_suspend2ram();
		} else if (current_settings->autoInactiveAction == "Standby") {
			return do_standby();
		} else {
			return false;
		}
	}
	else return false;
	
}

/*!
 * \b SLOT to lock the screen if a lidclose event was triggered. If the screen get locked
 * depends on the user specific settings.
 */
void kpowersave::lockscreen_on_lidcloseEvent(){
	myDebug ("kpowersave::lockscreen_on_lidcloseEvent");
	
	// get new general settings! This could maybe removed if we 
	// could be shure, that the settings are actuall
	current_settings->load_general_settings();
	
	if(current_settings->lockOnLidClose) {
		if(!new_screen->lockScreen( current_settings->lockmethod )) {
			KPassivePopup::message(i18n("WARNING"),
						i18n("Could not lock the screen. There may be a problem with the "
						     "selected \nlock method or something else."),
						SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 10000);
		
		}
	}
	if(current_settings->forceDpmsOffOnLidClose) {
		new_screen->forceDPMSOff();
	}

	if(!current_settings->disableNotifications)
		KNotifyClient::event( this->winId(), "lid_closed_event", i18n("The Lid was closed."));
}

/*!
 * \b SLOT to handle the lidopen event and send notification and do 
 * reset needed settings.
 */
void kpowersave::handleLidOpenEvent() {
	myDebug("kpowersave::handleLidOpenEvent");

	if(current_settings->forceDpmsOffOnLidClose) {
		// reset the scheme settings to avoid problems related to call xset on lidclose
		setSchemeSettings();
	}

	if(!current_settings->disableNotifications)
		KNotifyClient::event( this->winId(), "lid_opened_event", i18n("The Lid was opened."));
}

/*!
 * \b SLOT to lock the screen if a lockscreen event was triggered. If the screen get locked
 * depends on the user specific settings.
 */
void kpowersave::lockscreen_on_lockEvent(){
	myDebug("kpowersave::lockscreen_on_lockEvent");
        // get new general settings! This could maybe removed if we 
	// could be shure, that the settings are actuall
	current_settings->load_general_settings();
	
	if(current_settings->lockOnSuspend) {
		new_screen->lockScreen( current_settings->lockmethod );
		connect(pdaemon, SIGNAL(resumeFromSuspend(bool)), this, SLOT(activateLoginScreen()));
	}
}

/*!
 * \b SLOT to show the login dialog if the desktop was locked before the suspend.
 */
void kpowersave::activateLoginScreen(){
	myDebug("kpowersave::activateLoginScreen");
        // get new general settings! This could maybe removed if we 
	// could be shure, that the settings are actuall
	current_settings->load_general_settings();
	
	if(current_settings->lockOnSuspend) {
		QTimer::singleShot(current_settings->timeToFakeKeyAfterLock, new_screen, SLOT(fakeShiftKeyEvent()));
	}
}

/*!
 * \b SLOT to set the current suspend type for later use.
 */
void kpowersave::set_suspend_type( QString suspendtype){
    myDebug("kpowersave::set_suspend_type");

    suspendType = suspendtype;
}

/*!
 * \b SLOT which called if kpowersave is exited by the user. In this case the user
 * is asked through a yes/no box if "KPowersave start automatically on log in" and the 
 * result is written to the KDE configfile.
 */
void kpowersave::_quit (){
	myDebug ("kpowersave::_quit");
	
	// set the KDE-Settings back to user default
	if(getenv("KDE_FULL_SESSION")) {
		current_settings->load_kde_settings();
		// reset to KDE screensaver settings
		if(!current_settings->kde_settings->enabled) new_screen->setScreenSaver(false);
		else new_screen->setScreenSaver(true);
		
		if(!current_settings->kde_settings->displayEnergySaving) new_screen->setDPMS(false);
		else new_screen->setDPMS(true);
		
		new_screen->has_DPMS = new_screen->setDPMSTimeouts( current_settings->kde_settings->displayStandby,
								    current_settings->kde_settings->displaySuspend,
								    current_settings->kde_settings->displayPowerOff);
		
		current_settings->load_general_settings(); 
	}
	
	// set, if this is a GNOME session, XScreensaver settings back to user default
	QString session = getenv("DESKTOP_SESSION");
	if(session.startsWith("gnome")  && (new_screen->checkScreenSaverStatus() == 11)) {
		if (xscreensaver != NULL)
			delete xscreensaver;

		xscreensaver = new KProcess;
		*xscreensaver << "xscreensaver-command" << "-restart";
		xscreensaver->start(KProcess::DontCare);
	}
	
	if(!current_settings->autostartNeverAsk) { 
		QString tmp1 = i18n ("Start KPowersave automatically when you log in?");
		int tmp2 = KMessageBox::questionYesNo (0, tmp1, i18n("Question"), i18n("Start Automatically"), i18n("Do Not Start"));
		config->setGroup("General");
		config->writeEntry ("Autostart", tmp2 == KMessageBox::Yes);
		config->sync ();
	}
}

/*!
 * \b SLOT called if the user select a 'CPU Frequency Policy' from the menu ( \ref CPUFREQ_MENU_ID ). 
 */
void kpowersave::do_setSpeedPolicy(int menu_id){
	myDebug ("kpowersave::do_setSpeedPolicy");

	int retval = pdaemon->setCPUFreqPolicy(menu_id);
	if( retval < 0 ) {
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("CPU Freq Policy %1 could not be set.").arg(speed_menu->text(menu_id)),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 10000);
	}
	if( retval > 0) {
		update();
	}
}

/*!
 * \b SLOT called if the user select a scheme from the menu. The schemes a the same
 * as in the powersave daemon. If there is any erro while the powersave daemon try to set 
 * the selected scheme, the user get a messagebox. 
 */
void kpowersave::do_setActiveScheme(int i){
	myDebug ("kpowersave::do_setActiveScheme");

	int question = -1;

	if (org_schemenames[ i ] == "AdvancedPowersave") {
		QString _msg = i18n("This scheme support suspend and disable devices. This is a EXPERIMENTAL "
				    "feature of powersave to save more power on laptops. If you select this "
				    "scheme you can get in trouble (e.g. kernel oops). You can configure this "
				    "feature currently in the powersave config files. Please report all "
				    "problems via http://www.opensuse.org/Submit_a_bug \n\n"
				    "Would you switch the scheme (on your own risk)?");
		question = KMessageBox::warningContinueCancel(0, _msg, i18n("Warning"), 
							     i18n("Continue"), "dontAskAgainAdvancedPS",
							     KMessageBox::Dangerous);

		if (question == KMessageBox::Cancel)
			return;
	}
	// use the real name of the scheme now !!!
	int ret = pdaemon->setActiveScheme(org_schemenames[ i ]);
	if (ret < 0)
		KPassivePopup::message(i18n("WARNING"),
		                        i18n("Scheme %1 could not be activated.").arg(scheme_menu->text(i)),
		                        SmallIcon("messagebox_warning", 20), this, i18n("Warning"), 15000);
	else{
		for (int x = 0; x < (int) scheme_menu->count(); x++){
			if (x == i)
				scheme_menu->setItemChecked(x, true);
			else
				scheme_menu->setItemChecked(x, false);
		}
		pdaemon->updateSchemeInfo();
		update();
	}
}

/*!
 * This function is invoked if something has to be updated. This including also menu entries.
 * If the battery is in warning state (and powersave set pdaemon->send_battery_state_change_message) 
 * the function pop-up a messagebox.
 */
void kpowersave::update(){
	myDebug ("kpowersave::update");
	
	int redraw_pixmap = 0;
	QString justMins;
	
	
	/* need to redraw pixmap in toolbar */
	if (pdaemon->update_info_ac_changed == true){
		redraw_pixmap = 1;
		if(pdaemon->on_AC_power != HAL_ERROR && pdaemon->on_AC_power != UNKNOWN)
			hal_error_shown = false;

		pdaemon->update_info_ac_changed = FALSE;
	}
	if (pdaemon->update_info_sleep_state_changed == true){
		if (pdaemon->daemon_running <= 0){
			this->contextMenu()->setItemVisible(SUSPEND2DISK_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(SUSPEND2RAM_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(STANDBY_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(SLEEP_SEPARATOR_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(SCHEME_SEPARATOR_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(SCHEME_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(HELP_SEPARATOR_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(KPOWERSAVE_CONFIGURE_ID, FALSE);
			this->contextMenu()->setItemVisible(KPOWERSAVE_CONFIGURE_EVENTS_ID, FALSE);
		}
		else{
			this->contextMenu()->setItemVisible(SUSPEND2DISK_MENU_ID, true);
			this->contextMenu()->setItemVisible(SUSPEND2RAM_MENU_ID, true);
			this->contextMenu()->setItemVisible(STANDBY_MENU_ID, true);
			this->contextMenu()->setItemVisible(SLEEP_SEPARATOR_MENU_ID, true);
			this->contextMenu()->setItemVisible(SCHEME_SEPARATOR_MENU_ID, true);
			this->contextMenu()->setItemVisible(SCHEME_MENU_ID, true);
			this->contextMenu()->setItemVisible(HELP_SEPARATOR_MENU_ID, true);
			this->contextMenu()->setItemVisible(KPOWERSAVE_CONFIGURE_ID, true);
			this->contextMenu()->setItemVisible(KPOWERSAVE_CONFIGURE_EVENTS_ID, true);
			
			if (pdaemon->suspend2disk_allowed == 1)
				this->contextMenu()->setItemEnabled(SUSPEND2DISK_MENU_ID, true);
			else
				this->contextMenu()->setItemEnabled(SUSPEND2DISK_MENU_ID, FALSE);
			if (pdaemon->suspend2ram_allowed == 1)
				this->contextMenu()->setItemEnabled(SUSPEND2RAM_MENU_ID, true);
			else
				this->contextMenu()->setItemEnabled(SUSPEND2RAM_MENU_ID, FALSE);
			if (pdaemon->standby_allowed == 1)
				this->contextMenu()->setItemEnabled(STANDBY_MENU_ID, true);
			else
				this->contextMenu()->setItemEnabled(STANDBY_MENU_ID, FALSE);
		}
		pdaemon->update_info_sleep_state_changed = FALSE;
	}
	if (pdaemon->update_info_cpufreq_policy_changed == true){
		updateCPUFreqMenu();
		pdaemon->update_info_cpufreq_policy_changed = FALSE;
	}
	if (pdaemon->update_info_battery_state_changed == true){
		justMins.setNum(pdaemon->left % 60);
		justMins = justMins.rightJustify(2, '0');
		
		/* set fixed battery limits if daemon is not available */
		if (pdaemon->send_battery_state_change_message){
			if (pdaemon->battery_state != BAT_HAL_ERROR )
				hal_error_shown = false;

			if ((pdaemon->battery_state == BAT_HAL_ERROR) && !hal_error_shown) {
				if(!DISPLAY_HAL_ERROR_Timer->isActive() && !hal_error_shown) {
					hal_error_shown = true;
					DISPLAY_HAL_ERROR_Timer->start(HAL_ERROR_MSG_intervall, true);
				}
			}

			pdaemon->send_battery_state_change_message = FALSE;
		}
		redraw_pixmap = 1;
		pdaemon->update_info_battery_state_changed = FALSE;
	}
	if (pdaemon->update_info_battery_charge_changed == true){
		pdaemon->update_info_battery_charge_changed = FALSE;
	}
	if (pdaemon->update_info_battery_perc_changed == true){
		redraw_pixmap = 1;
		pdaemon->update_info_battery_perc_changed = FALSE;
	}
	if (pdaemon->update_info_cpufreq_speed_changed == true){
		pdaemon->update_info_cpufreq_speed_changed = FALSE;
	}
	if (pdaemon->update_info_scheme_info_changed == true){
		updateSchemeMenu();
		pdaemon->update_info_scheme_info_changed = FALSE;
	}
	if (redraw_pixmap){
		redrawPixmap();
	}
}

/*!
 * This function is involved if the CPUFreqMenu must be updated. The user selected
 * or by the powersave daemon set CPUFreq entry is set checked. 
 */
void kpowersave::updateCPUFreqMenu(){
	myDebug ("kpowersave::updateCPUFreqMenu");

	/* set CPU frequency menu entries *********/
	/* speed menu has id 3 in context menu */
	if (pdaemon->cpufreq_policy < 0 || pdaemon->cpufreq_policy == CPU_UNSUPP){
		/* there never were policies */
		if (!speed_menu)
			return ;
		else{
			/* there were CPU freq policies, but they are not accessible any more */
			/* delete speed_menu */
			contextMenu()->setItemVisible(CPUFREQ_MENU_ID, false);
			contextMenu()->setItemVisible(CPUFREQ_SEPARATOR_MENU_ID, false);
			return ;
		}
	}
	contextMenu()->setItemVisible(CPUFREQ_MENU_ID, true);
	contextMenu()->setItemVisible(CPUFREQ_SEPARATOR_MENU_ID, true);

	switch (pdaemon->cpufreq_policy){
		case CPU_HIGH:
			speed_menu->setItemChecked(CPU_HIGH, true);
			speed_menu->setItemChecked(CPU_AUTO, false);
			speed_menu->setItemChecked(CPU_LOW, false);
			break;
		case CPU_AUTO:
			speed_menu->setItemChecked(CPU_HIGH, false);
			speed_menu->setItemChecked(CPU_AUTO, true);
			speed_menu->setItemChecked(CPU_LOW, false);
			break;
		case CPU_LOW:
			speed_menu->setItemChecked(CPU_HIGH, false);
			speed_menu->setItemChecked(CPU_AUTO, false);
			speed_menu->setItemChecked(CPU_LOW, true);
			break;
	}
}

/*!
 * The function used to update the scheme menu. A update is needed if 
 * the powersave daemon send maybe new schemes or if the daemon change the 
 * active scheme. By this way also set the settings for screensaver related to
 * the selected scheme.
 */
void kpowersave::updateSchemeMenu(){
	myDebug ("kpowersave::updateSchemeMenu");

	if (pdaemon->schemes.count() == 0){
		/* there never were schemes */
		if (!scheme_menu)
			return ;
		else{
			/* there were schemes, but they are not accessible any more */
			/* delete scheme_menu */
			scheme_menu->clear();
			contextMenu()->setItemVisible(SCHEME_MENU_ID, false);
			contextMenu()->setItemVisible(SCHEME_SEPARATOR_MENU_ID, false);
			return ;
		}
	}

	/* redraw all scheme entries ... */
	scheme_menu->clear();
	// clear the list of real scheme names
	org_schemenames.clear();

	org_schemenames = pdaemon->schemes.schemeList();

	int x = 0;
	for ( QStringList::iterator it = org_schemenames.begin(); it != org_schemenames.end(); ++it ) {

		if ( *it == pdaemon->schemes.acScheme() ){
			scheme_menu->insertItem( SmallIcon("scheme_power", QIconSet::Automatic),
						 i18n( (QString)*it ), x, x);
		}
		else{
			if ( *it == pdaemon->schemes.batteryScheme() ){
				scheme_menu->insertItem(SmallIcon("scheme_powersave", QIconSet::Automatic),
							i18n( *it ), x, x);
			}
			else{
				if ((QString)*it == "Acoustic"){ 
					scheme_menu->insertItem(SmallIcon("scheme_acoustic", QIconSet::Automatic),
								i18n("Acoustic"), x, x);
				}
				else if ((QString)*it == "Presentation"){
					scheme_menu->insertItem(SmallIcon("scheme_presentation", QIconSet::Automatic),
								i18n("Presentation"), x, x);
					
				}
				else if((QString)*it == "AdvancedPowersave") {
					scheme_menu->insertItem(SmallIcon("scheme_advanced_powersave", QIconSet::Automatic),
								i18n( "Advanced Powersave" ), x, x);
				}
				else {
					scheme_menu->insertItem(i18n( *it ), x, x);
				}
			}
		}

		if ( *it == pdaemon->schemes.currentScheme()) {
			scheme_menu->setItemChecked(x, true);
		}
		++x;
	}
	
	if (x == 0 && scheme_menu){
		// this should not happen, scheme_list should have been NULL before
		// now we'd have an empty menu ...
	}
	else{
		contextMenu()->setItemVisible(SCHEME_MENU_ID, true);
		contextMenu()->setItemVisible(SCHEME_SEPARATOR_MENU_ID, true);
	}

	/* set screen saver and dpms settings new ... */
	setSchemeSettings();
}


/*!
 * Reimplemented eventhandler for mouse enterEvent. This is called if the mouse cursor 
 * enters the widget. In this case if the user move the mouse cursor over the kpowersave
 * trayicon. \n \n
 * We use this event to update the Tooltip with all needed information. The time beetween
 * the event and the automatically popup of the QToolTip should be long enought to collect
 * the needed values and add a updated Tooltip.
 */
void kpowersave::enterEvent( QEvent *qee ){

	updateTooltip();
	
	if(qee == NULL) ; // to prevent compiler warning
}


/*!
 * Reimplemented eventhandler for mousePressEvents which is involved if the user click 
 * on the icon on the kickerapplet. This was written to guarantee that a click with the 
 * right and the left mousebutton activate the menu. In the future this can also used 
 * to popup maybe a other menu. 
 */
void kpowersave::mousePressEvent(QMouseEvent *qme){
	myDebug ("kpowersave::mousePressEvent");

	KSystemTray::mousePressEvent(qme);
	if (qme->button() == RightButton){
		/* schemes are only checked if daemon is running... */
		if (pdaemon->daemon_running != 1){
			/* *
			 * force recheck independently from running QTimers
			 * this should fix problems if powersaved died somehow
			 * and is restart by user, hopefully (see Bugzilla #47113)
			 * */
			pdaemon->checkDaemon();
		}
		if (pdaemon->daemon_running == 1){
			pdaemon->checkDaemon();
			/* request schemes from daemon */
			pdaemon->updateSchemeInfo();
		}
		if (qme->button() == LeftButton){
			this->contextMenu()->exec(QCursor::pos());
		}
	} else if (qme->button() == LeftButton && pdaemon->daemon_running) {

		if (detailedIsShown) {
			detailedDlg->close();
			delete(detailedDlg);
			closedetaileddialog();
			return;
		}

		detailedDlg = new detaileddialog(pdaemon, &fullIcon);

		if (detailedDlg) {
			detailedDlg->show();
			detailedIsShown = true;
		}

		connect(detailedDlg, SIGNAL(destroyed()), this, SLOT(closedetaileddialog()));
	}
}

/*!
 * \b SLOT called if the detaileddialog is closed. With this we prevent open 
 * the dialog twice, use this function to reset the used variables.
 */
void kpowersave::closedetaileddialog() {
	detailedIsShown = false;
}

/*!
 * \b SLOT used to display messeges from powersave in kpowersave. This function 
 * block all messeges which we have in kpowersave!
 */
void kpowersave::showClientMessage( QString msg ){
	myDebug ("kpowersave::showClientMessage");

	if(!msg.endsWith("BATTERY WARNING") && !msg.endsWith("BATTERY LOW") && 
	   !msg.endsWith("BATTERY CRITICAL") && !msg.endsWith("BATTERY NORMAL")) {
		
		if(current_settings->psMsgAsPassivePopup) {
			KPassivePopup::message("Powersave Daemon", msg, 
					       SmallIcon("messagebox_warning", 20), 
			       		       this, i18n("Warning"), 10000);
		}
		else {
			kapp->updateUserTimestamp();
//			KMessageBox::error( 0, msg);
			KMessageBox::queuedMessageBox(0, KMessageBox::Error, msg);
		}
	}
}

/*!
 * \b SLOT used to display messeges from powersave in kpowersave. This function 
 * block all messeges which we have in kpowersave!
 */
void kpowersave::showPowersaveErrorMsg( ){
	myDebug ("kpowersave::showPowersaveErrorMsg");

	QString msg = i18n("The powersave daemon is not running.\n"
			   "Starting it will improve performance: "
			   "/etc/init.d/powersaved start");
	
	infoDialog *dlg = new infoDialog( config, i18n("Warning"), msg, 
					 i18n("Don't show this message again."),
					 "powersaveNotRunning" );

	if (!dlg->dialogIsDisabled()) dlg->show();
}


/*!
 * This function/SLOT is invoked if the powersave daemon send messages
 * with status information to kpowersave. This function open a dialog with a
 * Progressbar.
 * \todo 	fix for new event and update function docu
 * \todo	check for translation if new translation files with text from
 *		powersave scripts
 */
/*void kpowersave::showProgressDialog( QString msg, int percent ) {
	myDebug("kpowersave::showProgressDialog");
	
	
	// if conversion from string to int failed
	if ( !percent )
		return;

	// if the dialog should be closed
	if ( (percent < 0 || percent > 100) && suspend_dialog_shown ) {
		suspendDlg->close();
		suspend_dialog_shown = false;
		delete suspendDlg;
		suspendDlg = NULL;
		return;
	}
		
	// if it is not shown already, allocate memory and show it
	if ( ! suspend_dialog_shown ) {
		suspendDlg = new suspendDialog();
		suspendDlg->setPixmap(suspendType);
		kapp->updateUserTimestamp();
		suspendDlg->show();
		suspendDlg->setProgressbar(percent);
		if(!msg.isEmpty())
			suspendDlg->setTextLabel(msg);
		else 
			suspendDlg->setTextLabel("");

		suspend_dialog_shown = true;
		connect(suspendDlg, SIGNAL(destroyed()), this, SLOT(closeProgressDialog()));
	}

	// if the bar is already shown, set the progress
	if ( suspend_dialog_shown ) {
		suspendDlg->setProgressbar(percent);
		if(!msg.isEmpty())
			suspendDlg->setTextLabel(msg);
		else 
			suspendDlg->setTextLabel("");
	}
}*/
void kpowersave::showProgressDialog( QString msg ) {
	myDebug("kpowersave::showProgressDialog");

	QStringList msglist = QStringList::split("|", msg, false);
	int percent = 0;
	if(!msglist.isEmpty()) percent = msglist[0].toInt();
	else percent = msg.toInt();

	// if conversion from string to int failed
	if ( !percent )
		return;

	// if the dialog should be closed
	if ( (percent < 0 || percent > 100) && suspend_dialog_shown ) {
		suspendDlg->close();
		suspend_dialog_shown = false;
		delete suspendDlg;
		suspendDlg = NULL;
		return;
	}
		
	// if it is not shown already, allocate memory and show it
	if ( ! suspend_dialog_shown ) {
		suspendDlg = new suspendDialog();
		suspendDlg->setPixmap(suspendType);
		kapp->updateUserTimestamp();
		suspendDlg->show();
		suspendDlg->setProgressbar(percent);
		if(!msglist.isEmpty()) {
			if(msglist.count() > 1) suspendDlg->setTextLabel(translateProgressMsg(msglist[1]));
			else suspendDlg->setTextLabel("");
		}
		suspend_dialog_shown = true;
		connect(suspendDlg, SIGNAL(destroyed()), this, SLOT(closeProgressDialog()));
	}

	// if the bar is already shown, set the progress
	if ( suspend_dialog_shown ) {
		suspendDlg->setProgressbar(percent);
		if(!msglist.isEmpty()) {
			if(msglist.count() > 1) suspendDlg->setTextLabel(translateProgressMsg(msglist[1]));
			else suspendDlg->setTextLabel("");
		}
	}
}

/*!
 * This function translate the message for the suspend progress dialog.
 * This is needed to find the fix part of the message and preplace them 
 * with the new translated part.
 * \param message 	QString with the untranslated message
 * \return 		QString with the translated message
 */
QString kpowersave::translateProgressMsg( QString message ) {
	
	if (message.startsWith("checking resume kernel")) {
		return message.replace("checking resume kernel", i18n("checking resume kernel"));
	} else if (message.startsWith("checking for swap partition")) {
		return message.replace("checking for swap partition", i18n("checking for swap partition"));
	} else if (message.startsWith("unmounting FAT/NTFS partitions")) {
		return message.replace("unmounting FAT/NTFS partitions", i18n("unmounting FAT/NTFS partitions"));
	} else if (message.startsWith("stopping services")) {
		return message.replace("stopping services", i18n("stopping services"));
	} else if (message.startsWith("stopping service:")) {
		return message.replace("stopping service:", i18n("stopping service:"));
	} else if (message.startsWith("unloading modules")) {
		return message.replace("unloading modules", i18n("unloading modules"));
	} else if (message.startsWith("unloading module:")) {
		return message.replace("unloading module:", i18n("unloading module:"));
	} else if (message.startsWith("syncing file systems")) {
		return message.replace("syncing file systems", i18n("syncing file systems"));
	} else if (message.startsWith("preparing bootloader")) {
		return message.replace("preparing bootloader", i18n("preparing bootloader"));
	} else
		return message;
}


/*!
 * This SLOT is used to reset the suspend/progress dialog if the user closed 
 * the dialog.
 */
void kpowersave::closeProgressDialog() {
	myDebug("kpowersave::closeProgressDialog()");
	
	suspend_dialog_shown = false;
}


/*!
 * Use this function to set the SchemeSettings. This function currently set the 
 * e.g. the screensaver and dpms settings. Don't forget to call this function if
 * a scheme is changed or if the settings changed.
 */
void kpowersave::setSchemeSettings(){
	myDebug("kpowersave::setSchemeSettings()");

	if ( ! pdaemon->schemes.currentScheme().isEmpty()){
		current_settings->load_scheme_settings( pdaemon->schemes.currentScheme() );
		if(current_settings->autoSuspend) {
			setAutoSuspend(false);
		}
		else {
			this->contextMenu()->setItemVisible(AUTOSUSPEND_MENU_ID, FALSE);
			this->contextMenu()->setItemChecked(AUTOSUSPEND_MENU_ID, FALSE);
			this->contextMenu()->setItemVisible(AUTOSUSPEND_SEPARATOR_MENU_ID, FALSE);
			autoSuspend->stop();
		}
		
		if(current_settings->specSsSettings){
			if(current_settings->disableSs) new_screen->setScreenSaver(false);
			else {
				new_screen->setScreenSaver(true);
				if(current_settings->blankSs) new_screen->blankOnlyScreen(true);
				else {
					new_screen->blankOnlyScreen(false);
				}
			}
			
		}
		else if(getenv("KDE_FULL_SESSION")) {
			current_settings->load_kde_settings();
			// reset to global screensaver settings
			if(!current_settings->kde_settings->enabled) new_screen->setScreenSaver(false);
			else {
				new_screen->setScreenSaver(true);
				// What should we do with current_settings->kde_settings->lock ?
				// Maybe nothing ?!
			}	
		}
		// use this to set XScreensaver back to default settings this should
		// also cover the DPMS settings for GNOME/XScreensaver
		else if ((getenv("DESKTOP_SESSION") == "gnome") && (new_screen->checkScreenSaverStatus() == 11)) {
			if (xscreensaver != NULL)
				delete xscreensaver;

			xscreensaver = new KProcess;
			*xscreensaver << "xscreensaver-command" << "-restart";
			xscreensaver->start(KProcess::DontCare);
		}
		
		if(current_settings->specPMSettings){
			// set the new DPMS settings
			if(current_settings->disableDPMS) {
				new_screen->setDPMS(false);
			}
			else {
				new_screen->setDPMS(true);
				new_screen->has_DPMS = new_screen->setDPMSTimeouts(current_settings->standbyAfter,
										   current_settings->suspendAfter,
										   current_settings->powerOffAfter);
			}
		}
		else  if(getenv("KDE_FULL_SESSION")){
			current_settings->load_kde_settings();
			// reset to global screensaver settings
			if(!current_settings->kde_settings->displayEnergySaving) {
				new_screen->setDPMS(false);
			}
			else {
				new_screen->setDPMS(true);
				new_screen->has_DPMS = new_screen->setDPMSTimeouts(
									current_settings->kde_settings->displayStandby,
									current_settings->kde_settings->displaySuspend,
									current_settings->kde_settings->displayPowerOff);
			}
		}
		
		if(current_settings->brightness && brightness_supported) {
			// set to given values
			pdaemon->setBrightness(current_settings->brightnessValue, true);
		}
		else {
			//TODO: what should we do? reset back to 100 if changed?
		}
	}
}

/*! 
 * \b SLOT to display the HAL error message. We use this 
 * function to delay the message, and prevent display them is HAL or
 * powersaved restarted.
 */
void kpowersave::showHalErrorMsg() {
	myDebug("kpowersave::showHalErrorMsg");

	if(hal_error_shown && !DISPLAY_HAL_ERROR_Timer->isActive()) {
		KPassivePopup::message(i18n("ERROR"),
				       i18n("Could not get information from HAL. The haldaemon is maybe not running."),
				       SmallIcon("messagebox_warning", 20), this, i18n("Error"), 3000);
	}

	return;
}

/*!
 * \b SLOT which called to set and start the autosuspend monitoring.
 * \param resumed boolean value which represent information if machine 
 *		  currently back from suspend/standby
 */
void kpowersave::setAutoSuspend( bool resumed ){
	myDebug("kpowersave::setAutoSuspend");

	if(current_settings->autoInactiveActionAfter > 0 && current_settings->autoSuspend) {
		if( current_settings->autoInactiveAction.startsWith("_NONE_")) {
			autoSuspend->stop();
			return;
		}
		if (resumed) {
			delete autoSuspend;
			autoSuspend = NULL;
			autoSuspend = new autosuspend();
			connect(autoSuspend, SIGNAL(inactivityTimeExpired()), this, SLOT(do_autosuspend()));
			connect(autoSuspend, SIGNAL(inactivityTimeExpired()), this, SLOT(notifyAutosuspend()));
		}
		
		if(current_settings->autoInactiveSBlistEnabled) {
			autoSuspend->start( current_settings->autoInactiveActionAfter * 60, 
					    current_settings->autoInactiveSBlist );
		}
		else {
			autoSuspend->start( current_settings->autoInactiveActionAfter * 60, 
					    current_settings->autoInactiveGBlist );
		}
		this->contextMenu()->setItemVisible(AUTOSUSPEND_SEPARATOR_MENU_ID, true);
		this->contextMenu()->setItemVisible(AUTOSUSPEND_MENU_ID, true);
	}
	else {
		this->contextMenu()->setItemVisible(AUTOSUSPEND_MENU_ID, FALSE);
		this->contextMenu()->setItemVisible(AUTOSUSPEND_SEPARATOR_MENU_ID, FALSE);
	}
}

// -------- start KNotify functions ------------- //

/*!
 * \b SLOT called if a battery warning state reached and related signal recieved. 
 * Here we emit the related KNotify event, if not disabled.
 * \param state QString represent the reached event
 */
void kpowersave::notifyBatteryStatusChange ( QString state ) {
	if(!current_settings->disableNotifications) {
		if(state == "battery.warning") {
			KNotifyClient::event(this->winId(), "battery_warning_event",
					     i18n("Battery state changed to WARNING -- remaining time: "
						  "%1 hours and %2 minutes.").arg(pdaemon->left / 60).arg(pdaemon->left % 60));
		}
		else if(state == "battery.low") {
			KNotifyClient::event(this->winId(), "battery_low_event",
					     i18n("Battery state changed to LOW -- remaining time: "
						  "%1 hours and %2 minutes.").arg(pdaemon->left / 60).arg(pdaemon->left % 60));
	
		}
		else if(state == "battery.critical") {
			KNotifyClient::event(this->winId(), "battery_critical_event",
					     i18n("Battery state changed to CRITICAL -- remaining time: "
						  "%1 hours and %2 minutes. Shut down your system or plug "
						  "in the power cable immediately.").arg(pdaemon->left / 60).arg(pdaemon->left % 60));
	
		} 
	}
}

/*!
 * \b SLOT called if ac stated changed. Here we emit the related KNotify event.
 * \param acstate boolean represent the state of AC (true == AC plugged in ...)
 */
void kpowersave::notifyACStatusChange ( bool acstate ) {
	if(!current_settings->disableNotifications) {
		if (acstate) {
			KNotifyClient::event(this->winId(), "plug_event", i18n("AC adapter plugged in"));
		} else {
			KNotifyClient::event(this->winId(), "unplug_event", i18n("AC adapter unplugged"));
		}
	}
}

/*!
 * \b SLOT called if autosuspend started. Here we emit the KNotify event.
 */
void kpowersave::notifyAutosuspend() {
	myDebug("kpowersave::notifyAutosuspend");

	if(!current_settings->disableNotifications) {
		KNotifyClient::event( this->winId(), "autosuspend_event", 
				      i18n("System is going into suspend mode now"));
	}
}

/*!
 * \b SLOT called if scheme switched. Here we emit the related KNotify events
 * if they are not disabled.
 */
void kpowersave::notifySchemeSwitch() {
	myDebug("kpowersave::notifySchemeSwitch");

	if(!current_settings->disableNotifications && pdaemon->update_info_scheme_info_changed ) {
		QString _scheme = pdaemon->schemes.currentScheme();
		QString eventType;

		if( _scheme != "Performance" && _scheme != "Powersave" && _scheme != "Acoustic" &&
		    _scheme != "Presentation" && _scheme != "AdvancedPowersave" )
			eventType = "scheme_Unknown";
		else
			eventType = "scheme_" + _scheme;

		KNotifyClient::event( this->winId(), eventType, 
				      i18n("Powersave switched to scheme: %1").arg(i18n(_scheme)));
	}
}

/*!
 * \b SLOT called if the machine suspend. Here we emit the related KNotify events
 * if they are not disabled.
 */
void kpowersave::notifySuspend( QString suspendType ) {
	myDebug("kpowersave::notifySuspend");

	if(!current_settings->disableNotifications) {
		if (suspendType == "suspend2disk")
			KNotifyClient::event( this->winId(), "suspend2disk_event", 
					      i18n("System is going into %1 now.").arg(i18n("Suspend to Disk")));
		else if (suspendType == "suspend2ram")
			KNotifyClient::event( this->winId(), "suspend2ram_event", 
					      i18n("System is going into %1 now.").arg(i18n("Suspend to RAM")));
		else if(suspendType == "standby")
			KNotifyClient::event( this->winId(), "standby_event", 
					      i18n("System is going into %1 now.").arg(i18n("Standby")));
	}
}

/*!
 * \b SLOT called if the machine suspend. Here we emit the related KNotify events
 * if they are not disabled.
 */
void kpowersave::notifyResume( QString resumeType ) {
	myDebug("kpowersave::notifyResume");

	if(!current_settings->disableNotifications) {
		if (resumeType == "suspend2disk")
			KNotifyClient::event( this->winId(), "resume_from_suspend2disk_event", 
					      i18n("System is resumed from %1.").arg(i18n("Suspend to Disk")));
		else if (resumeType == "suspend2ram")
			KNotifyClient::event( this->winId(), "resume_from_suspend2ram_event", 
					      i18n("System is resumed from %1.").arg(i18n("Suspend to RAM")));
		else if(resumeType == "standby")
			KNotifyClient::event( this->winId(), "resume_from_standby_event", 
					      i18n("System is resumed from %1.").arg(i18n("Standby")));
	}
}

// -------- end KNotify functions ------------- //
// ------------ DCOP functions ---------------- //

/*!
 * DCOP Interface funtion to lock the screen with the userdefined methode.
 * \return boolean with the result of locking the screen
 * \retval true if locking the screen was successful
 * \retval false if locking the screen failed or the user don't wan't to lock
 */
bool kpowersave::lock_screen(){
	myDebug("kpowersave::lock_screen");
	
	current_settings->load_general_settings();
	
	if(current_settings->lockOnSuspend) {
		return new_screen->lockScreen( current_settings->lockmethod );
	}
	else return false;
}

/*!
 * DCOP Interface funtion to return the name of the current powersave scheme.
 * \return QString with the name of the current scheme
 */
QString kpowersave::currentScheme (){
	myDebug("kpowersave::currentScheme");
	
	if(pdaemon->daemon_running == 1) {
		return pdaemon->schemes.currentScheme();
	}
	else return "ERROR: powersaved not running";
}

/*!
 * DCOP Interface funtion to return the name of the current cpuFreqPolicy.
 * \return QString with the name of the current cpuFreqPolicy
 */
QString kpowersave::currentCPUFreqPolicy() {
	myDebug("kpowersave::currentCPUFreqPolicy");

	if(pdaemon->daemon_running == 1) {
		QString _cpuFreq = "";
		switch (pdaemon->cpufreq_policy){
			case CPU_HIGH:
				_cpuFreq = "PERFORMANCE";
				break;
			case CPU_AUTO:
				_cpuFreq = "DYNAMIC";
				break;
			case CPU_LOW:
				_cpuFreq = "POWERSAVE";
				break;
		}
		return _cpuFreq;
	}
	else return "ERROR: powersaved not running";
}

/*!
 * DCOP Interface funtion to send a list with the supported and enabled 
 * sleeping states.
 * \return QStringList with the supported spleeping states
 */
QStringList kpowersave::allowed_sleepingStates(){
	myDebug("kpowersave::allowed_sleepingStates");
	
	QStringList sleepList;	
	if(pdaemon->daemon_running == 1) {
		if (pdaemon->suspend2disk_allowed){ 
			sleepList.append("suspendToDisk");
		}
		if (pdaemon->suspend2ram_allowed){
			sleepList.append("suspendToRAM");
		}
		if (pdaemon->standby_allowed){
			sleepList.append("standBy");
		}	
		if(sleepList.isEmpty()){
			sleepList.append("NO_SLEEPING_STATES_SUPPORTED");
		}
	}
	else {
		sleepList.append("ERROR: powersaved not running");
	}
	return sleepList;
}

/*!
 * DCOP Interface funtion to send a list with the all schemes.
 * \return QStringList with all schemes
 */
QStringList kpowersave::list_schemes(){
	myDebug("kpowersave::list_schemes");

	QStringList _schemeList;	
	if(pdaemon->daemon_running == 1) {
		if (pdaemon->schemes.count() > 0){
			_schemeList = pdaemon->schemes.schemeList();
		}
	}
	else {
		_schemeList.append("ERROR: powersaved not running");
	}
	return _schemeList;
}

/*!
 * DCOP Interface funtion to send the suspend to disk command to powersave.
 * \return boolean with the result of calling do_suspend2disk()
 * \retval true if successful
 * \retval false if not supported or powersaved not running
 */
bool kpowersave::do_suspendToDisk(){
	myDebug("kpowersave::do_suspendToDisk");

	return do_suspend2disk();
}

/*!
 * DCOP Interface funtion to send the suspend to disk command to powersave.
 * \return boolean with the result of calling do_suspend2ram()
 * \retval true if successful
 * \retval false if not supported or powersaved not running
 */
bool kpowersave::do_suspendToRAM(){
	myDebug("kpowersave::do_suspendToRAM");

	return do_suspend2ram();
}

/*!
 * DCOP Interface funtion to send the suspend to disk command to powersave.
 * \return boolean with the result of calling do_standby()
 * \retval true if successful
 * \retval false if not supported or powersaved not running
 */
bool kpowersave::do_standBy(){
	myDebug("kpowersave::do_standBy");

	return do_standby();
}

/*!
 * DCOP Interface funtion to stop/start the Autosuspend 
 * \param activate boolean which tell if the autosuspend should be stopped (true)
 *		   or started (flase).
 */
void  kpowersave::disableAutosuspend( bool activate ){
	myDebug("kpowersave::disableAutosuspend");	

	if(activate) autoSuspend->stop();
	else setAutoSuspend(true);
}


/*!
 * DCOP Interface funtion to open the configure dialog.
 * \return boolean with the result of open the dialog
 * \retval false if failed (e.g. powersave daemon is not running)
 * \retval true if correct opend 
 */
bool kpowersave::openConfigureDialog (){
	myDebug("kpowersave::openConfigureDialog");
	
	if(pdaemon->daemon_running == 1) {
		show_configure_dialog();
		return config_dialog_shown;
	}
	else return false;
}

#include "kpowersave.moc"
