/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "radiostation.h"

#include <KStandardDirs>
#include <KConfig>
#include <KTemporaryFile>

#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

radioStation::radioStation(const QPointer<QObject> parent,
                            const QPointer<QWidget> mainWidget,
                            const QString &configFileName) : ripping(parent)
{
  m_mainWidget = mainWidget;
  helper_setupConfigSystem(configFileName);
}

radioStation::radioStation(const QPointer<QObject> parent,
                            const QPointer<QWidget> mainWidget,
                            const QString &configFileName,
                            const qlonglong &index) : ripping(parent)
{
  setIndex(index);
  m_mainWidget = mainWidget;
  helper_setupConfigSystem(configFileName);
}

radioStation::~radioStation()
{
  delete config_skeleton;  /*  We have to delete the object on which config_skeleton
    points to - _befor_ deleting the object shared_config points on because the first
    makes use of the second. As the object shared_config points on is deleted when it's
    pointer (shared_config is a special pointer managing this) is deleted (and that's
    done - as the pointer is a member of this class - after the descturctor call) we
    can make sure by deleting config_skeleton explicitly here that the object config_skeleton
    points on is deleted _befor_ the object shared_config points on. I don't want to
    relay on the QObject-parentship-system because I don't know _when_ exactly the child
    object is deleted, and this way it is sure. */
  delete settingsDialog;  /* Maybe there exists still a settingsDialog. (Because settingsDialog
    deletes itself after closing, this is only the case if the settingsDialog is still open.)
    Because the parent isn't this object, but m_mainWidget, the settingsDialog will not be
    automatically deleted by Qt and we have to do it manually. */
}

inline void radioStation::helper_setupConfigSystem(const QString & configFileName)
{
  // variables
  QFile file;

  /* make sure that internal_configFileName contains the path of a file which is
  *  readable and writeable. If possible, using "configFile" option passed to the
  *  contructor. If not, create a new (empty) one. */
  file.setFileName(configFileName);
  // TODO: catch error message if file name is empty!
  // TODO: replace the following block with code, that doesn't open the file,
  // but only collects information
  //       if the files are acessible.
  if (file.open(QIODevice::ReadWrite)) { // file (which was passed through
    // "configFileName" option in the constructor) working fine. Use it.
    file.close();
    internal_configFileName = configFileName;
  } else {                                   // create a new file
    KTemporaryFile newConfigFile;
    newConfigFile.setPrefix(KStandardDirs::locateLocal("appdata", "stream_")); // TODO warum
    // nicht locate() ?
    newConfigFile.setSuffix("_rc");
    if (!newConfigFile.open()) {
      // TODO handle error...
    }
    newConfigFile.setAutoRemove(false);
    internal_configFileName = newConfigFile.fileName();
    newConfigFile.close();
  };

  // setup configuration system
  shared_config = KSharedConfig::openConfig(internal_configFileName, KConfig::SimpleConfig);
  // We don't use a full-featured config system (with kiosk system and system-wide settings),
  // but we only read our streamConfig file.
  config_skeleton = new settings_stream(shared_config); // no constructor where I
  // can pass a parent
  /* config_skeleton->setParent(this) would also be a bad idea;
  So here I don't use QObject's parent-child-system but delete
  the object in the destructor. See there for further explications. */
}

QString radioStation::configFileName() const
{
  return internal_configFileName;
}

PropertyValue radioStation::streamName() const
{
  return formatedStreamName(config_skeleton->info_streamName());
}

void radioStation::setStreamName(const QString streamName)
{
  if (config_skeleton->info_streamName() != streamName) {
    config_skeleton->setInfo_streamName(streamName);
    config_skeleton->writeConfig();
    ripping::setStreamName(streamName);
  };
}

PropertyValue radioStation::serverName() const
{
  return formatedServerName(config_skeleton->info_serverName());
}

void radioStation::setServerName(const QString serverName)
{
  if (config_skeleton->info_serverName() != serverName) {
    config_skeleton->setInfo_serverName(serverName);
    config_skeleton->writeConfig();
    ripping::setServerName(serverName);
  };
}

PropertyValue radioStation::bitrate() const
{
  return formatedBitrate(config_skeleton->info_bitrate());
}

void radioStation::setBitrate(qint64 bitrate)
{
  // config_skeleton makes sure that no invalid value is stored
  if (config_skeleton->info_bitrate() != bitrate) {
    config_skeleton->setInfo_bitrate(bitrate);
    config_skeleton->writeConfig();
    ripping::setBitrate(bitrate);
  };
}

PropertyValue radioStation::metaInterval() const
{
  return formatedMetaInterval(config_skeleton->info_metaInterval());
}

void radioStation::setMetaInterval(qint64 metaInterval)
{
  // config_skeleton makes sure that no value < -1 is stored...
  if (config_skeleton->info_metaInterval() != metaInterval) {
    config_skeleton->setInfo_metaInterval(metaInterval);
    config_skeleton->writeConfig();
    ripping::setMetaInterval(metaInterval);
  };
}

QString radioStation::serverUri() const
{
  return config_skeleton->serverUri();
}

QStringList radioStation::parameterList() const
{
  // variables
  QStringList parameters;

  //code
  //TODO: options who aren't in the kcfg's
  // calculate all parameters
  parameters += ripping::parameterList();

  parameters.append(QString("-u"));
  parameters.append(config_skeleton->advanced_userAgentString());  //TODO test if this
  // works when the string is empty

  parameters.append(QString("--xs_offset=%1").
                     arg(config_skeleton->offset()));

  /* The following code produces a search window which length corresponds exactly
  to the value in the settings. When the value is odd, the last half of the window
  is 1 ms longer. */
  parameters.append(QString("--xs_search_window=%1:%2").
                     arg(config_skeleton->searchWindow() / 2).
                     arg((config_skeleton->searchWindow() / 2) +
                          (config_skeleton->searchWindow() % 2)));

  parameters.append(QString("--xs_silence_length=%1").
                     arg(config_skeleton->silenceWindow()));

  if (config_skeleton->writeSingleFile()) {
    parameters.append(QString("-a"));    //TODO file name schemes for single files: append here
  };

  if (config_skeleton->writeSplittedFiles()) {
    //TODO file name schemes splitted files
    parameters.append(QString("--xs_padding=%1:%2").
                       arg(config_skeleton->splittedFilesPostpad()).
                       arg(config_skeleton->splittedFilesPrepad()));
  } else {
    parameters.append(QString("-A"));
  };

  parameters.append(QString("-k"));
  parameters.append(QString::number(config_skeleton->skipFirstXTracks()));

  if (config_skeleton->advanced_useTimeout()) {
    parameters.append(QString("-m"));
    parameters.append(QString::number(config_skeleton->advanced_timeoutAfterXSeconds()));
  }

  if (config_skeleton->truncateDuplicatesInIncomplete()) {
    parameters.append(QString("-T"));
  };

  return parameters;
}

int radioStation::helper_displaySettingsDialog(bool returnImmediately)
{
  // variables
  int exitCode;

  // code
  /* If the dialog object yet exists, we can display it directly and return.
  (Should normally not happen, because we delete it after the use. And the dialog
  is modal, so the user can't open a second dialog. We do this here just to be
  absolutly sure.) We are using name of the config file as name of the dialog because
  this name is unique! */
  if (settingsDialog.isNull()) {
    settingsDialog = new settings_stream_dialog(m_mainWidget,
                                                 internal_configFileName,
                                                 config_skeleton);
    connect(settingsDialog,
             SIGNAL(finished()),
             settingsDialog,
             SLOT(deleteLater()));
  };
  if (returnImmediately) {
    settingsDialog->show();
    exitCode = KDialog::Accepted;
  } else {
    exitCode = settingsDialog->exec();
  };
  return exitCode;
}

void radioStation::showSettingsDialog()
{
  helper_displaySettingsDialog(true);
}

int radioStation::execSettingsDialog()
{
  return helper_displaySettingsDialog(false);
}
