/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef STREAMRIPPER_BASE_H
#define STREAMRIPPER_BASE_H

#include "console_reader.h"

/** \brief This abstract class implements access to \e %streamripper.
*
*   It provides only some minimalistic infrastructure. To do so, it inherits from console_reader.
*
*   Furthermore it implements the property #index and the functions kill() and waitForFinished().
*/
class streamripper_base : public console_reader
{

     Q_OBJECT

     /** This property has no function in this class. You can use it
     * for whatever purpose you want. It's default value is \e 0.
     *
     * This property is quite useful when you
     * have many objects of a child class of this class and you want to identify the sender
     * of a signal - just emit \e index as parameter of the signal.
     * \li \e read: \code virtual qlonglong index() const \endcode
     * \li \e write: \code virtual void setIndex(const qlonglong value) \endcode
     * \li \e notify: \code indexChanged(qlonglong newIndex) \endcode
     *
     * \sa \code qlonglong internal_index \endcode */
     Q_PROPERTY(qlonglong index READ index WRITE setIndex NOTIFY indexChanged)

  public:
     //contructor(s) and destructor(s)
     /** Constructor of the class.
     * @param parent Sets the parent of this object. */
     streamripper_base(const QPointer<QObject> parent = 0);
     /** Conveniance constructor of the class.
     * @param parent Sets the parent of this object.
     * @param index      Initializes the property #index. */
     streamripper_base(const QPointer<QObject> parent, const qlonglong index);
     /** The destructor. */
     virtual ~streamripper_base();

     //properties
     /** See property #index. */
     virtual qlonglong index() const;
     /** See property #index. */
     virtual void setIndex(const qlonglong value);

  signals:
     /** See property #index. */
     void indexChanged(qlonglong newIndex);

  public slots:
     /** Calls <tt>#m_process</tt><tt>.%kill()</tt> and
     *   <tt>#m_process</tt><tt>.%waitForFinished()</tt>.
     *   This takes usually only some milliseconds, but sometimes up to 20 sec. */
     void abortStreamripper();
     /** This slot starts a \e %streamripper instance using #m_process. It doesn't care about if
     *   #m_process is yet running or not.
     *
     *   Uses:
     *   \li  <tt> streamripperCommand() </tt> as command
     *   \li  <tt> QStringList parameterList() const </tt> as list of parameters */
     virtual void startStreamripper();

  protected:
     /** This abstract function is called by startStreamripper() to determinate the parameters
     *   with which \e %streamripper will be called.
     *   \returns A \c QStringList with the parameters. */
     virtual QStringList parameterList() const = 0;
     /** This abstract function is called by startStreamripper() to determinate the command
     *   which is used to start \e %streamripper.
     *   \returns A \c QString containing the command. */
     virtual QString streamripperCommand() const = 0;

  private:
     /** Used internally to store the property #index.*/
     qlonglong internal_index;
 };

#endif
