/*
    Copyright (C) 2008-2009  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef ICECAST_INTERNALTHREAD_H
#define ICECAST_INTERNALTHREAD_H

#include <QList>
#include <QThread>
#include <QXmlStreamReader>
#include "streamdirectoryentry_stream.h"

/** \brief A thread that downloads and processes an Icecast stream list.
*
* This class provides downloading of the "yellow page" file from icecast.org which provides a list
* of all available Icecast streams.
*
* For each stream it creates a #streamDirectoryEntry_stream objects on the heap. After finishing,
* it emits the signal #streamlist_ready with a list of pointers to these objects. It is up to you
* to make sure that these objects on the head will get deleted! */

class icecast_internalThread : public QThread
{

  Q_OBJECT

  public:
    /** Defines a type for a list of pointers to #streamDirectoryEntry_stream objects. This type
    *   is declared as meta type through Q_DECLARE_METATYPE.
    *   \note Additionally, you have to register this data type to Qt's meta object system by
    *   calling <code>qRegisterMetaType<icecast_internalThread::streamEntryList>();</code> before
    *   you can use it in queued signal-slot-connections. */
    typedef QList<streamDirectoryEntry_stream *> streamEntryList;
    /** The constructor. */
    icecast_internalThread();
    /** The destructor.
    * \note Different from QThread, it is save to execute the desctructor on any time, also when
    * the thread is running. However, this may take some seconds... */
    virtual ~icecast_internalThread();

  signals:
    /** This signal is emitted after the thread has prepared the data structure.
    *   \note This signal is emitted from within the thread. So, when you connect it, this will
    *   be a connection between thread, which is a queued connection. Because of this, you have
    *   to register the data type #icecast_internalThread::streamEntryList to Qt's meta type
    *   system before the connection. See the respective documentation of the data type for
    *   details.
    *   @param list A list with pointers the the objects on the heap. It is up to you to make sure
    *   that these objects on the head will get deleted! */
    void streamlist_ready(icecast_internalThread::streamEntryList list);

  protected:
    /** Internal implementation of the thread. */
    virtual void run();

  private:
    /** Helper function that expects that the actual tocken is <tt>\<entry\></tt>. It reads the
    *   entry, constructs the corresponding object on the heap and adds a pointer to it at
    *   #streamList.
    *   @param reader A reference to the QXmlStreamReader object that you're using for
    *   processing. */
    void readStreamEntry(QXmlStreamReader & reader);
    /** Internal list of pointers to the objects on the heap. */
    streamEntryList streamList;

};

Q_DECLARE_METATYPE(icecast_internalThread::streamEntryList)

#endif
