/***************************************************************************
                          profiledlg.cpp  -  description
                             -------------------
    begin                : Sat Nov 20 2004
    copyright            : (C) 2004 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "profiledlg.h"
#include "krenameimpl.h"
#include "kmyhistorycombo.h"
#include "pluginloader.h"
#include "kmylistbox.h"

#include <kapplication.h>
#include <kcombobox.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kinputdialog.h> 
#include <klistbox.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>

#include <qcheckbox.h>
#include <qdom.h>
#include <qfile.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qtooltip.h>

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#ifndef VERSION
    #define VERSION "unknown"
#endif

#define PROFILE_WIZARD_INDEX 0
#define PROFILE_TABBED_INDEX 1 

ProfileDlg::ProfileDlg(KRenameImpl* krename, QWidget *parent, const char *name)
    : KDialogBase( KDialogBase::Plain, i18n("Profiles"),
      KDialogBase::Close, KDialogBase::Close, parent, name, true, true ), m_krename( krename )
{
    QHBoxLayout* layout = new QHBoxLayout( plainPage(), 6, 6 );
    QVBoxLayout* button = new QVBoxLayout( 0, 6, 6 );
    
    profiles = new KListBox( plainPage() );
    profiles->insertItem( i18n("KRename: Wizard default profile"), PROFILE_WIZARD_INDEX );
    profiles->insertItem( i18n("KRename: Tabbed default profile"), PROFILE_TABBED_INDEX );
    
    createProfile = new KPushButton( i18n("&Create Profile..."), plainPage() );
    loadProfile = new KPushButton( i18n("&Load Profile"), plainPage() );
    deleteProfile = new KPushButton( i18n("&Delete Profile"), plainPage() );
    
    createProfile->setIconSet( SmallIconSet( "filenew") );
    loadProfile->setIconSet( SmallIconSet( "fileopen" ) );
    deleteProfile->setIconSet( SmallIconSet( "edittrash" ) );
    
    QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Expanding );
    button->addWidget( createProfile );
    button->addWidget( loadProfile );
    button->addWidget( deleteProfile );
    button->addItem( spacer );
    
    layout->addWidget( profiles );
    layout->addLayout( button );
    layout->setStretchFactor( profiles, 2 );
    
    QToolTip::add( createProfile, i18n("<qt>Create a new profile from KRename's current settings. " 
                                       "The settings are saved and can be restored with Load Profile later.</qt>" ) );
    QToolTip::add( loadProfile, i18n("<qt>Load all settings stored in this profile.</qt>") );
    
    enableControls();
    
    connect( createProfile, SIGNAL( clicked() ), this, SLOT( slotCreateProfile() ) );
    connect( loadProfile, SIGNAL( clicked() ), this, SLOT( slotLoadProfile() ) );
    connect( deleteProfile, SIGNAL( clicked() ), this, SLOT( slotDeleteProfile() ) );
    connect( profiles, SIGNAL( selectionChanged () ), this, SLOT( enableControls() ) );
    
    KConfig* conf = kapp->config();
    conf->setGroup( "ProfilesHeader" );
    profiles->insertStringList( conf->readListEntry( "list" ) );
}


ProfileDlg::~ProfileDlg()
{
}


void ProfileDlg::enableControls()
{
    loadProfile->setEnabled( profiles->selectedItem() );
    deleteProfile->setEnabled( profiles->selectedItem() );
}

void ProfileDlg::slotLoadProfile()
{
    QString profile = profiles->currentText();
    QString msg = QString( i18n("Do you really want to load the profile and overwrite the current settings: %1") ).arg( profile );

    QString path;    
    if( profiles->currentItem() == PROFILE_WIZARD_INDEX )
        path = locate( "data", "krename/krename_system_default_wizard.xml" );
    else if( profiles->currentItem() == PROFILE_TABBED_INDEX )
        path = locate( "data", "krename/krename_system_default_tabbed.xml" );
    else
        path = getProfilePath( profile );

    if( path.isEmpty() )
    {
        KMessageBox::error( this, i18n("The profile \"%1\" could not be found.").arg( profile ) );
        return;
    }
    
    if( KMessageBox::questionYesNo( this, msg ) == KMessageBox::Yes )
    {
        loadXML( path );
        enableControls();
        
        m_krename->enableControls();
        m_krename->updatePre();
    }
}

void ProfileDlg::slotCreateProfile()
{
    bool ok = false;
    const char* mask = "xXXXXXXXXXXXXXXXXXXX"; // allows for 20 characters
    QString name = KInputDialog::getText( i18n("Profile Name"), i18n("Please enter a name for the new profile:"), 
                   "KRename", &ok, this, 0, 0, mask );
                   
    if( !ok )
        return;
        
    if( profiles->findItem( name, Qt::ExactMatch ) )
    {
        KMessageBox::error( this, i18n("This profile does already exist. Please choose another name.") );
        slotCreateProfile();
        return;
    }
    
    profiles->insertItem( name );
    writeXML( name );
    
    enableControls();
}

void ProfileDlg::slotDeleteProfile()
{
    if( profiles->currentItem() == PROFILE_WIZARD_INDEX || profiles->currentItem() == PROFILE_TABBED_INDEX )
    {
        KMessageBox::error( this, i18n("You cannot delete default profiles!") );
        return;
    }
    
    QString profile = profiles->currentText();
    QString msg = QString( i18n("Do you really want to delete the profile: %1") ).arg( profile );
    
    if( KMessageBox::questionYesNo( this, msg ) == KMessageBox::Yes )
    {
        QString path = readProfilePath( profile );
        QFile::remove( path );
        
        QListBoxItem* item = profiles->findItem( profile, Qt::ExactMatch );
        if( item )
            delete item;
        
        KConfig* conf = kapp->config();
        
        conf->setGroup( "ProfilesHeader" );
        QStringList list = conf->readListEntry( "list" );
        list.remove( profile );
        conf->writeEntry( "list", list );
        conf->sync();
            
        enableControls();
    }
}

const QString ProfileDlg::readProfilePath( const QString & name )
{
    QString path;
    KConfig* conf = kapp->config();
    
    conf->setGroup( "Profiles" );
    path = conf->readEntry( name, QString::null );
    
    return path;    
}

const QString ProfileDlg::getProfilePath( const QString & name )
{
    QStringList list;
    QString path = locateLocal( "data", QString( "krename/%1.xml" ).arg( name ) );
    
    KConfig* conf = kapp->config();

    conf->setGroup( "ProfilesHeader" );
    list = conf->readListEntry( "list" );
    if( !list.contains( name ) )
        list.append( name );
    conf->writeEntry( "list", list );
        
    conf->setGroup( "Profiles" );
    conf->writeEntry( name, path );
    conf->sync();
    
    return path;    
}

void ProfileDlg::writeXML( const QString & name )
{
    QString path = getProfilePath( name );
    QFile file( path );
    
    if( !file.open( IO_WriteOnly ) )
    {
        qDebug("Cannot write to: %s", path.latin1() );
        return;
    }
    
    QDomDocument doc( "KRenameProfile" );
    QDomElement root = doc.createElement( "krename" );
    doc.appendChild( root );

    QDomElement v = doc.createElement( "version" );
    v.setAttribute( "version", VERSION );
    root.appendChild( v );
    
    // General settings of Krename
    QDomElement settings = doc.createElement( "settings" );
    settings.setAttribute( "wizard", m_krename->m_wizard );
    settings.setAttribute( "fileplugins", m_krename->plugin->filePluginsLoaded() );
    root.appendChild( settings );
    
    // Filename settings
    QDomElement filename = doc.createElement( "filename" );
    filename.setAttribute( "filename", m_krename->filename->text() );
    filename.setAttribute( "extension", m_krename->extemplate->text() );
    filename.setAttribute( "extension_enabled", m_krename->checkExtension->isChecked() );
    filename.setAttribute( "extension_start", m_krename->comboExtension->currentItem() );
    
    QDomElement numbering = doc.createElement( "numbering" );
    numbering.setAttribute( "start", m_krename->m_index );
    numbering.setAttribute( "step", m_krename->m_step );
    numbering.setAttribute( "skip", listToString( m_krename->skip ) );
    
    QDomElement replace = doc.createElement( "replace" );
    for( unsigned int i=0;i<m_krename->rep.count();i++)
    {
        QDomElement r = doc.createElement( "item" );
        r.setAttribute( "find", m_krename->rep[i].find );
        r.setAttribute( "replace", m_krename->rep[i].replace );
        r.setAttribute( "reg", m_krename->rep[i].reg );
        
        replace.appendChild( r );
    }
    
    // destination settings
    QDomElement dest = doc.createElement( "destination" );
    dest.setAttribute( "mode", m_krename->currentRenameMode() );
    dest.setAttribute( "overwrite", QString::number( m_krename->checkOverwrite->isChecked() ) );
    dest.setAttribute( "directory", m_krename->dirname->text() );
    dest.setAttribute( "undo", QString::number( m_krename->checkUndoScript->isChecked() ) );
    dest.setAttribute( "undoscript", m_krename->undorequester->url() );
    
    // file adding settings
    QDomElement files = doc.createElement( "files" );
    files.setAttribute( "sorting", QString::number( m_krename->comboSort->currentItem() ) );
    files.setAttribute( "preview", QString::number( m_krename->checkPreview->isChecked() ) );
    files.setAttribute( "names", QString::number( m_krename->checkName->isChecked() ) );
    
    filename.appendChild( replace );
    filename.appendChild( numbering );
    root.appendChild( settings );
    root.appendChild( filename );
    root.appendChild( dest );
    root.appendChild( files );
    
    QCString xml = doc.toCString();
    file.writeBlock( xml, xml.length() );
    file.close();
}

void ProfileDlg::loadXML( const QString & path )
{
    QFile file( path );
    
    if( !file.open( IO_ReadOnly ) )
    {
        qDebug("Cannot read from: %s", path.latin1() );
        return;
    }

    QDomDocument doc( "KRenameProfile" );
    if ( !doc.setContent( &file ) ) 
    {
        file.close();
        return;
    }

    QDomNode n = doc.documentElement().firstChild();
    while( !n.isNull() ) 
    {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if( !e.isNull() )
        {
            if( e.tagName() == "settings" ) 
            {
                bool wiz, plug;
                wiz = (bool)e.attribute( "wizard", 
                            QString( "%1" ).arg( m_krename->m_wizard ) ).toInt();
                if( wiz != m_krename->m_wizard )
                {
                    m_krename->m_wizard = wiz;
                    m_krename->changeGUIMode();
                }
                    
                plug = (bool)e.attribute( "fileplugins", 
                             QString( "%1" ).arg( m_krename->plugin->filePluginsLoaded() ) ).toInt(); 
                if( plug && !m_krename->plugin->filePluginsLoaded() )
                    m_krename->plugin->loadPlugins( true );
            }
            else if( e.tagName() == "filename" )
            {
                m_krename->filename->setText( e.attribute("filename", m_krename->filename->text() ) );
                m_krename->extemplate->setText( e.attribute("extension", m_krename->extemplate->text() ) );
                m_krename->checkExtension->setChecked( 
                           (bool)e.attribute("extension_enabled",
                           QString("%1").arg(m_krename->checkExtension->isChecked() ) ).toInt() );
                m_krename->comboExtension->setCurrentItem( 
                            e.attribute("extension_start", 
                            QString::number( m_krename->comboExtension->currentItem() ) ).toInt() );
                            
                QDomNode n = e.firstChild();
                while( !n.isNull() ) 
                {
                    QDomElement e = n.toElement(); // try to convert the node to an element.
                    if( !e.isNull() )
                    {
                        if( e.tagName() == "numbering" )
                        {
                            m_krename->m_index = e.attribute( "start", QString::number( m_krename->m_index ) ).toInt();
                            m_krename->m_step = e.attribute( "step", QString::number( m_krename->m_step ) ).toInt();
                            m_krename->skip = stringToList( e.attribute("skip", listToString( m_krename->skip ) ) );
                        }
                        else if( e.tagName() == "replace" )
                        {
                            m_krename->rep.clear();
                            QDomNode n = e.firstChild();
                            while( !n.isNull() ) 
                            {
                                QDomElement e = n.toElement(); // try to convert the node to an element.
                                if( !e.isNull() && e.tagName() == "item" )
                                {
                                    replacestrings r;
                                    r.find = e.attribute( "find", QString::null );
                                    r.replace = e.attribute( "replace", QString::null );
                                    r.reg = (bool)e.attribute( "reg", "0" ).toInt();
                                    m_krename->rep.append( r );
                                }
                                n = n.nextSibling();
                            }
                        }
                    }
                    n = n.nextSibling();
                }
            }
            else if( e.tagName() == "destination" )
            {
                int mode = e.attribute( "mode", QString::number( m_krename->currentRenameMode() ) ).toInt();
                m_krename->optionRename->setChecked( false );
                m_krename->optionCopy->setChecked( false );
                m_krename->optionMove->setChecked( false );
                m_krename->optionLink->setChecked( false );
                
                switch( mode )
                {
                    default:
                    case RENAME:
                        m_krename->optionRename->setChecked( true ); break;
                    case COPY:
                        m_krename->optionCopy->setChecked( true ); break;
                    case MOVE:
                        m_krename->optionMove->setChecked( true ); break;
                    case LINK:
                        m_krename->optionLink->setChecked( true ); break;
                }
                    
                m_krename->checkOverwrite->setChecked( e.attribute( "overwrite", 
                           QString::number( m_krename->checkOverwrite->isChecked() ) ).toInt() );
                m_krename->dirname->setText( e.attribute( "directory", m_krename->dirname->text() ) );
                m_krename->checkUndoScript->setChecked( e.attribute( "undo",
                            QString::number( m_krename->checkUndoScript->isChecked() ) ).toInt() );
                m_krename->undorequester->setURL( e.attribute( "undoscript", m_krename->undorequester->url() ) );
            }
            else if( e.tagName() == "files" )
            {
                m_krename->comboSort->setCurrentItem( e.attribute( "sorting", 
                            QString::number( m_krename->comboSort->currentItem() ) ).toInt() );
                m_krename->checkPreview->setChecked( e.attribute( "preview", 
                            QString::number( m_krename->checkPreview->isChecked() ) ).toInt() );
                m_krename->checkName->setChecked( e.attribute( "names",
                            QString::number( m_krename->checkName->isChecked() ) ).toInt() );
            }
        }
        n = n.nextSibling();
    }
    
    if( m_krename->m_wizard )
        m_krename->parseWizardMode();
        
    file.close();
    
    close();
}

const QString ProfileDlg::listToString( QValueList<int> & list )
{
    QString data;
    for( unsigned int i = 0; i < list.count(); i++ )
        data += QString( "%1;" ).arg( list[i] );
        
    return data;
}

const QValueList<int> ProfileDlg::stringToList( const QString & data )
{
    QValueList<int> list;
    int c = data.contains( ";" );
    if( c > 0 )
    {
        for( int i = 0;i<c;i++)
            list.append( data.section( ';', i, i ).toInt() );
    }
    
    return list;
}

#include "profiledlg.moc"
