/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "Burn.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

#include "DodgeBurnConfigurationWidget.h"

KisBurnFilter::KisBurnFilter() 
    : KisFilter(id(), "Burn", i18n("&Burn..."))
{
}

std::list<KisFilterConfiguration*> KisBurnFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisBurnFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("type", 0);
    config->setProperty("exposure", 0.5);
    return config;
};

KisFilterConfigWidget * KisBurnFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    return new DodgeBurnConfigurationWidget(parent, "");
}

KisFilterConfiguration* KisBurnFilter::configuration(QWidget* nwidget)
{
    DodgeBurnConfigurationWidget* widget = (DodgeBurnConfigurationWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        return widget->configuration(id().id(),1);
    }
}

typedef void (*funcBurn)(const Q_UINT8* , Q_UINT8* , uint, double );

template<typename _TYPE, int norm>
void BurnShadow(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        double v = (sT[i] / (double)norm - exposure) / ( 1.0 - exposure) * norm;
        if(v < 0.0) v = 0.0;
        dT[i] = (_TYPE)(v ) ;
    }
}

template<typename _TYPE, int norm>
void BurnMidtones(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        dT[i] = (_TYPE)(pow( sT[i] / (double)norm, exposure) * norm) ;
    }
}

template<typename _TYPE, int max>
void BurnHighlights(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        double v = exposure * sT[i];
        if(v > max) v = max;
        dT[i] = (_TYPE)( v ) ;
    }
}

void KisBurnFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);
    
    double exposure;
    int type;
    if(config)
    {
        exposure = config->getDouble("exposure", 0.5);
        type = config->getInt("type", SHADOWS);
    } else {
        type = SHADOWS;
        exposure = 0.5;
    }
    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();

    funcBurn F;
    KisChannelInfo::enumChannelValueType cT = cs->channels()[0]->channelValueType();

    switch(type)
    {
        case SHADOWS:
        {
            exposure /= 3.0;
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & BurnShadow<Q_UINT8, 255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & BurnShadow<Q_UINT16, 65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & BurnShadow<float, 1>;
            } else {
                return;
            }
        }
            break;
        case MIDTONES:
        {
            exposure = ( 1.0 + exposure / 3.0);
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & BurnMidtones<Q_UINT8, 255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & BurnMidtones<Q_UINT16, 65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & BurnMidtones<float, 1>;
            } else {
                return;
            }
        }
            break;
        case HIGHLIGHTS:
        {
            exposure = 1.0 - exposure / 3.0;
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & BurnHighlights<Q_UINT8,255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & BurnHighlights<Q_UINT16,65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & BurnHighlights<float,1000000000>;
            } else {
                return;
            }
        }
            break;
    }
    Q_INT32 nC = cs->nColorChannels();

    while( ! srcIt.isDone() )
    {
        if(srcIt.isSelected())
        {
            F( srcIt.oldRawData(), dstIt.rawData(), nC, exposure);
        }
        setProgress(++pixelsProcessed);
        ++srcIt;
        ++dstIt;
    }


    
    
    setProgressDone(); // Must be called even if you don't really support progression
}
