/***************************************************************************
                         fileview.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 2004
    copyright            : (C) 2004 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpushbutton.h>
#include <qfileinfo.h>
#include <qtabwidget.h>
#include <kfiledialog.h>
#include "fileview.h"
#include "filelist.h"
#include "kscopepixmaps.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 * @param	fl		Widget creation flags
 */
FileView::FileView(QWidget* pParent, const char* szName, WFlags fl) :
	FileViewLayout(pParent, szName, fl),
	m_pCurBranch(0)
{
	QWidget* pPage;
	
	// Set the tab widget icons
	pPage = m_pTabWidget->page(0);
	m_pTabWidget->setTabIconSet(pPage, 
		Pixmaps().getPixmap(KScopePixmaps::TabFileList));
	pPage = m_pTabWidget->page(1);
	m_pTabWidget->setTabIconSet(pPage, 
		Pixmaps().getPixmap(KScopePixmaps::TabFileTree));
	
	// Create a single column for the file tree
	m_pFileTree->addColumn("");
	
	// Send the fileRequested() signal whenever a file is selected in either
	// the list or the tree
	connect(m_pFileList, SIGNAL(fileRequested(const QString&, uint)), this,
		SIGNAL(fileRequested(const QString&, uint)));
	connect(m_pFileTree, SIGNAL(doubleClicked(QListViewItem*)), 
		this, SLOT(slotTreeItemSelected(QListViewItem*)));
	connect(m_pFileTree, SIGNAL(returnPressed(QListViewItem*)), this, 
		SLOT(slotTreeItemSelected(QListViewItem*)));
	
	// Prompt the user for a new root for the file tree when the "Set Root..."
	// button is clicked
	connect(m_pRootButton, SIGNAL(clicked()), this, SLOT(slotChangeRoot()));
	
	// Emit the findFile() signal when the "Find File..." button is clicked
	connect(m_pFindButton, SIGNAL(clicked()), this, SIGNAL(findFile()));
}

/**
 * Class destructor.
 */
FileView::~FileView()
{
}

/**
 * Changes the root of the file tree.
 * @param	sRoot	The full path of the new root
 */
void FileView::setTreeRoot(const QString& sRoot)
{
	// Remove the current branch
	if (m_pCurBranch)
		m_pFileTree->removeBranch(m_pCurBranch);
	
	// Do nothing for an empty root directory
	if (sRoot.isEmpty())
		return;
	
	// Create the and open new branch, with the newly specified root
	QFileInfo fi(sRoot);
	m_pCurBranch = m_pFileTree->addBranch(KURL(sRoot), fi.fileName());
	m_pCurBranch->setChildRecurse(false);
	m_pFileTree->setOpen(m_pCurBranch->root(), true);
}

/**
 * Clears the contents of the file view and file tree.
 */
void FileView::clear()
{
	m_pFileList->clear();
	setTreeRoot("");
} 

/**
 * Prompts the user for a new root for the file tree.
 * This slot is connected to the clicked() signal of the "Set Root" button.
 */
void FileView::slotChangeRoot()
{
	QString sRoot;
	
	// Get a new root directory from the user
	sRoot = KFileDialog::getExistingDirectory();
	if (!sRoot.isEmpty()) {
		// Set the new directory as the root of the file tree
		setTreeRoot(sRoot);
		emit rootChanged(sRoot);
	}
}

/**
 * Emits the fileRequested() signal when a file name is selected in the file 
 * tree. An item is selected by either double-clicking it or by hittin 
 * "ENTER" when it is highlighted.
 * This slot is connected to the doubleClicked() and returnPressed() signals
 * of the KFileTreeView object.
 * @param	pItem	The selected tree item
 */
void FileView::slotTreeItemSelected(QListViewItem* pItem)
{
	KFileTreeViewItem* pTreeItem;
	 
	pTreeItem = (KFileTreeViewItem*)pItem;
	if (!pTreeItem->isDir())
		emit fileRequested(pTreeItem->path(), 0);
}
