/***************************************************************************
                          kscopeconfig.cpp  -  description
                             -------------------
    begin                : Fri Jun 13 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kconfig.h>
#include <kapplication.h>
#include <kglobalsettings.h>
#include "kscopeconfig.h"

KScopeConfig::ConfParams KScopeConfig::s_cpDef = {
	"/usr/bin/cscope",
	true,
	"/usr/bin/ctags",
	true,
	SPLIT_SIZES(),
	QColor(255, 255, 255),
	QColor(0, 0, 0),
	QColor(255, 255, 255),
	QColor(0, 0, 0),
	QColor(255, 255, 255),
	QColor(0, 0, 0),
	QFont(), // Font definitions are overriden in load()
	QFont(),
	QFont(),
	100,
	NameAsc,
	false,
	true,
	true,
	false,
	true,
	true,
	"kate --line %L %F", // External editor example
	Fast,
	Embedded
};

/**
 * Class constructor.
 */
KScopeConfig::KScopeConfig() : m_bFontsChanged(false)
{
}

/**
 * Class destructor.
 */
KScopeConfig::~KScopeConfig()
{
}

/**
 * Reads KScope's parameters from the standard configuration file.
 */
void KScopeConfig::load()
{
	KConfig* pConf = kapp->config();

	// Need a working instance to get the system's default font (cannot be
	// initialised statically)
	s_cpDef.fontFileList = KGlobalSettings::generalFont();
	s_cpDef.fontTagList = KGlobalSettings::generalFont();
	s_cpDef.fontQueryList = KGlobalSettings::generalFont();
	
	// Read the paths to required executables
	pConf->setGroup("Programs");
	m_cp.sCscopePath = pConf->readEntry("CScope");
	m_cp.bVerboseCscope = pConf->readBoolEntry("VerboseCScope",
		s_cpDef.bVerboseCscope);
	m_cp.sCtagsPath = pConf->readEntry("CTags");

	// Read size and position parameters
	pConf->setGroup("Geometry");
	m_cp.bShowTagList = pConf->readBoolEntry("ShowTagList", 
		s_cpDef.bShowTagList);
	m_cp.siEditor = pConf->readIntListEntry("Editor");
	if (m_cp.siEditor.empty())
		m_cp.siEditor << 200 << 800;
	m_cp.bShowQueryButtons = pConf->readBoolEntry("ShowQueryButtons",
		s_cpDef.bShowQueryButtons);

	// Read the recent projects list
	pConf->setGroup("Projects");
	m_slProjects = pConf->readListEntry("Recent");

	// Read colour settings
	pConf->setGroup("Colors");
	m_cp.clrFileListBack = pConf->readColorEntry("FileListBack",
		&s_cpDef.clrFileListBack);
	m_cp.clrFileListFore = pConf->readColorEntry("FileListFore",
		&s_cpDef.clrFileListFore);
	m_cp.clrTagListBack = pConf->readColorEntry("TagListBack",
		&s_cpDef.clrTagListBack);
	m_cp.clrTagListFore = pConf->readColorEntry("TagListFore",
		&s_cpDef.clrTagListFore);
	m_cp.clrQueryListBack = pConf->readColorEntry("QueryListBack",
		&s_cpDef.clrQueryListBack);
	m_cp.clrQueryListFore = pConf->readColorEntry("QueryListFore",
		&s_cpDef.clrQueryListFore);

	// Read font settings
	pConf->setGroup("Fonts");
	m_cp.fontFileList = pConf->readFontEntry("FileList", 
		&s_cpDef.fontFileList);
	m_cp.fontTagList = pConf->readFontEntry("TagList",
		&s_cpDef.fontTagList);
	m_cp.fontQueryList = pConf->readFontEntry("QueryList",
		&s_cpDef.fontQueryList);
	
	// Other options
	pConf->setGroup("Options");
	m_cp.nMaxPosHistory = pConf->readUnsignedNumEntry("MaxPosHistory", 
		s_cpDef.nMaxPosHistory);
	m_cp.ctagSortOrder = 
		(CtagSort)pConf->readUnsignedNumEntry("CtagSortOrder",
		s_cpDef.ctagSortOrder);
	m_cp.bReadOnlyMode = pConf->readBoolEntry("ReadOnlyMode", 
		s_cpDef.bReadOnlyMode);
	m_cp.bLoadLastProj = pConf->readBoolEntry("LoadLastProj", 
		s_cpDef.bLoadLastProj);
	m_cp.bAutoTagHl = pConf->readBoolEntry("AutoTagHl", 
		s_cpDef.bAutoTagHl);
	m_cp.bBriefQueryCaptions = pConf->readBoolEntry("BriefQueryCaptions", 
		s_cpDef.bBriefQueryCaptions);
	m_cp.bWarnModifiedOnDisk = pConf->readBoolEntry("WarnModifiedOnDisk", 
		s_cpDef.bWarnModifiedOnDisk);
	m_cp.sExtEditor = pConf->readEntry("ExternalEditor", s_cpDef.sExtEditor);
	m_cp.profile = (SysProfile)pConf->readUnsignedNumEntry("SystemProfile",
		s_cpDef.profile);
	m_cp.popup = (EditorPopup)pConf->readUnsignedNumEntry("EditorPopup",
		s_cpDef.popup);
}

/**
 * Sets default values to he configuration parameters (except for those where
 * a default value has no meaning, such as the recent projects list).
 */
void KScopeConfig::loadDefault()
{
	m_cp = s_cpDef;
}

/**
 * Loads the layout of the main window.
 * @param	pMainWindow	Pointer to the main docking window
 */
void KScopeConfig::loadWorkspace(KDockMainWindow* pMainWindow)
{
	pMainWindow->readDockConfig(kapp->config(), "Workspace");
}
 
/**
 * Writes KScope's parameters from the standard configuration file.
 */
void KScopeConfig::store()
{
	KConfig* pConf = kapp->config();

	// Write the paths to required executables
	pConf->setGroup("Programs");
	pConf->writeEntry("CScope", m_cp.sCscopePath);
	pConf->writeEntry("VerboseCScope", m_cp.bVerboseCscope);
	pConf->writeEntry("CTags", m_cp.sCtagsPath);

	// Write size and position parameters
	pConf->setGroup("Geometry");
	pConf->writeEntry("ShowTagList", m_cp.bShowTagList);
	pConf->writeEntry("Editor", m_cp.siEditor);
	pConf->writeEntry("ShowQueryButtons", m_cp.bShowQueryButtons);

	// Write the recent projects list
	pConf->setGroup("Projects");
	pConf->writeEntry("Recent", m_slProjects);

	// Write colour settings
	pConf->setGroup("Colors");
	pConf->writeEntry("FileListBack", m_cp.clrFileListBack);
	pConf->writeEntry("FileListFore", m_cp.clrFileListFore);
	pConf->writeEntry("TagListBack", m_cp.clrTagListBack);
	pConf->writeEntry("TagListFore", m_cp.clrTagListFore);
	pConf->writeEntry("QueryListBack", m_cp.clrQueryListBack);
	pConf->writeEntry("QueryListFore", m_cp.clrQueryListFore);

	// Write font settings
	if (m_bFontsChanged) {
		pConf->setGroup("Fonts");
		pConf->writeEntry("FileList", m_cp.fontFileList);
		pConf->writeEntry("TagList", m_cp.fontTagList);
		pConf->writeEntry("QueryList", m_cp.fontQueryList);
	}
		
	// Other options
	pConf->setGroup("Options");
	pConf->writeEntry("CtagSortOrder", (uint)m_cp.ctagSortOrder);
	pConf->writeEntry("ReadOnlyMode", m_cp.bReadOnlyMode);
	pConf->writeEntry("LoadLastProj", m_cp.bLoadLastProj);
	pConf->writeEntry("AutoTagHl", m_cp.bAutoTagHl);
	pConf->writeEntry("BriefQueryCaptions", m_cp.bBriefQueryCaptions);
	pConf->writeEntry("WarnModifiedOnDisk", m_cp.bWarnModifiedOnDisk);
	pConf->writeEntry("ExternalEditor", m_cp.sExtEditor);
	pConf->writeEntry("SystemProfile", (uint)m_cp.profile);
	pConf->writeEntry("EditorPopup", (uint)m_cp.popup);
	
	// Do not report it's the first time on the next run
	pConf->setGroup("General");
	pConf->writeEntry("FirstTime", false);
}

/**
 * Stores the layout of the main window.
 * @param	pMainWindow	Pointer to the main docking window
 */
void KScopeConfig::storeWorkspace(KDockMainWindow* pMainWindow)
{
	pMainWindow->writeDockConfig(kapp->config(), "Workspace");
}

/**
 * Determines if this is the first time KScope was launched by the current
 * user.
 * @return	true if this is the first time, false otherwise
 */
bool KScopeConfig::isFirstTime()
{
	KConfig* pConf = kapp->config();

	pConf->setGroup("General");
	return pConf->readBoolEntry("FirstTime", true);
}

/**
 * @return	The full path of the Cscope executable
 */
const QString& KScopeConfig::getCscopePath() const
{
	return m_cp.sCscopePath;
}

/**
 * @param	sPath	The full path of the Cscope executable
 */
void KScopeConfig::setCscopePath(const QString& sPath)
{
	m_cp.sCscopePath = sPath;
}

/**
 * @return	true to use the '-v' option for Cscope, false otherwise
 */
bool KScopeConfig::useVerboseCscope() const
{
	return m_cp.bVerboseCscope;
}

/**
 * @param	bVerbose	true to use the '-v' option for Cscope, false
 *						otherwise
 */
void KScopeConfig::setVerboseCscope(bool bVerbose)
{
	m_cp.bVerboseCscope = bVerbose;
}

/**
 * @return	The full path of the Ctags executable
 */
const QString& KScopeConfig::getCtagsPath() const
{
	return m_cp.sCtagsPath;
}

/**
 * @param	sPath	The full path of the Cscope executable
 */
void KScopeConfig::setCtagsPath(const QString& sPath)
{
	m_cp.sCtagsPath = sPath;
}

/**
 * @return	A sorted list of recently used project paths.
 */
const QStringList& KScopeConfig::getRecentProjects() const
{
	return m_slProjects;
}

/**
 * Adds the given project path to the beginning of the recently used projects
 * list.
 * @param	sProjPath	The path of the project to add
 */
void KScopeConfig::addRecentProject(const QString& sProjPath)
{
	QStringList::Iterator itr;
	
	itr = m_slProjects.find(sProjPath);
	if (itr != m_slProjects.end())
		m_slProjects.remove(itr);
			
	m_slProjects.prepend(sProjPath);
}

/**
 * Removes the given project path from recently used projects list.
 * @param	sProjPath	The path of the project to remove
 */
void KScopeConfig::removeRecentProject(const QString& sProjPath)
{
	m_slProjects.remove(sProjPath);
}

/**
 * @return	true if the tag list should be visible, false otherwise
 */
bool KScopeConfig::getShowTagList() const
{
	return m_cp.bShowTagList;
}

/**
 * @param	bShowList	true to make the tag list visible, false otherwise
 */
void KScopeConfig::setShowTagList(bool bShowTagList)
{
	m_cp.bShowTagList = bShowTagList;
}

/**
 * @return	A list containing the widths of the Ctags list part and the
 * editor part in an editor page.
 */
const SPLIT_SIZES& KScopeConfig::getEditorSizes() const
{
	return m_cp.siEditor;
}

/**
 * @param	siEditor	A list containing the widths of the Ctags list part
 * and the editor part in an editor page.
 */
void KScopeConfig::setEditorSizes(const SPLIT_SIZES& siEditor)
{
	m_cp.siEditor = siEditor;
}

/**
 * Finds a colour to use for either the foreground or the background of a
 * window. 
 * The windows for which KScope maintains colour preferences are the
 * children of the main widget (the query window and file list) and the tag
 * list. The colours of editor windows are managed by the editor part.
 * @param	cw		Identifies the window
 * @param	type	Whether the requested colour is for the foreground or the
 *					background of the window
 * @return	A reference to the colour object to use
 */
const QColor& KScopeConfig::getColor(ConfWidget cw, ColorType type) const
{
	switch (cw) {
	case FileList:
		return (type == Background) ? m_cp.clrFileListBack :
			m_cp.clrFileListFore;

	case TagList:
		return (type == Background) ? m_cp.clrTagListBack :
			m_cp.clrTagListFore;

	case QueryList:
		return (type == Background) ? m_cp.clrQueryListBack :
			m_cp.clrQueryListFore;
	}

	// Default return value, shouldn't happen (but else the compiler may
	// complain)
	return m_cp.clrFileListBack;
}

/**
 * Sets a new colour to a managed child window (@see getColor() for a list of
 * these windows.)
 * @param	cw		Identifies the window
 * @param	type	Whether the given colour is for the foreground or
 *					background of the window
 * @param	clr		The colour to use
 */ 
void KScopeConfig::setColor(ConfWidget cw, ColorType type, const QColor& clr)
{
	switch (cw) {
	case FileList:
		(type == Background) ? m_cp.clrFileListBack = clr :
			m_cp.clrFileListFore = clr;
		break;
		
	case TagList:
		(type == Background) ? m_cp.clrTagListBack = clr :
			m_cp.clrTagListFore = clr;
		break;

	case QueryList:
		(type == Background) ? m_cp.clrQueryListBack = clr :
			m_cp.clrQueryListFore = clr;
		break;
	}
}

/**
 * Finds a font to use for either the foreground or the background of a
 * window. 
 * The windows for which KScope maintains font preferences are the
 * children of the main widget (the query window and file list) and the tag
 * list. The font used by editor windows is managed by the editor part.
 * @param	cw		Identifies the window
 * @return	A reference to the font object to use
 */
const QFont& KScopeConfig::getFont(ConfWidget cw) const
{
	switch (cw) {
	case FileList:
		return m_cp.fontFileList;
		
	case TagList:
		return m_cp.fontTagList;
		
	case QueryList:
		return m_cp.fontQueryList;
	}

	// Default return value, shouldn't happen (but else the compiler may
	// complain)
	return m_cp.fontFileList;
}

/**
 * Sets a new font to a managed child window (@see getFont() for a list of
 * these windows.)
 * @param	cw		Identifies the window
 * @param	font	The font to use
 */ 
void KScopeConfig::setFont(ConfWidget cw, const QFont& font)
{
	switch (cw) {
	case FileList:
		m_cp.fontFileList = font;
		break;
		
	case TagList:
		m_cp.fontTagList = font;
		break;

	case QueryList:
		m_cp.fontQueryList = font;
		break;
	}

	m_bFontsChanged = true;
}

/**
 * @return	The maximal size of the positions history list
 */
uint KScopeConfig::getMaxPosHistory()
{
	return m_cp.nMaxPosHistory;
}

/**
 * @return	The column and direction by which the tags should be sorted
 */
KScopeConfig::CtagSort KScopeConfig::getCtagSortOrder()
{
	return m_cp.ctagSortOrder;
}

/**
 * @param	ctagSortOrder	The column and direction by which the tags should
 *							be sorted
 */
void KScopeConfig::setCtagSortOrder(CtagSort ctagSortOrder)
{
	m_cp.ctagSortOrder = ctagSortOrder;
}

/**
 * @return	true to work in Read-Only mode, false otherwise
 */
bool KScopeConfig::getReadOnlyMode()
{
	return m_cp.bReadOnlyMode;
}

/**
 * @param	bReadOnlyMode	true to work in Read-Only mode, false otherwise
 */
void KScopeConfig::setReadOnlyMode(bool bReadOnlyMode)
{
	m_cp.bReadOnlyMode = bReadOnlyMode;
}

/**
 * @return	true to load the last project on start-up, false otherwise
 */
bool KScopeConfig::getLoadLastProj()
{
	return m_cp.bLoadLastProj;
}

/**
 * @param	bLoadLastProj	true to load the last project on start-up, false
 *							otherwise
 */
void KScopeConfig::setLoadLastProj(bool bLoadLastProj)
{
	m_cp.bLoadLastProj = bLoadLastProj;
}

/**
 * @return	true to enable tag highlighting based on cursor position, false
 *			to disable this feature
 */
bool KScopeConfig::getAutoTagHl()
{
	return m_cp.bAutoTagHl;
}

/**
 * @param	bAutoTagHl	true to enable tag highlighting based on cursor
 *			position, false to disable this feature
 */
void KScopeConfig::setAutoTagHl(bool bAutoTagHl)
{
	m_cp.bAutoTagHl = bAutoTagHl;
}

/**
 * @return	true to use the short version of the query captions, false to use
 *			the long version
 */
bool KScopeConfig::getUseBriefQueryCaptions()
{
	return m_cp.bBriefQueryCaptions;
}

/**
 * @param	bBrief	true to use the short version of the query captions, false
 *			to use the long version
 */
void KScopeConfig::setUseBriefQueryCaptions(bool bBrief)
{
	m_cp.bBriefQueryCaptions = bBrief;
}

/**
 * @return	true to show the toolbar to the side of the query window, false
 *			otherwise
 */
bool KScopeConfig::getShowQueryButtons()
{
	return m_cp.bShowQueryButtons;
}

/**
 * @param	bShow	true to show the toolbar to the side of the query window,
 *					false otherwise
 */
void KScopeConfig::setShowQueryButtons(bool bShow)
{
	m_cp.bShowQueryButtons = bShow;
}

/**
 * @return	true to warn user when file is modified on disk, false
 *			otherwise
 */
bool KScopeConfig::getWarnModifiedOnDisk()
{
	return m_cp.bWarnModifiedOnDisk;
}

/**
 * @param	bWarn	true to warn user when file is modified on disk,
 *					false otherwise
 */
void KScopeConfig::setWarnModifiedOnDisk(bool bWarn)
{
	m_cp.bWarnModifiedOnDisk = bWarn;
}

/**
 * @return	A command line for launching an external editor
 */
const QString& KScopeConfig::getExtEditor()
{
	return m_cp.sExtEditor;
}

/**
 * @param	sExtEditor	A command line for launching an external editor
 */
void KScopeConfig::setExtEditor(const QString& sExtEditor)
{
	m_cp.sExtEditor = sExtEditor;
}

/**
 * Determines if an external editor should be used.
 * An external editor is used if KScope is in Read-Only mode, and a command-
 * line for the editor was specified.
 * @return	true to use an external editor, false otherwise
 */
bool KScopeConfig::useExtEditor()
{
	return !m_cp.sExtEditor.isEmpty();
}

/**
 * @return	The chosen profile for this system (@see SysProfile)
 */
KScopeConfig::SysProfile KScopeConfig::getSysProfile() const
{
	return m_cp.profile;
}

/**
 * @param	profile	The system profile to use (@see SysProfile)
 */
void KScopeConfig::setSysProfile(KScopeConfig::SysProfile profile)
{
	m_cp.profile = profile;
}

/**
 * @return	The chosen popup menu type for the embedded editor (@see
 *			EditorPopup)
 */
KScopeConfig::EditorPopup KScopeConfig::getEditorPopup() const
{
	return m_cp.popup;
}

/**
 * @return	The name of the popup menu to use in the embedded editor
 */
QString KScopeConfig::getEditorPopupName() const
{
	switch (m_cp.popup) {
	case Embedded:
		return "ktexteditor_popup";
		
	case KScopeOnly:
		return "kscope_popup";
	}
	
	// Will not happen, but the compiler complains if no return statement is
	// given here
	return "";
}

/**
 * @param	ep	The popup menu to use for the embedded editor (@see
 *				EditorPopup)
 */
void KScopeConfig::setEditorPopup(KScopeConfig::EditorPopup popup)
{
	m_cp.popup = popup;
}

/**
 * Returns a reference to a global configuration object.
 * The static object defined is this function should be the only KSCopeConfig
 * object in this programme. Any code that wishes to get or set global
 * configuration parameters, should call Config(), instead of defining its
 * own object.
 * @return	Reference to a statically allocated configuration object
 */
KScopeConfig& Config()
{
	static KScopeConfig conf;
	return conf;
}
