/***************************************************************************
                          projectmanager.cpp  -  description
                             -------------------
    begin                : Sat May 10 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <unistd.h>
#include <kmessagebox.h>
#include <klocale.h>
#include "projectmanager.h"
#include "kscopeconfig.h"

#define CONFIG_VER 2

#define DEF_IS_KERNEL		false
#define DEF_INV_INDEX		true
#define DEF_AC_MIN_CHARS	3
#define DEF_AC_DELAY		500
#define DEF_AC_MAX_ENTRIES	100

/**
 * Class constructor.
 */
ProjectManager::ProjectManager() :
	m_pFiles(NULL),
	m_pConf(NULL),
	m_bOpen(false),
	m_bTempProj(false),
	m_nAutoRebuildTime(-1)
{
}

/**
 * Class destructor.
 */
ProjectManager::~ProjectManager()
{
	close();
}

/**
 * Creates a project's directory, and associates this directory with the
 * current object. This directory is created under the given path, and using
 * the project's name (which, thus, has to be a legal file name). 
 * Note: this function attempts to create a new directory, so the given path
 * and name must not lead to an existing one.
 * @param	sName		The project's name
 * @param	sPath		The parent directory under which to create the 
 *						project's directory
 * @param	opt			A structure containing project options
 * @return	true if successful, false otherwise
 */
bool ProjectManager::create(const QString& sName, const QString& sPath, 
	const Options& opt)
{
	QFile* pFilesFile;
	KConfig* pConf;
	
	// If the project is open, close it first
	if (m_bOpen)
		close();

	// Make sure the directory doesn't exist
	QDir dir(sPath);
	if (dir.exists(sName)) {
		KMessageBox::error(0, i18n("Cannot create a project inside an "
			"existing directory"));
		return false;
	}

	// Try to create the projcet's directory
	if (!dir.mkdir(sName, false) || !dir.cd(sName, false)) {
		KMessageBox::error(0, i18n("Failed to create the project's "
			"directory"));
		return false;
	}
	
	// Prepare the project's files
	initConfig(dir.absPath(), &pFilesFile, &pConf);

	// Write the configuration file version
	pConf->setGroup("");
	pConf->writeEntry("Version", CONFIG_VER);
	
	// Write project properties in the configuration file
	pConf->setGroup("Project");
	pConf->writeEntry("Name", sName);
	writeOptions(pConf, opt);
	
	// Flush the config file data, so the project is created even if KScope
	// crashes...
	pConf->sync();
	delete pConf;

	// Create the "files" file
	if (!pFilesFile->open(IO_WriteOnly)) {
		delete pFilesFile;
		KMessageBox::error(0, i18n("Failed to open the \"files\" file"));
		return false;
	}

	pFilesFile->close();
	delete pFilesFile;
	
	return true;
}

/**
 * Opens a project and makes it the current one.
 * @param	sPath	The directory containing the project's files
 * @return	true if successful, false otherwise
 */
bool ProjectManager::open(const QString& sPath)
{
	QString sConfFile;
	Options opt;
	
	// Associate the object with the project directory
	m_dir.setPath(sPath);
	if (!m_dir.exists()) {
		KMessageBox::error(0, i18n("Project directory does not exist"));
		return false;
	}

	// Open the configuration files
	initConfig(sPath, &m_pFiles, &m_pConf);

	// Verify the configuration file's version is compatible
	m_pConf->setGroup("");
	if (m_pConf->readUnsignedNumEntry("Version", 0) != CONFIG_VER) {
		KMessageBox::error(0, i18n("Your project is not compatible with this "
		"version of KScope.\nPlease re-create the project."));
		return false;
	}
	
	// Get the project name
	m_pConf->setGroup("Project");
	m_sName = m_pConf->readEntry("Name");
	if (m_sName == QString::null) {
		KMessageBox::error(0, i18n("Cannot read project name"));
		return false;
	}
		
	// Indicate that a project has been opened
	m_bOpen = true;
	m_bTempProj = false;

	// Create the argument list for invoking Cscope
	getOptions(opt);
	if (opt.bKernel)
		m_slArgs << "-k";
	if (opt.bInvIndex)
		m_slArgs << "-q";
	
	// Get the auto-rebuild	time value
	m_nAutoRebuildTime = opt.nAutoRebuildTime;
	
	// Add to the list of recently opened projects
	Config().addRecentProject(sPath);
	
	return true;
}

/**
 * Opens a Cscope.out file as a temporary project.
 * @param	sFilePath	The full path of the Cscope.out file
 * @return	true if successful, false otherwise
 */
bool ProjectManager::openCscopeOut(const QString& sFilePath)
{
	QFileInfo fi(sFilePath);
	
	// Make sure the file exists, and that is is a cross-reference file
	if (!fi.exists() || !isCscopeOut(fi.absFilePath()))
		return false;
		
	// Set the project's directory
	m_dir = fi.dirPath(true);
	
	// Set the name of the project to be the full path of the file
	m_sName = fi.absFilePath();

	// Indicate that a project has been opened
	m_bOpen = true;
	m_bTempProj = true;
	
	return true;
}

/**
 * Performs clean-up on the project's variables, and detaches the associated
 * directory.
 */
void ProjectManager::close()
{
	if (!m_bTempProj) {
		m_dir = QDir();
		delete m_pFiles;
		m_pFiles = NULL;
		delete m_pConf;
		m_pConf = NULL;
	}
	
	m_slArgs = QStringList();
	m_slSymHistory.clear();
	m_nAutoRebuildTime = -1;
	m_bOpen = false;
}

/**
 * Adds a set of files to the current project.
 * @param	slFiles	A list of file names to add
 * @return	The total number of files added to the project
 */
int ProjectManager::addFiles(const QStringList& slFiles)
{
	QStringList::const_iterator itr;
	int nFiles = 0;

	// Cannot add files if no project is open, or the project is temporary
	if (!m_bOpen || m_bTempProj)
		return 0;

	// Open the source files file, to which the file paths will be added
	if (!m_pFiles->open(IO_WriteOnly | IO_Append))
		return 0;

	// Add each file to the project
	QTextStream str(m_pFiles);
	for (itr = slFiles.begin(); itr != slFiles.end(); ++itr) {
		str << *itr << "\n";
		nFiles++;
	}

	m_pFiles->close();
	emit filesAdded(slFiles);
	
	return nFiles;
}

/**
 * Fills a list object with all files in the project.
 * List items are created by reading and parsing all file name entries from
 * the project's 'cscope.files' file.
 * Note that the file may contain option lines, beginning with a dash. These
 * should be ignored.
 * @param	pList	Pointer to the object to fill
 */
void ProjectManager::fillList(FileListTarget* pList)
{
	QString sFilePath;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_ReadOnly))
		return;

	// Read all file names from the file
	QTextStream str(m_pFiles);
	while ((sFilePath = str.readLine()) != QString::null) {
		// Skip option lines
		if (sFilePath.at(0) == '-')
			continue;

		// Set the new list item
		pList->addItem(sFilePath);
	}

	m_pFiles->close();
}

/**
 * Writes all file entries in a list view widget to the project's
 * 'cscope.files' file (replacing current file contents.)
 * @param	pList	Pointer to the object from which to take the new entries
 */
void ProjectManager::writeList(FileListSource* pList)
{
	QString sFilePath;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_WriteOnly))
		return;

	QTextStream str(m_pFiles);

	// Write all file names
	if (pList->firstItem(sFilePath)) {
		do {
			str << sFilePath << "\n";
		} while (pList->nextItem(sFilePath));
	}

	m_pFiles->close();
	emit fileListChanged();
}

/**
 * Determines whether the project includes any files.
 * Reads the 'cscope.files' file and looks for any file names in it. If none
 * is present, then the project is considered empty.
 * Note that the file may contain option lines, beginning with a dash. These
 * should be ignored.
 * @return	true if no files are included in the project, false otherwise
 */
bool ProjectManager::isEmpty()
{
	QString sPath, sFileName;
	bool bResult = true;
	
	// Must have an active project
	if (!m_bOpen)
		return true;

	// Assume there are some files if using only a cscope.out file
	if (m_bTempProj)
		return false;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_ReadOnly))
		return true;

	// Find at least one file name entry in the file
	QTextStream str(m_pFiles);
	while ((sPath = str.readLine()) != QString::null) {
		if (sPath.at(0) != '-') {
			bResult = false;
			break;
		}
	}

	m_pFiles->close();
	return bResult;
}

/**
 * Returns a semi-colon separated list of the file types included in the
 * current project.
 */
QString ProjectManager::getFileTypes() const
{
	QString sTypes;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return QString::null;

	m_pConf->setGroup("Project");
	return m_pConf->readEntry("FileTypes");
}

/**
 * Reads the project's options from the configuration file.
 * @param	opt	A structure to fill with the read options
 */
void ProjectManager::getOptions(Options& opt) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Get project properties
	m_pConf->setGroup("Project");
	opt.slFileTypes = m_pConf->readListEntry("FileTypes", ' ');
	opt.bKernel = m_pConf->readBoolEntry("Kernel", DEF_IS_KERNEL);
	opt.bInvIndex = m_pConf->readBoolEntry("InvIndex", DEF_INV_INDEX);
	opt.nAutoRebuildTime = m_pConf->readNumEntry("AutoRebuildTime");
	
	// Get auto-completion options
	m_pConf->setGroup("AutoCompletion");
	opt.bACEnabled = m_pConf->readBoolEntry("Enabled");
	opt.nACMinChars = m_pConf->readUnsignedNumEntry("MinChars",
		DEF_AC_MIN_CHARS);
	opt.nACDelay = m_pConf->readUnsignedNumEntry("Delay", DEF_AC_DELAY);
	opt.nACMaxEntries = m_pConf->readUnsignedNumEntry("MaxEntries",
		DEF_AC_MAX_ENTRIES);
}

/**
 * Sets project options.
 * @param	opt	A structure containing the new parameters to set
 */
void ProjectManager::setOptions(const Options& opt)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the options to the configuratio nfile
	writeOptions(m_pConf, opt);	
			
	// Create the argument list for invoking Cscope
	m_slArgs = QStringList();
	if (opt.bKernel)
		m_slArgs << "-k";
	if (opt.bInvIndex)
		m_slArgs << "-q";
}

/**
 * Reads the paths of the files that were open when the project was closed,
 * and fills a list used to restore the previous session.
 * @param	slOpenFiles	The list to fill
 */
void ProjectManager::getOpenFiles(QStringList& slOpenFiles) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slOpenFiles = m_pConf->readListEntry("OpenFiles");
}

/**
 * Writes a list of open files to the project's configuration file.
 * The method is called before a project is closed, so these file paths can
 * be read when the project is restored.
 * @param	slOpenFiles	The list of file paths to write
 */
void ProjectManager::setOpenFiles(QStringList& slOpenFiles)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the file paths
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("OpenFiles", slOpenFiles);
}

/**
 * Reads the names of the files corresponding to queries that were locked
 * when the project was closed, and fills a list used to restore the
 * previous session.
 * @param	slQueryFiles	The list to fill
 */
void ProjectManager::getQueryFiles(QStringList& slQueryFiles) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slQueryFiles = m_pConf->readListEntry("QueryFiles");
}

/**
 * Writes a list of query file names to the project's configuration file.
 * The method is called before a project is closed, so these file names can
 * be read when the project is restored.
 * @param	slQueryFiles	The list of file names to write
 */
void ProjectManager::setQueryFiles(QStringList& slQueryFiles)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the file names
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("QueryFiles", slQueryFiles);
}

/**
 * Reads the names of the files corresponding to call trees that were opened
 * when the project was closed, and fills a list used to restore the
 * previous session.
 * @param	slQueryFiles	The list to fill
 */
void ProjectManager::getCallTreeFiles(QStringList& slQueryFiles) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slQueryFiles = m_pConf->readListEntry("CallTreeFiles");
}

/**
 * Writes a list of call tree file names to the project's configuration file.
 * The method is called before a project is closed, so these file names can
 * be read when the project is restored.
 * @param	slQueryFiles	The list of file names to write
 */
void ProjectManager::setCallTreeFiles(QStringList& slQueryFiles)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the file names
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("CallTreeFiles", slQueryFiles);
}

/**
 * Reads the value of the project's root directory from the project's
 * configuration file.
 * @return	The root directory of the active project
 */
QString ProjectManager::getRoot() const
{
	// Empty root if no project is currently open
	if (!m_bOpen)
		return "";
	
	// Return the file-system root if using a temporary project
	if (m_bTempProj)
		return "/";

	m_pConf->setGroup("Project");
	return m_pConf->readEntry("RootPath", "/");
}

/**
 * Sets a new root directory for a project.
 * @param	sRoot	The full path of the new root directory
 */
void ProjectManager::setRoot(const QString& sRoot)
{
	// Do nothing if there is no open project, or using a temporary project
	if (!m_bOpen || m_bTempProj)
		return;
	
	m_pConf->setGroup("Project");
	m_pConf->writeEntry("RootPath", sRoot);
}
	
/**
 * Copies the list of previously queried symbols to the target object.
 * @param	slSymHistory	The list object to copy into
 */
void ProjectManager::getSymHistory(QStringList& slSymHistory) const
{
	slSymHistory = m_slSymHistory;
}

/**
 * Copies the list of previously queried symbols from the target object.
 * @param	slSymHistory	The list object to copy from
 */
void ProjectManager::setSymHistory(QStringList& slSymHistory)
{
	m_slSymHistory = slSymHistory;
}

/**
 * Determines if the cscope.out file for this project exists.
 * @return	true if the database exists, false otherwise
 */
bool ProjectManager::dbExists()
{
	if (!m_bOpen)
		return false;
		
	return m_dir.exists("cscope.out");
}

/**
 * Fills a structure with default properties for a new project.
 * Default properties are partly based on the system profile.
 * @param	opt	The structure to fill
 */
void ProjectManager::getDefOptions(Options& opt)
{
	// Initialise MIME-type list
	opt.slFileTypes.append("*.c");
	opt.slFileTypes.append("*.h");

	// Set other options
	opt.bKernel = DEF_IS_KERNEL;
	opt.bInvIndex = DEF_INV_INDEX;
	opt.nACMinChars = DEF_AC_MIN_CHARS;
	opt.nACDelay = DEF_AC_DELAY;
	opt.nACMaxEntries = DEF_AC_MAX_ENTRIES;
	
	// Set profile-dependant options
	if (Config().getSysProfile() == KScopeConfig::Fast) {
		opt.nAutoRebuildTime = 10;
		opt.bACEnabled = true;
	}
	else {
		opt.nAutoRebuildTime = -1;
		opt.bACEnabled = false;
	}
}

/**
 * Creates the "cscope.files" and "cscope.proj" file objects for the given
 * project.
 * @param	sPath		The project directory, in which the files reside
 * @param	ppFilesFile	Holds the new "cscope.files" object, upon return
 * @param	ppConf		Holds the new "cscope.proj" object, upon return
 */
void ProjectManager::initConfig(const QString& sPath, QFile** ppFilesFile,
	KConfig** ppConf)
{
	QString sFilesFile, sConfFile;

	// Attach to the source files file
	sFilesFile = sPath + "/cscope.files";
	*ppFilesFile = new QFile(sFilesFile);
	
	// Open the configuraton file
	sConfFile = sPath + "/cscope.proj";	
	*ppConf = new KConfig(sConfFile);
}

/**
 * Determines if the given file is a Cscope cross-reference database.
 * @param	sPath	The full path of the file to check
 * @return	true if the given file is a cscope.out file, false otherwise
 */
bool ProjectManager::isCscopeOut(const QString& sPath)
{
	QFile file(sPath);
	QString sLine;
	int nVer;
	char szDir[PATH_MAX];

	// Try to open the file
	if (!file.open(IO_ReadOnly))
		return false;
		
	// Check if the first line matches the expected format
	sLine = QTextStream(&file).readLine();
	return sscanf(sLine.latin1(), "cscope %d %s", &nVer, szDir) == 2;
}

/**
 * Writes a set of project options to the given configuration file.
 * @param	pConf	An initialised configuration file
 * @param	opt		The options to write
 */
void ProjectManager::writeOptions(KConfig* pConf, const Options& opt)
{
	// Set new project options
	pConf->setGroup("Project");
	pConf->writeEntry("FileTypes", opt.slFileTypes.join(" "));
	pConf->writeEntry("Kernel", opt.bKernel);
	pConf->writeEntry("InvIndex", opt.bInvIndex);		
	pConf->writeEntry("AutoRebuildTime", opt.nAutoRebuildTime);
	
	// Set auto-completion options
	pConf->setGroup("AutoCompletion");
	pConf->writeEntry("Enabled", opt.bACEnabled);
	pConf->writeEntry("MinChars", opt.nACMinChars);
	pConf->writeEntry("Delay", opt.nACDelay);
	pConf->writeEntry("MaxEntries", opt.nACMaxEntries);
}
