/***************************************************************************
                          groupstatsview.cpp  -  description
                             -------------------
    begin                : Mon Sep 2 2002
    copyright            : (C) 2002 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlabel.h>
#include <qlayout.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpushbutton.h>

#include "groupstatsview.h"

const double AvgTeraFLOPs = 3.9;
const double AvgCpF = 6.9;

GroupStatsView::GroupStatsView(QWidget *parent, const char *name)
              : KSetiSpyView(Text, parent, name)
{
  QBoxLayout *layout = new QVBoxLayout(this);

  QWidgetStack *stack = new QWidgetStack(this, "GroupStatsView::stack");
  layout->addWidget(stack);

  QLabel *label = new QLabel(stack, "GroupStatsView::message");
  label->setAlignment(AlignCenter);
  stack->addWidget(label, 0);

  QWidget *w = new QWidget(stack, "GroupStatsView::main");
  stack->addWidget(w, 1);

  QGridLayout *grid = new QGridLayout(w, 8, 2);
  grid->setSpacing(8);

  grid->addMultiCellWidget(addField(w, "GroupStatsView::group"), 0, 0, 0, 1);
  setFieldName("GroupStatsView::group", i18n("Group:"));

  grid->addWidget(addField(w, "GroupStatsView::total_results"), 1, 0);

  grid->addWidget(addField(w, "GroupStatsView::daily_results"), 1, 1);

  grid->addWidget(addField(w, "GroupStatsView::total_time"), 2, 0);

  grid->addWidget(addField(w, "GroupStatsView::cpus"), 2, 1);

  grid->addWidget(addField(w, "GroupStatsView::avg_time"), 3, 0);

  grid->addWidget(addField(w, "GroupStatsView::speed"), 3, 1);

  grid->addMultiCellWidget(addField(w, "GroupStatsView::founder"), 4, 4, 0, 1);

  grid->addWidget(addField(w, "GroupStatsView::members"), 5, 0);

  grid->setRowStretch(6, 1);

  QBoxLayout *hbox = new QHBoxLayout(8);
  grid->addMultiCellLayout(hbox, 7, 7, 0, 1);

  hbox->addStretch(1);

  KPushButton *button = new KPushButton(i18n("&Update"), w, "GroupStatsView::update");
  button->installEventFilter(this);
  hbox->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(updateGroupStats()));
  connect(kdoc, SIGNAL(updatedUserStats(const user_stats *)),
          this, SLOT(handleUserStatsUpdates(const user_stats *)));
  connect(kdoc, SIGNAL(updatedGroupStats(const group_stats *, const group_stats *)),
          this, SLOT(handleGroupStatsUpdates(const group_stats *, const group_stats *)));

  group = QString::null;
  history = stats = NULL;

  updateContent(false);

  showMain();
}

GroupStatsView::~GroupStatsView()
{
}

void GroupStatsView::updateContent(bool force)
{
  // optimization: do not update if this is not the visible widget
  const QWidgetStack *views = (QWidgetStack *) this->parent();
  if(views->visibleWidget() != this && !force) return;

  const SetiClientMonitor::State state = kdoc->setiMonitorState();
  const KLocale *locale = KGlobal::locale();

  if(state >= SetiClientMonitor::No_WU)
  {
    SetiClientMonitor *monitor = kdoc->setiMonitor();
    const seti_data *data = monitor->setiData();

    const QString email = data->user.email;

    if(stats != NULL && stats->email != email)
    {
      group = QString::null;

      delete stats;
      stats = NULL;

      if(history != NULL)
      {
        delete history;
        history = NULL;
      }
    }
  }

  if(!group.isEmpty() || group.isNull())
  {
    if(!group.isNull())
    {
      const QString groupStatsURL = kdoc->groupStatsURL(group);
      if(!groupStatsURL.isEmpty())
        setFieldContentURL("GroupStatsView::group", group, groupStatsURL,
                           i18n("Click to visit your SETI@home group stats web page"),
                           this, SLOT(handleURL(const QString&)));
    }
    else
      setFieldContentText("GroupStatsView::group", i18n(unknownContent));

    setFieldName("GroupStatsView::total_results", i18n("Results returned:"));

    setFieldName("GroupStatsView::total_time", i18n("Total CPU time:"));

    setFieldName("GroupStatsView::avg_time", i18n("Avg. CPU time:"));

    setFieldName("GroupStatsView::founder", i18n("Founder:"));

    setFieldName("GroupStatsView::members", i18n("Members:"));

    if(stats != NULL)
    {
      setFieldContentText("GroupStatsView::total_results", locale->formatNumber(stats->n_results, 0));

      const double total_cpu_years = stats->total_cpu / (60 * 60 * 24 * 365);
      setFieldContentText("GroupStatsView::total_time",
                          i18n("%1 years").arg(locale->formatNumber(total_cpu_years, 2)));

      const double avg = (stats->n_results > 0) ? stats->total_cpu/stats->n_results : 0;
      setFieldContentText("GroupStatsView::avg_time", SetiClientMonitor::timeToString(int(avg)));

      setFieldContentText("GroupStatsView::founder", stats->founder);

      setFieldContentText("GroupStatsView::members", locale->formatNumber(stats->n_members, 0));
    }
    else
    {
      setFieldContentText("GroupStatsView::total_results", i18n(unknownContent));

      setFieldContentText("GroupStatsView::total_time", i18n(unknownContent));

      setFieldContentText("GroupStatsView::avg_time", i18n(unknownContent));

      setFieldContentText("GroupStatsView::founder", i18n(unknownContent));

      setFieldContentText("GroupStatsView::members", i18n(unknownContent));
    }  
  }
  else
  {
    setFieldContentText("GroupStatsView::group", i18n("no group"));

    setFieldName("GroupStatsView::total_results", QString::null);
    setFieldContentText("GroupStatsView::total_result", QString::null);

    setFieldName("GroupStatsView::total_time", QString::null);
    setFieldContentText("GroupStatsView::total_time", QString::null);

    setFieldName("GroupStatsView::avg_time", QString::null);
    setFieldContentText("GroupStatsView::avg_time", QString::null);

    setFieldName("GroupStatsView::founder", QString::null);
    setFieldContentText("GroupStatsView::founder", QString::null);

    setFieldName("GroupStatsView::members", QString::null);
    setFieldContentText("GroupStatsView::members", QString::null);
  }

  double days = 0.0;
  if(!group.isEmpty() && stats != NULL && history != NULL)
    days = history->timestamp.secsTo(stats->timestamp) / double(60 * 60 * 24);

  // optional fields
  if(days >= 1.0)
  {
    const double daily_results = (stats->n_results - history->n_results)/days;
    setFieldName("GroupStatsView::daily_results", i18n("Results per day:"));
    setFieldContentText("GroupStatsView::daily_results", locale->formatNumber(daily_results, 0));

    const double daily_cpu = (stats->total_cpu - history->total_cpu)/days;
    const double n_cpus = daily_cpu / (24 * 60 * 60);
    const double avg_MHz = (daily_cpu != 0.0) ? daily_results * AvgTeraFLOPs * AvgCpF/daily_cpu * 1e6 : 0.0;
    setFieldName("GroupStatsView::cpus", i18n("CPUs:"));
    setFieldContentText("GroupStatsView::cpus",
                        i18n("%1 @ %2 MHz").arg(locale->formatNumber(n_cpus, 0))
                                           .arg(locale->formatNumber(avg_MHz, 0)));

    const double speed = daily_results * AvgTeraFLOPs / (24 * 60 * 60);
    setFieldName("GroupStatsView::speed", i18n("Speed:"));
    setFieldContentText("GroupStatsView::speed",
                        i18n("%1 TeraFLOPS").arg(locale->formatNumber(speed, 3)));
  }
  else
  {
    setFieldName("GroupStatsView::daily_results", QString::null);
    setFieldContentText("GroupStatsView::daily_results", QString::null);

    setFieldName("GroupStatsView::cpus", QString::null);
    setFieldContentText("GroupStatsView::cpus", QString::null);

    setFieldName("GroupStatsView::speed", QString::null);
    setFieldContentText("GroupStatsView::speed", QString::null);
  }
}

bool GroupStatsView::messageVisible()
{
  QWidgetStack *stack = (QWidgetStack *) child("GroupStatsView::stack", "QWidgetStack");
  return(stack->id(stack->visibleWidget()) == 0);
}

void GroupStatsView::showMessage(const QString& msg)
{
  QLabel *label = (QLabel *) child ("GroupStatsView::message", "QLabel");
  label->setText(msg);

  QWidgetStack *stack = (QWidgetStack *) child("GroupStatsView::stack", "QWidgetStack");
  stack->raiseWidget(0);
}

void GroupStatsView::showMain()
{
  QWidgetStack *stack = (QWidgetStack *) child("GroupStatsView::stack", "QWidgetStack");
  stack->raiseWidget(1);
}

void GroupStatsView::updateGroupStats()
{
  showMessage(i18n("Updating from the SETI@home web site..."));
  kdoc->updateGroupStats();
}

void GroupStatsView::handleURL(const QString& url)
{
  kapp->invokeBrowser(url);
}

void GroupStatsView::handleUserStatsUpdates(const user_stats *current)
{
  if(current == NULL)
  {
    if(messageVisible())
    {
      showMessage(i18n("Update failed. Resetting to local values..."));
      QTimer::singleShot(4*1000, this, SLOT(showMain()));
    }
    return;
  }

  QString email;
  if(stats != NULL)
    email = stats->email;
  else if(kdoc->setiMonitorState() > SetiClientMonitor::No_Data)
    email = kdoc->setiMonitor()->setiData()->user.email;
  else
    email = QString::null;

  if(current->email == email)
  {
    if(current->group != group)
    {
      if(stats != NULL)
      {
        delete stats;
        stats = NULL;
      }

      if(history != NULL)
      {
        delete history;
        history = NULL;
      }

      group = current->group;
      if(group.isNull()) group = "";
    }

    updateContent(false);
  }

  showMain();
}


void GroupStatsView::handleGroupStatsUpdates(const group_stats *old, const group_stats *current)
{
  if(current == NULL)
  {
    if(messageVisible())
    {
      showMessage(i18n("Update failed. Resetting to local values..."));
      QTimer::singleShot(4*1000, this, SLOT(showMain()));
    }
    return;
  }

  QString email;
  if(stats != NULL)
    email = stats->email;
  else if(kdoc->setiMonitorState() > SetiClientMonitor::No_Data)
    email = kdoc->setiMonitor()->setiData()->user.email;
  else
    email = QString::null;

  if(current->email == email)
  {
    if(stats == NULL) stats = new group_stats;
    *stats = *current;

    if(old != NULL)
    {
      if(history == NULL) history = new group_stats;
      *history = *old;
    }
    else if(history != NULL)
    {
      delete history;
      history = NULL;
    }

    updateContent(false);
  }

  showMain();
}

#include "groupstatsview.moc"

