/***************************************************************************
                          ksetispytray.cpp  -  description
                             -------------------
    begin                : Tue Jun 26 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qtooltip.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include "ksetispytray.h"

const QString IdleIcon            = "ksetispy_idle";
const QString UninterestingIcon    = "ksetispy_uninteresting";
const QString ReturnedIcon        = "ksetispy_returned";
const QString InterestingIcon      = "ksetispy_interesting";

KSetiSpyTray::KSetiSpyTray(KSetiSpyDoc *doc, QWidget *parent, const char *name)
             : KSystemTray(parent, name)
{
  kdoc = doc;

  setFixedSize(24, 24);

  KPopupMenu *clientMenu = new KPopupMenu(this, "KSetiSpyTray::client_menu");
  contextMenu()->insertItem(i18n("&Client"), clientMenu);

  clientMenu->insertItem(i18n("&Stop"), this, SLOT(handleClient()), 0, 0, 0);

  KPopupMenu *priorityMenu = new KPopupMenu(this, "KSetiSpyTray::priority_menu");
  clientMenu->insertItem(i18n("&Lower priority"), priorityMenu, 1, 1);

  KPopupMenu *tooltipMenu = new KPopupMenu(this, "KSetiSpyTray::tooltip_menu");
  contextMenu()->insertItem(i18n("&Tooltip text"), tooltipMenu);

  tooltipMenu->insertItem(i18n("&Progress"), 0, 0);
  tooltipMenu->insertItem(i18n("&Time left"), 1, 1);
  tooltipMenu->insertItem(i18n("Time &done"), 2, 2);
  tooltipMenu->insertSeparator(3);
  tooltipMenu->insertItem(i18n("&Speed"), 4, 4);
  tooltipMenu->insertItem(i18n("&Efficiency"), 5, 5);
  tooltipMenu->insertItem(i18n("&Rate"), 6, 6);
  tooltipMenu->insertSeparator(7);
  tooltipMenu->insertItem(i18n("Si&gnals"), 8, 8);

  KPopupMenu *profileMenu = new KPopupMenu(this, "KSetiSpyTray::profile_menu");
  contextMenu()->insertItem(i18n("&Profiles"), profileMenu);

  contextMenu()->insertItem(i18n("&About KSetiSpy"), this, SIGNAL(about()));

  connect(kdoc, SIGNAL(updated()), this, SLOT(updateContent()));
  connect(priorityMenu, SIGNAL(activated(int)), this, SLOT(setClientPriority(int)));
  connect(profileMenu, SIGNAL(activated(int)), this, SLOT(handleProfileMenu(int)));
  connect(tooltipMenu, SIGNAL(activated(int)), this, SLOT(setTooltipType(int)));

  updateContent();
}

KSetiSpyTray::~KSetiSpyTray()
{
}

void KSetiSpyTray::readConfig()
{
  KConfig *config = kapp->config();

  config->setGroup("KSetiSpy Tray");

  tooltipType = config->readNumEntry("Tooltip", 0);

  setTooltipType(tooltipType);
}

void KSetiSpyTray::saveConfig()
{
  KConfig *config = kapp->config();

  config->setGroup("KSetiSpy Tray");

  config->writeEntry("Tooltip", tooltipType);
}

void KSetiSpyTray::contextMenuAboutToShow(KPopupMenu *menu)
{

  SetiClientMonitor *monitor = kdoc->setiMonitor();

  KPopupMenu *clientMenu = (KPopupMenu *) child("KSetiSpyTray::client_menu", "KPopupMenu");
  KPopupMenu *priorityMenu = (KPopupMenu *) child("KSetiSpyTray::priority_menu", "KPopupMenu");

  if(monitor != NULL)
  {
    const bool running = monitor->isRunning();

    clientMenu->changeItem(0, running ? i18n("&Stop") : i18n("&Start"));
    clientMenu->setItemEnabled(0, running || !monitor->command().isEmpty());

    priorityMenu->clear();
    clientMenu->setItemEnabled(1, running);

    if(running)
    {
      const int priority = monitor->getPriority();
      for(int i = priority; i < 20; i++)
        priorityMenu->insertItem(QString::number(i).rightJustify(3), i, i);
      priorityMenu->setItemChecked(priority, true);
    }
  }
  else
  {
    clientMenu->setItemEnabled(0, false);
    clientMenu->setItemEnabled(1, false);

    for(int i = 20; i >= -20; i-=5)
      priorityMenu->setItemChecked(20 - i, false);
    clientMenu->setItemEnabled(2, false);
  }

  QStringList profiles = kdoc->profiles();
  const int current = profiles.findIndex(kdoc->currentProfile());

  KPopupMenu *profileMenu = (KPopupMenu *) child("KSetiSpyTray::profile_menu", "KPopupMenu");
  profileMenu->clear();
  for(uint i = 0; i < profiles.count(); i++)
    profileMenu->insertItem(profiles[i], i, i);
  if (current != -1) profileMenu->setItemChecked(current, true);

  KPopupMenu *tooltipMenu = (KPopupMenu *) child("KSetiSpyTray::tooltip_menu", "KPopupMenu");
  tooltipMenu->setItemChecked(tooltipType, true);

  // hack: rewires the behavior of the Quit button
  int quitID = menu->idAt(menu->count()-1);
  if(menu->text(quitID) == menu->title())
    quitID = menu->idAt(menu->count()-2);;
  menu->disconnectItem(quitID, 0, 0);
  menu->connectItem(quitID, this, SIGNAL(quit()));
}

void KSetiSpyTray::updateContent()
{
  QToolTip::remove(this);

  const SetiClientMonitor::State state = kdoc->setiMonitorState();

  if(state >= SetiClientMonitor::Idle)
  {
    SetiClientMonitor *monitor = kdoc->setiMonitor();
    const seti_data *data = monitor->setiData();
    const double teraFLOPs = SetiDataMonitor::teraFLOPs(*data);
    const sys_info *info = monitor->sysInfo();
    const int cpu = monitor->currentCPU();

    const double secs = data->state.cpu;
    const double done = data->state.progress;

    const double completed = 1e2 * done;

    const double time_total = (done > 0) ? secs/done : 0.0;
    const double time_left = time_total - secs;

    const QDateTime date_time_done = QDateTime::currentDateTime().addSecs(int(time_left));
    const int days = QDateTime::currentDateTime().daysTo(date_time_done);
    QTime time_done = date_time_done.time();
    QString time_done_str = KGlobal::locale()->formatTime(time_done);
    if(days > 1) time_done_str += QString(" (+%1)").arg(days-1);

    const double speed = (secs > 0.0) ? (1e6 * teraFLOPs * done) / secs : 0.0;
    const double efficiency = (cpu >= 0 && speed > 0.0) ? info->cpus[cpu].MHz / speed : 0.0;
    const double rate = (secs > 0.0) ? 1e2 * (done/secs) * 60.0 * 60.0 : 0.0;

    const QString no_signals_str = i18n("no signals");
    QString signals_str = QString::null;
    if(data->output.spikes.count() > 0)
    {
      if(!signals_str.isEmpty()) signals_str += ", ";
      signals_str += i18n("%1 spikes").arg(data->output.spikes.count());
    }
    if(data->output.gaussians.count() > 0)
    {
      if(!signals_str.isEmpty()) signals_str += ", ";
      signals_str += i18n("%1 gaussians").arg(data->output.gaussians.count());
    }
    if(data->output.pulses.count() > 0)
    {
      if(!signals_str.isEmpty()) signals_str += ", ";
      signals_str += i18n("%1 pulses").arg(data->output.pulses.count());
    }
    if(data->output.triplets.count() > 0)
    {
      if(!signals_str.isEmpty()) signals_str += ", ";
      signals_str += i18n("%1 triplets").arg(data->output.triplets.count());
    }
    if(signals_str.isEmpty()) signals_str = no_signals_str;

    KLocale *locale = KGlobal::locale();

    switch(tooltipType) {
      case 0:
        QToolTip::add(this, i18n("KSetiSpy (%1%)").arg(locale->formatNumber(completed, 2)));
        break;
      case 1:
        QToolTip::add(this, i18n("KSetiSpy (%1 left)").arg(SetiClientMonitor::timeToString(time_left)));
        break;
      case 2:
        QToolTip::add(this, i18n("KSetiSpy (done at %1)").arg(time_done_str));
        break;
      case 4:
        QToolTip::add(this, i18n("KSetiSpy (%1 MegaFLOPS average)").arg(locale->formatNumber(speed, 2)));
        break;
      case 5:
        QToolTip::add(this, i18n("KSetiSpy (%1 CpF average)").arg(locale->formatNumber(efficiency, 2)));
        break;
      case 6:
        QToolTip::add(this, i18n("KSetiSpy (%1%/hour)").arg(locale->formatNumber(rate, 2)));
        break;
      case 8:
        QToolTip::add(this, i18n("KSetiSpy (%1)").arg(signals_str));
        break;
      default:
        break;
    }

    if(state != SetiClientMonitor::Running)
      setPixmap(KIconLoader().loadIcon(IdleIcon, KIcon::NoGroup, 22));
    else if(signals_str == no_signals_str)
      setPixmap(KIconLoader().loadIcon(UninterestingIcon, KIcon::NoGroup, 22));
    else if(SetiClientMonitor::isInteresting(data->state.best_gaussian))
      setPixmap(KIconLoader().loadIcon(InterestingIcon, KIcon::NoGroup, 22));
    else
      setPixmap(KIconLoader().loadIcon(ReturnedIcon, KIcon::NoGroup, 22));
  }
  else
  {
    QToolTip::remove(this);
    setPixmap(KIconLoader().loadIcon(IdleIcon, KIcon::Small, 22));
  }
}

void KSetiSpyTray::handleClient()
{
  SetiClientMonitor *monitor = kdoc->setiMonitor();
  if(monitor == NULL) return;

  if(monitor->isRunning())
    monitor->kill();
  else
    monitor->exec();
}

void KSetiSpyTray::setClientPriority(int priority)
{
  SetiClientMonitor *monitor = kdoc->setiMonitor();
  if(monitor == NULL) return;

  monitor->setPriority(priority);
}

void KSetiSpyTray::setTooltipType(int type)
{
  KPopupMenu *tooltipMenu = (KPopupMenu *) child("KSetiSpyTray::tooltip_menu", "KPopupMenu");
  tooltipMenu->setItemChecked(tooltipType, false);
  tooltipMenu->setItemChecked(type, true);

  tooltipType = type;

  updateContent();
}

void KSetiSpyTray::handleProfileMenu(int index)
{
  KPopupMenu *profileMenu = (KPopupMenu *) child("KSetiSpyTray::profile_menu", "KPopupMenu");
  kdoc->setCurrentProfile(profileMenu->text(index));
}

#include "ksetispytray.moc"

