/***************************************************************************
                          profiledialog.cpp  -  description
                             -------------------
    begin                : Wed Jun 27 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgrid.h>
#include <qgroupbox.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qtooltip.h>
#include <qvalidator.h>

#include <kcombobox.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ktabctl.h>

#include "profiledialog.h"

ProfileDialog::ProfileDialog(profileInfoStruct *profile, QWidget *parent, const char *name)
              : KDialog(parent, name, true)

{
  const bool isEdit = !profile->name.isEmpty();

  setCaption(isEdit ? i18n("Edit Profile") : i18n("New Profile"));
  // commented out temporarily
  // setMaximumSize(0, 0);

  QBoxLayout *layout = new QVBoxLayout(this);
  layout->setMargin(8);
  layout->setSpacing(8);

  KTabCtl *tabs = new KTabCtl(this);
  layout->addWidget(tabs, 1);

  KLocale *locale = KGlobal::locale();

  {
    QWidget *tab = new QWidget(tabs);
    tabs->addTab(tab, i18n("&Setup"));

    QGridLayout *gridlayout = new QGridLayout(tab, 4, 2, 8, 8);
    gridlayout->setRowStretch(1, 1);
    gridlayout->setRowStretch(2, 1);

    QGroupBox *group;

    {
      group = new QGroupBox(1, Horizontal, i18n("Main Settings"), tab);
      gridlayout->addMultiCellWidget(group, 0, 0, 0, 1);

      QHBox *hbox;

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Profile name:"), hbox);

      QLineEdit *edit = new QLineEdit(hbox, "ProfileDialog::name");
      edit->setText(profile->name);
      hbox->setStretchFactor(edit, 1);

      connect(edit, SIGNAL(textChanged(const QString&)), this, SLOT(validateOK(const QString&)));

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Files location:"), hbox);

      if(isEdit)
        hbox->setStretchFactor(new QLabel(profile->url.prettyURL(+1, KURL::StripFileProtocol), hbox), 1);
      else
      {
        KURLRequester *requester = new KURLRequester(hbox, "ProfileDialog::url");
        requester->fileDialog()->setMode(static_cast<KFile::Mode>(KFile::Directory | KFile::ExistingOnly));
        hbox->setStretchFactor(requester, 1);

        connect(requester, SIGNAL(textChanged(const QString&)), this, SLOT(validateOK(const QString&)));
      }
    }

    {
      group = new QGroupBox(1, Horizontal, i18n("Client"), tab);
      gridlayout->addMultiCellWidget(group, 1, 1, 0, 1);

      QHBox *hbox;

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Client location:"), hbox);

      KURLRequester *requester = new KURLRequester(hbox, "ProfileDialog::client_url");
      requester->fileDialog()->setMode(static_cast<KFile::Mode>(KFile::File | KFile::ExistingOnly | KFile::LocalOnly));
      requester->setURL(profile->clientURL.regular.prettyURL(-1, KURL::StripFileProtocol));

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Client parameters:"), hbox);

      new QLineEdit(profile->clientArgs, hbox, "ProfileDialog::client_arguments");

      if(isEdit) group->setEnabled(profile->url.isLocalFile());
    }

    {
      group = new QGroupBox(1, Horizontal, i18n("Monitoring"), tab);
      gridlayout->addWidget(group, 2, 0);

      QGrid *grid = new QGrid(2, group);
      grid->setSpacing(8);

      QSpinBox *spinbox;

      new QLabel(i18n("Client idle timeout (min):"), grid);

      spinbox = new QSpinBox(1, 20, 1, grid, "ProfileDialog::client_timeout");
      spinbox->setValue(profile->timeout / 60);

      new QLabel(i18n("Poll interval (sec):"), grid);

      spinbox = new QSpinBox(1, 99, 1, grid, "ProfileDialog::poll_interval");
      spinbox->setValue(profile->interval);
      if(isEdit) spinbox->setEnabled(!profile->url.isLocalFile());
    }

    {
      group = new QGroupBox(1, Horizontal, i18n("Process Control"), tab);
      gridlayout->addWidget(group, 2, 1);

      QCheckBox *checkbox;

      checkbox = new QCheckBox(i18n("&Launch client on startup"), group, "ProfileDialog::launch_on_startup");
      checkbox->setChecked(profile->launchOnStartup);

      checkbox = new QCheckBox(i18n("&Kill client process on exit"), group, "ProfileDialog::kill_on_exit");
      checkbox->setChecked(profile->killOnExit);

      checkbox = new QCheckBox(i18n("Keep SETI@home client &alive"), group, "ProfileDialog::keep_alive");
      checkbox->setEnabled(profile->launchOnStartup);
      checkbox->setChecked(profile->keepAlive);

      if(isEdit) group->setEnabled(profile->url.isLocalFile());
    }

    {
      group = new QGroupBox(1, Horizontal, i18n("Caching"), tab);
      gridlayout->addWidget(group, 3, 0);

      QCheckBox *checkbox;

      checkbox = new QCheckBox(i18n("&Use work unit cache"), group, "ProfileDialog::uses_cache");
      checkbox->setChecked(profile->usesCache);

      connect(checkbox, SIGNAL(toggled(bool)), this, SLOT(handleCache(bool)));

      if(isEdit) group->setEnabled(profile->url.isLocalFile());
    }

    {
      group = new QGroupBox(1, Horizontal, i18n("User Statistics"), tab);
      gridlayout->addWidget(group, 3, 1);

      QHBox *hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("User e-mail:"), hbox);

      QLineEdit *edit = new QLineEdit(profile->email, hbox, "ProfileDialog::email");
      edit->setEnabled(!profile->usesCache);
      hbox->setStretchFactor(edit, 1);
    }

    tab->setMinimumSize(gridlayout->minimumSize());
  }

  {
    QWidget *tab = new QWidget(tabs);
    tabs->addTab(tab, i18n("Lo&gging"));

    QBoxLayout *vbox = new QVBoxLayout(tab);
    vbox->setMargin(8);
    vbox->setSpacing(8);

    QGroupBox *group;

    {
      group = new QGroupBox(1, Horizontal, i18n("Work Unit Log"), tab);
      vbox->addWidget(group);

      QWidget *widget = new QWidget(group);
      QGridLayout *gridlayout = new QGridLayout(widget, 3, 3, 8, 8);
      gridlayout->setColStretch(2, 1);

      gridlayout->addWidget(new QLabel(i18n("Read from:"), widget), 0, 0);

      QRadioButton *radio;
      QButtonGroup *buttongroup = new QButtonGroup(widget, "ProfileDialog::uses_log");
      buttongroup->hide();

      radio = new QRadioButton(i18n("SETI Spy log"), widget, "ProfileDialog::uses_setispy_log");
      buttongroup->insert(radio, 0);
      gridlayout->addWidget(radio, 0, 1);

      radio = new QRadioButton(i18n("SetiWatch log (CSV)"), widget, "ProfileDialog::uses_csv_log");
      buttongroup->insert(radio, 1);
      gridlayout->addWidget(radio, 0, 2);

      buttongroup->setExclusive(true);
      buttongroup->setButton(profile->usesSETISpy ? 0 : 1);

      gridlayout->addWidget(new QLabel(i18n("Write to:"), widget), 1, 0);

      QCheckBox *checkbox;

      checkbox = new QCheckBox(i18n("SETI Spy log"), widget, "ProfileDialog::writes_setispy_log");
      checkbox->setChecked(profile->writesSETISpy);
      gridlayout->addWidget(checkbox, 1, 1);

      checkbox = new QCheckBox(i18n("SetiWatch log (CSV)"), widget, "ProfileDialog::writes_csv_log");
      checkbox->setChecked(profile->writesCSV);
      gridlayout->addWidget(checkbox, 1, 2);

      gridlayout->addWidget(new QLabel(i18n("Location:"), widget), 2, 0);

      KURLRequester *requester = new KURLRequester(widget, "ProfileDialog::log_url");
      requester->fileDialog()->setMode(static_cast<KFile::Mode>(KFile::Directory | KFile::ExistingOnly));
      requester->setURL(profile->logURL.prettyURL(+1, KURL::StripFileProtocol));
      gridlayout->addMultiCellWidget(requester, 2, 2, 1, 2);
    }
    {
      group = new QGroupBox(1, Horizontal, i18n("Best Gaussians"), tab);
      vbox->addWidget(group);

      QHBox *hbox;

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Save best score gaussian:"), hbox);

      KComboBox *menu;

      menu = new KComboBox(false, hbox, "ProfileDialog::best_gaussian_criterion");
      menu->insertItem(i18n("none"));
      menu->insertItem(i18n("all"));
      menu->insertItem(i18n("interesting"));
      for(int i = 4; i < 11; i++)
        menu->insertItem(i18n("S.R. > %1").arg(locale->formatNumber(i/2.0, 1)));
      menu->setCurrentItem((profile->bestGaussian.filter < 3) ? profile->bestGaussian.filter
                                                               : int(profile->bestGaussian.threshold * 2 - 1));
      QToolTip::add(menu, i18n("S.R. = Signal Ratio"));

      QLabel *label = new QLabel(i18n("Format:"), hbox);
      label->setAlignment(AlignRight | AlignVCenter);
      hbox->setStretchFactor(label, 1);

      menu = new KComboBox(false, hbox, "ProfileDialog::best_gaussian_format");
      menu->insertItem(i18n("Bitmap"));
      menu->insertItem(i18n("PNG"));
      menu->insertItem(i18n("JPEG"));
      menu->setCurrentItem(profile->bestGaussian.format);

      new QLabel(i18n("Size:"), hbox);

      menu = new KComboBox(false, hbox, "ProfileDialog::best_gaussian_size");
      menu->insertItem(i18n("Default"));
      menu->insertItem(i18n("SETI@home"));
      menu->setCurrentItem(profile->bestGaussian.size);

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Location:"), hbox);

      KURLRequester *requester = new KURLRequester(hbox, "ProfileDialog::best_gaussian_url");
      requester->fileDialog()->setMode(static_cast<KFile::Mode>(KFile::Directory | KFile::ExistingOnly));
      requester->setURL(profile->bestGaussian.url.prettyURL(+1, KURL::StripFileProtocol));
      hbox->setStretchFactor(requester, 1);
    }
    {
      group = new QGroupBox(1, Horizontal, i18n("Returned Gaussians"), tab);
      vbox->addWidget(group);

      QHBox *hbox;

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Save returned gaussians:"), hbox);

      KComboBox *menu;

      menu = new KComboBox(false, hbox, "ProfileDialog::returned_gaussians_criterion");
      menu->insertItem(i18n("none"));
      menu->insertItem(i18n("all"));
      menu->insertItem(i18n("interesting"));
      for(double i = 7; i < 11; i++)
        menu->insertItem(i18n("S.R. > %1").arg(locale->formatNumber(double(i)/2, 1)));
      menu->setCurrentItem((profile->returnedGaussians.filter < 3) ? profile->returnedGaussians.filter
                                                                   : int(profile->returnedGaussians.threshold * 2 - 4));
      QToolTip::add(menu, i18n("S.R. = Signal Ratio"));

      QLabel *label = new QLabel(i18n("Format:"), hbox);
      label->setAlignment(AlignRight | AlignVCenter);
      hbox->setStretchFactor(label, 1);

      menu = new KComboBox(false, hbox, "ProfileDialog::returned_gaussians_format");
      menu->insertItem(i18n("Bitmap"));
      menu->insertItem(i18n("PNG"));
      menu->insertItem(i18n("JPEG"));
      menu->setCurrentItem(profile->returnedGaussians.format);

      new QLabel(i18n("Size:"), hbox);

      menu = new KComboBox(false, hbox, "ProfileDialog::returned_gaussians_size");
      menu->insertItem(i18n("Default"));
      menu->insertItem(i18n("SETI@home"));
      menu->setCurrentItem(profile->returnedGaussians.size);

      hbox = new QHBox(group);
      hbox->setSpacing(8);

      new QLabel(i18n("Location:"), hbox);

      KURLRequester *requester = new KURLRequester(hbox, "ProfileDialog::returned_gaussians_url");
      requester->fileDialog()->setMode(static_cast<KFile::Mode>(KFile::Directory | KFile::ExistingOnly));
      requester->setURL(profile->returnedGaussians.url.prettyURL(+1, KURL::StripFileProtocol));
      hbox->setStretchFactor(requester, 1);
    }

    tab->setMinimumSize(vbox->minimumSize());
  }

  {
    QWidget *tab = new QWidget(tabs);
    tabs->addTab(tab, i18n("Cali&bration"));

    QGridLayout *gridlayout = new QGridLayout(tab, 2, 3, 8, 8);
    gridlayout->setRowStretch(1, 1);

    QBoxLayout *hbox = new QHBoxLayout();
    hbox->setSpacing(8);
    gridlayout->addMultiCellLayout(hbox, 0, 0, 0, 2);

    hbox->addWidget(new QLabel(i18n("Presets:"), tab));

    KComboBox *menu = new KComboBox(false, tab, "ProfileDialog::presets_menu");
    hbox->addWidget(menu);
    connect(menu, SIGNAL(activated(int)), this, SLOT(handlePresets(int)));

    hbox->addStretch(1);

    QGroupBox *group;
    QDoubleValidator *validator = new QDoubleValidator(0.0, 100.0, 4, this);
    const QString groupTitle[3] = {i18n("Low AR"), i18n("Medium AR"), i18n("High AR")};

    const QString reportedText = i18n("Reported %");
    const QString actualText = i18n("Actual %");

    const int reportedTextWidth = fontMetrics().width(reportedText);
    const int actualTextWidth = fontMetrics().width(actualText);
    const int colWidth = (reportedTextWidth > actualTextWidth) ? reportedTextWidth : actualTextWidth;

    for(int i = 0; i < 3; i++)
    {
      group = new QGroupBox(1, Horizontal, groupTitle[i], tab);
      gridlayout->addWidget(group, 1, i);

      QGrid *grid = new QGrid(2, group);
      grid->setSpacing(8);

      QLabel *label;

      label = new QLabel(reportedText, grid);
      label->setAlignment(AlignCenter);
      label->setMinimumWidth(colWidth);

      label = new QLabel(actualText, grid);
      label->setAlignment(AlignCenter);
      label->setMinimumWidth(colWidth);

      QLineEdit *edit;
      QString editID;

      for(int j = 0; j < 7; j++)
      {
        editID = QString("ProfileDialog::reported_%1_%2").arg(i).arg(j);
        edit = new QLineEdit(grid, editID.latin1());
        edit->setValidator(validator);
        editID = QString("ProfileDialog::actual_%1_%2").arg(i).arg(j);
        edit = new QLineEdit(grid, editID.latin1());
        edit->setValidator(validator);
      }
    }

    tab->setMinimumSize(gridlayout->minimumSize());
  }

  {
    QBoxLayout *hbox = new QHBoxLayout(layout);
    hbox->setSpacing(8);

    hbox->addStretch(1);

    KPushButton *okButton = new KPushButton(i18n("&OK"), this, "ProfileDialog::OK");
    hbox->addWidget(okButton);

    okButton->setDefault(true);

    KPushButton *cancelButton = new KPushButton(i18n("&Cancel"), this, "ProfileDialog::cancel");
    hbox->addWidget(cancelButton);

    connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
  }

  this->profile = profile;

  names.clear();
  presets.clear();

  handlePresets(0);

  validateOK(QString::null);
}

ProfileDialog::~ProfileDialog()
{
}

void ProfileDialog::setForbiddenNames(const QStringList& names)
{
  this->names = names;
}

void ProfileDialog::setCalibrationPresets(const QValueList<calibrationDataStruct>& presets)
{
  this->presets = presets;

  KComboBox *menu = (KComboBox *) child("ProfileDialog::presets_menu", "KComboBox");
  menu->clear();
  menu->insertItem(i18n("Original Settings"));
  menu->insertItem(i18n("No Calibration"));
  for(uint i = 0; i < presets.count(); i++)
    menu->insertItem(presets[i].cpu);
}

void ProfileDialog::accept()
{
  QLineEdit *edit;

  edit = (QLineEdit *) child("ProfileDialog::name", "QLineEdit");
  profile->name = edit->text().stripWhiteSpace();

  edit = (QLineEdit *) child("ProfileDialog::email", "QLineEdit");
  profile->email = edit->text().stripWhiteSpace();

  KURLRequester *requester;

  requester = (KURLRequester *) child("ProfileDialog::url", "KURLRequester");
  if(requester != NULL) profile->url = requester->url().stripWhiteSpace();

  requester = (KURLRequester *) child("ProfileDialog::client_url", "KURLRequester");
  profile->clientURL.regular = requester->url().stripWhiteSpace();

  edit = (QLineEdit *) child("ProfileDialog::client_arguments", "QLineEdit");
  profile->clientArgs = edit->text().stripWhiteSpace();

  QSpinBox *spinbox;

  spinbox = (QSpinBox *) child("ProfileDialog::client_timeout", "QSpinBox");
  profile->timeout = spinbox->value() * 60;

  spinbox = (QSpinBox *) child("ProfileDialog::poll_interval", "QSpinBox");
  profile->interval = spinbox->value();

  QCheckBox *checkbox;

  checkbox = (QCheckBox *) child("ProfileDialog::launch_on_startup", "QCheckBox");
  profile->launchOnStartup = checkbox->isChecked();

  checkbox = (QCheckBox *) child("ProfileDialog::kill_on_exit", "QCheckBox");
  profile->killOnExit = checkbox->isChecked();

  checkbox = (QCheckBox *) child("ProfileDialog::keep_alive", "QCheckBox");
  profile->keepAlive = checkbox->isChecked();

  checkbox = (QCheckBox *) child("ProfileDialog::uses_cache", "QCheckBox");
  profile->usesCache = checkbox->isChecked();

  QButtonGroup *buttongroup = (QButtonGroup *) child("ProfileDialog::uses_log", "QButtonGroup");
  profile->usesSETISpy = buttongroup->selected() == buttongroup->find(0);

  checkbox = (QCheckBox *) child("ProfileDialog::writes_setispy_log", "QCheckBox");
  profile->writesSETISpy = checkbox->isChecked();

  checkbox = (QCheckBox *) child("ProfileDialog::writes_csv_log", "QCheckBox");
  profile->writesCSV = checkbox->isChecked();

  requester = (KURLRequester *) child("ProfileDialog::log_url", "KURLRequester");
  profile->logURL = requester->url().stripWhiteSpace();
  if(profile->logURL.isEmpty())
    profile->logURL = profile->url;

  KComboBox *menu;
  int index;

  menu = (KComboBox *) child("ProfileDialog::best_gaussian_criterion", "KComboBox");
  index = menu->currentItem();
  profile->bestGaussian.filter = (index < 3) ? imageLogStruct::Filter(index) : imageLogStruct::SNRAbove;
  profile->bestGaussian.threshold = (index < 3) ? 2.0 : (index + 1)/2.0;

  menu = (KComboBox *) child("ProfileDialog::best_gaussian_format", "KComboBox");
  index = menu->currentItem();
  profile->bestGaussian.format = imageLogStruct::Format(index);

  menu = (KComboBox *) child("ProfileDialog::best_gaussian_size", "KComboBox");
  index = menu->currentItem();
  profile->bestGaussian.size = imageLogStruct::Size(index);

  requester = (KURLRequester *) child("ProfileDialog::best_gaussian_url", "KURLRequester");
  profile->bestGaussian.url = requester->url().stripWhiteSpace();
  if(profile->bestGaussian.url.isEmpty())
    profile->bestGaussian.url = profile->logURL;

  menu = (KComboBox *) child("ProfileDialog::returned_gaussians_criterion", "KComboBox");
  index = menu->currentItem();
  profile->returnedGaussians.filter = (index < 3) ? imageLogStruct::Filter(index) : imageLogStruct::SNRAbove;
  profile->returnedGaussians.threshold = (index < 3) ? 3.5 : (index + 4)/2.0;

  menu = (KComboBox *) child("ProfileDialog::returned_gaussians_format", "KComboBox");
  index = menu->currentItem();
  profile->returnedGaussians.format = imageLogStruct::Format(index);

  menu = (KComboBox *) child("ProfileDialog::returned_gaussians_size", "KComboBox");
  index = menu->currentItem();
  profile->returnedGaussians.size = imageLogStruct::Size(index);

  requester = (KURLRequester *) child("ProfileDialog::returned_gaussians_url", "KURLRequester");
  profile->returnedGaussians.url = requester->url().stripWhiteSpace();
  if(profile->returnedGaussians.url.isEmpty())
    profile->returnedGaussians.url =  profile->logURL;

  profile->calibration = getCalibration();

  KDialog::accept();
}

void ProfileDialog::validateOK(const QString&)
{
  bool enabled = true;

  QLineEdit *edit = (QLineEdit *) child("ProfileDialog::name", "QLineEdit");
  const QString name = edit->text();
  enabled = enabled && !name.isEmpty() && (name == profile->name || !names.contains(name));

  KURLRequester *requester = (KURLRequester *) child("ProfileDialog::url", "KURLRequester");
  enabled = enabled && (requester == NULL || !requester->url().isEmpty());

  KPushButton *okButton = (KPushButton *) child("ProfileDialog::OK", "KPushButton");
  okButton->setEnabled(enabled);
}

calibrationDataStruct ProfileDialog::getCalibration()
{
  calibrationDataStruct out;

  out.cpu = QString::null;

  out.low.clear();
  out.medium.clear();
  out.high.clear();

  for(int i = 0; i < 3; i++)
  {
    KLocale *locale = KGlobal::locale();
    QLineEdit *edit;
    QString editID;

    for(int j = 0; j < 7; j++)
    {
      calibrationItemStruct item;

      editID = QString("ProfileDialog::reported_%1_%2").arg(i).arg(j);
      edit = (QLineEdit *) child(editID.latin1(), "QLineEdit");
      item.reported = locale->readNumber(edit->text()) * 1e-2;

      editID = QString("ProfileDialog::actual_%1_%2").arg(i).arg(j);
      edit = (QLineEdit *) child(editID.latin1(), "QLineEdit");
      item.actual = locale->readNumber(edit->text()) * 1e-2;

      if(i == 0) out.low += item;
      else if(i == 1) out.medium += item;
      else /* i == 2 */ out.high += item;
    }
  }

  return(out);
}

void ProfileDialog::setCalibration(calibrationDataStruct data)
{
  for(int i = 0; i < 3; i++)
  {

    QValueList<calibrationItemStruct> list;

    if(i == 0) list = data.low;
    else if(i == 1) list = data.medium;
    else /* i == 2 */ list = data.high;

    KLocale *locale = KGlobal::locale();
    QLineEdit *edit;
    QString editID;

    for(int j = 0; j < 7 && j < int(list.count()); j++)
    {
      editID = QString("ProfileDialog::reported_%1_%2").arg(i).arg(j);
      edit = (QLineEdit *) child(editID.latin1(), "QLineEdit");
      edit->setText(locale->formatNumber(list[j].reported * 1e2, 2));

      editID = QString("ProfileDialog::actual_%1_%2").arg(i).arg(j);
      edit = (QLineEdit *) child(editID.latin1(), "QLineEdit");
      edit->setText(locale->formatNumber(list[j].actual * 1e2, 2));
    }
  }
}

void ProfileDialog::handleCache(bool checked)
{
  QLineEdit *edit = (QLineEdit *) child("ProfileDialog::email", "QLineEdit");
  edit->setEnabled(!checked);
}

void ProfileDialog::handlePresets(int index)
{
  if(index == 0)
    setCalibration(profile->calibration);
  else if(index == 1)
    setCalibration(KSetiSpyDoc::defaultCalibration());
  else // index > 1
    setCalibration(presets[index-2]);
}

#include "profiledialog.moc"

