/*
	extras.cpp - extras
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "extras.h"
#include "miscutils.h"
#include "mmainwindow.h"

#include <qdir.h>

#include <kdesktopfile.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kstringhandler.h>

/* contructor */

Extras::Extras(QObject *parent, const char *name)
	: QObject(parent, name),
	// init
	_idToFileMap(0) // initialized in idToFileMap()
{
	// init popup menu
	_menu = new KPopupMenu(mainWindow, "KPopupMenu::menu");
	connect(
		_menu, SIGNAL(aboutToShow()),
		this, SLOT(showMenuSlot()));
	connect(
		_menu, SIGNAL(activated(int)),
		this, SLOT(activatedSlot(int)));

	_actionDescription = QString::null;
	_fileToExecute = QString::null;
	_userExtrasDir = QString::null; // initialized in userExtrasDir()
}

/* destructor */

Extras::~Extras()
{
	if (_idToFileMap)
	{
		delete _idToFileMap;
		_idToFileMap = 0;
	}
	if (_menu)
	{
		delete _menu;
		_menu = 0;
	}
}

/* public */

bool Extras::execAction()
{
	if (_fileToExecute.isNull())
		return FALSE; // error

	if (!execFile(_fileToExecute))
	{
		MiscUtils::showRunErrorMessage(_fileToExecute);

		return FALSE; // error
	}

	return TRUE; // ok
}

QString Extras::getActionName() const
{
	if (_actionDescription.isEmpty())
		return QString::null;

	// "Substitute characters at the middle of a string by \"...\""
	return KStringHandler::csqueeze(_actionDescription, 100);
}

void Extras::modify()
{
	KMessageBox::information(
		mainWindow,
		// text
		"<qt>" +
		i18n("Use context menu to add/edit/remove links.") +
		"<ul>" +
		"<li>" + i18n("Use <b>Context Menu</b> " \
			"to create a new link to application") + "</li>" +
		"<li>" + i18n("Use <b>Create New|Folder...</b> " \
			"to create a new submenu") + "</li>" +
		"<li>" + i18n("Use <b>Properties</b> " \
			"to change icon, name, or comment") + "</li>" +
		"</ul></qt>",
		i18n("Extras"), // title
		"ModifyExtras" // config entry
	);

	// launch Konqueror as an items editor
	KRun::run("konqueror \"" + userExtrasDir() + "\"", KURL::List());
}

void Extras::setAction(const QString &path, const QString &description)
{
	_fileToExecute = path;
	_actionDescription = description;
}

/* private */

void Extras::buildMenu(KPopupMenu *parentMenu, const QString &subdir)
{
	int id;
	QDir d(subdir);
	QFileInfo fi;
	QString description, icon, menuText, mimeType, name, path;

	QStringList list =
		d.entryList(QDir::DefaultFilter, QDir::Name | QDir::DirsFirst);
	list.remove(".");
	list.remove("..");

	uint count = list.count();
	for (uint i = 0; i < count; i++)
	{
		fi.setFile(d, list[i]);
		path = fi.filePath();

		// dir
		if (fi.isDir())
		{
			KPopupMenu *submenu =
				new KPopupMenu(parentMenu, "KPopupMenu::submenu");
			connect(
				submenu, SIGNAL(activated(int)),
				this, SLOT(activatedSlot(int)));

			KDesktopFile *desktopFile =
				new KDesktopFile(path + "/.directory", TRUE);
			if (desktopFile->getConfigState() != KConfigBase::NoAccess)
			{
				icon = desktopFile->readIcon();
				if (icon.isEmpty())
					icon = "folder";
				name = desktopFile->readName();
				if (name.isEmpty())
					name = fi.baseName();
				description = desktopFile->readComment();
				if (description.isEmpty())
					menuText = name;
				else
					menuText = description;

				// add item
				parentMenu->insertItem(
					SmallIcon(icon),
					menuText,
					submenu
				);
			}
			else
			{
				// add item
				parentMenu->insertItem(
					SmallIcon("folder"),
					fi.baseName(),
					submenu
				);
			}
			delete desktopFile;

			// recurse
			buildMenu(submenu, fi.filePath());
		}

		// file
		else if (fi.isFile())
		{
			mimeType = KMimeType::findByPath(path)->name();

			if (mimeType == "application/x-trash")
				continue;

			if (mimeType != "application/x-desktop")
			{
				// add item
				id = parentMenu->insertItem(
					KMimeType::pixmapForURL(KURL::fromPathOrURL(path)),
					fi.baseName()
				);
				// map item
				idToFileMap()->insert(id, path);

				continue;
			}

			// application/x-desktop
			KDesktopFile *desktopFile =
				new KDesktopFile(path, TRUE);
			if (desktopFile->getConfigState() != KConfigBase::NoAccess)
			{
				name = desktopFile->readName();
				if (name.isEmpty())
					name = desktopFile->readEntry("Exec");
				description = desktopFile->readComment();
				if (description.isEmpty())
					menuText = name;
				else
					menuText = description + " (" + name + ")";

				// add item
				id = parentMenu->insertItem(
					SmallIcon(desktopFile->readIcon()),
					menuText
				);
				// map item
				idToFileMap()->insert(id, path);
			}
			delete desktopFile;
		}
	}
}

bool Extras::execFile(const QString &path)
{
	pid_t pid;
	pid = KRun::runURL(
		KURL::fromPathOrURL(path),
		KMimeType::findByPath(path)->name()
	);

	return pid != 0;
}

QMap<int, QString> *Extras::idToFileMap()
{
	if (!_idToFileMap)
		_idToFileMap = new QMap<int, QString>;

	return _idToFileMap;
}

QString Extras::userExtrasDir()
{
	if (_userExtrasDir.isNull())
	{
		_userExtrasDir =
			KGlobal::dirs()->saveLocation("data", "kshutdown/extras");
	}

	return _userExtrasDir;
}

/* private slots */

void Extras::activatedSlot(int id)
{
	if (id == _modifyID)
		return;

	QMap<int, QString>::iterator i = idToFileMap()->find(id);
	if (i == idToFileMap()->end())
	{
		KMessageBox::error(mainWindow,
			i18n("Internal error!\nSelected menu item is broken."),
			i18n("Extras")
		);
	}
	else
	{
		setAction(i.data(), _menu->text(id));
		mainWindow->cancel();
		mainWindow->setActive(MMainWindow::What_Extras, TRUE);
	}
}

void Extras::modifySlot()
{
	modify();
}

void Extras::showMenuSlot()
{
	_menu->clear(); // reset menu
	idToFileMap()->clear(); // reset map

	_modifyID = _menu->insertItem(
		SmallIcon("edit"),
		i18n("&Modify..."),
		this, SLOT(modifySlot())
	);

	QStringList extrasDirs(KGlobal::dirs()->findDirs("data", "kshutdown/extras"));
	QStringList::ConstIterator
		begin = extrasDirs.begin(),
		end = extrasDirs.end();
	for (QStringList::ConstIterator it = begin; it != end; it++)
	{
		_menu->insertSeparator();
		buildMenu(_menu, *it);
	}
}
