/*
	links.cpp - A Link Creator
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "links.h"
#include "mmainwindow.h"

#include <qcombobox.h>
#include <qfile.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>

#include <kdesktopfile.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>

/* public */

Links::Links(QWidget *parent, const char *name)
	: QGrid(2, parent, name)
{
	setSpacing(4);
	int alignment = AlignVCenter;
	alignment |= QApplication::reverseLayout() ? AlignLeft : AlignRight;

	// location
	QLabel *l_location =
		new QLabel(i18n("Locatio&n:"), this, "QLabel::l_location");
	l_location->setAlignment(alignment);
	cb_location = new QComboBox(this, "QComboBox::cb_location");
	cb_location->insertItem(SmallIcon("desktop"), i18n("Desktop"));
	cb_location->insertItem(SmallIcon("kmenu"), i18n("K Menu"));
	l_location->setBuddy(cb_location);

	// type
	QLabel *l_type = new QLabel(i18n("&Type:"), this, "QLabel::l_type");
	l_type->setAlignment(alignment);
	cb_type = new QComboBox(this, "QComboBox::cb_type");
	cb_type->insertItem(SmallIcon("kshutdown"), i18n("KShutDown"));
	cb_type->insertItem(SmallIcon("wizard"), i18n("KShutDown Wizard"));
	cb_type->insertItem(SmallIcon("exit"), i18n("Standard Logout Dialog"));
	l_type->setBuddy(cb_type);

	// create link
	KPushButton *b_createLink = new KPushButton(SmallIcon("filenew"),
		i18n("Cr&eate Link"), this, "KPushButton::b_createLink");
	connect(
		b_createLink, SIGNAL(clicked()),
		this, SLOT(createLinkSlot()));

	// remove link
	KPushButton *b_removeLink = new KPushButton(SmallIcon("editdelete"),
		i18n("&Remove Link"), this, "KPushButton::b_removeLink");
	connect(
		b_removeLink, SIGNAL(clicked()),
		this, SLOT(removeLinkSlot()));
}

void Links::createLink(const QString &path, const QString &command,
	const QString &icon, const QString &name)
{
	if (path.isNull())
		return;

	KDesktopFile *df = new KDesktopFile(path);
	df->setGroup("Desktop Entry");
	df->writeEntry("Encoding", "UTF-8");
	df->writeEntry("Exec", command);
	df->writeEntry("Icon", icon);
	if (!name.isNull())
		df->writeEntry("Name", name);
	df->writeEntry("StartupNotify", "false");
	df->writeEntry("Type", "Application");
	delete df;

	if (!QFile::exists(path))
	{
		KMessageBox::error(
			mainWindow,
			"<qt>" +
			QString(i18n("Could not create file <b>%1</b>!"))
				.arg(path) +
			"</qt>"
		);
	}
}

void Links::removeLink(const QString &path)
{
	if (QFile::exists(path) && !QFile::remove(path))
	{
		KMessageBox::error(
			mainWindow,
			"<qt>" +
			QString(i18n("Could not remove file <b>%1</b>!"))
				.arg(path) +
			"</qt>"
		);
	}
}

/* private */

QString Links::getCurrentLocationPath() const
{
	QString path;
	switch (cb_location->currentItem())
	{
		case 0:
			path = KGlobalSettings::desktopPath() + "/";
			break;
		case 1:
			path = locateLocal("apps", "") + "/";
			break;
		default:
			return QString::null;
	}

	switch (cb_type->currentItem())
	{
		case 0: return path += "kshutdown-classic.desktop";
		case 1: return path += "kshutdown-wizard.desktop";
		case 2: return path += "kshutdown-standard.desktop";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeCommand() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "kshutdown --wizard";
		case 2: return "kshutdown --standard";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeIcon() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "kshutdown";
		case 2: return "exit";
		default: return QString::null;
	}
}

/* private slots */

void Links::createLinkSlot()
{
	createLink(
		getCurrentLocationPath(),
		getCurrentTypeCommand(),
		getCurrentTypeIcon(),
		cb_type->currentItem() == 2 ? i18n("Logout") : cb_type->currentText()
	);
}

void Links::removeLinkSlot()
{
	removeLink(getCurrentLocationPath());
}
