/*
	miscutils.cpp - Misc Utils
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "miscutils.h"
#include "msystemtray.h"

#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kconfig.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <krun.h>
#include <kwin.h>

void MiscUtils::closeCDTray()
{
	if (kshutdownrc->cdTrayClose)
		runShellCommand(kshutdownrc->cdTrayCloseCommand, KProcess::Block);
}

bool MiscUtils::confirmAction(const MMainWindow::What action)
{
	if (!kshutdownrc->confirmAction)
		return TRUE; // no confirmation = Yes

	KWin::setOnDesktop(mainWindow->winId(), KWin::currentDesktop());
	bool result = (KMessageBox::questionYesNo(
		mainWindow,
		i18n("Are you sure?"), // text
		mainWindow->caption(), // title
		mainWindow->getActionName(action), // yes button
		i18n("&Cancel"), // no button
		"ShutDown" // config entry
	) == KMessageBox::Yes);

	KConfig *conf = kapp->config();
	conf->setGroup("Notification Messages");
	kshutdownrc->confirmAction = conf->readEntry("ShutDown", "").isEmpty();

	return result;
}

void MiscUtils::customMessage(const QString &text, const QString &testCommand)
{
	QString c;
	if (testCommand.isNull())
		c = kshutdownrc->customMessageCommand;
	else
		c = testCommand;

	if (kshutdownrc->customMessageEnabled)
	{
		c.replace("%appname", "KShutDown");
		c.replace("%text", text);
		c.replace("%title", mainWindow->caption());
		runShellCommand(c, KProcess::DontCare);
	}
}

QString MiscUtils::formatDateTime(const int secs, const QString &format)
{
	if (secs >= 86400 || secs < 0) // 24h
		return ">24";

	int h = secs / 3600;
	int m = (secs / 60) % 60;
	int s = secs % 60;
	QTime t(h, m, s);

	if (format.isNull())
		return t.toString();

	return t.toString(format);
}

QString MiscUtils::formatDateTime(const QDateTime &dt)
{
	QDateTime now = QDateTime::currentDateTime();

	return
		dt.toString(KGlobal::locale()->use12Clock() ? "hh:mm ap, dddd" : "hh:mm, dddd") + " (+" +
		formatDateTime(now.secsTo(dt), "hh:mm") +
		")";
}

bool MiscUtils::isActionEnabled(const MMainWindow::What action)
{
	switch (action)
	{
		case MMainWindow::What_Nothing:
			return FALSE;
		case MMainWindow::What_ShutDown:
			return !mainWindow->isRestricted("kshutdown_action_shutdown");
		case MMainWindow::What_Reboot:
			return !mainWindow->isRestricted("kshutdown_action_reboot");
		case MMainWindow::What_LockScreen:
			return !mainWindow->isRestricted("kshutdown_action_lockscreen");
		case MMainWindow::What_Logout:
			return !mainWindow->isRestricted("kshutdown_action_logout");
		case MMainWindow::What_Extras:
			return !mainWindow->isRestricted("kshutdown_action_extras");
		default:
			return FALSE;
	}
}

void MiscUtils::launchHelp(const QString &anchor)
{
#if KDE_VERSION < KDE_MAKE_VERSION(3,2,0)
	kapp->invokeHelp(anchor);
#else
	kapp->invokeHelp(anchor, "", "");
#endif // KDE_VERSION
}

void MiscUtils::playSound(const QString &name)
{
	KNotifyClient::event(mainWindow->winId(), name);
}

bool MiscUtils::runCommand(const QString &command)
{
	int pid;
	pid = KRun::run(command, KURL::List());

	if (pid)
		return TRUE; // ok

	showRunErrorMessage(command);

	return FALSE; // error
}

void MiscUtils::runCommandBeforeAction(const QString &configEntry)
{
	KConfig *conf = kapp->config();

	if (!conf->hasGroup(configEntry))
		return;

	conf->setGroup(configEntry);

	if (!conf->readBoolEntry("RunCommandBeforeAction", FALSE))
		return;

	QString command = conf->readEntry("CommandBeforeAction", "");
	int pause = conf->readNumEntry("CommandBeforeActionPause", 10);
	if (pause < 0 || pause > 300)
		pause = 10;
	runShellCommand(command, KProcess::DontCare, pause);
}

bool MiscUtils::runShellCommand(const QString &command, const KProcess::RunMode mode, const int pause)
{
	if (command.isEmpty())
		return FALSE;

	KProcess *p = new KProcess();
	if (!p)
	{
		showRunErrorMessage(command);

		return FALSE; // error
	}

	bool retVal = TRUE; // assume ok

	p->setUseShell(TRUE);
	*p << command;
	if (!p->start(mode))
	{
		showRunErrorMessage(command);

		retVal = FALSE; // error
	}
#if KDE_VERSION >= KDE_MAKE_VERSION(3,2,0)
	else
	{
		// pause
		if (pause > 0)
			p->wait(pause);
	}
#endif // KDE_VERSION
	delete p;

	return retVal;
}

void MiscUtils::setHint(QWidget *widget, const QString &value)
{
	QToolTip::add(widget, value);
	QWhatsThis::add(widget, value);
}

void MiscUtils::showRunErrorMessage(const QString &command)
{
	mainWindow->tray()->popupMessage(
		MSystemTray::Reason_Error,
		QString(i18n("Could not run \"%1\"!"))
			.arg(command)
	);
}

void MiscUtils::showTestMessage(const QString &message)
{
	KMessageBox::information(mainWindow, message, i18n("Test"));
}
