/*
	mmainwindow.cpp - main window
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef TEST_KSHUTDOWN_DCOP_API
//	#define TEST_KSHUTDOWN_DCOP_API
// uncomment above line to test kshutdowndcopapi.*
// see also "taskTest" below
#endif // TEST_KSHUTDOWN_DCOP_API

#include "appscheduler.h"
#include "configuration.h"
#include "extras.h"
#include "miscutils.h"
#include "mmainwindow.h"
#include "mmessagedialog.h"
#include "mschedulertab.h"
#include "msettingsdialog.h"
#include "mstatstab.h"
#include "msystemtray.h"
#include "mwizard.h"

#ifdef TEST_KSHUTDOWN_DCOP_API
	#include "kshutdowndcopapi.h"
#endif // TEST_KSHUTDOWN_DCOP_API

#include <qdatetimeedit.h>
#include <qhbox.h>
#include <qhgroupbox.h>
#include <qlayout.h>
#include <qspinbox.h>
#include <qtabwidget.h>
#include <qtimer.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qvbox.h>
#include <qvgroupbox.h>

#include <dcopclient.h>
#include <kaction.h>
#include <kconfig.h>
#include <kdatewidget.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kpushbutton.h>
#include <kwin.h>

/* globals */

#ifdef TEST_KSHUTDOWN_DCOP_API
KShutDownDCOPAPI *taskTest = 0;
#endif // TEST_KSHUTDOWN_DCOP_API

MMainWindow *MMainWindow::mainApplicationWindow = 0;

/* constructor */

MMainWindow::MMainWindow(QWidget *parent, const char *name)
	// Iface
	: DCOPObject("KShutdownIface"),
	// window
	KMainWindow(parent, name, WStyle_ContextHelp),
	// init
	_active(FALSE),
	_cancelled(FALSE),
	_testMode(FALSE),
	_totalExit(FALSE),
	_delayMisc(DelayMisc_Seconds),
	_lastSettingsPage(0), // general
	_oldSec(-1),
	_modalMessageDialog(0),
	_systemTray(0), // init in "tray()"
	_what(What_Nothing),
	_delayType(DelayType_Now)
{
	mainApplicationWindow = this;

	connect(
		kapp, SIGNAL(aboutToQuit()),
		this, SLOT(aboutToQuitSlot()));

	// init date/time
	dt_end = QDateTime();
	dt_start = QDateTime();

	// init "extras" manager
	_extras = new Extras(this, "Extras::_extras");

	// action timeout timer
	_timer = new QTimer(this);
	connect(
		_timer, SIGNAL(timeout()),
		this, SLOT(checkTimeSlot()));

	// main widget
	QHBox *mainBox = new QHBox(this, "QHBox::mainBox");
	setCentralWidget(mainBox);

	// vertical (left) - tabs + bottom buttons
	QVBox *leftBox = new QVBox(mainBox);

	// add tabs
	_tabs = new QTabWidget(leftBox, "QTabWidget::_tabs");
	// time tab
	_tabs->addTab(initTimeTab(leftBox), i18n("T&ime"));
	// scheduler tab
	_schedulerTab = new MSchedulerTab(_tabs, "MSchedulerTab::_schedulerTab");
	_tabs->addTab(_schedulerTab, "");
	// statistics tab
	_statsTab = new MStatsTab(_tabs, "MStatsTab::_statsTab");
	_tabs->addTab(_statsTab, i18n("St&atistics"));
	connect(
		_tabs, SIGNAL(currentChanged(QWidget *)),
		this, SLOT(tabChangedSlot(QWidget *)));
	_tabs->setMargin(0);

	// init bottom buttons
	initBottomButtons(leftBox);

	// init main buttons (right)
	initButtons(mainBox);

	// init DCOP
	DCOPClient *client = kapp->dcopClient();
	client->registerAs(kapp->name(), FALSE);

	// init scheduler
	_appScheduler = new AppScheduler();
	client->setDefaultObject(objId()); // KShutdownIface

	_configuration = new Config();
	kshutdownrc->read();

	updateSchedulerTab();
	setCaptions();
	updateTimeInfoSlot();

#ifdef TEST_KSHUTDOWN_DCOP_API
	taskTest = new KShutDownDCOPAPI(client, "Hello, World!", ACTION_REBOOT);
	qDebug("taskTest->id(): %d", taskTest->id());
#endif // TEST_KSHUTDOWN_DCOP_API
}

/* destructor */

MMainWindow::~MMainWindow()
{
#ifdef TEST_KSHUTDOWN_DCOP_API
	if (taskTest)
		delete taskTest;
#endif // TEST_KSHUTDOWN_DCOP_API

	delete _appScheduler;
	delete _configuration;
	delete _extras;
}

/* public */

// TODO: 0.7: force application exit (possibly without data save)
bool MMainWindow::endSession(const KApplication::ShutdownType type,
	const What action)
{
	_totalExit = FALSE;

	// test mode
	if (isTestMode())
	{
		MiscUtils::showTestMessage(getActionName(action));

		return TRUE;
	}

	setCaption(i18n("Please wait..."));
	_totalExit = TRUE;
	if (
		!kapp->requestShutDown(
			KApplication::ShutdownConfirmNo,
			type,
			KApplication::ShutdownModeForceNow
		)
	)
	{
		setCaption("");
		KMessageBox::error(
			this,
			i18n(
				"Could not logout properly.\n" \
				"The session manager cannot be contacted."
			)
		);
		_totalExit = FALSE;

		return FALSE; // error
	}

	return TRUE; // ok
}

bool MMainWindow::execAction(const What action, const bool stopTimer)
{
	if (stopTimer)
		setActive(action, FALSE);

	_totalExit = FALSE;

	Method m = Method_KDE;
	QString c;

	if (!MiscUtils::isActionEnabled(action))
		return FALSE; // error

	switch (action)
	{
		// nothing
		case What_Nothing:
			return FALSE; // error

		// shut down
		case What_ShutDown:
			_totalExit = TRUE;
			MiscUtils::closeCDTray();
			getMethod("Shut Down", m, c, DEFAULT_SHUT_DOWN_COMMAND);
			MiscUtils::runCommandBeforeAction("Shut Down");

			if (m == Method_KDE)
				return endSession(KApplication::ShutdownTypeHalt, action);

			break;

		// reboot
		case What_Reboot:
			_totalExit = TRUE;
			MiscUtils::closeCDTray();
			getMethod("Reboot", m, c, DEFAULT_REBOOT_COMMAND);
			MiscUtils::runCommandBeforeAction("Reboot");

			if (m == Method_KDE)
				return endSession(KApplication::ShutdownTypeReboot, action);

			break;

		// lock screen
		case What_LockScreen:
			getMethod("Lock Screen", m, c, DEFAULT_LOCK_SCREEN_COMMAND);
			MiscUtils::runCommandBeforeAction("Lock Screen");
			if (m == Method_KDE)
			{
				// test mode
				if (isTestMode())
				{
					MiscUtils::showTestMessage(getActionName(action));

					return TRUE; // ok
				}

				// this is a modified "void Lockout::lock()"
				// from "Lock/Logout Applet" (lockout.cpp [3.1.4])
				QCString kdesktop("kdesktop");
				int kshutdown_screen_number = qt_xscreen();
				if (kshutdown_screen_number)
				{
					kdesktop.sprintf(
						"kdesktop-screen-%d",
						kshutdown_screen_number
					);
				}

				if (
					!kapp->dcopClient()->send(
						kdesktop,
						"KScreensaverIface",
						"lock()",
						""
					)
				)
				{
					KMessageBox::error(
						this,
						i18n("kdesktop: DCOP call failed!")
					);

					return FALSE; // error
				}
				hide(); // hide window

				return TRUE; // ok
			}
			break;

		// logout
		case What_Logout:
			_totalExit = TRUE;
			MiscUtils::closeCDTray();
			getMethod("Logout", m, c, DEFAULT_LOGOUT_COMMAND);
			MiscUtils::runCommandBeforeAction("Logout");

			if (m == Method_KDE)
				return endSession(KApplication::ShutdownTypeNone, action);

			break;

		// "extras" action
		case What_Extras:
			// test mode
			if (isTestMode())
			{
				MiscUtils::showTestMessage(
					extras()->actionDescription() + "\n" +
					extras()->fileToExecute()
				);

				return TRUE; // ok
			}

			return extras()->execAction();
	}

	// test mode
	if (isTestMode())
	{
		MiscUtils::showTestMessage(i18n("Command: %1").arg(c));

		return TRUE; // ok
	}

	// run default or user command
// TODO: 0.7: add option to change /sbin/* permissions (some apps)
// TODO: 0.7: save session before /sbin/* execution
// TODO: 0.7: detect permissions
	if (MiscUtils::runCommand(c))
	{
		if (action == What_LockScreen)
			hide(); // hide window

		return TRUE; // ok
	}

	_totalExit = FALSE;

	return FALSE; // error
}

QPixmap MMainWindow::getActionIcon(const What action) const
{
	// compatible with the standard KDE logout dialog
	switch (action)
	{
		case What_Nothing:
			return SmallIcon("misc");

		case What_ShutDown:
			return SmallIcon("exit");

		case What_Reboot:
			return SmallIcon("reload");

		case What_LockScreen:
			return SmallIcon("lock");

		case What_Logout:
			return SmallIcon("undo");

		case What_Extras:
			return SmallIcon("bookmark");
	}

	return SmallIcon("misc");
}

QString MMainWindow::getActionName(const What value) const
{
	switch (value)
	{
		case What_Nothing:
			return i18n("Nothing");

		case What_ShutDown:
			return i18n("Turn Off Computer");

		case What_Reboot:
			return i18n("Restart Computer");

		case What_LockScreen:
			return i18n("Lock Session");

		case What_Logout:
			return i18n("End Current Session");

		case What_Extras:
			return extras()->getActionName();

		default:
			return i18n("Unknown");
	}
}

QString MMainWindow::getCurrentActionName() const
{
	return getActionName(_what);
}

int MMainWindow::getCurrentDelayMisc() const
{
	return cb_delayMisc->currentItem();
}

int MMainWindow::getCurrentDelayType() const
{
	return cb_delayType->currentItem();
}

void MMainWindow::setDelayType(const DelayType value)
{
	if (isRestricted("kshutdown_tab_time"))
	{
		_delayType = DelayType_Now;
		cb_delayType->setEnabled(FALSE);
	}
	else
	{
		_delayType = value;
		cb_delayType->setEnabled(TRUE);
	}

	cb_delayType->setCurrentItem(_delayType);

	switch (_delayType)
	{
		case DelayType_Now:
			gb_delayValues->setTitle(i18n("No delay"));
			sb_delay->setEnabled(FALSE);
			cb_delayMisc->setEnabled(FALSE);
			te_time->setDisplay(
				KGlobal::locale()->use12Clock()
				? QTimeEdit::Hours | QTimeEdit::Minutes | QTimeEdit::AMPM
				: QTimeEdit::Hours | QTimeEdit::Minutes
			);
			te_time->setEnabled(FALSE);
			b_misc->setEnabled(FALSE);
			MiscUtils::setHint(b_misc, QString::null);
			de_date->setEnabled(FALSE);
			l_date->setEnabled(FALSE);
			l_timeFromNow->setEnabled(FALSE);
			l_timeHHMM->setEnabled(FALSE);
			break;
		case DelayType_Misc:
			gb_delayValues->setTitle(i18n("Enter delay:"));
			sb_delay->setEnabled(TRUE);
			cb_delayMisc->setEnabled(TRUE);
			te_time->setDisplay(
				KGlobal::locale()->use12Clock()
				? QTimeEdit::Hours | QTimeEdit::Minutes | QTimeEdit::AMPM
				: QTimeEdit::Hours | QTimeEdit::Minutes
			);
			te_time->setEnabled(FALSE);
			b_misc->setEnabled(TRUE);
			MiscUtils::setHint(b_misc, i18n("Set delay to 0 seconds"));
			de_date->setEnabled(FALSE);
			l_date->setEnabled(FALSE);
			l_timeFromNow->setEnabled(TRUE);
			l_timeHHMM->setEnabled(FALSE);
			break;
		case DelayType_TimeFromNow:
			gb_delayValues->setTitle(i18n("Enter hour and minute:"));
			sb_delay->setEnabled(FALSE);
			cb_delayMisc->setEnabled(FALSE);
			te_time->setDisplay(QTimeEdit::Hours | QTimeEdit::Minutes);
			te_time->setEnabled(TRUE);
			MiscUtils::setHint(b_misc, i18n("Set delay to 00:00"));
			b_misc->setEnabled(TRUE);
			de_date->setEnabled(FALSE);
			l_date->setEnabled(FALSE);
			l_timeFromNow->setEnabled(FALSE);
			l_timeHHMM->setEnabled(TRUE);
			l_timeHHMM->setText(i18n("Time from &now (HH:MM):"));
			break;
		case DelayType_DateTime:
			gb_delayValues->setTitle(i18n("Enter time and date:"));
			sb_delay->setEnabled(FALSE);
			cb_delayMisc->setEnabled(FALSE);
			te_time->setDisplay(
				KGlobal::locale()->use12Clock()
				? QTimeEdit::Hours | QTimeEdit::Minutes | QTimeEdit::AMPM
				: QTimeEdit::Hours | QTimeEdit::Minutes
			);
			te_time->setEnabled(TRUE);
			MiscUtils::setHint(
				b_misc,
				i18n("Set date/time to the current date/time")
			);
			b_misc->setEnabled(TRUE);
			de_date->setEnabled(TRUE);
			l_date->setEnabled(TRUE);
			l_timeFromNow->setEnabled(FALSE);
			l_timeHHMM->setEnabled(TRUE);
			l_timeHHMM->setText(i18n("Ti&me (HH:MM):"));
			break;
	}
}

void MMainWindow::getMethod(const QString &configEntry, Method &method,
	QString &command, const QString &defaultCommand)
{
	KConfig *conf = kapp->config();

	if (!conf->hasGroup(configEntry))
	{
		method = Method_KDE;
		command = defaultCommand;

		return;
	}

	conf->setGroup(configEntry);

	// read method
	method = (Method)conf->readNumEntry("Method", Method_KDE);
	if (method < Method_KDE || method > Method_UserCommand)
		method = Method_KDE;

	switch (method)
	{
		case Method_KDE:
			command = conf->readEntry("Command", defaultCommand);
			break;
		case Method_DefaultCommand:
			command = defaultCommand;
			break;
		case Method_UserCommand:
			command = conf->readEntry("Command", defaultCommand);
			if (command.isEmpty())
				method = Method_KDE;
			break;
	}
}

QTime MMainWindow::getTime() const
{
	return te_time->time();
}

/** Sets editor time to @c time. */
void MMainWindow::setTime(const QTime &time)
{
	te_time->setTime(QTime(time.hour(), time.minute(), 0));
}

bool MMainWindow::isRestricted(const QString &key) const
{
	return !kapp->authorize(key);
}

bool MMainWindow::messageDialogAccepted(const bool modal,
	const What actionToExecute, const QString &text)
{
	if (!kshutdownrc->warningMessageEnabled)
		return TRUE; // accept

	KWin::setOnDesktop(mainWindow->winId(), KWin::currentDesktop());

	// show message dialog
	MMessageDialog *messageDialog = new MMessageDialog(
		modal,
		kshutdownrc->warningMessageDelay,
		actionToExecute,
		text,
		this,
		"MMessageDialog::messageDialog"
	);
	Q_CHECK_PTR(messageDialog);

	// modal
	if (modal)
	{
		_modalMessageDialog = messageDialog;
		if (_modalMessageDialog->exec() == QDialog::Accepted) // exec dialog
		{
			delete _modalMessageDialog;
			_modalMessageDialog = 0;

			return TRUE; // accepted
		}
		delete _modalMessageDialog;
		_modalMessageDialog = 0;

		return FALSE; // rejected
	}

	// non-modal
	messageDialog->show();

	return TRUE;
}

void MMainWindow::notifyUser(const int secs)
{
	QString s;
	switch (secs)
	{
		case 3600: // 1 hour
			s = i18n("1 hour warning");
			MiscUtils::customMessage(s);
			tray()->popupMessage(MSystemTray::Reason_Warning, s);
			break;
		case 300: // 5 minutes
			MiscUtils::playSound("kshutdown-5m");
			s = i18n("5 minutes warning");
			MiscUtils::customMessage(s);
			tray()->popupMessage(MSystemTray::Reason_Warning, s);
			break;
		case 60: // 1 minute
			MiscUtils::playSound("kshutdown-1m");
			s = i18n("1 minute warning");
			MiscUtils::customMessage(s);
			tray()->popupMessage(MSystemTray::Reason_Warning, s);
			break;
		case 10: // 10 seconds
			s = i18n("10 seconds warning");
			MiscUtils::customMessage(s);
			tray()->popupMessage(MSystemTray::Reason_Warning, s);
			break;
		case 3: // 3 seconds
			MiscUtils::playSound("kshutdown-three");
			break;
		case 2: // 2 seconds
			MiscUtils::playSound("kshutdown-two");
			break;
		case 1: // 1 second
			MiscUtils::playSound("kshutdown-one");
			break;
	}
}

bool MMainWindow::runWizard()
{
	if (!MWizard::canRun())
		return FALSE;

	MWizard *w = new MWizard(0, "MWizard::w");

	Q_CHECK_PTR(w);

	bool retVal = w->runWizard();
	delete w;

	return retVal;
}

void MMainWindow::setActive(const What value, const bool yes,
	const bool confirmation)
{
	tray()->hidePopupMessages();

	setCancelled(FALSE);

	if (isActive() == yes)
		return; // this should never happen, because buttons are disabled

	if (confirmation)
	{
		if (yes && !MiscUtils::confirmAction(value))
			return; // action not confirmed by user; do nothing
	}

	// immediate action
	if (_delayType == DelayType_Now)
	{
		execAction(value);

		return;
	}

	_active = yes;
	_what = value;

	// disable widgets
	if (_active)
	{
		cb_delayType->setEnabled(FALSE);
		b_misc->setEnabled(FALSE);
		sb_delay->setEnabled(FALSE);
		cb_delayMisc->setEnabled(FALSE);
		te_time->setEnabled(FALSE);
		de_date->setEnabled(FALSE);
	}
	// enable widgets
	else
	{
		setDelayType(_delayType); // this will enable widgets correctly
	}

	b_shutDown->setEnabled(!_active);
	b_reboot->setEnabled(!_active);
	b_logout->setEnabled(!_active);
	b_extras->setEnabled(!_active);

	updateTimeInfoSlot();

	if (isActive())
	{
		_oldSec = -1;

		b_cancel->setGuiItem(KStdGuiItem::cancel());
		MiscUtils::setHint(b_cancel, i18n("Cancel an active action"));

		tray()->setIdle(FALSE);

		_tabs->setCurrentPage(0); // show "Time" tab

		// time machine is not supported yet
		if (dt_end < dt_start)
		{
			QString selDT = QString(i18n("Selected date/time: %1"))
				.arg("<b>" + KGlobal::locale()->formatDateTime(dt_end, FALSE, TRUE)+ "</b>");
			QString curDT = QString(i18n("Current date/time: %1"))
				.arg("<b>" + KGlobal::locale()->formatDateTime(dt_start, FALSE, TRUE) + "</b>");

			setActive(What_Nothing, FALSE);
			KMessageBox::error(
				this,
				"<qt>" +
				i18n(
					"Selected date/time is earlier " \
					"than current date/time!"
				) +
				"<br><br>" +
				selDT + "<br>" +
				curDT +
				"</qt>"
			);

			return;
		}

		_timer->start(500); // this timer value is good ..

		int secsTo = dt_start.secsTo(dt_end);
		updateTimeInfo(secsTo);
	}
	else
	{
		_timer->stop();

		setCaptions();
		b_cancel->setGuiItem(KStdGuiItem::close());
		MiscUtils::setHint(b_cancel, i18n("Quit the application"));

		tray()->setIdle(TRUE);
	}
}

void MMainWindow::setDate(const QDate &date)
{
	de_date->setDate(date);
}

void MMainWindow::setDelayMisc(const DelayMisc value)
{
	QString s = "";
	_delayMisc = value;
	cb_delayMisc->setCurrentItem(_delayMisc);
	switch (_delayMisc)
	{
		case DelayMisc_Seconds:
			s = i18n("Enter delay in seconds.");
			break;
		case DelayMisc_Minutes:
			s = i18n("Enter delay in minutes.");
			break;
		case DelayMisc_Hours:
			s = i18n("Enter delay in hours.");
			break;
	}
	MiscUtils::setHint(sb_delay, s);
}

MSystemTray *MMainWindow::tray()
{
	if (_systemTray == 0)
		_systemTray = new MSystemTray(this, "MSystemTray::_systemTray");

	return _systemTray;
}

void MMainWindow::updateSchedulerTab(const bool focus)
{
	_schedulerTab->update();

	if (appScheduler()->isEnabled())
	{
		_tabs->setTabLabel(_schedulerTab, i18n("Sched&uler"));
	}
	else
	{
		_tabs->setTabLabel(
			_schedulerTab,
			i18n("Sched&uler") + " [" + i18n("Disabled") + "]"
		);
	}

	if (focus)
		_tabs->setCurrentPage(_tabs->indexOf(_schedulerTab));
}

// DCOP:

// general:

void MMainWindow::cancel()
{
	setActive(What_Nothing, FALSE);
	setCancelled(TRUE);
	if (_modalMessageDialog)
		_modalMessageDialog->cancelAction();
}

bool MMainWindow::shutDown()
{
	return execAction(What_ShutDown);
}

bool MMainWindow::shutdown()
{
	return shutDown();
	//.........^^
}

bool MMainWindow::reboot()
{
	return execAction(What_Reboot);
}

bool MMainWindow::lockScreen()
{
	return execAction(What_LockScreen, FALSE); // don't stop timer
}

bool MMainWindow::logout()
{
	return execAction(What_Logout);
}

// scheduler:

bool MMainWindow::activateAction(int id)
{
	return appScheduler()->activateAction(id);
}

int MMainWindow::registerTask(const QString &name, const QString &description,
	int action)
{
	return appScheduler()->registerTask(name, description, action);
}

bool MMainWindow::unregisterTask(int id)
{
	return appScheduler()->unregisterTask(id);
}

// misc:

void MMainWindow::configure()
{
	MSettingsDialog *dialog = new MSettingsDialog(this, "MSettingsDialog::dialog");

	Q_CHECK_PTR(dialog);

	dialog->showPage(_lastSettingsPage); // restore page
	dialog->exec(); // show dialog
	_lastSettingsPage = dialog->activePageIndex(); // remember page

	delete dialog;

	// refresh views
	updateSchedulerTab();
	if (_tabs->currentPage() == _statsTab)
		_statsTab->initOnDemand();
}

QString MMainWindow::getStatusInfo()
{
	return caption();
}

void MMainWindow::makeVisible()
{
	show();
	KWin::setOnDesktop(winId(), KWin::currentDesktop());
	raise();
}

void MMainWindow::setTestMode(bool yes)
{
	_testMode = yes;
	setCaptions(); // add/remove "[ TEST MODE ]" to/from window caption.

	QString s;
	if (_testMode)
	{
		s = i18n("Test mode enabled");
		tray()->popupMessage(MSystemTray::Reason_Warning, s);
	}
	else
	{
		s = i18n("Test mode disabled");
		tray()->popupMessage(MSystemTray::Reason_Info, s);
	}
	MiscUtils::customMessage(s);
}

void MMainWindow::wizard()
{
	if (runWizard())
		makeVisible();
}

/* private */

void MMainWindow::calcSelectedTime()
{
	// start time = now
	dt_start.setDate(QDate::currentDate());
	dt_start.setTime(QTime::currentTime());
	// end time = start time
	dt_end.setDate(dt_start.date());
	dt_end.setTime(dt_start.time());

	// round down msec
	int ms = dt_end.time().msec();
	if (ms > 0)
		dt_end.setTime(dt_end.time().addMSecs(-ms));

	QDateTime dtDelay;
	switch (_delayMisc)
	{
		case DelayMisc_Seconds:
			dtDelay = dt_end.addSecs(sb_delay->value());
			break;
		case DelayMisc_Minutes:
			dtDelay = dt_end.addSecs(sb_delay->value() * 60);
			break;
		case DelayMisc_Hours:
			dtDelay = dt_end.addSecs(sb_delay->value() * 3600);
			break;
	}
	switch (_delayType)
	{
		case DelayType_Now:
			break;
		case DelayType_Misc:
			// end time = start time + delay
			dt_end.setDate(dtDelay.date());
			dt_end.setTime(dtDelay.time());
			break;
		case DelayType_TimeFromNow:
			// end time = start time + delay time
			dt_end = dt_end.addSecs(
				(getTime().hour() * 3600) +
				(getTime().minute() * 60)
			);
			break;
		case DelayType_DateTime:
			// end time = selected date/time
			dt_end.setDate(de_date->date());
			dt_end.setTime(getTime());
			break;
	}
}

void MMainWindow::initBottomButtons(QWidget *parent)
{
	QWidget *w = new QWidget(parent, "QWidget::w");
	QHBoxLayout *l = new QHBoxLayout(w);

	// help
	KPushButton *b_help = new KPushButton(SmallIcon("help"), i18n("&Help"), w,
		"KPushButton::b_help");
	b_help->setFlat(TRUE);
	b_help->setPopup(helpMenu());
	l->addWidget(b_help);

	// configure
	KPushButton *b_configure = new KPushButton(SmallIcon("configure"),
		i18n("C&onfigure..."), w, "KPushButton::b_configure");
	b_configure->setFlat(TRUE);
	connect(
		b_configure, SIGNAL(clicked()),
		this, SLOT(settingsSlot()));
	l->addWidget(b_configure);
}

void MMainWindow::initButtons(QWidget *parent)
{
	QWidget *w = new QWidget(parent, "QWidget::w");
	QVBoxLayout *l = new QVBoxLayout(w, 4, 0);

	// shut down
	b_shutDown = new KPushButton(getActionIcon(What_ShutDown),
		i18n("&Turn Off Computer"), w, "KPushButton::b_shutDown");
	b_shutDown->setFocus();
	l->addWidget(b_shutDown);
	connect(
		b_shutDown, SIGNAL(clicked()),
		this, SLOT(shutDownSlot()));
	if (!MiscUtils::isActionEnabled(MMainWindow::What_ShutDown))
		b_shutDown->hide();

	// reboot
	b_reboot = new KPushButton(getActionIcon(What_Reboot),
		i18n("&Restart Computer"), w, "KPushButton::b_reboot");
	l->addWidget(b_reboot);
	connect(
		b_reboot, SIGNAL(clicked()),
		this, SLOT(rebootSlot()));
	if (!MiscUtils::isActionEnabled(MMainWindow::What_Reboot))
		b_reboot->hide();

	// -
	l->addStretch();

	// lock
	b_lockScreen = new KPushButton(getActionIcon(What_LockScreen),
		i18n("&Lock Session"), w, "KPushButton::b_lockScreen");
	MiscUtils::setHint(
		b_lockScreen,
		i18n("Lock the screen using a screen saver")
	);
	l->addWidget(b_lockScreen);
	connect(
		b_lockScreen, SIGNAL(clicked()),
		this, SLOT(lockScreenSlot()));
	if (!MiscUtils::isActionEnabled(MMainWindow::What_LockScreen))
		b_lockScreen->hide();

	// logout
	b_logout = new KPushButton(getActionIcon(What_Logout),
		i18n("&End Current Session"), w, "KPushButton::b_logout");
	MiscUtils::setHint(b_logout, i18n("Logout"));
	l->addWidget(b_logout);
	connect(
		b_logout, SIGNAL(clicked()),
		this, SLOT(logoutSlot()));
	if (!MiscUtils::isActionEnabled(MMainWindow::What_Logout))
		b_logout->hide();

	l->addStretch();

	// wizard
	b_wizard = new KPushButton(SmallIcon("wizard"), i18n("&Wizard..."), w,
		"KPushButton::b_wizard");
	MiscUtils::setHint(b_wizard, i18n("Run the Wizard"));
	connect(
		b_wizard, SIGNAL(clicked()),
		this, SLOT(wizardSlot()));
	l->add(b_wizard);
	if (!MWizard::canRun())
		b_wizard->hide();

	// extras
	b_extras = new KPushButton(getActionIcon(What_Extras), i18n("E&xtras..."),
		w, "KPushButton::b_extras");
	b_extras->setPopup(extras()->menu());
	MiscUtils::setHint(
		b_extras,
		i18n(
			"More commands...<br>" \
			"Click <b>Modify...</b> to add/edit/remove items."
		)
	);
	l->add(b_extras);
	if (!MiscUtils::isActionEnabled(MMainWindow::What_Extras))
		b_extras->hide();

	// -
	l->addStretch();

	// close/cancel
	b_cancel =
		new KPushButton(KStdGuiItem::close(), w, "KPushButton::b_cancel");
	MiscUtils::setHint(b_cancel, i18n("Quit the application"));
	l->addWidget(b_cancel);
	connect(
		b_cancel, SIGNAL(clicked()),
		this, SLOT(cancelSlot()));

	// key bindings
	(void) new KAction(i18n("&Cancel"), KShortcut(Key_Escape),
		this, SLOT(cancelSlot()), actionCollection(), "ks_cancel");
/* TODO: 0.7: bind F1
	// conflicts with Shift+F1 (?)
	(void) new KAction(i18n("&Help"), KShortcut(Key_F1),
		this, SLOT(launchHelpSlot()), actionCollection(), "ks_help");
*/
	(void) new KAction(i18n("&Quit"), KShortcut(CTRL + Key_Q),
		this, SLOT(cancelSlot()), actionCollection(), "ks_quit");
}

QWidget *MMainWindow::initTimeTab(QWidget *parent)
{
	QWidget *w = new QWidget(parent, "QWidget::w");
	QVBoxLayout *l = new QVBoxLayout(w, 2);

	QHGroupBox *gb_delayMisc =
		new QHGroupBox(i18n("Select the type o&f delay:"), w,
		"QHGroupBox::gb_delayMisc");
	gb_delayMisc->setInsideSpacing(0);
	// misc button (current/reset time)
	b_misc = new QToolButton(gb_delayMisc, "QToolButton::b_misc");
	QString clearIcon =
		QApplication::reverseLayout() ? "clear_left" : "locationbar_erase";
	b_misc->setIconSet(SmallIcon(clearIcon));
	connect(
		b_misc, SIGNAL(clicked()),
		this, SLOT(miscSlot()));
	// combo box
	cb_delayType = new QComboBox(gb_delayMisc, "QComboBox::cb_delayType");
	cb_delayType->insertItem(SmallIcon("messagebox_warning"), i18n("Now!"));
	cb_delayType->insertItem(i18n("Time From Now"));
	cb_delayType->insertItem(
		i18n("Time From Now") + " (" + i18n("HH:MM") + ")"
	);
	cb_delayType->insertItem(i18n("At Date/Time"));
	MiscUtils::setHint(cb_delayType, i18n("Select the type of delay"));
	connect(
		cb_delayType, SIGNAL(activated(int)),
		this, SLOT(delayTypeChangeSlot(int)));
	connect(
		cb_delayType, SIGNAL(activated(int)),
		this, SLOT(updateTimeInfoSlot()));
	l->addWidget(gb_delayMisc);

	gb_delayValues = new QVGroupBox("null", w, "QVGroupBox::gb_delayValues");

	// delay layout
	QHBox *delayBox = new QHBox(gb_delayValues);
	// delay label
	l_timeFromNow =
		new QLabel(i18n("Time from &now:"), delayBox, "QLabel::l_timeFromNow");
	// delay edit
	sb_delay = new QSpinBox(delayBox, "QSpinBox::sb_delay");
	sb_delay->setRange(0, 10000);
	sb_delay->setSteps(1, 1);
	connect(
		sb_delay, SIGNAL(valueChanged(int)),
		this, SLOT(updateTimeInfoSlot()));
	l_timeFromNow->setBuddy(sb_delay);
	// delay type
	cb_delayMisc = new QComboBox(delayBox, "QComboBox::cb_delayMisc");
	cb_delayMisc->insertItem(i18n("Second(s)"));
	cb_delayMisc->insertItem(i18n("Minute(s)"));
	cb_delayMisc->insertItem(i18n("Hour(s)"));
	connect(
		cb_delayMisc, SIGNAL(activated(int)),
		this, SLOT(delayMiscChangeSlot(int)));
	connect(
		cb_delayMisc, SIGNAL(activated(int)),
		this, SLOT(updateTimeInfoSlot()));

	// time edit layout
	QHBox *timeEditBox = new QHBox(gb_delayValues);
	// time edit label
	l_timeHHMM =
		new QLabel(i18n("Ti&me (HH:MM):"), timeEditBox, "QLabel::l_timeHHMM");
	// time edit
	te_time = new QTimeEdit(timeEditBox, "QTimeEdit::te_time");
	te_time->setAutoAdvance(TRUE);
	MiscUtils::setHint(te_time, i18n("HH:MM"));
	connect(
		te_time, SIGNAL(valueChanged(const QTime&)),
		this, SLOT(updateTimeInfoSlot()));
	l_timeHHMM->setBuddy(te_time);

	// date edit layout
	QHBox *dateEditBox = new QHBox(gb_delayValues);
	// date edit label
	l_date = new QLabel(i18n("&Date:"), dateEditBox, "QLabel::l_date");
	// date edit
	de_date = new KDateWidget(dateEditBox, "KDateWidget::de_date");
	setDate(QDate::currentDate());
	MiscUtils::setHint(de_date, i18n("Enter date"));
	connect(
		de_date, SIGNAL(changed(QDate)),
		this, SLOT(updateTimeInfoSlot()));
	l_date->setBuddy(de_date);

	l->addWidget(gb_delayValues);

	// status line
	l_statusLine = new QLabel(w, "QLabel::l_statusLine");
	l->addWidget(l_statusLine);

	return w;
}

void MMainWindow::setCaptions(const QString &remainingTime, const QString &selectedTime)
{
	// window tool tip
	QString s;
	if (isActive())
	{
		s = remainingTime + " / " + selectedTime + " - " + getCurrentActionName();
	}
	if (isTestMode())
		s += " [" + i18n("TEST MODE") + "]";
	setCaption(s);

	// system tray tool tip
	if (isActive())
	{
		s = "<br>";
		s += i18n("Remaining time: <b>%1</b>").arg(remainingTime) + "<br>";
		s += i18n("Selected time: <b>%1</b>").arg(selectedTime) + "<br>";
		s += i18n("Selected action: <b>%1</b>").arg(getCurrentActionName()) + "<br>";
		if (isTestMode())
		{
			s += "<br>" + i18n("<b>Note: </b> The test mode is enabled") + "<br>";
		}
	}
	else
	{
		s = "";
	}
	QToolTip::add(
		tray(),
		"<qt>" \
		"<b>KShutDown</b><br>" +
		s +
		"<br>" +
		i18n("<b>Tip:</b> Use the <b>Middle Mouse Button</b> to display the actions menu") +
		"</qt>"
	);
}

void MMainWindow::totalShutDown()
{
	// shut down, reboot, etc. (now)
	if (!execAction(_what))
	{
		QString s = QString(i18n("Action failed! (%1)"))
			.arg(_what);
		KMessageBox::sorry(this, s);
	}
}

void MMainWindow::updateTimeInfo(const int secs)
{
	setCaptions(
		MiscUtils::formatDateTime(secs),
		KGlobal::locale()->formatDateTime(dt_end, TRUE, TRUE)
	);
}

/* public slots */

void MMainWindow::cancelSlot()
{
	// cancel shut down
	if (isActive())
		cancel();
	// close window
	else
		closeSlot();
}

void MMainWindow::closeSlot()
{
	_totalExit = TRUE; // do not hide window to system tray
	close();
}

void MMainWindow::lockScreenSlot()
{
	if (isActive())
		lockScreen();
	else
		setActive(What_LockScreen, TRUE);
}

void MMainWindow::logoutSlot()
{
	setActive(What_Logout, TRUE);
}

void MMainWindow::rebootSlot()
{
	setActive(What_Reboot, TRUE);
}

void MMainWindow::shutDownSlot()
{
#ifdef TEST_KSHUTDOWN_DCOP_API
	if (taskTest)
	{
		if (!taskTest->id())
		{
			KMessageBox::information(this, "taskTest->id(): NULL");
		}
		else if (taskTest->activate())
		{
			KMessageBox::information(this, "taskTest->activate(): TRUE");
		}
		else
		{
			KMessageBox::information(this, "taskTest->activate(): FALSE");
		}
	}
#else // TEST_KSHUTDOWN_DCOP_API
	setActive(What_ShutDown, TRUE);
#endif // TEST_KSHUTDOWN_DCOP_API
}

void MMainWindow::updateTimeInfoSlot()
{
	calcSelectedTime();
	QString s;
	if (_delayType == DelayType_Now)
		s = i18n("No delay");
	else
		s = MiscUtils::formatDateTime(dt_end.addSecs(1));
	l_statusLine->setText(QString(i18n("Selected time: <b>%1</b>")).arg(s));
}

/* private slots */

void MMainWindow::aboutToQuitSlot()
{
	_totalExit = TRUE;
}

void MMainWindow::checkTimeSlot()
{
	// check timeout

	QDateTime now = QDateTime::currentDateTime();

	int curSec = now.time().second();

	if (curSec == _oldSec)
		return; // time not changed since last check

	_oldSec = curSec;

	// timeout
	if (now >= dt_end)
	{
		totalShutDown();

		return;
	}

	int secsTo = now.secsTo(dt_end);
	updateTimeInfo(secsTo);

	// show message
	if (
		kshutdownrc->warningMessageEnabled &&
		(secsTo == (kshutdownrc->warningMessageDelay * 60))
	)
	{
		setActive(_what, FALSE); // stop timer
		// create modal dialog
		if (
			messageDialogAccepted(
				TRUE,
				currentAction(),
				getCurrentActionName()
			)
		)
		{
			totalShutDown();
		}

		return;
	}

	notifyUser(secsTo);
}

void MMainWindow::delayMiscChangeSlot(int index)
{
	setDelayMisc((DelayMisc)index);
}

void MMainWindow::delayTypeChangeSlot(int index)
{
	setDelayType((DelayType)index);
}

void MMainWindow::launchHelpSlot()
{
	MiscUtils::launchHelp();
}

void MMainWindow::miscSlot()
{
	switch (_delayType)
	{
		case DelayType_Now:
			break;
		case DelayType_Misc:
			delay()->setValue(0);
			break;
		case DelayType_TimeFromNow:
			setTime(QTime(0, 0, 0)); // 00:00:00
			break;
		case DelayType_DateTime:
			setTime(QTime::currentTime());
			setDate(QDate::currentDate());
			break;
	}
}

void MMainWindow::settingsSlot()
{
	configure();
}

void MMainWindow::tabChangedSlot(QWidget *tab)
{
	if (!tab)
		return;

	// init on demand
	if (tab == _schedulerTab)
		_schedulerTab->initOnDemand();
	else if (tab == _statsTab)
		_statsTab->initOnDemand();
}

void MMainWindow::wizardSlot()
{
	runWizard();
}

/* protected */

bool MMainWindow::queryClose()
{
	// exit
	if (_totalExit || kapp->sessionSaving())
		return TRUE;

	// hide in system tray
	tray()->flashIcon();
	hide();

	return FALSE;
}

bool MMainWindow::queryExit()
{
	_totalExit = TRUE;
	kshutdownrc->write();

	return TRUE; // ok to exit
}
