/*
	mstatstab.cpp - statistics tab
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "mmainwindow.h"
#include "mstatstab.h"

#include <qlayout.h>
#include <qprocess.h>
#include <qregexp.h>
#include <qtextedit.h>
#include <qwhatsthis.h>

#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpushbutton.h>

/* constructor */

MStatsTab::MStatsTab(QWidget *parent, const char *name)
	: QWidget(parent, name),
	// init
	_initDone(FALSE),
	b_refresh(0),
	_process(0),
	_buf(""),
	te_output(0)
{
}

/* destructor */

MStatsTab::~MStatsTab()
{
	killProcess();
}

/* public */

void MStatsTab::initOnDemand()
{
	if (_initDone)
	{
		refreshSlot();

		return;
	}

	_process = new QProcess(this);

	connect(
		_process, SIGNAL(processExited()),
		this, SLOT(processExitSlot()));
	connect(
		_process, SIGNAL(readyReadStdout()),
		this, SLOT(readStdoutSlot()));

	QVBoxLayout *l = new QVBoxLayout(this, 2);

	// output
	te_output = new QTextEdit(this, "QTextEdit (stats view)");
	te_output->setReadOnly(TRUE);
	te_output->setTextFormat(RichText); // HTML tags
	te_output->setWordWrap(QTextEdit::NoWrap);

	// based on the "Linux User's Manual" (man w)
	QWhatsThis::add(te_output,
		"<qt>" +
		i18n(
		"This view displays information about the users currently on the " \
		"machine, and their processes.<br>" \
		"The header shows how long the system has been running."
		) +
		"</qt>"
	);
	l->addWidget(te_output);

	// refresh button
	b_refresh = new KPushButton(SmallIcon("reload"), i18n("Re&fresh"), this,
		"KPushButton::b_refresh");
	b_refresh->setFlat(TRUE);
	connect(
		b_refresh, SIGNAL(clicked()),
		this, SLOT(refreshSlot()));
	l->addWidget(b_refresh);

	// show created widgets
	te_output->show();
	b_refresh->show();

	refreshSlot();

	_initDone = TRUE;
}

/* private */

void MStatsTab::killProcess() const
{
	// kill previous process if any
	if (_process && _process->isRunning())
		_process->kill();
}

/* private slots */

void MStatsTab::processExitSlot()
{
	QString s_header = "#a0a0a0";
	QString s_warning = "<font color=\"#ff0000\">";

	_buf.replace("\n", "<br>");
	_buf.replace(" ", "&nbsp;");

	// underline header
	_buf.replace("USER", "<font color=\"" + s_header + "\">USER</font>");
	_buf.replace("TTY", "<font color=\"" + s_header + "\">TTY</font>");
	_buf.replace("FROM", "<font color=\"" + s_header + "\">FROM</font>");
	_buf.replace("LOGIN@", "<font color=\"" + s_header + "\">LOGIN@</font>");
	_buf.replace("JCPU", "<font color=\"" + s_header + "\">JCPU</font>");
	_buf.replace("PCPU", "<font color=\"" + s_header + "\">PCPU</font>");
	_buf.replace("IDLE", "<font color=\"" + s_header + "\">IDLE</font>");
	_buf.replace("WHAT", "<font color=\"" + s_header + "\">WHAT</font>");

	_buf.replace(QRegExp("\\broot\\b"), s_warning + "root</font>");
	_buf.replace(QRegExp("\\bssh\\b"), s_warning + "ssh</font>");
	_buf.replace(QRegExp("\\bsu\\b"), s_warning + "su</font>");
	_buf.replace(QRegExp("\\bsudo\\b"), s_warning + "sudo</font>");
	_buf.replace(QRegExp("\\btelnet\\b"), s_warning + "telnet</font>");

	_buf.prepend("<qt><pre style=\"font-family: " + KGlobalSettings::fixedFont().family() + "\"><b>");
	_buf.append("</b></pre></qt>");

	te_output->setText(_buf);

	b_refresh->setEnabled(TRUE);
}

void MStatsTab::readStdoutSlot()
{
	_buf.append(_process->readStdout());
}

void MStatsTab::refreshSlot()
{
	te_output->setPaletteBackgroundColor(QColor(255, 255, 255)); // white
	te_output->setPaletteForegroundColor(QColor(0, 0, 0)); // black

	if (mainWindow->isRestricted("kshutdown_tab_stats"))
	{
		te_output->setText(i18n("Disabled by the Administrator."));
		setEnabled(FALSE);

		return;
	}

	te_output->setText(i18n("Please wait..."));

	killProcess();

	b_refresh->setEnabled(FALSE);

	_buf = "";

	_process->clearArguments();
	_process->addArgument("w");
	if (!kshutdownrc->statsLongFormat)
		_process->addArgument("-s"); // -s - short format
	if (kshutdownrc->statsToggleFromField)
		_process->addArgument("-f"); // -f - toggle "from" field
	if (!_process->start())
	{
		_buf = "<b><font color=\"#ff0000\">" + i18n("Error") + "</font><br><br>";
		_buf.append(i18n("Command:") + " ");
		_buf.append(_process->arguments().join(" ") + "</b>");
		te_output->setText(_buf);
		b_refresh->setEnabled(TRUE);
	}
}
