/*
	msystemtray.h - system tray
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MSYSTEMTRAY_H__
#define __MSYSTEMTRAY_H__

#include "mmainwindow.h"

#include <ksystemtray.h>

class QListBox;

class KPassivePopup;
class KPixmap;

/** @short System tray icon and popup messages. */
class MSystemTray: public KSystemTray
{
	Q_OBJECT
public:
	/** This value tells what kind of icon will be used in the list. */
	enum Reason {
		/** Information icon. */
		Reason_Info = 0,
		/** Warning icon. */
		Reason_Warning = 1,
		/** Error icon. */
		Reason_Error = 2
	};

	/**
	 * Constructor.
	 * @param parent A parent widget
	 * @param name A widget name
	 */
	MSystemTray(QWidget *parent, const char *name = 0);

	/** Destructor. */
	virtual ~MSystemTray();

	/** Flashes the system tray icon. */
	void flashIcon();

	/** Hiddes the popup messages. */
	void hidePopupMessages();

	/**
	 * Constructs and returns a popup window.
	 * @param caption A popup window caption
	 */
	KPassivePopup *popup(const QString &caption = QString::null);

	/**
	 * Displays a popup message.
	 * @param reason A reason; icon type
	 * @param caption A popup title
	 * @param text A text to display
	 * @param timeout A timeout in seconds
	 * @param test @c TRUE if called from the settings dialog ("Test" button)
	 */
	void popupMessage(const Reason reason, const QString &caption,
		const QString &text, const int timeout = -1, const bool test = FALSE);

	/**
	 * This is an overloaded member function.
	 * Uses main window caption as a popup title.
	 * @param reason A reason; icon type
	 * @param text A text to display
	 */
	void popupMessage(const Reason reason, const QString &text);

	/**
	 * Switches icon between idle and active states.
	 * @param yes If @c TRUE the state is "idle" (normal icon);
	 * otherwise the state is "active" (highlighted icon)
	 */
	void setIdle(const bool yes);

	/**
	 * Displays the popup message for testing purpose.
	 * @param timeout A timeout in seconds
	 */
	void testMessage(const int timeout);
protected:
	/** Disables/enables items before context menu becomes visible. */
	virtual void contextMenuAboutToShow(KPopupMenu *menu);

	/**
	 * Overriden to handle middle button click; displays the actions menu.
	 */
	virtual void mousePressEvent(QMouseEvent *e);
private:
	bool
		_idle,
		_test;
	int
		_cancelItem,
		_lockScreenItem,
		_logoutItem,
		_rebootItem,
		_shutDownItem,
		_wizardItem,
		_flashCount;
	KPassivePopup *_popup;
	KPixmap
		*_activeIcon,
		*_flashIcon;
	KPopupMenu *_actionsMenu;
	KPushButton *b_cancel;
	QLabel *l_popupCaption;
	QListBox *lb_popupMessagesList;
	QPixmap *_saveIcon;
	QTimer *_flashTimer;

	/** @ref MMainWindow::execAction with confirmation. */
	void doExecAction(const MMainWindow::What action);
private slots:
	void actionsMenuAboutToShowSlot();
	void cancelSlot();
	void flashTimeoutSlot();
	void lockScreenSlot();
	void logoutSlot();
	void rebootSlot();
	void shutDownSlot();
	void wizardSlot();
};

#endif // __MSYSTEMTRAY_H__
