/*
	mwizard.cpp - wizard
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "miscutils.h"
#include "mradiobutton.h"
#include "mwizard.h"

#include <qdatetimeedit.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include <qvbuttongroup.h>
#include <qwhatsthis.h>

#include <kdialog.h>
#include <kiconloader.h>
#include <klocale.h>

/* constructor */

MWizard::MWizard(QWidget *parent, const char *name)
	: KWizard(parent, name, TRUE)
{
	setCaption(i18n("Wizard"));
	initActionPage();
	initTimePage();
	connect(
		helpButton(), SIGNAL(clicked()),
		this, SLOT(launchHelpSlot()));
	connect(
		this, SIGNAL(selected(const QString&)),
		this, SLOT(pageSelectedSlot(const QString&)));
	// pack();
	// setFixedSize(size());
}

/* destructor */

MWizard::~MWizard()
{
}

/* public */

bool MWizard::canRun()
{
	return !mainWindow->isRestricted("kshutdown_action_wizard");
}

bool MWizard::runWizard()
{
// FIXME: 0.7: center wizard dialog
	KDialog::centerOnScreen(this);

	if (exec() == QDialog::Accepted)
	{
		mainWindow->cancel(); // cancel action before "setDelayType"

		MMainWindow::What action = getSelectedAction();

		if (action == MMainWindow::What_Nothing)
			return FALSE;

		if (rb_now->isChecked())
		{
			mainWindow->setDelayType(MMainWindow::DelayType_Now);
		}
		else if (rb_atDateTime->isChecked())
		{
			mainWindow->setDelayType(MMainWindow::DelayType_DateTime);
			mainWindow->setDate(dateEdit->date());
			mainWindow->setTime(timeEdit->time());
		}
		else if (rb_timeFromNow->isChecked())
		{
			mainWindow->setDelayType(MMainWindow::DelayType_TimeFromNow);
			mainWindow->setTime(timeEdit->time());
		}

		mainWindow->setActive(action, TRUE);

		return TRUE;
	}

	return FALSE;
}

/* private */

MMainWindow::What MWizard::getSelectedAction()
{
	if (rb_logout->isChecked())
		return MMainWindow::What_Logout;

	if (rb_shutDown->isChecked())
		return MMainWindow::What_ShutDown;

	if (rb_reboot->isChecked())
		return MMainWindow::What_Reboot;

	return MMainWindow::What_Nothing;
}

void MWizard::initActionPage()
{
	QWidget *w = new QWidget(this);
	QHBoxLayout *l = new QHBoxLayout(w, 2);

	// zzz konq
	QLabel *konq = new QLabel(w, "QLabel::konq");
	konq->setAlignment(AlignCenter);
	konq->setPixmap(UserIcon("zzzkonq.png"));
	QWhatsThis::add(konq, "Konqi, the Desktop Dragon");
	l->addWidget(konq);

	// actions
	QVButtonGroup *bg_actions = new QVButtonGroup(w);
	rb_logout = new MRadioButton(i18n("&End current session"), bg_actions);
	rb_logout->setEnabled(MiscUtils::isActionEnabled(MMainWindow::What_Logout));
	rb_logout->setChecked(rb_logout->isEnabled());
	connect(
		rb_logout, SIGNAL(doubleClicked()),
		this, SLOT(next()));
	rb_shutDown = new MRadioButton(i18n("&Turn off computer"), bg_actions);
	rb_shutDown->setEnabled(
		MiscUtils::isActionEnabled(MMainWindow::What_ShutDown)
	);
	connect(
		rb_shutDown, SIGNAL(doubleClicked()),
		this, SLOT(next()));
	rb_reboot = new MRadioButton(i18n("&Restart computer"), bg_actions);
	rb_reboot->setEnabled(MiscUtils::isActionEnabled(MMainWindow::What_Reboot));
	connect(
		rb_reboot, SIGNAL(doubleClicked()),
		this, SLOT(next()));
	l->addWidget(bg_actions);

	addPage(w, i18n("What do you want to do?"));
}

void MWizard::initTimePage()
{
	timePage = new QWidget(this);
	QVBoxLayout *l = new QVBoxLayout(timePage, 2);

	// radio buttons + date/time editors layout
	QHBox *mainBox = new QHBox(timePage, "QHBox::mainBox");
	mainBox->setSpacing(2);

	// radio buttons
	QVButtonGroup *bg_delayType = new QVButtonGroup(mainBox);
	connect(
		bg_delayType, SIGNAL(clicked(int)),
		this, SLOT(delayTypeClickedSlot(int)));
	rb_now = new QRadioButton(i18n("&Now (no delay)"), bg_delayType);
	rb_now->setChecked(TRUE);
	rb_atDateTime = new QRadioButton(i18n("At &date/time"), bg_delayType);
	rb_atDateTime->setEnabled(!mainWindow->isRestricted("kshutdown_tab_time"));
	rb_timeFromNow = new QRadioButton(i18n("&Time from now"), bg_delayType);
	rb_timeFromNow->setEnabled(!mainWindow->isRestricted("kshutdown_tab_time"));

	// date/time editors
	QVBox *editBox = new QVBox(mainBox);

	timeEdit = new QTimeEdit(editBox, "QTimeEdit::timeEdit");
	timeEdit->setAutoAdvance(TRUE);
	MiscUtils::setHint(timeEdit, i18n("HH:MM"));
	connect(
		timeEdit, SIGNAL(valueChanged(const QTime&)),
		this, SLOT(timeEditValueChangedSlot(const QTime&)));

	dateEdit =
		new QDateEdit(QDate::currentDate(), editBox, "QDateEdit::dateEdit");
	dateEdit->setAutoAdvance(TRUE);
	dateEdit->setMinValue(QDate::currentDate());
	connect(
		dateEdit, SIGNAL(valueChanged(const QDate&)),
		this, SLOT(dateEditValueChangedSlot(const QDate&)));

	l->addWidget(mainBox);

	l_time = new QLabel(timePage, "QLabel::l_time");
	l->addWidget(l_time);

	l_action = new QLabel(timePage, "QLabel::l_action");
	l->addWidget(l_action);

	addPage(timePage, i18n("When to run the selected action?"));
	setFinishEnabled(timePage, TRUE);

	updateTimeInfo();
	updateTimeWidgets();
}

void MWizard::updateTimeInfo()
{
	QString s_timeHint;
	if (rb_now->isChecked())
	{
		s_timeHint = i18n("No delay");
	}
	else if (rb_atDateTime->isChecked())
	{
		QDateTime dt(dateEdit->date(), timeEdit->time());
		s_timeHint = MiscUtils::formatDateTime(dt);
	}
	else if (rb_timeFromNow->isChecked())
	{
		QDateTime dt = QDateTime::currentDateTime();
		s_timeHint = MiscUtils::formatDateTime(
			dt.addSecs(
				(timeEdit->time().hour() * 3600) +
				timeEdit->time().minute() * 60
			)
		);
	}

	l_time->setText(
		QString(i18n("Selected time: <b>%1</b>"))
			.arg(s_timeHint)
	);
}

void MWizard::updateTimeWidgets()
{
	if (rb_now->isChecked())
	{
		dateEdit->setEnabled(FALSE);
		timeEdit->setDisplay(
			KGlobal::locale()->use12Clock()
			? QTimeEdit::Hours | QTimeEdit::Minutes | QTimeEdit::AMPM
			: QTimeEdit::Hours | QTimeEdit::Minutes
		);
		timeEdit->setEnabled(FALSE);
	}
	else if (rb_atDateTime->isChecked())
	{
		dateEdit->setDate(QDate::currentDate());
		dateEdit->setEnabled(TRUE);
		timeEdit->setDisplay(
			KGlobal::locale()->use12Clock()
			? QTimeEdit::Hours | QTimeEdit::Minutes | QTimeEdit::AMPM
			: QTimeEdit::Hours | QTimeEdit::Minutes
		);
		timeEdit->setTime(QTime::currentTime());
		timeEdit->setEnabled(TRUE);
	}
	else if (rb_timeFromNow->isChecked())
	{
		dateEdit->setDate(QDate::currentDate());
		dateEdit->setEnabled(FALSE);
		timeEdit->setDisplay(QTimeEdit::Hours | QTimeEdit::Minutes);
		timeEdit->setTime(QTime(0, 0, 0));
		timeEdit->setEnabled(TRUE);
	}
}

/* private slots */

void MWizard::dateEditValueChangedSlot(const QDate &/*date*/)
{
	updateTimeInfo();
}

void MWizard::delayTypeClickedSlot(int /*id*/)
{
	updateTimeInfo();
	updateTimeWidgets();
}

void MWizard::launchHelpSlot()
{
	MiscUtils::launchHelp("anchor.using-wizard");
}

void MWizard::pageSelectedSlot(const QString &selectedTitle)
{
	if (selectedTitle == title(timePage))
	{
		// update info
		l_action->setText(
			QString(i18n("Selected action: <b>%1</b>"))
				.arg(mainWindow->getActionName(getSelectedAction()))
		);
		// update "finish" icon
		finishButton()->setIconSet(
			mainWindow->getActionIcon(getSelectedAction())
		);

		rb_now->setFocus();
		finishButton()->setDefault(TRUE);
	}
	else
	{
		rb_logout->setFocus();
		nextButton()->setDefault(TRUE);
	}
}

void MWizard::timeEditValueChangedSlot(const QTime &/*time*/)
{
	updateTimeInfo();
}
