/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <stdlib.h>
#include <netinet/in.h>

#include <qstring.h>
#include <qstringlist.h>

#include <klocale.h>
#include <kdebug.h>

#include "arp-protocol.h"
#include "../packet.h"
#include "../packetstructures.h"

ArpProtocol::ArpProtocol(Packet *packet)
{
  m_packet = packet;
  m_length = packet->frameHeaderLength();
}


ArpProtocol::ArpProtocol(ptrPacketType packet)
{
  m_packet = (Packet *)packet;
  m_length = m_packet->frameHeaderLength();
}


// format of hardware address
unsigned short ArpProtocol::hwType() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );

  return ( (unsigned short) ntohs(arp->ar_hrd) );
}


// format of protocol address
unsigned short ArpProtocol::protocolType() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );

  return ( (unsigned short) ntohs(arp->ar_pro) );
}


// length of hardware address
unsigned char ArpProtocol::hwSize() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );

  return ( (unsigned char) arp->ar_hln );
}


// length of protocol address
unsigned char ArpProtocol::protocolSize() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );

  return ( (unsigned char) arp->ar_pln );
}


// ARP opcode (command)
unsigned char ArpProtocol::opCode() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );

  return ( (unsigned short)  ntohs(arp->ar_op) );
}


// sender hardware address
QString ArpProtocol::senderMacAddress() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );
  unsigned char *x = (unsigned char *) (&arp->ar_sha);

  return QString("%1:%2:%3:%4:%5:%6").arg(x[0], 0, 16).arg(x[1], 0, 16).
      arg(x[2], 0, 16).arg(x[3], 0, 16).arg(x[4], 0, 16).arg(x[5], 0, 16);
}


// sender IP address
QString ArpProtocol::senderIpAddress() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );
  unsigned char *x = (unsigned char *) (&arp->ar_sip);

  return QString("%1.%2.%3.%4").arg(x[0]).arg(x[1]).arg(x[2]).arg(x[3]);
}


// target hardware address
QString ArpProtocol::targetMacAddress() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );
  unsigned char *x = (unsigned char *) (&arp->ar_tha);

  return QString("%1:%2:%3:%4:%5:%6").arg(x[0], 0, 16).arg(x[1], 0, 16).
      arg(x[2], 0, 16).arg(x[3], 0, 16).arg(x[4], 0, 16).arg(x[5], 0, 16);
}


// target IP address
QString ArpProtocol::targetIpAddress() const
{
  struct ArpHdr *arp = (struct ArpHdr*) ( (ptrPacketType)m_packet + m_length );
  unsigned char *x = (unsigned char *) (&arp->ar_tip);

  return QString("%1.%2.%3.%4").arg(x[0]).arg(x[1]).arg(x[2]).arg(x[3]);
}


QString ArpProtocol::strProtocol() const
{
  return "ARP";
}


QStringList ArpProtocol::headerLines() const
{
  QStringList list;

  list << i18n("ARP protocol field", "Hardware address format: 0x%1").arg( hwType(), 0, 16 );
  list << i18n("ARP protocol field", "Protocol type: 0x%1").arg( protocolType(), 0, 16);
  list << i18n("ARP protocol field", "Hardware address length: %1").arg( hwSize() );
  list << i18n("ARP protocol field", "Protocol address length: %1").arg( protocolSize() );
  list << i18n("ARP protocol field: ARP request type", "Opcode: 0x%1").arg( opCode(), 0, 16 );
  list << i18n("ARP protocol field", "Sender MAC address: %1").arg( senderMacAddress() );
  list << i18n("ARP protocol field", "Sender IP address: %1").arg( senderIpAddress() );
  list << i18n("ARP protocol field", "Target MAC address: %1").arg( targetMacAddress() );
  list << i18n("ARP protocol field", "Target IP address: %1").arg( targetIpAddress() );

  return list;
}
