/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>

#include <sys/socket.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <time.h>

#ifndef _WIN32
#include <netinet/in.h>
#endif

#include <arpa/inet.h>
#include <unistd.h>

#include <iostream>

#include <pcap.h>

using namespace std;

#include "../packetstructures.h"
#include "../version.h"

#include "consts.h"
#include "parseconfig.h"
#include "ksniff.h"

void logging(char *str)
{
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "%s", str);
}

void loggingInt(char *str, int i)
{
  char msg[100];

  sprintf(msg, "%s %d", str, i);
  logging(msg);
}


void loggingString(char *str, char *str2)
{
  char msg[100];

  sprintf(msg, "%s %s", str, str2);
  logging(msg);
}


void initSniff(ParseConfig *config)
{
  signal( SIGTERM, exitHandler );
  signal( SIGINT, exitHandler );

  // SIGUSR1 pause and resume the sniffing
  signal( SIGUSR1, pauseContinueSniffHandler );

  m_canSniff = m_bSniffing = true;
  m_bPauseSniffing = false;
  cerr << endl << endl << APPNAME << "You choose to use interface: '" << config->interface() << "'" << endl;
  cerr << APPNAME << "sniffing in: '" << config->sniffFile() << "'" << endl;
  cerr << APPNAME << "type of capture: " << config->sniffType() << endl;
  if (config->sniffType() != STOP_CAPTURE_MANUALLY)
    cerr << APPNAME << "PackSizeTime parameter: " << config->nPackSizeTime() << endl;
  switch (config->sniffType())
  {
    case STOP_CAPTURE_MANUALLY:
      cerr << APPNAME << "the capture will stopped manually for the user" << endl;
      break;
    case STOP_AFTER_X_PACKETS:
      cerr << APPNAME << "the packets capture will stop after: " << config->nPackSizeTime() * config->nUnits() << " packets" << endl;
      break;
    case STOP_AFTER_X_BYTES:
      cerr << APPNAME << "the packets capture will stop after: " << config->nPackSizeTime() * config->nUnits()  << " bytes" << endl;
      break;
    case STOP_AFTER_X_SECONDS:
      cerr << APPNAME << "the packets capture will stop after: " << config->nPackSizeTime() * config->nUnits()  << " seconds" << endl;
      break;
  }
  fflush( stderr );
}


void startSniffing(ParseConfig *config)
{
  // set all parameters to prepare sniffing structures
  //struct bpf_program bpfp;
  char errbuf[PCAP_ERRBUF_SIZE];
  int frameLength, frameType;

  // needed to avoid calculate delay in sniffing
  long limit = config->nPackSizeTime() * config->nUnits();
  int sniffType = config->sniffType();

  long packetsSize = 24;  // total size (in byte) of all captured packets

  #ifdef MUST_DO_SELECT
    int selRet = 0;
    m_pcap_fd = 0;
  #endif

  // delete the file, to avoid problem if cause of some strange reason (crash?) it was not deleted at the KSniffer quit
  cerr << APPNAME << "deleting " << config->sniffFile() << endl;
  if (unlink(config->sniffFile()) == -1)
    // if we don't reset errno and the file was not delete because all was ok on last quit
    // we cannot start sniffing: we'll get an annoying error on the wrong statement
    errno = 0;

  // open libpcap using ifname interface
  if (config->interface() == "any")
    // promiscue mode not supported by "any" interface
    m_pcapfp = pcap_open_live(config->interface(), PKTMAX, 0, -1, errbuf);
  else
    m_pcapfp = pcap_open_live(config->interface(), PKTMAX, 1, -1, errbuf);

  if (m_pcapfp == NULL)
  {
    cerr << APPNAME << "Error opening libpcap: " << errbuf << endl;
    fflush(stderr);
    return;
  }
  else
  {
    m_pdump = pcap_dump_open( m_pcapfp, config->sniffFile() );
    chmod(config->sniffFile(), S_IRUSR|S_IWUSR|S_IRGRP);
    cerr << APPNAME << "writing packets into " << config->sniffFile() << endl;
    loggingString( "capturing from interface:", config->interface() );
    loggingString( "writing packets into file:", config->sniffFile() );
    fflush(stderr);
  }
#ifdef MUST_DO_SELECT

#ifdef HAVE_PCAP_GET_SELECTABLE_FD
    m_pcap_fd = pcap_get_selectable_fd(m_pcapfp);
#else
    m_pcap_fd = pcap_fileno(m_pcapfp);
#endif

#endif

  // reset the number of counted packets
  m_countPackets = 0;

  // start reading packets on the selected source interface
  time_t rawtime;
  time ( &rawtime );
  long startTime = rawtime;
  while (m_canSniff)
  {
    struct pcap_pkthdr hdr;
    unsigned char *p;

    #ifdef MUST_DO_SELECT
    /*
     * snipped from ethereal code.
     */

    FD_ZERO(&m_fdset);
    FD_SET(m_pcap_fd, &m_fdset);
    m_fdtimeout.tv_sec = 0;
    m_fdtimeout.tv_usec = CAP_READ_TIMEOUT*1000;
    selRet = select(m_pcap_fd+1, &m_fdset, NULL, NULL, &m_fdtimeout);
    if (selRet > 0)
    {
      /*
       * "select()" says we can read from it without blocking; go for
       * it.
       */
      // get next tail packet calling pcap_next()
      p = (unsigned char*) pcap_next(m_pcapfp, &hdr);
    }
    else
    {
       p = NULL;
       if (selRet < 0 && errno != EINTR)
       {
         cerr << "Unexpected error from select: " << strerror(errno) << endl;
         fflush(stderr);
         break;
       }
    }
#else
    // get next tail packet calling pcap_next()
    p = (unsigned char*) pcap_next(m_pcapfp, &hdr);
#endif /* MUST_DO_SELECT */

    // treat the error just if it's sniffing and not quitting 'sniff'
    if (errno != 0 && m_canSniff)
      if ((p == NULL) && (errno != EINTR))
      {
        // errno == 100 : 'Network is down' --> occurs when the interface go down
        cerr << endl << APPNAME << strerror(errno) << ": a grave error occurred... Quitting..." << endl;
        m_returnValue = SNIFF_NETWORK_ERROR;
        fflush(stderr);
        break;
      }

    if (p == NULL)
    {
      if (config->sniffType() == STOP_AFTER_X_SECONDS)
      {
        time ( &rawtime );  // evaluing time again to know how much seconds has spent
        if (rawtime - startTime > limit)
          m_canSniff = false;
        else
          continue;  // we didn't get any packet
      }
      else continue;
    }
    else
      if (!m_bPauseSniffing)
      {
        // if the capture is not paused the packet can be saved into the libpcap file
        // if the capture condition is respected: not excedeed capure time or packets size
        time ( &rawtime );
        switch (sniffType)
        {
          case STOP_AFTER_X_PACKETS:
            if (m_countPackets >= limit)
              m_canSniff = false;
            break;
          case STOP_AFTER_X_BYTES:
            if (limit <= packetsSize)
              m_canSniff = false;
            break;
          case STOP_AFTER_X_SECONDS:
            if (rawtime - startTime > limit)
            {
              cerr << APPNAME << "Ok capturing time's ended..." << endl;
              m_canSniff = false;
            }
            break;
        }
        if (m_canSniff)
        {
          pcap_dump((u_char *)m_pdump, &hdr, p);
          pcap_dump_flush( m_pdump );
          m_countPackets++;

         // got a packet: raw bytes (p), the timestamp and the captured packet length and the real packet length (hdr)
         /**
          *  hdr is a struct pcap_pkthdr:
          *    ts - a struct timeval containing the time when the packet was captured
          *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
          *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
          *      of bytes available from the capture, if the length of the packet is larger than the maximum
          *      number of bytes to capture)
          */

         setHeaderData(m_pcapfp, frameType, frameLength);
         if (frameLength < 0)
         {
           cerr << APPNAME << "unknown packet" << endl;
           continue;
         }
         else
           packetsSize += hdr.caplen + 16;
        }
      }
  }
  cerr << endl << APPNAME << "total captured packets: " << m_countPackets << endl;
  loggingInt( "total captured packets:", m_countPackets );
  if (m_countPackets == 0)
  {
    cerr << APPNAME << "total captured packets size: 0" << endl;
    loggingInt( "total captured packets size:", 0 );
  }
  else
  {
    cerr << APPNAME << "total captured packets size: " << packetsSize << endl;
    loggingInt( "total captured packets size:",  packetsSize);
  }
}


void setHeaderData(pcap_t *pHandler, int &frameType, int &hdrLen)
{
  frameType = pcap_datalink(pHandler);

  switch(frameType)
  {
    case DLT_EN10MB:
      hdrLen = 14;
      break;
    case DLT_LINUX_SLL:
      hdrLen = 16;
      break;
    case DLT_LOOP:
      hdrLen = 4;
      break;
    case DLT_NULL:
      hdrLen = 4;
      break;
    case DLT_RAW:
      hdrLen = 0;
      break;
    default:
      hdrLen = -1;
      break;
  }
}


void stopSniffing()
{
  m_canSniff = false;
}


int isSniffing()
{
  return m_bSniffing;
}


void exitHandler( int )
{
  cerr << APPNAME << "Thread stopped. Got " << m_countPackets << " packets" << endl;
  fflush(stderr);

  // release libpcap resource
  pcap_close(m_pcapfp);

  // now can close dumped file
  pcap_dump_close(m_pdump);

  // now the thread has stopped and the boolean variable could be "resetted"
  m_bSniffing = false;

  // thread's stopped
  m_canSniff = false;
}



void pauseContinueSniffHandler( int )
{
  if (!m_bPauseSniffing)
  {
    m_bPauseSniffing = true;
    cerr << APPNAME << "Thread paused." << endl;
    logging( "paused sniffing..." );
  }
  else
  {
    m_bPauseSniffing = false;
    cerr << APPNAME << "Thread resumed." << endl;
    logging( "resumed sniffing..." );
  }
}


int main( int argc, char *argv[] )
{
  if ( argc == 2 )
  {
    logging( "starting...");
    if (strcmp(argv[1], "devs") == 0)
    {
      pcap_if_t *alldevs;
      char errbuf[PCAP_ERRBUF_SIZE] = "";
      struct sockaddr_in *ai;

      if (pcap_findalldevs(&alldevs, errbuf) == 0)
        cerr << APPNAME << "found capturable devices" << endl;
      else
      {
        cerr << APPNAME << "errbuf: '" << errbuf << "'" << endl;
        m_returnValue = SNIFF_NO_SNIFFING_DEVICE_FOUND;
        return m_returnValue;
      }
      m_returnValue = SNIFF_NO_ERROR;
      while (alldevs)
      {
        cerr << APPNAME << "****************************************" << endl;
        cerr << APPNAME << "interface: " << alldevs->name << endl;
        if (alldevs->description != NULL)
          cerr << APPNAME << "description: " << alldevs->description << endl;
        else
          cerr << APPNAME << "no description available" << endl;
        if (alldevs->addresses != NULL)
        {
          cerr << APPNAME << "addresses not NULL" << endl;
          if (alldevs->addresses->addr != NULL)
            switch (alldevs->addresses->addr->sa_family)
            {
              case AF_INET:
              //case 17:
                cerr << APPNAME << "I'm AF_INET" << endl;
                ai = (struct sockaddr_in *)alldevs->addresses->addr;
                cerr << inet_ntoa(ai->sin_addr) << ":" << ai->sin_addr.s_addr << endl;
                break;
            }
          else
            cerr << APPNAME << "addr is NULL" << endl;
          if (alldevs->addresses->netmask != NULL)
            switch (alldevs->addresses->addr->sa_family)
            {
              case AF_INET:
                //case 17:
                cerr << APPNAME << "I'm AF_INET" << endl;
                ai = (struct sockaddr_in *)alldevs->addresses->netmask;
                cerr << inet_ntoa(ai->sin_addr) << ":" << ai->sin_addr.s_addr << endl;
                break;
            }
          else
            cerr << APPNAME << "netmask is NULL" << endl;
          if (alldevs->addresses->broadaddr != NULL)
            switch (alldevs->addresses->addr->sa_family)
            {
              case AF_INET:
                  //case 17:
                cerr << APPNAME << "I'm AF_INET" << endl;
                ai = (struct sockaddr_in *)alldevs->addresses->broadaddr;
                cerr << inet_ntoa(ai->sin_addr) << ":" << ai->sin_addr.s_addr << endl;
                break;
            }
          else
            cerr << APPNAME << "broadaddr is NULL" << endl;
        }
        alldevs = alldevs->next;
        cerr << endl;
      }
      pcap_freealldevs(alldevs);
    }
    else
    {
      ParseConfig *config = new ParseConfig(argv[1]);
      if (config->error())
        cerr << endl << APPNAME << "The XML file has a syntax error or file '" << argv[1] << "' doesn't exist!" << endl << endl;
      else
      {
        initSniff(config);
        startSniffing(config);
        logging( "quitting..." );
      }
    }
  }
  else
    cerr << APPNAME << "Usage:" << endl << "'sniff xml-config-file' OR 'sniff devs'" << endl << endl;

  sleep(2);
  cerr << APPNAME << "Quitting with error code: " << m_returnValue << endl;
  return m_returnValue;
}
