/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   Copyright (C) 2001 by Alexander Neundorf                              *
 *   neundorf@kde.org                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifdef HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif

#include <sys/types.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>

#ifdef USE_SOLARIS
/* net/if.h is incompatible with STL on Solaris 2.6 - 2.8, redefine
   map in the header file because we don't need it. -- Simon Josefsson */
#define map junkmap
#endif
#  include <net/if.h>
#ifdef USE_SOLARIS
#undef map
#endif

#include <sys/ioctl.h>

#ifndef HAVE_STRUCT_SOCKADDR_SA_LEN
        #undef HAVE_GETNAMEINFO
        #undef HAVE_GETIFADDRS
#endif

#include <kdebug.h>
#include <klocale.h>

#include "networkdevice.h"

NetworkDevice::NetworkDevice()
{
    m_netDeviceList = new DeviceList;
    detect();
}


QString NetworkDevice::HWaddr2String( char *hwaddr )
{
  QString ret;
  int i;

  for (i=0; i<6; i++, hwaddr++) {
    int v = (*hwaddr & 0xff);
    QString num = QString("%1").arg(v,0,16).upper();
    if (num.length() < 2)
      num.prepend("0");
    if (i>0)
      ret.append(":");
    ret.append(num);
  }
  return ret;
}

void NetworkDevice::detect()
{
    QString upMessage(   i18n("State of network card is connected",    "Up") );
    QString downMessage( i18n("State of network card is disconnected", "Down") );
    QStringList macaddr;
    bool duplicatedMacAddress;    // true if the MAC address is duplicated

    m_netDeviceList->setAutoDelete(true);
    m_netDeviceList->clear();
    m_hasDevices = false;

#if !defined(HAVE_GETIFADDRS) || !defined(HAVE_GETNAMEINFO)

   int sockfd = socket(AF_INET, SOCK_DGRAM, 0);

   char buf[8*1024];
   struct ifconf ifc;
   ifc.ifc_len = sizeof(buf);
   ifc.ifc_req = (struct ifreq *) buf;
   int result = ioctl(sockfd, SIOCGIFCONF, &ifc);

   for (char* ptr = buf; ptr < buf + ifc.ifc_len; )
   {
      struct ifreq *ifr = (struct ifreq *) ptr;

#ifdef HAVE_STRUCT_SOCKADDR_SA_LEN
      int len = sizeof(struct sockaddr);
      if (ifr->ifr_addr.sa_len > len)
         len = ifr->ifr_addr.sa_len;            /* length > 16 */
      ptr += sizeof(ifr->ifr_name) + len;       /* for next one in buffer */
#else
      ptr += sizeof(*ifr);                      /* for next one in buffer */
#endif

      int flags;
      struct sockaddr_in *sinptr;
      DeviceData *tmp = 0;
      switch (ifr->ifr_addr.sa_family)
      {
        case AF_INET:
        {
          sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
          flags = 0;

          struct ifreq ifcopy;
          ifcopy = *ifr;
          result = ioctl(sockfd, SIOCGIFFLAGS, &ifcopy);
          flags = ifcopy.ifr_flags;

          tmp = new DeviceData;  // a network device with all its details

          if ((flags & IFF_UP) == IFF_UP)
          {
            tmp->m_strName = ifr->ifr_name;
            tmp->m_strState = upMessage;
          }
          else
            tmp->m_strState = downMessage;

          if ((flags & IFF_BROADCAST) == IFF_BROADCAST)
            tmp->m_strType = i18n("Broadcast");
          else if ((flags & IFF_POINTOPOINT) == IFF_POINTOPOINT)
            tmp->m_strType = i18n("Point to Point");
#ifndef _AIX
          else if ((flags & IFF_MULTICAST) == IFF_MULTICAST)
            tmp->m_strType = i18n("Multicast");
#endif
          else if ((flags & IFF_LOOPBACK) == IFF_LOOPBACK)
             tmp->m_strType = i18n("Loopback");
          else
             tmp->m_strType = i18n("Unknown");

          tmp->m_strAddress = inet_ntoa(sinptr->sin_addr);

          ifcopy = *ifr;
          result = ioctl(sockfd, SIOCGIFNETMASK, &ifcopy);
          if (result == 0)
          {
            sinptr = (struct sockaddr_in *) &ifcopy.ifr_addr;
            tmp->m_strNetmask = inet_ntoa(sinptr->sin_addr);
          }
          else
             tmp->m_strNetmask = i18n("Unknown");

          duplicatedMacAddress = false;  // initialize the condition
          ifcopy=*ifr;
          result=-1; // if none of the two #ifs below matches, ensure that result!=0 so that "Unknown" is returned as result
#ifdef SIOCGIFHWADDR
          result=ioctl(sockfd,SIOCGIFHWADDR,&ifcopy);
          if (result==0)
          {
            char *n = &ifcopy.ifr_ifru.ifru_hwaddr.sa_data[0];
            tmp->m_strHWaddr = HWaddr2String(n);
            QStringList::Iterator it = macaddr.begin();
            while ((it != macaddr.end()) && (!duplicatedMacAddress))
            {
              duplicatedMacAddress = (tmp->m_strHWaddr == *it);
              ++it;
            }
          }
#elif defined SIOCGENADDR
          result=ioctl(sockfd,SIOCGENADDR,&ifcopy);
          if (result==0)
          {
            char *n = &ifcopy.ifr_ifru.ifru_enaddr[0];
            tmp->m_strHWaddr = HWaddr2String(n);
            QStringList::Iterator it = macaddr.begin();
            while ((it != macaddr.end()) && (!duplicatedMacAddress))
            {
              duplicatedMacAddress = (tmp->m_strHWaddr == *it);
              ++it;
            }
          }
#elif defined SIOCGIFMAP
          result=ioctl(sockfd,SIOCGIFMAP,&ifcopy);
          if (result==0)
          {
            char *n=&ifcopy.ifr_ifru.ifru_data[0];
            tmp->m_strHWaddr = HWaddr2String(n);
            QStringList::Iterator it = macaddr.begin();
            while ((it != macaddr.end()) && (!duplicatedMacAddress))
            {
              duplicatedMacAddress = (tmp->m_strHWaddr == *it);
              ++it;
            }
          }
#endif
          if (result!=0)
          {
            tmp->m_strHWaddr = i18n("Unknown");
          }

          m_hasDevices = true;
          if (!duplicatedMacAddress)
          {
            m_netDeviceList->append(tmp);
            macaddr << tmp->m_strHWaddr;
            kdDebug() << "**********************************" << endl;
            kdDebug() << "- name: " << tmp->m_strName << endl;
            kdDebug() << "- address: " << tmp->m_strAddress << endl;
            kdDebug() << "- HW address: " << tmp->m_strHWaddr << endl;
            kdDebug() << "- netmask: " << tmp->m_strNetmask << endl;
            kdDebug() << "- state: " << tmp->m_strState << endl;
            kdDebug() << "- type: " << tmp->m_strType << endl;
            kdDebug() << endl;
          }
          break;
        }

        default:
          break;
      }
   }
#else
  struct ifaddrs *ifap, *ifa;
  if (getifaddrs(&ifap) != 0) {
    return;
  }

  deviceData *tmp = 0;
  for (ifa = ifap; ifa; ifa = ifa->ifa_next) {
    switch (ifa->ifa_addr->sa_family) {
    case AF_INET6:
    case AF_INET: {
      tmp = new deviceData;  // a network device with all its details
      tmp->m_strName = ifa->ifa_name;

      char buf[128];

      bzero(buf, 128);
      getnameinfo(ifa->ifa_addr, ifa->ifa_addr->sa_len, buf, 127, 0, 0, NI_NUMERICHOST);
      tmp->m_addrress = buf;

      if (ifa->ifa_netmask != NULL) {
        bzero(buf, 128);
        getnameinfo(ifa->ifa_netmask, ifa->ifa_netmask->sa_len, buf, 127, 0, 0, NI_NUMERICHOST);
        tmp->m_strNetmask = buf;
      }

      if (ifa->ifa_flags & IFF_UP)
        tmp->m_strState = upMessage;
      else
        tmp->m_strState = downMessage;

      tmp->m_strType = flags_tos(ifa->ifa_flags);

      hasDevices = true;
      netDeviceList->append(tmp);
      break;
    }
    default:
      break;
    }
  }

  freeifaddrs(ifap);
#endif

}


NetworkDevice::~NetworkDevice()
{
}


// the following function returns a StringList for the ComboBox
QStringList NetworkDevice::interfaceList() const
{
    QStringList list;

    if (m_hasDevices)
    {
       for (DeviceData* tmp = m_netDeviceList->first(); tmp != 0; tmp = m_netDeviceList->next())
           list << tmp->m_strName;
       list << "any";
       list.sort();
    }

    return list;
}

DeviceData NetworkDevice::interfaceData(const QString & strInterface) const
{
    for (DeviceData* tmp = m_netDeviceList->first(); tmp != 0; tmp = m_netDeviceList->next())
           if (tmp->m_strName == strInterface)
           {
               // I can return *tmp with no problem because in this class constructor
               // I made a new() and the assigned *tmp in the for cycle corresponds to
               // a real element in the memory
               return *tmp;
           }

    // if strInterface is not valid return a "null" deviceData
    DeviceData ret;
    ret.m_strAddress = QString::null;
    ret.m_strName = QString::null;
    ret.m_strNetmask = QString::null;
    ret.m_strState = QString::null;
    ret.m_strType = QString::null;

    return ret;
}
