/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <pcap.h>
#include <errno.h>
#include <unistd.h>

#include <kurl.h>
#include <kdebug.h>
#include <klocale.h>

#include "sniffer.h"
#include "packetmanager.h"
#include "packetstructures.h"

// set all sniffing options
Sniffer::Sniffer( CaptureOptions *options ) : QThread()
{
  setCaptureOptions( options );
  m_bSniffing = false;       // not sniffing yet
  m_frameNumber = 1;         // set the frame number
  m_bFromFile = false;       // the default sniffing is live, not from file
  m_canSniff = true;         // ok, now you can sniff
  m_bPauseSniffing = false;  // no pause at start-up
  m_packetsTotalSize = 0;    // 0 bytes sniffed
}


Sniffer::~Sniffer()
{
}


// set all sniffing options
void Sniffer::setCaptureOptions(CaptureOptions *captOpt)
{
  m_strInterface = captOpt->sniffingInterface();
  m_strFilename = captOpt->sniffingFilename();
  m_strTmpFilePath = captOpt->tmpFilePath();
  /*m_packetsNumber = captOpt->packetsNumber();
  m_packetsSize = captOpt->packetsSize();
  m_packetsTime = captOpt->packetsTime();
  m_sniffType = captOpt->sniffType();*/
}


// start sniffing on the selected interface
void Sniffer::startSniffing()
{
  m_frameNumber = 1;  // reset the number
  m_canSniff = true;  // ok, now you can sniff
  if (!m_bSniffing)
  {
    // if the sniffing thread is not active yet, active it now
    m_bSniffing = true;
    if ( m_strInterface.isEmpty() )
      m_bFromFile = true;
    else
      m_bFromFile = false;
    start();
  }
  else
    kdDebug() << "still sniffing" << endl;
}


// stop sniffing on the selected interface
void Sniffer::stopSniffing()
{
  // stop sniffing: the capture cycle into run() exits at next packet arrived and
  // m_bSniffing will be false and the Sniffing Thread stops
  m_canSniff = false;
  m_bPauseSniffing = false;
}


// check if a file is in a valid libpcap format
int Sniffer::checkFile( QString strFile )
{
  int ret;   // return condition
  char errbuf[PCAP_ERRBUF_SIZE];

  // pcap_open_offline(...) for capture from a file needs it's a local one
  pcap_t *pd;
  pd = pcap_open_offline( strFile, errbuf );

  if (pd == NULL)
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    ret = BAD_FORMAT_FILE;
  }
  else
  {
    ret = RIGHT_FORMAT_FILE;
    pcap_close(pd);
  }

  return ret;
}


bool Sniffer::isSniffing()
{
  return m_bSniffing;
}


void Sniffer::pauseSniffing()
{
  m_bPauseSniffing = true;
}


void Sniffer::continueSniffing()
{
  m_bPauseSniffing = false;
}


bool Sniffer::pausedSniffing() const
{
  return m_bPauseSniffing;
}


long Sniffer::packetsTotalSize() const
{
  return m_packetsTotalSize;
}


long Sniffer::countPackets() const
{
  return m_countPackets;
}


void Sniffer::run()
{
  if (m_bFromFile)
    startSniffingFromFile();
  else
    startSniffingInterface();
}


void Sniffer::startSniffingFromFile()
{
  // set all parameters to prepare sniffing structures
  //struct bpf_program bpfp;
  char errbuf[PCAP_ERRBUF_SIZE];

//#ifdef MUST_DO_SELECT
//  int selRet = 0;
//  m_pcap_fd = 0;
//#endif

  kdDebug() << "Sniffer::startSniffingFromFile()... Ok, sniffing from file: '" << m_strFilename << "'" << endl;

  // reset the number of counted packets
  m_countPackets = 0;

  m_packetsTotalSize = 24;
  m_pcapfp = pcap_open_offline( m_strFilename.ascii(), errbuf );
  if ( m_pcapfp == NULL )
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    pcap_close( m_pcapfp );
    return;
  }
#ifdef MUST_DO_SELECT

#ifdef HAVE_PCAP_GET_SELECTABLE_FD
    m_pcap_fd = pcap_get_selectable_fd(m_pcapfp);
#else
    m_pcap_fd = pcap_fileno(m_pcapfp);
#endif

#endif

  // start reading packets on the selected source
  while (m_canSniff)
  {
    struct pcap_pkthdr hdr;
    unsigned char *p;

    p = (unsigned char*) pcap_next(m_pcapfp, &hdr);

    if (p == NULL)
      break;     // file is end

    // we got a packet: raw bytes (p), the timestamp and the captured packet length and the real packet length (hdr)
    /*
     *  hdr is a struct pcap_pkthdr:
     *    ts - a struct timeval containing the time when the packet was captured
     *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
     *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
     *      of bytes available from the capture, if the length of the packet is larger than the maximum
     *      number of bytes to capture)
     */

    int frameLength, frameType;
    setHeaderData(m_pcapfp, frameType, frameLength);
    if (frameLength < 0)
    {
      kdDebug() << "unknown packet" << endl;
      continue;
    }
    m_packetsTotalSize += hdr.caplen + 16;
    m_countPackets++;

    // now we call PacketManager to store:
    //  1 - the raw data
    //  2 - the timestamp and the captured packet length and the real packet length
    //  3 - the frame type string
    //  4 - the frame length: we need this to know how to get the data from the frame
    emit gotPacket(p, hdr, m_frameNumber++, frameType, frameLength);
  }

  kdDebug() << "Thread stopped. Got " << m_countPackets << " packets" << endl;

  kdDebug() << "----------- total data: " << m_packetsTotalSize << endl;

  // release libpcap resource
  pcap_close(m_pcapfp);

  // now the thread has stopped and the boolean variable could be "resetted"
  m_bSniffing = false;

  // thread's stopped
  m_canSniff = false;
}


void Sniffer::startSniffingInterface()
{
  // set all parameters to prepare sniffing structures
  //struct bpf_program bpfp;
  char errbuf[PCAP_ERRBUF_SIZE];
  unsigned short numberOfNULL = 0;

  // reset the number of counted packets
  m_countPackets = 0;

  m_packetsTotalSize = 24;
  m_pcapfp = NULL;

  // when you start sniffing the GUI read ksniffer.pcap, but it's not created untill packets doesn't arrive
  // so it's not a real problem if m_pcapfp is NULL... the loop will not be infinite and than Sniffer class
  // inherits by QThread so the GUI will not freeze
  while (m_pcapfp == NULL)
  {
    m_pcapfp = pcap_open_offline( m_strTmpFilePath.ascii(), errbuf );
    sleep(1);
  }

  // an error occurred
  if ( m_pcapfp == NULL )
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    pcap_close( m_pcapfp );
    return;
  }
#ifdef MUST_DO_SELECT
  m_pcap_fd = pcap_fileno(m_pcapfp);
#endif

  kdDebug() << "Sniffer::startSniffingInterface()... Ok, sniffing from file: '" << m_strTmpFilePath << "'" << endl;

  // start reading packets on the selected source
  while (m_canSniff)
  {
    struct pcap_pkthdr hdr;
    unsigned char *p;

    p = (unsigned char*) pcap_next(m_pcapfp, &hdr);

    /*
     * file is not ended: we are doing live capture using the file capture:
     * the "sniff" executable produts a sniffer packets file and we read from it
     * the thread stops when user wants or when problems occur
     */
    if (p == NULL)
    {
      usleep(200 * numberOfNULL);
      if (numberOfNULL < 5)
        // wait max one millisecond
        numberOfNULL++;
      continue;
    }
    else
      if (numberOfNULL > 0)
        numberOfNULL--;

    // we got a packet: raw bytes (p), the timestamp and the captured packet length and the real packet length (hdr)
    /*
     *  hdr is a struct pcap_pkthdr:
     *    ts - a struct timeval containing the time when the packet was captured
     *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
     *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
     *      of bytes available from the capture, if the length of the packet is larger than the maximum
     *      number of bytes to capture)
     */

    int frameLength, frameType;
    setHeaderData(m_pcapfp, frameType, frameLength);
    if (frameLength < 0)
    {
      kdDebug() << "unknown packet" << endl;
      continue;
    }
    m_packetsTotalSize += hdr.caplen + 16;
    m_countPackets++;

    // now we call PacketManager to store:
    //  1 - the raw data
    //  2 - the timestamp and the captured packet length and the real packet length
    //  3 - the frame type string
    //  4 - the frame length: we need this to know how to get the data from the frame
    emit gotPacket(p, hdr, m_frameNumber++, frameType, frameLength);
  }

  kdDebug() << "Thread stopped. Got " << m_countPackets << " packets" << endl;

  if (m_packetsTotalSize == 24)
    kdDebug() << "----------- total data: 0 byte" << endl;
  else
    kdDebug() << "----------- total data: " << m_packetsTotalSize << " bytes" << endl;

  // release libpcap resource
  pcap_close(m_pcapfp);

  // now the thread has stopped and the boolean variable could be "resetted"
  m_bSniffing = false;

  // thread's stopped
  m_canSniff = false;
}


// parameters:
//   1 - pcap_t pHandler: packet handler
//   2 - int frameType: frame type (input/output data)
//   3 - int hdrLen: frame header length (input/output data)
// we need the header length to extract the data from the frame
// we got -1 for an unknown Link Layer
void Sniffer::setHeaderData(pcap_t *pHandler, int &frameType, int &hdrLen)
{
  frameType = pcap_datalink(pHandler);

  switch(frameType)
  {
    case DLT_EN10MB:
      hdrLen = 14;
      break;
    case DLT_LINUX_SLL:
      hdrLen = 16;
      break;
    case DLT_LOOP:
      hdrLen = 4;
      break;
    case DLT_NULL:
      hdrLen = 4;
      break;
    case DLT_RAW:
      hdrLen = 0;
      break;
    default:
      hdrLen = -1;
      break;
  }
}

#include "sniffer.moc"
