/***************************************************************************
 *   Copyright (C) 2004 by Stefan Kombrink                                 *
 *   katakombi@web.de                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
 
#include "synparam.h"
#include "syntp.h"

#include <stdlib.h>

#include <kdebug.h> 


SynTouchPad::SynTouchPad()
{
    // initialize pressure sentitivity default presets
    pressSens[0]=53;
    pressSens[1]=38;
    pressSens[2]=25;
    pressSens[3]=18;
    pressSens[4]=10;
    
    // fail save defaults
    detectedSynapticsVersion = "0.13.2 or below";
    
    // create the configuration instances
    cfgSettings = 0;
    cfgHWSettings = new KConfig();
}

SynTouchPad::~SynTouchPad()
{
    if (cfgSettings) delete cfgSettings;    
    delete cfgHWSettings;
}

bool SynTouchPad::hasSynaptics()
{
    KProcess proc;
    proc << "synclient" << "-l";
    
    // try to run synclient
    if (!proc.start(KProcess::Block, KProcess::AllOutput))
        // running synclient wasn't even possible
        return FALSE;
    else 
        // check whether synclient exited normally
        return proc.normalExit();
}


bool SynTouchPad::hasSynDaemon()
{
    KProcess proc;
    proc << "syndaemon";
    
    // try to run syndaemon
    if (proc.start(KProcess::DontCare, KProcess::AllOutput))
    {
        proc.kill();
        return TRUE;
    }
    
    // it didn't start
    return FALSE;
}




QString SynTouchPad::detectSynapticsVersion()
{
    // erase the std out string
    shellStdOut="";
    KProcess proc;
    proc << "synclient" << "-V";
    
    connect(&proc, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(getProcessOutput(KProcess *, char *, int)));
    proc.start(KProcess::Block, KProcess::Stdout);
    
    if (proc.normalExit()) detectedSynapticsVersion = shellStdOut;
    
    return detectedSynapticsVersion;
}


bool SynTouchPad::hasSHMConfigurability()
{
    KProcess proc;
    proc.setUseShell(TRUE);
    proc << "synclient" << "-h" << ">/dev/null";
    
    proc.start(KProcess::Block, KProcess::NoCommunication);
    if (!proc.normalExit())
        if (proc.exitStatus() == 1)
            return FALSE;
            
    return TRUE;
}


bool SynTouchPad::disableTP()
{
    KProcess proc;
    proc << "synclient" << "TouchPadOff=1";
    proc.start(KProcess::Block);
    return proc.normalExit();
}


bool SynTouchPad::enableTP()
{
    KProcess proc;
    proc << "synclient" << "TouchPadOff=0";
    proc.start(KProcess::Block);
    return proc.normalExit();
}


void SynTouchPad::defaultConfig()
{
    cfgSettings->writeEntry(TOUCHPADOFF, (int)0);
    
    cfgSettings->writeEntry(TAPBUTTON1, (int)1);
    cfgSettings->writeEntry(TAPBUTTON2, (int)2);
    cfgSettings->writeEntry(TAPBUTTON3, (int)3);
    
    cfgSettings->writeEntry(MAXTAPTIME, (int)180);
    
    cfgSettings->writeEntry(VERTSCROLLDELTA, (int)100);
    cfgSettings->writeEntry(HORIZSCROLLDELTA, (int)100);
    cfgSettings->writeEntry(CIRCSCROLLDELTA, (int)80);
    
    cfgSettings->writeEntry(CIRCULARSCROLLING, (int)0);
    cfgSettings->writeEntry(CIRCSCROLLTRIGGER, (int)0);
    
    cfgSettings->writeEntry(SYNDAEMONTIMING, (int)600);
    cfgSettings->writeEntry(SYNDAEMONOFF, (int)1);
    cfgSettings->writeEntry(VSCROLLEMUOFF, (int)0);
    cfgSettings->writeEntry(HSCROLLEMUOFF, (int)0);
    cfgSettings->writeEntry(TAPPINGOFF, (int)0);
    cfgSettings->writeEntry(SCROLLINGMODE, (int)1);
    cfgSettings->writeEntry(PRESSURESENSITIVITY, (int)2);
}

void SynTouchPad::enhanceConfig()
{
    // if there are no user settings take the initial defaults
    
    if (!cfgSettings->hasKey(TOUCHPADOFF)) cfgSettings->writeEntry(TOUCHPADOFF, (int)0);
    
    if (!cfgSettings->hasKey(TAPBUTTON1)) cfgSettings->writeEntry(TAPBUTTON1, (int)1);
    if (!cfgSettings->hasKey(TAPBUTTON2)) cfgSettings->writeEntry(TAPBUTTON2, (int)2);
    if (!cfgSettings->hasKey(TAPBUTTON3)) cfgSettings->writeEntry(TAPBUTTON3, (int)3);
    
    if (!cfgSettings->hasKey(MAXTAPTIME)) cfgSettings->writeEntry(MAXTAPTIME, (int)180);
    
    if (!cfgSettings->hasKey(VERTSCROLLDELTA)) cfgSettings->writeEntry(VERTSCROLLDELTA, (int)100);
    if (!cfgSettings->hasKey(HORIZSCROLLDELTA)) cfgSettings->writeEntry(HORIZSCROLLDELTA, (int)100);
    if (!cfgSettings->hasKey(CIRCSCROLLDELTA)) cfgSettings->writeEntry(CIRCSCROLLDELTA, (int)80);
    
    if (!cfgSettings->hasKey(CIRCULARSCROLLING)) cfgSettings->writeEntry(CIRCULARSCROLLING, (int)0);
    if (!cfgSettings->hasKey(CIRCSCROLLTRIGGER)) cfgSettings->writeEntry(CIRCSCROLLTRIGGER, (int)0);
    
    if (!cfgSettings->hasKey(SYNDAEMONTIMING)) cfgSettings->writeEntry(SYNDAEMONTIMING, (int)600);
    if (!cfgSettings->hasKey(SYNDAEMONOFF)) cfgSettings->writeEntry(SYNDAEMONOFF, (int)1);
    if (!cfgSettings->hasKey(VSCROLLEMUOFF)) cfgSettings->writeEntry(VSCROLLEMUOFF, (int)0);
    if (!cfgSettings->hasKey(HSCROLLEMUOFF)) cfgSettings->writeEntry(HSCROLLEMUOFF, (int)0);
    if (!cfgSettings->hasKey(TAPPINGOFF)) cfgSettings->writeEntry(TAPPINGOFF, (int)0);
    if (!cfgSettings->hasKey(SCROLLINGMODE)) cfgSettings->writeEntry(SCROLLINGMODE, (int)1);
    if (!cfgSettings->hasKey(PRESSURESENSITIVITY)) cfgSettings->writeEntry(PRESSURESENSITIVITY, (int)2);
}


void SynTouchPad::readConfig()
{
    // if user config already used wipe all out
    if (cfgSettings) 
    {
        cfgSettings->rollback();
        delete cfgSettings;
    }
    // (re)create user config
    cfgSettings = new KConfig("ksynaptics");
    
    if (hasSynaptics())
    {
        // erase the stdout string
        shellStdOut="";
        
        KProcess proc;
        proc.setUseShell(TRUE);
        
        proc << "synclient -l|grep '='|tr -d ' '|tr '\n=' ','";
        connect(&proc, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(getProcessOutput(KProcess *, char *, int)));
        proc.start(KProcess::Block,KProcess::Stdout);
        
        QString token = shellStdOut.section(',',0,0);
        QString value = shellStdOut.section(',',1,1);
        unsigned int i = 0;
        unsigned int factor;
        
        while (!token.isNull() && !token.isEmpty())
        {
            if (token == QString(CIRCSCROLLDELTA)) 
                factor = 1000;
            else
                factor = 1;
            
            cfgHWSettings->writeEntry(token,(unsigned int)(atof(value.ascii())*factor));
            kdDebug() << token << "=" << value << endl;
            
            i++;
            value = shellStdOut.section(',',i,i);
            i++;
            token = shellStdOut.section(',',i,i);
        }
        
    
        // detect syndaemon timing value
        cfgHWSettings->writeEntry(SYNDAEMONTIMING,getSynDaemonTiming());
    }
    
    // fill in missing parameters by default values
    enhanceConfig();
}


bool SynTouchPad::isCustomParameter(QString paramName)
{
    if (paramName==VSCROLLEMUOFF) return TRUE;
    if (paramName==HSCROLLEMUOFF) return TRUE;
    if (paramName==SYNDAEMONOFF) return TRUE;
    if (paramName==SYNDAEMONTIMING) return TRUE;
    if (paramName==TAPPINGOFF) return TRUE;
    if (paramName==SCROLLINGMODE) return TRUE;
    if (paramName==PRESSURESENSITIVITY) return TRUE;
    
    return FALSE;
}


bool SynTouchPad::settingsChanged()
{
    return cfgSettings->isDirty();
}








void SynTouchPad::applyConfig()
{
    kdDebug() << "saving user settings" << endl;
    cfgSettings->sync();
    
    
    QMap<QString, float> param;
    QMap<QString, float>::Iterator it;
    
    
    param[TOUCHPADOFF] = cfgSettings->readUnsignedNumEntry(TOUCHPADOFF);

    param[TAPBUTTON1] =  cfgSettings->readUnsignedNumEntry(TAPBUTTON1);
    param[TAPBUTTON2] =  cfgSettings->readUnsignedNumEntry(TAPBUTTON2);                   
    param[TAPBUTTON3] =  cfgSettings->readUnsignedNumEntry(TAPBUTTON3);                   
    param[MAXTAPTIME] =  cfgSettings->readUnsignedNumEntry(MAXTAPTIME);                   
                    
    param[VERTSCROLLDELTA]  = cfgSettings->readUnsignedNumEntry(VERTSCROLLDELTA);
    param[HORIZSCROLLDELTA] = cfgSettings->readUnsignedNumEntry(HORIZSCROLLDELTA);              
    param[CIRCSCROLLDELTA]  = ((float)cfgSettings->readUnsignedNumEntry(CIRCSCROLLDELTA))/1000;               

    param[CIRCULARSCROLLING] = cfgSettings->readUnsignedNumEntry(CIRCULARSCROLLING);
    param[CIRCSCROLLTRIGGER] = cfgSettings->readUnsignedNumEntry(CIRCSCROLLTRIGGER);            

    param[SYNDAEMONTIMING] = cfgSettings->readUnsignedNumEntry(SYNDAEMONTIMING);                
    param[SYNDAEMONOFF]    = cfgSettings->readUnsignedNumEntry(SYNDAEMONOFF);                
    param[VSCROLLEMUOFF]   = cfgSettings->readUnsignedNumEntry(VSCROLLEMUOFF);                
    param[HSCROLLEMUOFF]   = cfgSettings->readUnsignedNumEntry(HSCROLLEMUOFF);                
    param[TAPPINGOFF]      = cfgSettings->readUnsignedNumEntry(TAPPINGOFF);                
    param[SCROLLINGMODE]   = cfgSettings->readUnsignedNumEntry(SCROLLINGMODE);                
    
    
    
    // special cases
    switch (cfgSettings->readUnsignedNumEntry(SCROLLINGMODE))
    {
        case 0:
            param[VSCROLLEMUOFF] = 1;
            param[HSCROLLEMUOFF] = 1;
            param[CIRCULARSCROLLING] = 0;
            break;
        case 1:
            param[CIRCULARSCROLLING] = 0;
            break;
        case 2:
            param[VSCROLLEMUOFF] = 1;
            param[HSCROLLEMUOFF] = 1;
            param[CIRCULARSCROLLING] = 1;
            break;
        default:
            break;
    }
   
    // calculate fingerhigh/fingerlow
    param[FINGERLOW] = pressSens[cfgSettings->readUnsignedNumEntry(PRESSURESENSITIVITY)];
    //param[FINGERHIGH] = pressSens[cfgSettings->readUnsignedNumEntry(PRESSURESENSITIVITY)] + param[FINGERLOW]/5;
    param[FINGERHIGH] = param[FINGERLOW] + 5;
   
    if (param[VSCROLLEMUOFF] == 1) param[VERTSCROLLDELTA] = 0;
    if (param[HSCROLLEMUOFF] == 1) param[HORIZSCROLLDELTA] = 0;

    if (param[TAPPINGOFF] == 1) param[MAXTAPTIME] = 0;
    
    for (it = param.begin(); it != param.end(); it++)
    {
        kdDebug() << "new " << it.key() << " is " << it.data() << endl;

        // skip the loop if there is a custom parameter
        if (isCustomParameter(it.key()))
            continue;
                       
        KProcess proc;
        QString cmdStr;
        proc << "synclient" << cmdStr.sprintf("%s=%g",it.key().ascii(), it.data());
        proc.start(KProcess::Block);
   }
    
   if ((param[TOUCHPADOFF] == 1) || (param[SYNDAEMONOFF] == 1))
       applySynDaemonTiming(0);
   else
       applySynDaemonTiming((int)param[SYNDAEMONTIMING]);
       
   if (param[TOUCHPADOFF] == 1 ) disableTP(); else enableTP();
}

void SynTouchPad::cancelConfig()
{
    cfgSettings->rollback();
}


int SynTouchPad::getParameter(QString paramName)
{
    kdDebug() << "read " << paramName << endl;
    return cfgSettings->readUnsignedNumEntry(paramName);
}


bool SynTouchPad::setParameter(QString paramName, int param)
{
    bool existed = (cfgSettings->hasKey(paramName));
    kdDebug() << "write " << paramName << "=" << param << endl;
    if (!existed) kdDebug() << "(new)";
    cfgSettings->writeEntry(paramName,(int)param);
    return existed;
}


int SynTouchPad::getSynDaemonTiming()
{
    // erase the stdout string
    shellStdOut = "";
    KProcess proc, proc2;
    
    int runningInstances = 0;
    
    proc.setUseShell(TRUE);
    proc << "ps ax|grep -v grep|grep -c syndaemon";
    
    connect(&proc, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(getProcessOutput(KProcess *, char *, int)));    
    proc.start(KProcess::Block,KProcess::Stdout);

    runningInstances = atoi(shellStdOut.ascii());
    
    if (runningInstances == 0) return 0; // in that case syndaemon timing is disabled
    
    // erase the stdout string
    shellStdOut = "";
    
    proc2.setUseShell(TRUE);
    proc2 << "ps ax|grep -v grep|grep syndaemon|sed 's/^.*syndaemon//g'|sed 's/-i//g'";
    
    connect(&proc2, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(getProcessOutput(KProcess *, char *, int)));    
    proc2.start(KProcess::Block,KProcess::Stdout);
    
    return (int)(1000 * atof(shellStdOut.ascii()));
}


bool SynTouchPad::applySynDaemonTiming(int delay)
{
    if (!hasSynDaemon()) return FALSE;
    
    KProcess killInst;
    killInst.setUseShell(TRUE);
    killInst << "killall -9 syndaemon";
    killInst.start(KProcess::Block);
    
    if (delay == 0) return TRUE;
    
    QString arg;
    arg.sprintf("-i %g",(float)cfgSettings->readUnsignedNumEntry(SYNDAEMONTIMING)/1000.0);
    KProcess startInst;
    startInst << "syndaemon" << arg <<  ">/dev/null";
    startInst.start(KProcess::DontCare);
    
    return startInst.isRunning();
}



void SynTouchPad::getProcessOutput(KProcess *, char *s, int len)
{
    shellStdOut.append(QString::fromAscii(s, len));
}

#include "syntp.moc"
