/***************************************************************************
 *   Copyright (C) 2004 by Raul Fernandes                                  *
 *   rgfbr@yahoo.com.br                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "ktranslator.h"
#include "ktranslator.moc"

#include "ktranslatordock.h"
#include "resultswindow.h"
#include "generalsettings.h"
#include "colorssettings.h"
#include "dictionariessettings.h"
#include "config.h"
#include "ktranslatordictionary.h"

#include <dictionary/dictionaryplugin.h>
#include <ui/uiplugin.h>

#include <qlabel.h>
#include <qlineedit.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qlayout.h>
#include <qtextbrowser.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qpushbutton.h>
#include <qstringlist.h>
#include <qcheckbox.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qlistbox.h>
#include <qdatetime.h>
#include <qregexp.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kactionclasses.h>
#include <kpopupmenu.h>
#include <kparts/componentfactory.h>
#include <kconfigdialog.h>

#ifndef NDEBUG

#include <kdebug.h>

#endif



ktranslator::ktranslator()
    : KMainWindow( 0, "ktranslator" )
{

#ifndef NDEBUG

    kdDebug() << "ktranslator::ktranslator()" << endl;

#endif

    // set the shell's ui resource file
    setXMLFile("ktranslatorui.rc");

    indexFilePath = locate( "appdata", "english.idx" );

    vbox = new QVBox( this );

    new QLabel( i18n( "Enter the word to be translated:" ), vbox );

    QHBox *hbox = new QHBox( vbox );

    lineedit = new QLineEdit( hbox );
    QWhatsThis::add( lineedit, i18n( "Enter the word to be searched here and click in \"Translate\" to translate the word" ) );
    connect( lineedit, SIGNAL( returnPressed() ), this, SLOT( searchLineedit() ) );

    QPushButton *button = new QPushButton ( i18n( "Translate" ), hbox );
    button->resize( 30, 20 );
    QWhatsThis::add( button, i18n( "Click here to translate the word above" ) );
    connect( button, SIGNAL( clicked() ), this, SLOT( searchLineedit() ) );

    main = new QTextBrowser( vbox );
    main->setText( i18n( "To translate a word, select the word. The easiest way to select the word is double clicking in it.\nIt will appears a passive popup with the translation of word." ) );
    connect( main, SIGNAL( linkClicked( const QString & ) ), this, SLOT( linkClicked( const QString & ) ) );

    vbox->adjustSize();

    // dock the app
    dock = new ktranslatorDock( this, "dock" );
    dock->setPixmap( dock->loadIcon( "ktranslator" ) );

    // Actions

    KAction *actionConfig = new KAction( i18n( "Preferences" ), "configure", 0, this, SLOT( slotConfig() ), actionCollection(), "preferences" );
    actionConfig->setWhatsThis( i18n( "Click here to configure the application" ) );
    dock->actionCollection()->insert( actionConfig );
    actionConfig->plug( dock->contextMenu() );

    setCentralWidget( vbox );
    createGUI();

    dock->show();

    QTimer::singleShot( 0, this, SLOT( loadPlugins() ) );

    wordToTranslate = "";
}

ktranslator::~ktranslator()
{
}

void ktranslator::search( const QString &text )
{
  if( text.isEmpty() ) return;
  QString result = translate( text );
  resultsWindow *pop = new resultsWindow( result, this );
  pop->show();
  pop->setActiveWindow();
  pop->setFocus();
}


void ktranslator::searchLineedit()
{
  QString text = lineedit->text();
  if( text.isEmpty() ) return;
  QString result = translate( text );
  main->clear();
  main->setText( result );
}


void ktranslator::linkClicked( const QString &word )
{
  if( word.isEmpty() ) return;
  QString result = translate( word );
  main->clear();
  main->setText( result );
}


QString ktranslator::translate( const QString &text )
{
  QString word = text;
  wordToTranslate = text;
  QString result = "<html><body><table>";

  #ifndef NDEBUG
    QTime time;
  #endif

  word.remove( "." ); // remove dots from the word
  word = word.lower().stripWhiteSpace();
  result += "<tr><td><b>" + i18n( "Word to translate:" ) + "<br><font color=" + config::headwordColor().name() + ">" + word + "</font><br></b></td></tr>";

  // search the word in index file
  QString indexResult = searchIndex( word );
  // test if searchIndex returns something
  // if yes, show it
  if( !indexResult.isEmpty() ) result += "<tr><td>" + indexResult + "</td></tr>";

  QString dicResult;
  // search in dictionaries
  for ( KTranslatorDictionary *dic = plugins->list.first(); dic; dic = plugins->list.next() )
  {
    if( !dic->plugin->enabled() ) continue;

    #ifndef NDEBUG
      time.start();
    #endif

    dicResult = dic->plugin->search( word );
    if( config::showEmptyResult() || dicResult.find( i18n( "Word not found" ) ) == -1 ) result += "<tr><td>" + dicResult + "</td></tr>";

    #ifndef NDEBUG
      kdDebug() << "Procurar: " << dic->name() << " - search in " << time.elapsed() << "ms" << endl;
    #endif

  }

  if( result.find( "<dicName>" ) == -1 ) result = "<html><body><b>Word to translate:<br><font color=" + config::headwordColor().name() + ">" + word + "</font><br><br>No results found</b></body></html>";
  else
  {
    result += "</table></body></html>";

    // dictionary name
    result.replace( "<dicName>", "<font color=" + config::nameColor().name() + "><b>" );
    result.replace( "</dicName>", "</b></font><br>" );
    // headword
    result.replace( "<headword>", "<br><b><font color=" + config::headwordColor().name() + ">" );
    result.replace( "</headword>", "</font></b><br>" );
    // inflections
    result.replace( "<inflection>", "<font color=" + config::inflectionColor().name() + ">" );
    result.replace( "</inflection>", "</font><br>" );
    // part of speech
    result.replace( "<pos>", "<font color=" + config::posColor().name() + ">(" );
    result.replace( "</pos>", ".)</font><br>" );
    // definition
    result.replace( "<definition>", "<font color=" + config::definitionColor().name() + ">" );
    result.replace( "</definition>", "</font><br>" );
    // link
    result.replace( QRegExp( "<link>([^<]*)</link>" ), "<b><font color=" + config::linkColor().name() + "><a href=\"\\1\">\\1</a></font></b>" );
  }

  return result;
}


// Keep the app running when the main window is closed,
// unless the session is being saved for a logout
bool ktranslator::queryClose()
{
  if( !kapp->sessionSaving() )
  {
    hide();
    return false;
  }
  return true;
}

void ktranslator::speech()
{
   QString word = wordToTranslate;
   KProcess *proc = new KProcess();
   // text2wave???
   *proc << config::festivalBinary();
   //*proc << "--language english";
   *proc << "--batch";
   //*proc << "(voice_don_diphone)";
   *proc << "(SayText \"" + word + "\")";
   proc->start( KProcess::Block );
   if( !proc->normalExit() ) KMessageBox::error( this, i18n( "Error when running Festival." ), i18n( "Error" ) );
   delete proc;
}

void ktranslator::searchInInternet()
{
   QString word = wordToTranslate;
   QStringList addresses;
   addresses.append( "http://www.m-w.com/cgi-bin/dictionary?book=Dictionary&va=" + word );
   addresses.append( "http://www.dict.org/bin/Dict?Form=Dict2&Database=*&Query=" + word );
   addresses.append( "http://en.wikipedia.org/wiki/" + word );
   addresses.append( "http://en.wiktionary.org/wiki/" + word );
   /*kapp->startServiceByName( "Web Browser", addresses, NULL, NULL, NULL, NULL, true );*/
   KApplication::kdeinitExec( "konqueror", addresses );
}


void ktranslator::slotConfig()
{
   if ( KConfigDialog::showDialog( "settings" ) ) return;

   dialog = new KConfigDialog( this, "settings", config::self() );

   // General settings page
   GeneralSettings *general = new GeneralSettings( 0, "General" );
   dialog->addPage( general, i18n( "General" ), "configure" );

   // Colors settings page
   ColorsSettings *colors = new ColorsSettings( 0, "Colors" );
   dialog->addPage( colors, i18n( "Colors" ), "configure" );

   // Dictionaries page
   DictionariesSettings *dictionaries = new DictionariesSettings( 0, "Dictionaries" );
   dialog->addPage( dictionaries, i18n( "Dictionaries" ), "configure" );
   dictionaries->editButton->hide();
   //dictionaries->dicList->addColumn( "Dictionaries" );
   for( KTranslatorDictionary *dic = plugins->list.first(); dic; dic = plugins->list.next() )
   {
     dictionaries->dicList->insertItem( dic->name() );
   }
   connect( dictionaries->addButton, SIGNAL( clicked() ), this, SLOT( addDictionary() ) );
   dictionaries->editButton->setEnabled( false );
   //connect( dictionaries->editButton, SIGNAL( clicked() ), this, SLOT( editDictionary() ) );
   connect( dictionaries->removeButton, SIGNAL( clicked() ), this, SLOT( removeDictionary() ) );

   connect( dialog, SIGNAL( settingsChanged() ), this, SLOT( updateConfig() ) );
   connect( dialog, SIGNAL( okClicked() ), this, SLOT( updateConfig() ) );

   dialog->show();
}


void ktranslator::addDictionary()
{
  // Add the dictionary
  plugins->addDictionary();

  // Refresh list of dictionaries in config dialog
  KConfigDialog *dialog = KConfigDialog::exists( "settings" );
  if( dialog == NULL ) return;

  QListBox *dicList = (QListBox *)(dialog->child( "Dictionaries" )->child( "dicList" ));
  dicList->clear();
  for( KTranslatorDictionary *dic = plugins->list.first(); dic; dic = plugins->list.next() )
  {
    dicList->insertItem( dic->name() );
  }
}


void ktranslator::editDictionary()
{
  KConfigDialog *dialog = KConfigDialog::exists( "settings" );
  if( dialog == NULL ) return;

  QListBox *dicList = (QListBox *)(dialog->child( "Dictionaries" )->child( "dicList" ));
  int item = dicList->currentItem();

  plugins->editDictionary( item );

  // Refresh the dictionary list
  dicList->clear();
  for( KTranslatorDictionary *dic = plugins->list.first(); dic; dic = plugins->list.next() )
  {
    dicList->insertItem( dic->name() );
  }
}


void ktranslator::removeDictionary()
{
  KConfigDialog *dialog = KConfigDialog::exists( "settings" );
  if( dialog == NULL ) return;

  QListBox *dicList = (QListBox *)(dialog->child( "Dictionaries" )->child( "dicList" ));
  int item = dicList->currentItem();

  // Remove the dictionary
  plugins->removeDictionary( item );

  // Refresh the dictionary list
  dicList->clear();
  for( KTranslatorDictionary *dic = plugins->list.first(); dic; dic = plugins->list.next() )
  {
    dicList->insertItem( dic->name() );
  }
}


void ktranslator::updateConfig()
{
  // Call the function in UI plugins to change the
  // modifier key that triggers the translation
  emit loadModifierKey();
}


/*!
    \fn ktranslator::loadPlugins()
 */
void ktranslator::loadPlugins()
{

  // Load dictionaries
  plugins = new KTranslatorDictionaryManager();
  plugins->loadPlugins();

  // load UI plugins
  KTrader::OfferList offers = KTrader::self()->query( "KTranslator/UIPlugin" );
  KService::Ptr service;
  KTrader::OfferList::ConstIterator iter;
  int errCode;
  KTranslator::UIPlugin *uiplugin;
  for( iter = offers.begin(); iter != offers.end(); ++iter )
  {
    service = *iter;
    errCode = 0;
    uiplugin = KParts::ComponentFactory::createInstanceFromService<KTranslator::UIPlugin>( service, this, 0, NULL, &errCode );
    if( uiplugin )
    {
      guiFactory()->addClient(uiplugin);
      // Signal to start the search in dictionary plugins
      connect( uiplugin, SIGNAL( search( const QString & ) ), this, SLOT( search( const QString & ) ) );
      // Signal to change the modifier key that triggers the translation
      connect( this, SIGNAL( loadModifierKey() ), uiplugin, SLOT( loadModifierKey() ) );
    }
  }
}



/*!
    \fn ktranslator::searchIndex( QString &word )
 */
QString ktranslator::searchIndex( QString &word )
{
  QString headword;
  bool found = false;
  QString plural, plural2, trdPerson, preterite, pastParticiple, presentParticiple;
  QString result = QString::null;

  QFile indexFile( indexFilePath );

  // search in index file
  if( !indexFile.open( IO_ReadOnly ) ) return result;
  QTextStream index( &indexFile );
  QString line;
  int pos;

  while ( !index.atEnd() && !found )
  {
    line = index.readLine();
    if( line[ 0 ] == '#' ) continue; // comment line
    // a little optimize
    // don't make unnecessary comparations
    // if the first letter differs, the word is different
    if( line[ 0 ] != word[ 0 ] ) continue;
    pos = line.find( "|" );
    headword = line.left( pos );

    // plural
    switch( line[ pos + 1 ].latin1() )
    {
      case '0':
      {
        // noun with plural equal to singular
        plural = headword;
        break;
      }
      case '1':
      {
        // noun with simple plural: +s
        plural = headword + "s";
        break;
      }
      case '2':
      {
        // noun ending in ss, s, sh, ch, x, o: +es
        plural = headword + "es";
        break;
      }
      case '3':
      {
        // noun ending y not preceded by a vowel: -y +ies
        plural = headword.left( headword.length() - 1 ) + "ies";
        break;
      }
      case '4':
      {
        // noun ending in y preceded by a vowel: +s
        plural = headword + "s";
        break;
      }
      case '5':
      {
        // noun ending in f: -f +ves
        plural = headword.left( headword.length() - 1 ) + "ves";
        break;
      }
      case '6':
      {
        // noun ending in fe: -fe +ves
        plural = headword.left( headword.length() - 2 ) + "ves";
        break;
      }
      case '7':
      {
        // noun ending in um: -um +a
        plural = headword.left( headword.length() - 2 ) + "a";
        break;
      }
      case '8':
      {
        // noun ending in on: -on +a
        plural = headword.left( headword.length() - 2 ) + "a";
        break;
      }
      case '9':
      {
        // noun ending in us: -us +i
        plural = headword.left( headword.length() - 2 ) + "i";
        break;
      }
      case 'A':
      {
        // noun of old english: +en
        plural = headword + "en";
        break;
      }
      case 'B':
      {
        // noun ending in sis: -is +es
        plural = headword.left( headword.length() - 2 ) + "es";
        break;
      }
      case 'Z':
      {
        // irregular plural: plural after |
        plural = line.right( line.length() - pos - 4 );
        break;
      }
      default:
      {
        plural = ""; // don't exists plural*/
      }
    }

    // have two plurals??
    if( line[ pos + 3 ] == '*' ) plural2 = line.section( '*', 1, 1 );
    else plural2 = "";

    // verbal forms
    switch( line[ pos + 2 ].latin1() )
    {
      case '1':
      {
        // regular verb ending in consonant (3p +s; p +ed; present participle +ing)
        trdPerson = headword + "s";
        preterite = headword + "ed";
        pastParticiple = headword + "ed";
        presentParticiple = headword + "ing";
        break;
      }
      case '2':
      {
        // regular verb ending in e (3p +s; p +d; present participle -e +ing)
        trdPerson = headword + "s";
        preterite = headword + "d";
        pastParticiple = headword + "d";
        presentParticiple = headword.left( headword.length() - 1 ) + "ing";
        break;
      }
      case '3':
      {
        // regular verb ending in ss or x (3p; p +ed; present participle +ing)
        trdPerson = headword;
        preterite = headword + "ed";
        pastParticiple = headword + "ed";
        presentParticiple = headword + "ing";
        break;
      }
      default:
      {
        // don't exist verbal forms
        trdPerson = "";
        preterite = "";
        pastParticiple = "";
        presentParticiple = "";
      }
    }

      // word found?
    if( headword == word ) found = true;
    else if( plural == word ) found = true;
    else if( plural2 == word ) found = true;
    else if( trdPerson == word ) found = true;
    else if( preterite == word ) found = true;
    else if( pastParticiple == word ) found = true;
    else if( presentParticiple == word ) found = true;
  }
  indexFile.close();

  if( found )
  {
    result += "<inflection><dicName>" + i18n( "Inflections of " ) + "<headword>" + headword + "</headword></dicName>";
    word = headword;
    if( !plural.isEmpty() )
    {
      result += i18n( "plural: %1" ).arg( plural );
      if( !plural2.isEmpty() ) result += i18n( " or %1<br>" ).arg( plural2 );
      else result += "<br>";
    }
    if( !trdPerson.isEmpty() ) result += i18n( "3rd person: %1<br>" ).arg( trdPerson  );
    if( !preterite.isEmpty() ) result += i18n( "preterite: %1<br>" ).arg( preterite  );
    if( !pastParticiple.isEmpty() ) result += i18n( "past participle: %1<br>" ).arg( pastParticiple  );
    if( !presentParticiple.isEmpty() ) result += i18n( "present participle: %1<br>" ).arg( presentParticiple );
    result.truncate( result.length() - 4 );
    result += "</inflection>";
  }
  return result;
}
