/***************************************************************************
 *   Copyright (C) 2006 by Raul Fernandes                                  *
 *   rgfbr@yahoo.com.br                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "sdict.h"

#include <qfile.h>

#include <klocale.h>
#include <kdebug.h>

#include <zlib.h>

#define CHUNK 0xffffL

Sdict::Sdict( const QString &arq )
{
  uint size;
  char block[256];
  uint titlePos;
  uint copyrightPos;
  uint versionPos;

  // dct file
  if( !QFile::exists( arq ) )
  {
    m_isOk = false;
    return;
  }
  file = new QFile( arq );

  // Read the header
  file->open( IO_ReadOnly );
  file->readBlock( block, 43 );
  m_inlang[0] = block[4];
  m_inlang[1] = block[5];
  m_inlang[2] = block[6];
  m_outlang[0] = block[7];
  m_outlang[1] = block[8];
  m_outlang[2] = block[9];
  m_compress = (uchar)block[10] & '\x0f';
  if( m_compress > 1 ){
    // Don't support bzip compression
    m_isOk = false;
    return;
  }
  m_idxlevels = (uchar)block[10] >> 4;
  m_size = (uchar)block[11] | (uchar)block[12] << 8 | (uchar)block[13] << 16 | (uchar)block[14] << 24;
  m_shortidxlen = (uchar)block[15] | (uchar)block[16] << 8 | (uchar)block[17] << 16 | (uchar)block[18] << 24;
  titlePos = (uchar)block[19] | (uchar)block[20] << 8 | (uchar)block[21] << 16 | (uchar)block[22] << 24;
  copyrightPos = (uchar)block[23] | (uchar)block[24] << 8 | (uchar)block[25] << 16 | (uchar)block[26] << 24;
  versionPos = (uchar)block[27] | (uchar)block[28] << 8 | (uchar)block[29] << 16 | (uchar)block[30] << 24;
  m_shortidx = (uchar)block[31] | (uchar)block[32] << 8 | (uchar)block[33] << 16 | (uchar)block[34] << 24;
  m_fullidx = (uchar)block[35] | (uchar)block[36] << 8 | (uchar)block[37] << 16 | (uchar)block[38] << 24;
  m_articles = (uchar)block[39] | (uchar)block[40] << 8 | (uchar)block[41] << 16 | (uchar)block[42] << 24;

  // Read title
  file->at( titlePos );
  file->readBlock( block, 4 );
  size = (uchar)block[0] | (uchar)block[1] << 8 | (uchar)block[2] << 16 | (uchar)block[3] << 24;
  if( m_compress == 1 )
  {
    size -= 2;
    file->at( file->at() + 2 );
  }
  file->readBlock( block, size );
  block[size] = '\0';
  if( m_compress == 0 ) m_title = QString::fromUtf8( block );
  else{
    m_title = QString::fromUtf8( Inflate( block, size ).data() );
  }

  // Read copyright
  file->at( copyrightPos );
  file->readBlock( block, 4 );
  size = (uchar)block[0] | (uchar)block[1] << 8 | (uchar)block[2] << 16 | (uchar)block[3] << 24;
  if( m_compress == 1 )
  {
    size -= 2;
    file->at( file->at() + 2 );
  }
  file->readBlock( block, size );
  block[size] = '\0';
  if( m_compress == 0 ) m_copyright = QString::fromUtf8( block );
  else{
    m_copyright = QString::fromUtf8( Inflate( block, size ).data() );
  }

  // Read version
  file->at( versionPos );
  file->readBlock( block, 4 );
  size = (uchar)block[0] | (uchar)block[1] << 8 | (uchar)block[2] << 16 | (uchar)block[3] << 24;
  if( m_compress == 1 )
  {
    size -= 2;
    file->at( file->at() + 2 );
  }
  file->readBlock( block, size );
  block[size] = '\0';
  if( m_compress == 0 ) m_version = QString::fromUtf8( block );
  else{
    m_version = QString::fromUtf8( Inflate( block, size ).data() );
  }

  QString word;
  ulong offset;
  dic.clear();

  file->at( m_fullidx );
  for(uint a = 0;a<m_size;a++)
  {
    file->readBlock( block, 8 );
    offset = (uchar)block[4] | (uchar)block[5] << 8 | (uchar)block[6] << 16 | (uchar)block[7] << 24;
    size = (uchar)block[0] | (uchar)block[1] << 8;
    size -= 8;
    file->readBlock( block, size );
    block[size] = '\0';
    word = QString::fromUtf8( block ).lower();
    dic.insert( word, offset );
  }

  file->close();

  m_isOk = true;
}


Sdict::~Sdict()
{
  delete file;
}


/*!
    \fn Sdict::search( const QString &word )
 */
QString Sdict::search( const QString &word )
{
  QString result;
  uint size;
  ulong offset;
  char block[256];

  Dictionary::ConstIterator it = dic.find( word );
  if( it != dic.constEnd() )
  {
    offset = it.data();

    file->open( IO_ReadOnly );

    file->at( m_articles + offset );
    file->readBlock( block, 4 );
    size = (uchar)block[0] | (uchar)block[1] << 8 | (uchar)block[2] << 16 | (uchar)block[3] << 24;
    if( m_compress == 1 )
    {
      size -= 2;
      file->at( file->at() + 2 );
    }
    char article[size];
    file->readBlock( article, size );
    file->close();
    article[size] = '\0';

    result = word + "\n";
    if( m_compress == 0 ) result += QString::fromUtf8( article );
    else{
      result += QString::fromUtf8( Inflate( article, size ) );
    }
    return result;
  }
  result += "<font color=#000000>" + i18n( "Word not found" ) + "</font>";
  return result;
}


QCString Sdict::Inflate( const char *data, uint size )
{
    //kdDebug()<< "Inflate()" << endl;
    int ret;
    z_stream strm;
    char out[CHUNK];
    for(uint a = 0;a<CHUNK;a++) out[a] = '\0';
    QCString result;

    // Inicialization of zlib
    strm.zalloc = Z_NULL;
    strm.zfree = Z_NULL;
    strm.opaque = Z_NULL;
    strm.avail_in = 0;
    strm.next_in = Z_NULL;
    ret = inflateInit2( &strm, -MAX_WBITS );
    if (ret != Z_OK)
      return "";

      // Compressed data
      strm.avail_in = size;
      strm.next_in = (Bytef*)data;

      /* run inflate() on input until output buffer not full */
      do {
        strm.avail_out = CHUNK;
        strm.next_out = (Bytef*)out;
        ret = inflate(&strm, Z_SYNC_FLUSH);
        switch (ret) {
          case Z_NEED_DICT:
            ret = Z_DATA_ERROR;     /* and fall through */
          case Z_DATA_ERROR:
          case Z_MEM_ERROR:
            (void)inflateEnd(&strm);
            return ""; // Error
        }
        result += out;
      } while (strm.avail_out == 0);

    /* clean up and return */
    ret = inflateEnd(&strm);
    return result;
}
