using System;
using Gtk;
using GtkExtra;
using Glade;

namespace Kurush
{

public class AccountPropertiesFrame : Frame
{
    public override void UpdateActions (Actions actions)
    {
        actions.EditMove.Enabled = false;
        actions.ViewExpandCollapse.Enabled = false;

        actions.EditProperties.Enabled = false;
        actions.EditDetails.Enabled = true;
        actions.EditAdd.Enabled = false;
        actions.EditRemove.Enabled = false;
        actions.FilePrint.Enabled = false;
    }

    [Glade.Widget] private OptionMenu typeOptionMenu;
    [Glade.Widget] private Entry nameEntry;
    [Glade.Widget] private Entry openingBalanceEntry;
    [Glade.Widget] private Label balanceLabel;
    [Glade.Widget] private Entry descriptionEntry;
    [Glade.Widget] private Button okButton;
    [Glade.Widget] private Table accountTable;


    private Data.Account account;
    private bool newAccount = false;
    private WidgetEntry accountEntry;
    private EventHandler AccountEntryChangedHandler;

    public Data.AccountType Type
    {
        set
        {
            typeOptionMenu.SetHistory ( (uint) value );
        }
    }

    public Data.Account ParentAccount
    {
        set
        {
            accountEntry.Content = value;
            accountEntry.Entry.Text = value.Name;
        }
    }

    public AccountPropertiesFrame (FrameManager owner, Data.Account account)
            : base (owner, "accountPropertiesFrame")
    {
        this.defaultWidget = nameEntry;

        accountEntry = new WidgetEntry (Owner.AccountsPopup);
        accountEntry.AutoSize = true;
        accountEntry.Entry.Editable = false;
        AccountEntryChangedHandler = new EventHandler (AccountEntryChanged);
        accountEntry.PopUp += new EventHandler (AccountEntryPopUp);
        accountEntry.PopDown += new EventHandler (AccountEntryPopDown);
        accountEntry.Show ();
        accountTable.Attach (accountEntry, 1, 4, 3, 4,
                             AttachOptions.Shrink | AttachOptions.Fill,
                             AttachOptions.Shrink,
                             0,0);

        if (account == null)
        {
            this.account = new Data.Account ();
            Title = "Properties for New Account";
            newAccount = true;
        }
        else
        {
            this.account = account;

            nameEntry.Text = account.Name;
            descriptionEntry.Text = account.Description;

            if (account.ParentAccount != null)
            {
                accountEntry.Content = account.ParentAccount;
                accountEntry.Entry.Text = account.ParentAccount.Name;
            }

            Title = "Properties for " + account.Name;
        }


        typeOptionMenu.SetHistory ( (uint) this.account.Type );

        balanceLabel.Text = Format.Value (this.account.Balance);

        this.key = this.account;
    }

    public override void Details ()
    {
        if (account.No < 0)
            return;

        if (!Owner.Activate ( typeof(DetailsFrame),
                              account) )
        {
            DetailsFrame detailsFrame =
                new DetailsFrame (Owner, account);

            Owner.Go (detailsFrame);
        }
    }

    private void on_nameEntry_changed (object o, EventArgs args)
    {
        if (nameEntry.Text != "")
            Title = "Properties for " + nameEntry.Text;
        else
        {
            if (newAccount)
                Title = "Properties for New Account";
            else
                Title = "Properties for " + account.Name;
        }
    }

    private void on_okButton_clicked (object o, EventArgs args)
    {
        account.Name = nameEntry.Text;
        account.Description = descriptionEntry.Text;
        account.ParentAccount = accountEntry.Content as Data.Account;

        if (newAccount)
        {
            account.Type = (Data.AccountType) typeOptionMenu.History;
            Owner.DataBase.AddAccount (account);
        }

        Owner.DataBase.Modified = true;
        Close ();
    }

    private void on_cancelButton_clicked (object o, EventArgs args)
    {
        Close ();
    }

    private void AccountEntryChanged (object obj, EventArgs args)
    {
        SummaryNode node = Owner.AccountsView.GetSelected () as SummaryNode;

        if (node != null && node.Account != null)
        {
            ParentAccount = node.Account;
            accountEntry.Active = false;
        }
        else
        {
            accountEntry.Content = null;
            accountEntry.Entry.Text = "";
        }
    }

    private void AccountEntryPopUp (object obj, EventArgs args)
    {
        Owner.AccountsView.Selection.Changed += AccountEntryChangedHandler;
        accountEntry.Content = null;
    }

    private void AccountEntryPopDown (object obj, EventArgs args)
    {
        Owner.AccountsView.Selection.Changed -= AccountEntryChangedHandler;
    }


}

}
