using System;
using System.Collections;
using Gnome;

namespace Kurush
{

public class Report : PrintJob
{
    static GLib.GType gtype = GLib.GType.Invalid;
    public static new GLib.GType GType
    {
        get
        {
            if (gtype == GLib.GType.Invalid)
                gtype = RegisterGType (typeof (Report));
            return gtype;
        }
    }

    public Report (GLib.GType gtype) : base (gtype)
    {

    }

    public static double ConvertToPixel (double cm)
    {
        return cm * (72.0 / 2.54);
    }

    public enum Alignment
    {
        Left,
        Center,
        Right
    }

    public class Column
    {
        public string Title = "";
        public double width = 0.0;
        public Alignment Align;

        public double Width
        {
            get
            {
                return width;
            }
            set
            {
                width = ConvertToPixel (value);
            }
        }
    }

    private double pageWidth;
    private double pageHeight;

    private double leftMargin = ConvertToPixel (2.5);
    private double rightMargin = ConvertToPixel (1.5);
    private double topMargin = ConvertToPixel (4.0);
    private double bottomMargin = ConvertToPixel (2.0);
    private double headerMargin = ConvertToPixel (1.0);
    private double footerMargin = ConvertToPixel (1.0);
    private double constSpacing = ConvertToPixel (0.2);

    private double baseSpacing;
    private double headerUserSpacing;
    private double headerTitleSpacing;
    private double headerDateSpacing;
    private double headerColumnSpacing;
    private double footerSpacing;

    private Font baseFont;
    private Font headerUserFont;
    private Font headerTitleFont;
    private Font headerDateFont;
    private Font headerColumnFont;
    private Font footerFont;

    public Column[] Columns;

    public string HeaderUser = "";
    public string HeaderTitle;
    public string HeaderDate;
    public bool ShowColumnTitles = false;

    public Report (PrintConfig config, int columnCount,
                   double baseFontSize) : base (config)
    {
        Columns = new Column[columnCount];
        for (int i = 0; i < columnCount; i++)
            Columns[i] = new Column ();

        GetPageSize (out pageWidth, out pageHeight);

        string baseFontName = "Sans Regular";

        baseFont = Font.FindClosest (baseFontName, baseFontSize);
        headerUserFont = Font.FindClosest (baseFontName + " Bold", 16);
        headerTitleFont = Font.FindClosest (baseFontName + " Bold", 16);
        headerDateFont = Font.FindClosest (baseFontName + " Bold", 12);
        headerColumnFont = Font.FindClosest (baseFontName + " Bold", baseFontSize);
        footerFont = Font.FindClosest (baseFontName, 8);

        baseSpacing = constSpacing + baseFont.CapHeight;
        headerUserSpacing = constSpacing + headerUserFont.CapHeight;
        headerTitleSpacing = constSpacing + headerTitleFont.CapHeight;
        headerDateSpacing = constSpacing + headerDateFont.CapHeight;
        headerColumnSpacing = constSpacing + headerColumnFont.CapHeight;
        footerSpacing = constSpacing + footerFont.CapHeight;

        currentRow = pageHeight - topMargin - baseFont.CapHeight;

        maxRows =
            Convert.ToInt32 ((pageHeight - topMargin - bottomMargin) / baseSpacing);
    }

    public void Open ()
    {
        double totalWidth = 0;
        ArrayList zeroCols = new ArrayList ();

        foreach (Column column in Columns)
        {
            if (column.Width == 0.0)
                zeroCols.Add (column);
            else
                totalWidth += column.Width;
        }

        int zeroColsCount = zeroCols.Count;

        if (zeroColsCount > 0)
        {
            double effectiveWidth = pageWidth - leftMargin - rightMargin;
            double autoWidth = (effectiveWidth - totalWidth) / zeroColsCount;

            if (autoWidth < 0)
                autoWidth = 0;

            foreach (Column column in zeroCols)
            {
                column.width = autoWidth;
            }
        }
    }

    private int rowCount = 0;
    private double currentRow;
    private int maxRows;
    private bool pageBeginned = false;
    private int pages = 1;

    private double headerCurrentRow;

    private void BeginPage ()
    {
        Gnome.Print.Beginpage (Context, "");
        pageBeginned = true;

        double effectiveWidth = pageWidth - leftMargin - rightMargin;
        headerCurrentRow = pageHeight - headerMargin - headerUserFont.XHeight;

        if (HeaderUser != "")
        {
            Gnome.Print.Setfont (Context, headerUserFont);
            Gnome.Print.Moveto (Context,
                                (effectiveWidth -
                                 headerUserFont.GetWidthUtf8 (HeaderUser)
                                ) / 2 + leftMargin, headerCurrentRow);
            Gnome.Print.Show (Context, HeaderUser);

            headerCurrentRow -= headerUserSpacing;
        }

        if (HeaderTitle != null)
        {
            Gnome.Print.Setfont (Context, headerTitleFont);
            Gnome.Print.Moveto (Context,
                                (effectiveWidth -
                                 headerTitleFont.GetWidthUtf8 (HeaderTitle)
                                ) / 2 + leftMargin, headerCurrentRow);
            Gnome.Print.Show (Context, HeaderTitle);
            headerCurrentRow -= headerTitleSpacing;
        }

        if (HeaderDate != null)
        {
            Gnome.Print.Setfont (Context, headerDateFont);
            Gnome.Print.Moveto (Context,
                                (effectiveWidth -
                                 headerDateFont.GetWidthUtf8 (HeaderDate)
                                ) / 2 + leftMargin, headerCurrentRow);
            Gnome.Print.Show (Context, HeaderDate);
            headerCurrentRow -= headerDateSpacing;
        }

        if (ShowColumnTitles)
        {
            double x = leftMargin;
            double y = pageHeight - topMargin + 2 * constSpacing;

            foreach (Column column in Columns)
            {
                Gnome.Print.Setfont (Context, headerColumnFont);

                switch (column.Align)
                {
                case Alignment.Left:
                    Gnome.Print.Moveto (Context, x, y);
                    break;
                case Alignment.Right:
                    Gnome.Print.Moveto (Context, x + column.Width -
                                        headerColumnFont.GetWidthUtf8 (column.Title), y);
                    break;
                }

                Gnome.Print.Show (Context, column.Title);

                x += column.Width;
            }
        }

        string footer = "Page " + pages.ToString ();

        Gnome.Print.Setfont (Context, footerFont);
        Gnome.Print.Moveto (Context,
                            (effectiveWidth - footerFont.GetWidthUtf8 (footer)) / 2 + leftMargin,
                            footerMargin);
        Gnome.Print.Show (Context, footer);

        pages++;
    }

    public void Add (params string[] data)
    {
        if (rowCount == 0)
            BeginPage ();

        Gnome.Print.Setfont (Context, baseFont);

        double x = leftMargin;
        int dataCols = data.Length;
        int col = 0;
        foreach (Column column in Columns)
        {
            if (col >= dataCols)
                break;

            switch (column.Align)
            {
            case Alignment.Left:
                Gnome.Print.Moveto (Context, x, currentRow);
                break;
            case Alignment.Right:
                Gnome.Print.Moveto (Context, x + column.Width -
                                    baseFont.GetWidthUtf8 (data[col]), currentRow);
                break;
            }

            Gnome.Print.Show (Context, data[col]);

            x += column.Width;
            col++;
        }

        currentRow -= baseSpacing;
        rowCount++;

        if (rowCount > maxRows)
        {
            Gnome.Print.Showpage (Context);
            pageBeginned = false;
            rowCount = 0;
            currentRow = pageHeight - topMargin - baseFont.CapHeight;
        }
    }

    public void AddSpace (int count)
    {
        for (int i = 0; i < count; i++)
            Add ("");
    }

    public new void Close ()
    {
        if (pageBeginned)
            Gnome.Print.Showpage (Context);

        base.Close ();
    }


}

}
