// =============================================================================
//
//      --- kvi_dcc_voice.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccVoice"

#include <qsplitter.h>

#include "kvi_dcc_event.h"
#include "kvi_dcc_manager.h"
#include "kvi_dcc_voice.h"
#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_frame.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_view.h"
#include "kvi_label.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"
#include "kvi_options.h"
#include "kvi_pushbutton.h"

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

/*
	@document: doc_dcc_voice_proto.kvihelp
	@title: The KVIrc DCC Voice protocol
		The DCC Voice protocol was initially inspired by the DCC TALK
		protocol implemented in the VIrc IRC client for Windows.<br>
		Since no technical documentation was available,
		I had to reimplement everything by myself.<br>
		The protocol is not excellent technically,
		but behaves well with slow connections.<br>
		The requesting client sets up a listening socket
		and sends a CTCP DCC to the target client:<br>
		<b>DCC VOICE &lt;codec&gt; &lt;host_address&gt; &lt;host_port&gt; &lt;sample_rate&gt;</b><br>
		&lt;codec&gt; is the name of the compression algorythm used for the data transfer.<br>
		Currently only the ADPCM codec is implemented (see below).<br>
		&lt;host_address&gt; is an unsigned integer in network byte order
		that specifies the IP address of the source host (as in a normal DCC CHAT or SEND).<br>
		&lt;host_port&gt; is the port number that the requesting client
		listens on.<br>
		&lt;sample_rate&gt; is the number of samples per second
		of the audio data.<br>
		Currently only 8000 Hz sample rate is implemented.<br>
		The target client receives the DCC request and connects to the
		source host on the specified port.<br>
		Once the connection is established, the audio data transfer can start at any time from
		any of the two remote ends.<br>
		The audio messages are sent in form of streams of compressed data packets.<br>
		The adpcm compression algorithm (see kvi_adpcm.cpp/h in the distribution)
		is fast and simple; it uses a 1:4 compression rate and compresses 16 bit samples.<br>
		In my implementation, I have chosen to compress 2048 byte packets of audio data (1024 16 bit samples),
		obtaining 512 byte compressed data packets.<br>
		So one of the ends starts sending the audio data packets and the other receives and places
		it in a queue (this is an optimisation for common (slow) connections: lose time but gain data integrity)
		after uncompressing it.<br>
		The receiver queues packets until a message-termination packet is received.<br>
		The termination packet is a 512 byte packet with at least 5 first bytes filled with zeros.<br>
		Yes, it is not so "technical" but it works excellently (this packet should be
		treated as a terminator and ignored, as the other 507 bytes are meaningless (We could use it as a "roger-beep" ? :) ).<br>
		Once the terminator has been received the playback can start on the other side dequeuing
		the uncompressed audio data.<br>
		The current implementation is for half-duplex audio cards but it could be easily ported to
		take advantage of full-duplex cards.<br>
*/

/**
 * ============ KviDccChat ============
 */
KviDccVoice::KviDccVoice(KviFrame *lpFrm, const char *name)
	: KviWindow(name, KVI_WND_TYPE_VOICE, lpFrm)
{
	m_pSplitter   = new QSplitter(QSplitter::Horizontal, this);
	m_pSplitter->setOpaqueResize();
	m_pView       = new KviIrcView(m_pSplitter, lpFrm, this);
	m_pTalkButton = new KviPushButton(_i18n_("&Talk"), this);
	m_pTalkButton->setEnabled(false);
	connect(m_pTalkButton, SIGNAL(pressed()),  this, SLOT(talkPressed()));
	connect(m_pTalkButton, SIGNAL(released()), this, SLOT(talkReleased()));
	m_pStatLabel  = new KviLabel(_i18n_("No connection"), this);
	m_pStatLabel->setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
	m_thread      = 0;

	m_pData = new KviDccVoiceData();
	m_pData->parent                      = this;
	m_pData->soundDevice                 = g_pOptions->m_szSoundDevice.ptr();
	m_pData->soundFd                     = -1;
	m_pData->fullDuplex                  = g_pOptions->m_bFullDuplexSoundDevice;
	m_pData->state                       = KVI_DCC_VOICE_STATE_IDLE;
	m_pData->prevState                   = KVI_DCC_VOICE_STATE_IDLE;
	m_pData->sendHead                    = 0;
	m_pData->sendTail                    = 0;
	m_pData->recvHead                    = 0;
	m_pData->recvTail                    = 0;
	m_pData->lastSentSize                = 0;
	m_pData->completeBuffers             = 0;
	m_pData->lastReadSize                = 0;
	m_pData->tempSample                  = 0;
	m_pData->recvQueueCount              = 0;
	m_pData->use090CompatibleCompression = g_pOptions->m_b090CompatibleVoiceCompression;
	m_pData->uPortToListenOn             = 0;
	m_pData->bPlaying                    = false;
	m_pData->playMutex                   = new KviMutex();

	memset(m_pData->soundReadBuffer,  '\0', KVI_READ_BUF_SIZE_IN_CHARS);
	memset(m_pData->soundWriteBuffer, '\0', KVI_WRITE_BUF_SIZE_IN_CHARS);

	m_szLocalNick = m_pFrm->m_global.szCurrentNick;
	m_szLocalMask = m_pFrm->m_global.szCurrentMaskFromServer;
	if( m_szLocalNick.isEmpty() ) m_szLocalNick = _i18n_("me");
	if( m_szLocalMask.isEmpty() ) m_szLocalMask = _i18n_("me!me@localhost");
}

/**
 * ============ ~KviDccChat ============
 */
KviDccVoice::~KviDccVoice()
{
	abortThread();
	if( m_pData ) {
		if( m_pData->soundFd != -1 ) close(m_pData->soundFd);
		if( m_pData->tempSample ) {
			delete m_pData->tempSample;
			m_pData->tempSample = 0;
		}
		if( m_pData->playMutex ) {
			delete m_pData->playMutex;
			m_pData->playMutex = 0;
		}
		while( m_pData->sendHead ) {
			SampleStruct *s = m_pData->sendHead;
			m_pData->sendHead = s->next;
			delete s;
		}
		while( m_pData->recvHead ) {
			SampleStruct *s = m_pData->recvHead;
			m_pData->recvHead = s->next;
			m_pData->recvQueueCount--;
			delete s;
		}
		__range_valid(m_pData->recvQueueCount == 0);
		delete m_pData;
		m_pData = 0;
	}
}

void KviDccVoice::abortThread()
{
	if( !m_thread )
		return;

	m_thread->stop();
	m_thread->wait();
	delete m_thread;
	m_thread = 0;
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviDccVoice::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_VOICE];
}

/**
 * =============== applyOptions ================
 */
void KviDccVoice::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages, false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);

	if( !g_pOptions->m_pixLabelsBack->isNull() )
		m_pStatLabel->setBackgroundPixmap(*(g_pOptions->m_pixLabelsBack));
	else
		m_pStatLabel->setBackgroundColor(g_pOptions->m_clrLabelsBack);
	QPalette pal(m_pStatLabel->palette());
	pal.setColor(QPalette::Normal, QColorGroup::Text, g_pOptions->m_clrLabelsActiveFore);
	m_pStatLabel->setPalette(pal);

	m_pStatLabel->setFont(g_pOptions->m_fntLabels);
}

bool KviDccVoice::event(QEvent *e)
{
	if( e->type() != QEvent::User )
		return KviWindow::event(e);
	KviDccEvent *ev = (KviDccEvent *) e;
	switch( ev->m_type ) {
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR, ev->m_dataString.ptr());
			m_pStatLabel->setText(_i18n_("No connection"));
			m_pTalkButton->setEnabled(false);
			abortThread();
			// The thread is dead now... m_pData is safe.
			output(KVI_OUT_DCCINFO,
				_i18n_("Connection terminated (%s:%d)"),
				m_pData->szAddress.ptr(), m_pData->uPort
			);
			break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO, ev->m_dataString.ptr());
			break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(ev->m_dataString.ptr(), ev->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO, _i18n_("Sent DCC voice request, waiting for reply..."));
			break;
		case KVI_DCC_EVENT_ENABLETALK:
			m_pTalkButton->setEnabled(true);
			break;
		case KVI_DCC_EVENT_DISABLETALK:
			m_pTalkButton->setEnabled(false);
			break;
		case KVI_DCC_EVENT_BUFFEREDTIME:
			m_pStatLabel->setText(_CHAR_2_QSTRING(ev->m_dataString.ptr()));
			break;
		case KVI_DCC_EVENT_CONNECTIONESTABLISHED:
			m_pStatLabel->setText(_i18n_("Connected"));
			break;
		default:
			return KviWindow::event(e);
			break;
	}
	return true;
}

void KviDccVoice::talkPressed()
{
	if( !m_thread ) return;

	m_pData->playMutex->lock();
	if( !m_pData->bPlaying && !m_pData->fullDuplex )
		m_pData->state = KVI_DCC_VOICE_STATE_TALK;
	m_pData->playMutex->unlock();
}

void KviDccVoice::talkReleased()
{
	if( !m_thread ) return;

	if( m_pData->state == KVI_DCC_VOICE_STATE_TALK )
		m_pData->state = KVI_DCC_VOICE_STATE_IDLE;
}

/**
 * ================ resizeEvent ===============
 */
void KviDccVoice::resizeEvent(QResizeEvent *)
{
	int btnSize = m_pTalkButton->sizeHint().height();
	m_pStatLabel->setGeometry(0, 0, width() >> 1, btnSize);
	m_pTalkButton->setGeometry(width() >> 1, 0, width() >> 1, btnSize);
	m_pSplitter->setGeometry(0, btnSize, width(), height() - btnSize);
}

int KviDccVoice::checkSoundCard(KviDccVoiceData *dcc)
{
	int iError = KviDccVoiceThread::openSoundCard(dcc, true);
	if( dcc->soundFd != -1 )
		close(dcc->soundFd);
	dcc->soundFd = -1;
	return iError;
}

// =====================================================================================================
//
// Main routines for the slave DCC Chat thread
//
void KviDccVoice::acceptDccRequest(
	const char *nick, const char *username, const char *host, unsigned long uAddress, unsigned short uPort)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s", nick, username, host);
	// Called from KviDccManager
	// A user sent us a CTCP DCC, and we accepted it.
	// Connect to that host.
	if( m_thread ) {
		debug("WARNING: acceptDccRequest was called twice!");
		return;
	}

	int iError = checkSoundCard(m_pData);
	if( iError != KVI_ERROR_Success ) {
		output(KVI_OUT_DCCERROR,
			_i18n_("Sound card check failed (%s). Cannot accept DCC voice"),
			kvi_getErrorString(iError)
		);
		return;
	}

	m_pData->nick      = nick;
	m_pData->username  = username;
	m_pData->host      = host;
	m_pData->uAddress  = uAddress;
	struct in_addr addr;
	addr.s_addr = uAddress;
	kvi_binaryIpToString(addr, m_pData->szAddress);
	m_pData->uPort     = uPort;

	// Create the slave thread...
	m_thread = new KviDccVoiceAcceptThread(m_pData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, _i18n_("Cannot create slave thread. DCC voice failed"));
}

void KviDccVoice::requestDcc(const char *nick, const char *username, const char *host)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s", nick, username, host);
	// Called from KviDccManager
	// Request a DCC connection to a user
	if( m_thread ) {
		debug("WARNING: requestDcc was called twice!");
		return;
	}

	int iError = checkSoundCard(m_pData);
	if( iError != KVI_ERROR_Success ) {
		output(KVI_OUT_DCCERROR,
			_i18n_("Sound card check failed (%s). Cannot request DCC voice"),
			kvi_getErrorString(iError)
		);
		return;
	}

	m_pData->nick            = nick;
	m_pData->username        = username;
	m_pData->host            = host;
	m_pData->uAddress        = m_pFrm->m_pSocket->getSockAddress();
	m_pData->uPortToListenOn = m_pFrm->m_pDccManager->getDccSendListenPort();

	if( g_pOptions->m_bUseUserDefinedIpForDccRequests || (m_pData->uAddress == 0) ) {
		struct in_addr addr;
		if( kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(), &addr) ) {
			output(KVI_OUT_DCCINFO, _i18n_("Using %s as local host address"), g_pOptions->m_szDccLocalIpAddress.ptr());
			m_pData->uAddress = addr.s_addr;
		} else {
			output(KVI_OUT_DCCERROR,
				_i18n_("IP address %s is invalid: using standard IP address"),
				g_pOptions->m_szDccLocalIpAddress.ptr()
			);
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}
	if( m_pData->uAddress == 0 ) {
		outputNoFmt(KVI_OUT_DCCERROR, _i18n_("Cannot resolve local host. DCC voice failed"));
		return;
	}
	m_pData->szAddress = _i18n_("unknown");
	m_pData->uPort     = 0;

	// Create the slave thread...
	m_thread = new KviDccVoiceRequestThread(m_pData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, _i18n_("Cannot create slave thread. DCC voice failed"));
}

#include "m_kvi_dcc_voice.moc"
