// =============================================================================
//
//      --- kvi_locale.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviLocale"

#define _KVI_LOCALE_CPP_

#include "kvi_application.h"
#include "kvi_string.h"

static KviStr g_szLang("en"); // Default locale name

char *kvi_getLocaleName()
{
	return g_szLang.ptr();
}

#include "kvi_settings.h"
#ifdef COMPILE_LOCALE_STUFF

#include <qdict.h>
#include <qfile.h>

#include "kvi_debug.h"
#include "kvi_locale.h"

/////////////////////////////////////////////////////////////////////////////////////////////////////
//
//   The following code was extracted and adapted from gettext.h and gettextP.h
//   from the GNU gettext package.
//
//   Internal header for GNU gettext internationalization functions.
//   Copyright (C) 1995, 1997 Free Software Foundation, Inc.
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2, or (at your option)
//   any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU Library General Public
//   License along with the GNU C Library; see the file COPYING.LIB.  If not,
//   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>

#if HAVE_LIMITS_H || _LIBC
	#include <limits.h>
#endif

/* The magic number of the GNU message catalog format.  */
#define KVI_LOCALE_MAGIC 0x950412de
#define KVI_LOCALE_MAGIC_SWAPPED 0xde120495

/* Revision number of the currently used .mo (binary) file format.  */
#define MO_REVISION_NUMBER 0

/* The following contortions are an attempt to use the C preprocessor
   to determine an unsigned integral type that is 32 bits wide.  An
   alternative approach is to use autoconf's AC_CHECK_SIZEOF macro, but
   doing that would require that the configure script compile and *run*
   the resulting executable.  Locally running cross-compiled executables
   is usually not possible.  */

#if __STDC__
	#define UINT_MAX_32_BITS 4294967295U
#else
	#define UINT_MAX_32_BITS 0xFFFFFFFF
#endif

/* If UINT_MAX is not defined, assume it is a 32-bit type.
   This should be valid for all systems GNU cares about because
   that does not include 16-bit systems, and only modern systems
   (that certainly have <limits.h>) have 64+-bit integral types.  */

#ifndef UINT_MAX
	#define UINT_MAX UINT_MAX_32_BITS
#endif

#if UINT_MAX == UINT_MAX_32_BITS
	typedef unsigned nls_uint32;
#else
	#if USHRT_MAX == UINT_MAX_32_BITS
		typedef unsigned short nls_uint32;
	#else
		#if ULONG_MAX == UINT_MAX_32_BITS
			typedef unsigned long nls_uint32;
		#else
				/* The following line is intended to throw an error. */
				/* Using #error is not portable enough. :)  */
				"Cannot determine unsigned 32-bit data type."
		#endif
	#endif
#endif


/* Header for binary .mo file format.  */
struct GnuMoFileHeader
{
	/* The magic number.  */
	nls_uint32 magic;
	/* The revision number of the file format.  */
	nls_uint32 revision;
	/* The number of strings pairs.  */
	nls_uint32 nstrings;
	/* Offset of table with start offsets of original strings.  */
	nls_uint32 orig_tab_offset;
	/* Offset of table with start offsets of translation strings.  */
	nls_uint32 trans_tab_offset;
	/* Size of hashing table.  */
	nls_uint32 hash_tab_size;
	/* Offset of first hashing entry.  */
	nls_uint32 hash_tab_offset;
};

struct GnuMoStringDescriptor
{
	/* Length of addressed string.  */
	nls_uint32 length;
	/* Offset of string in file.  */
	nls_uint32 offset;
};


static inline nls_uint32 kvi_swap_32(nls_uint32 i)
{
	return (i << 24) | ((i & 0xff00) << 8) | ((i >> 8) & 0xff00) | (i >> 24);
}

#define KVI_SWAP_IF_NEEDED(flag,value) (flag ? kvi_swap_32(value) : (value))

///////////////////////////////////////////////////////////////////////////////////////////////
//   End of gettext.h & gettextP.h
///////////////////////////////////////////////////////////////////////////////////////////////

/**
 * This function attempts to determine the current locale
 * and then load the corresponding translation file
 * from the KVIrc locale directory.
 * Returns true if the locale was correctly set
 * i.e. the locale is C or POSIX (no translation needed)
 *     or the locale is correctly defined and the
 *     translation map was loaded successfully
 */

#include <stdlib.h> // For getenv

static QDict<KviStr> *g_pMessages   = 0;
static KviTranslator *g_pTranslator = 0;

static int somePrimeNumbers[90] =
{
	 257,  521,  769, 1031, 1087, 1091, 1103, 1117, 1123, 1151, // Incomplete *.mo files
	1163, 1171, 1181, 1193, 1201, 1213, 1217, 1223, 1229, 1231, // Complete *.mo files
	1237, 1249, 1259, 1277, 1283, 1289, 1291, 1297, 1307, 1319,
	1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423, 1433,
	1447, 1459, 1471, 1481, 1493, 1511, 1523, 1531, 1543, 1553,
	1567, 1571, 1583, 1597, 1609, 1619, 1627, 1637, 1657, 1667, // Too big for KVIrc *.mo files
	1693, 1709, 1721, 1733, 1741, 1753, 1777, 1789, 1811, 1831,
	1907, 2069, 2111, 2221, 2309, 2441, 2531, 2617, 2731, 2837,
	2903, 3121, 3329, 3331, 3767, 4127, 5051, 6089, 7039, 9973
};

int kvi_getFirstBiggerPrime(int number)
{
	for( int i = 0; i < 90; i++ ) {
		if( somePrimeNumbers[i] >= number ) return somePrimeNumbers[i];
	}
	return 9973; // Error!
}

bool kvi_initLocale(KviApplication *app, const char *localeDir)
{
	// Find the current locale
	g_szLang = getenv("LANG");
	g_szLang.toLower();
	g_szLang.stripWhiteSpace();

	if( g_szLang.isEmpty() ) {
		g_szLang = "en";
		return true; // No locale specified... use default (english is built-in)
	}

	KviStr tmpLang = g_szLang;

	if( tmpLang.findFirstIdx('_') != -1 ) {
		// Oops! The locale is set in the "strange" form (it_IT or things like that).
		// We'll use also the first part
		tmpLang.cutFromFirst('_', true);
	}

	if( kvi_strEqualCI(tmpLang.ptr(),"c")  || kvi_strEqualCI(tmpLang.ptr(),"en") ||
		kvi_strEqualCI(tmpLang.ptr(),"uk") || kvi_strEqualCI(tmpLang.ptr(),"us") ||
		kvi_strEqualCI(tmpLang.ptr(),"gb") || kvi_strEqualCI(tmpLang.ptr(),"posix") ) {
		g_szLang = "en";
		return true; // English is built-in
	}

	__debug_1arg("Locale is set to %s", g_szLang.ptr());

	// Compute the name of the messages file to load: first lookup with the real LANG variable contents
	KviStr messagesFile = localeDir;
	messagesFile.append(g_szLang);
	messagesFile.append(".mo");

	// Try to load the header
	QFile f(messagesFile.ptr());
	if( !f.open(IO_ReadOnly) ) {
		debug("KviLocale: failed to read messages file %s: probably does not exist", messagesFile.ptr());
		// Ok.. maybe try the hacked contents... (just the first part)
		messagesFile = localeDir;
		messagesFile.append(tmpLang);
		messagesFile.append(".mo");
		f.setName(messagesFile.ptr());

		if( !f.open(IO_ReadOnly) ) {
			debug("KviLocale: failed to read messages file %s: probably does not exist", messagesFile.ptr());
			return false;
		}
	}

	GnuMoFileHeader hdr;
	if( f.readBlock((char *) &hdr, sizeof(GnuMoFileHeader)) < (int) sizeof(GnuMoFileHeader) ) {
		debug("KviLocale: failed to read header of %s", messagesFile.ptr());
		f.close();
		return false;
	}

	bool bMustSwap = false;
	if( hdr.magic != KVI_LOCALE_MAGIC ) {
		if( hdr.magic == KVI_LOCALE_MAGIC_SWAPPED ) {
			debug("KviLocale: swapped magic for file %s: swapping data too", messagesFile.ptr());
			bMustSwap = true;
		} else {
			debug("KviLocale: bad locale magic for file %s: not a *.mo file?", messagesFile.ptr());
			f.close();
			return false;
		}
	}

	if( KVI_SWAP_IF_NEEDED(bMustSwap, hdr.revision) != MO_REVISION_NUMBER ) {
		debug("KviLocale: invalid *.mo file revision number for file %s", messagesFile.ptr());
		f.close();
		return false;
	}

	int numberOfStrings = KVI_SWAP_IF_NEEDED(bMustSwap, hdr.nstrings);
	if( numberOfStrings <= 0 ) {
		debug("KviLocale: no translated messages found in file %s", messagesFile.ptr());
		f.close();
		return false;
	}
	if( numberOfStrings >= 9972 ) {
		debug("Number of strings too big... sure that it is a KVIrc catalog file?");
		numberOfStrings = 9972;
	}
	// Return back
	f.at(0);

	unsigned int fSize = f.size();
	char *buffer = new char[fSize];

	if( f.readBlock(buffer, fSize) < (int) fSize ) {
		debug("KviLocale: error while reading the translation file %s", messagesFile.ptr());
		if( buffer ) delete buffer;
		f.close();
		return false;
	}

	// Check for broken *.mo files
	if( fSize < (24 + (sizeof(GnuMoStringDescriptor) * numberOfStrings)) ) {
		debug("KviLocale: broken translation file %s (too small for all descriptors)", messagesFile.ptr());
		if( buffer ) delete buffer;
		f.close();
		return false;
	}

	GnuMoStringDescriptor *origDescriptor  = (GnuMoStringDescriptor *) (buffer + KVI_SWAP_IF_NEEDED(bMustSwap, hdr.orig_tab_offset));
	GnuMoStringDescriptor *transDescriptor = (GnuMoStringDescriptor *) (buffer + KVI_SWAP_IF_NEEDED(bMustSwap, hdr.trans_tab_offset));

	// Check again for broken *.mo files
	int expectedFileSize = KVI_SWAP_IF_NEEDED(bMustSwap, transDescriptor[numberOfStrings - 1].offset) +
	                       KVI_SWAP_IF_NEEDED(bMustSwap, transDescriptor[numberOfStrings - 1].length);

	if( fSize < (unsigned int) expectedFileSize ) {
		debug("KviLocale: broken translation file %s (too small for all the message strings)", messagesFile.ptr());
		if( buffer ) delete buffer;
		f.close();
		return false;
	}

	// Ok... we can run now
	int dictSize = kvi_getFirstBiggerPrime(numberOfStrings);
	g_pMessages = new QDict<KviStr> (dictSize);
	g_pMessages->setAutoDelete(true);

	__debug_2arg("Number of strings : %d, dict size %d", numberOfStrings, dictSize);

	for( int i = 0; i < numberOfStrings; i++ ) {
		KviStr original((char *) (buffer + KVI_SWAP_IF_NEEDED(bMustSwap, origDescriptor[i].offset)), KVI_SWAP_IF_NEEDED(bMustSwap, origDescriptor[i].length));
		// In some (or all?) *.mo files the first string
		// is zero bytes long and the translated one contains
		// information about the translation
		if( original.len() == 0 ) continue;
		KviStr translated((char *) (buffer + KVI_SWAP_IF_NEEDED(bMustSwap, transDescriptor[i].offset)), KVI_SWAP_IF_NEEDED(bMustSwap, transDescriptor[i].length));
		g_pMessages->insert(original.ptr(), new KviStr(translated));
	}

	if( buffer ) delete buffer;
	f.close();

	g_pTranslator = new KviTranslator(app, "kvirc_translator");
	app->installTranslator(g_pTranslator);

	return true;
}

void kvi_destroyLocale(KviApplication *app)
{
	if( g_pMessages ) {
		delete g_pMessages;
		g_pMessages = 0;
	}

	if( g_pTranslator ) {
		app->removeTranslator(g_pTranslator);
		delete g_pTranslator;
		g_pTranslator = 0;
	}
}

const char *kvi_translate(const char *text)
{
	if( g_pMessages ) {
		KviStr *aux = g_pMessages->find(text);
		return aux ? aux->ptr() : text;
	}
	return text;
}


KviTranslator::KviTranslator(QObject *parent, const char *name)
	: QTranslator(parent, name)
{
	// Nothing here
}

KviTranslator::~KviTranslator()
{
	// Nothing here
}

QString KviTranslator::find(const char *, const char *message) const
{
	if( g_pMessages ) {
		KviStr *aux = g_pMessages->find(message);
		return aux ? _CHAR_2_QSTRING(aux->ptr()) : QString::null;
	}
	return QString::null;
}

#include "m_kvi_locale.moc"

#else // COMPILE_LOCALE_STUFF

bool kvi_initLocale(KviApplication *, const char *)
{
	g_szLang = "en";
	debug("This program was build without locale support.");
	return false;
}

void kvi_destroyLocale(KviApplication *)
{
	// Nothing here
}

#endif // COMPILE_LOCALE_STUFF
