// =============================================================================
//
//      --- kvi_input.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviInput"

#include <qclipboard.h>
#include <qcursor.h>
#include <qdragobject.h>
#include <qpainter.h>

#define _KVI_INPUT_CPP_

#include "kvi_app.h"
#include "kvi_colorwindow.h"
#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_user.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"
#include "kvi_statusbar.h"
#include "kvi_strsub.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"
#include "kvi_window.h"

/*
	@quickhelp: KviInput
	@widget: Commandline input

	This is the commandline input widget.<br>
	<docsubtitle>Default key bindings:</docsubtitle><br>
	Ctrl+B: Inserts the 'bold' mIRC control character<br>
	Ctrl+K: Inserts the 'color' mIRC control character<br>
	Ctrl+R: Inserts the 'reverse' mIRC control character<br>
	Ctrl+U: Inserts the 'underline' mIRC control character<br>
	Ctrl+O: Inserts the 'reset' mIRC control character<br>
	Ctrl+C: Copy selected text to clipboard<br>
	Ctrl+X: Cuts the selected text<br>
	Ctrl+V: Paste the clipboard contents (same as middle mouse click)<br>
	CursorUp: Moves backward in the commandline history<br>
	CursorDown: Moves forward in the commandline history<br>
	CursorRight: Moves cursor right<br>
	CursorLeft: Moves cursor left :)<br>
	Shift+CursorLeft: Moves selection left<br>
	Shift+RightCursor: Moves selection right<br>
	Tab: Nick completion or function/command completion (see below)<br>
	Shift+Tab: Mask completion or function/command completion (see below)<br>
	Alt+&lt;numeric_sequence&gt;: Inserts the character by ASCII code<br>
	<example>
	Alt+32: Inserts the ASCII character 32 = space
	Alt+00032: Same as above:)
	Alt+13: Inserts the Carriage Return (CR) control character
	Alt+77: Inserts the ASCII character 77: M
	</example>
	Take a look also at the <a href="shortcuts.kvihelp">global shortcuts</a> description.<br>
	If you drop a file on this widget, a <a href="parse.kvihelp">/PARSE &lt;filename&gt;</a> will be executed.<br>
	You can enable word substitution in the misc options dialog.<br>
	For example, if you choose to substitute "afaik" with "As far as I know", <br>
	when you will type "afaik" somewhere in the commandline, and then
	press space or return, that word will be replaced with "As far as I know".<br>
	Experiment with it :)<br>
	The Tab key activates the completion of the current word.<br>
	If the word start with a '/' sign, it is treated as a command to be completed,
	if it starts with a '$' sign it is treated as a function or identifier to be completed,
	otherwise it is treated as a nickname to be completed.<br>
	<example>
		/ec&lt;tab&gt; will produce /echo&lt;space&gt
		/echo $loca&lt;tab&gt; will produce /echo $localhost
	</example>
	Multiple matches are listed in the view window and the word is completed
	to the common part of all the matches.<br>
	<example>
		$sel&lt;tab;&gt; will find multiple matches and produce $selected
	</example>
	Experiment with that too :)
*/

// This comes from kvi_app.cpp:
extern KviColorWindow  *g_pColorWindow;
extern KviPopupMenu    *g_pInputPopup;
extern KviEventManager *g_pEventManager;

// This comes from the KviIrcView class
extern const char *getColorBytes(const char *data_ptr, char *byte_1, char *byte_2);

// This one is our own
static int g_iInputFontCharWidth[256];

/**
 * =============== Constructor ==============
 */
KviInput::KviInput(KviWindow *parent, KviUserParser *pParser, KviUserListBox *listBox)
	: QWidget(parent, "input")
{
	m_szTextBuffer               = "";                             // Main data buffer
	m_szSaveTextBuffer           = "";                             // Save the data while browsing history
	m_pMemBuffer                 = new QPixmap(width(), height()); // Pixmap buffer for painting
	m_iCursorPosition            = 0;                              // Index of the char AFTER the cursor
	m_iFirstVisibleChar          = 0;                              // Index of the first visible character
	m_iSelectionBegin            = -1;                             // Index of the first char in the selection
	m_iSelectionEnd              = -1;                             // Index of the last char in the selection
	m_bCursorOn                  = false;                          // Cursor state
	m_iCursorTimer               = 0;                              // Timer that iverts the cursor state
	m_iDragTimer                 = 0;                              // Timer for drag selection updates
	m_iLastCursorXPosition       = KVI_INPUT_BORDER;               // Calculated in paintEvent
	m_iSelectionAnchorChar       = -1;                             // Character clicked at the start of the selection process
	m_pHistory                   = new QPtrList<QString>;          // History buffer
	m_pHistory->setAutoDelete(true);
	m_iCurHistoryIdx             = -1;                             // No data in the history
	m_bUpdatesEnabled            = true;
	m_pUserParser                = pParser;
	m_pParent                    = parent;
	m_szAltKeyCode               = "";
	m_pListBoxToControl          = listBox;
	m_iLastCompletionIndex       = -1;
	m_szStringBeforeSubstitution = "";
	m_bLastEventWasASubstitution = false;
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);
	setAcceptDrops(true);
	recalcFontMetrics();
}

/**
 * ================= Destructor ================
 */
KviInput::~KviInput()
{
	if( m_pHistory ) {
		delete m_pHistory;
		m_pHistory = 0;
	}
	if( m_iCursorTimer )
		killTimer(m_iCursorTimer);
	killDragTimer();
	if( m_pMemBuffer ) {
		delete m_pMemBuffer;
		m_pMemBuffer = 0;
	}
}

void KviInput::recalcFontMetrics()
{
	// TODO: this should be shared between all input widgets and calculated
	// only once per "Apply Options" operation
	QFontMetrics fm(g_pOptions->m_fntInput);
	unsigned short i;
	for( i = 1; i < 32; i++ ) {
		QChar c = getSubstituteChar(i);
		if( i == KVI_TEXT_TAB )
			g_iInputFontCharWidth[i] = KVI_TAB_WIDTH * fm.width(c);
		else
			g_iInputFontCharWidth[i] = fm.width(c);
		if( c != QChar(i) )
			g_iInputFontCharWidth[i] += 4;
	}
	for( i = 32; i < 256; i++ )
		g_iInputFontCharWidth[i] = fm.width(QChar(i));
}

void KviInput::applyOptions()
{
	update();
	recalcFontMetrics();
}

void KviInput::dragEnterEvent(QDragEnterEvent *e)
{
	if( QUriDrag::canDecode(e) ) {
		e->accept(true);
		m_pParent->m_pFrm->m_pStatusBar->tempText(__tr("Drop the file to /PARSE it"), 5000);
	} else e->accept(false);
}

void KviInput::dropEvent(QDropEvent *e)
{
	QStringList list;
	if( QUriDrag::decodeLocalFiles(e, list) ) {
		if( !list.isEmpty() ) {
			QStringList::ConstIterator it = list.begin();
			for( ; it != list.end(); ++it ) {
				QString tmp = *it;
				tmp.prepend("/PARSE ");
				m_pUserParser->parseUserCommand(tmp, m_pParent);
			}
		}
	}
}

int KviInput::heightHint() const
{
	return g_pOptions->m_iInputFontLineSpacing + (KVI_INPUT_BORDER << 1);
}

#define KVI_INPUT_DEF_BACK 100
#define KVI_INPUT_DEF_FORE 101

void KviInput::paintEvent(QPaintEvent *)
{
	if( !isVisible() ) return;

	int widgetWidth       = width();
	int widgetHeight      = height();

	static QPen pen(colorGroup().dark(), 1, Qt::SolidLine, Qt::FlatCap, Qt::MiterJoin);
	QPainter pa(m_pMemBuffer);
	pa.setPen(pen);

	// Draw the background
	if( g_pOptions->m_pixInputBack->isNull() ) {
		pa.fillRect(0, 0, widgetWidth, widgetHeight, g_pOptions->m_clrInputBack);
	} else {
		QPoint pnt = mapToGlobal(QPoint(0, 0));
		pa.drawTiledPixmap(0, 0, widgetWidth, widgetHeight, *(g_pOptions->m_pixInputBack), pnt.x(), pnt.y());
	}
	pa.setFont(g_pOptions->m_fntInput);
	QFontMetrics fm(pa.fontMetrics());

	int curXPos      = KVI_INPUT_BORDER;
	int maxXPos      = widgetWidth - KVI_INPUT_BORDER;
	m_iCurBack       = KVI_INPUT_DEF_BACK; // Transparent
	m_iCurFore       = KVI_INPUT_DEF_FORE; // Normal foreground color
	m_bCurBold       = false;
	m_bCurUnderline  = false;

	int bottom       = widgetHeight - KVI_INPUT_BORDER;
	int textBaseline = bottom - fm.descent();
	int top          = bottom - fm.height();

	runUpToTheFirstVisibleChar();
	int charIdx = m_iFirstVisibleChar;

	pa.setClipRect(
		KVI_INPUT_BORDER, KVI_INPUT_BORDER,
		widgetWidth - (KVI_INPUT_BORDER << 1), widgetHeight - (KVI_INPUT_BORDER << 1) + 1
	);

	// Control the selection state
	if( (m_iSelectionEnd < m_iSelectionBegin) || (m_iSelectionEnd == -1) || (m_iSelectionBegin == -1) ) {
		m_iSelectionEnd   = -1;
		m_iSelectionBegin = -1;
	}

	while( ((unsigned int) charIdx < m_szTextBuffer.length()) && (curXPos < maxXPos) ) {
		extractNextBlock(charIdx, fm, curXPos, maxXPos);
		if( m_bControlBlock ) {
			// Only a control char
			pa.setPen(g_pOptions->m_clrInputCursor);

			QString s = getSubstituteChar(m_szTextBuffer[charIdx].unicode());
			// The block width is 4 pixels more than the actual character
			pa.drawText(curXPos + 2, textBaseline, s, 1);
			pa.drawLine(curXPos, top, curXPos + m_iBlockWidth - 1, top);
			pa.drawLine(curXPos, top, curXPos, bottom);
			pa.drawLine(curXPos, bottom, curXPos + m_iBlockWidth, bottom);
			pa.drawLine(curXPos + m_iBlockWidth - 1, top, curXPos + m_iBlockWidth - 1, bottom);
		} else {
			drawTextBlock(&pa, fm, curXPos, textBaseline, charIdx, m_iBlockLen, m_iBlockWidth);
		}
		curXPos += m_iBlockWidth;
		charIdx += m_iBlockLen;
	}

	if( (m_iSelectionBegin != -1) && (m_iSelectionEnd >= m_iFirstVisibleChar) ) {
		int iSelStart = m_iSelectionBegin;
		if( iSelStart < m_iFirstVisibleChar )
			iSelStart = m_iFirstVisibleChar;
		int xLeft  = xPositionFromCharIndex(iSelStart);
		int xRight = xPositionFromCharIndex(m_iSelectionEnd + 1);

		pa.setRasterOp(Qt::NotROP);
		pa.fillRect(xLeft, KVI_INPUT_BORDER, xRight - xLeft, widgetWidth - (KVI_INPUT_BORDER << 1), Qt::black);
		pa.setRasterOp(Qt::CopyROP);
	}

	pa.setClipping(false);
	// Now the cursor
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iBlockLen = m_iFirstVisibleChar;

	while( m_iBlockLen < m_iCursorPosition ) {
		QChar c = m_szTextBuffer.at(m_iBlockLen);
		m_iLastCursorXPosition += (c.unicode() < 256) ? g_iInputFontCharWidth[c.unicode()] : fm.width(c);
		m_iBlockLen++;
	}

	if( m_bCursorOn ) {
		pa.setPen(g_pOptions->m_clrInputCursor);
		pa.drawLine(
			m_iLastCursorXPosition, KVI_INPUT_BORDER - 1, m_iLastCursorXPosition, (widgetHeight - KVI_INPUT_BORDER) + 2
		);
	} else {
		pa.setPen(g_pOptions->m_clrInputFore);
	}

	pa.drawLine(m_iLastCursorXPosition - 2, KVI_INPUT_BORDER - 2, m_iLastCursorXPosition + 3, KVI_INPUT_BORDER - 2);
	pa.drawLine(m_iLastCursorXPosition - 3, KVI_INPUT_BORDER - 3, m_iLastCursorXPosition + 4, KVI_INPUT_BORDER - 3);
	pa.drawLine(m_iLastCursorXPosition - 1, KVI_INPUT_BORDER - 1, m_iLastCursorXPosition + 2, KVI_INPUT_BORDER - 1);
	pa.drawLine(m_iLastCursorXPosition, KVI_INPUT_BORDER, m_iLastCursorXPosition + 1, KVI_INPUT_BORDER);
	// Need to draw the sunken rect around the view now
	pa.setPen(colorGroup().dark());
	pa.drawLine(0, 0, widgetWidth, 0);
	pa.drawLine(0, 0, 0, widgetHeight);
	pa.setPen(colorGroup().light());
	pa.drawLine(1, widgetHeight - 1, widgetWidth - 1, widgetHeight - 1);
	pa.drawLine(widgetWidth - 1, 1, widgetWidth - 1, widgetHeight);

	// Copy to the display
	bitBlt(this, 0, 0, m_pMemBuffer, 0, 0, widgetWidth, widgetHeight,Qt::CopyROP);
}

void KviInput::drawTextBlock(QPainter *pa, QFontMetrics &fm, int curXPos, int bottom, int charIdx, int len, int wdth)
{
	if( m_iCurFore == KVI_INPUT_DEF_FORE )
		pa->setPen(g_pOptions->m_clrInputFore);
	else {
		if( m_iCurFore > 16 )
			pa->setPen(g_pOptions->m_clrInputBack);
		else
			pa->setPen(*(g_pOptions->m_pMircColor[(unsigned char) m_iCurFore]));
	}
	if( m_iCurBack != KVI_INPUT_DEF_BACK ) {
		if( m_iCurBack > 16 ) {
			pa->fillRect(
				curXPos, height() - (KVI_INPUT_BORDER + fm.height()), wdth, fm.height(),
				g_pOptions->m_clrInputFore
			);
		} else {
			pa->fillRect(
				curXPos, height() - (KVI_INPUT_BORDER + fm.height()), wdth, fm.height(),
				*(g_pOptions->m_pMircColor[(unsigned char) m_iCurBack])
			);
		}
	}
	QString tmp = m_szTextBuffer.mid(charIdx, len);
	pa->drawText(curXPos, bottom, tmp, len);
	if( m_bCurBold )
		pa->drawText(curXPos + 1, bottom, tmp, len);
	if( m_bCurUnderline )
		pa->drawLine(curXPos, bottom + fm.descent(), curXPos + wdth, bottom + fm.descent());
}

inline QChar KviInput::getSubstituteChar(const char control_code)
{
	switch( control_code ) {
		case KVI_TEXT_COLOR:
			return QChar('K');
			break;
		case KVI_TEXT_BOLD:
			return QChar('B');
			break;
		case KVI_TEXT_RESET:
			return QChar('O');
			break;
		case KVI_TEXT_REVERSE:
			return QChar('R');
			break;
		case KVI_TEXT_UNDERLINE:
			return QChar('U');
			break;
		case KVI_TEXT_TAB:
			return QChar(' ');
			break;
		default:
			return QChar(control_code);
			break;
	}
}

void KviInput::extractNextBlock(unsigned int idx, QFontMetrics &fm, int curXPos, int maxXPos)
{
	m_iBlockLen   = 0;
	m_iBlockWidth = 0;

	QChar c = m_szTextBuffer[idx];
	if( (c.unicode() > 32 ) || (
		(c != QChar(KVI_TEXT_TAB))       &&
		(c != QChar(KVI_TEXT_COLOR))     &&
		(c != QChar(KVI_TEXT_BOLD))      &&
		(c != QChar(KVI_TEXT_UNDERLINE)) &&
		(c != QChar(KVI_TEXT_RESET))     &&
		(c != QChar(KVI_TEXT_REVERSE)))
	) {
		// Not a control code
		m_bControlBlock = false;
		while( (idx < m_szTextBuffer.length()) && (curXPos < maxXPos) ) {
			c = m_szTextBuffer[idx];
			if( (c.unicode() > 32) || (
				(c != QChar(KVI_TEXT_TAB))       &&
				(c != QChar(KVI_TEXT_COLOR))     &&
				(c != QChar(KVI_TEXT_BOLD))      &&
				(c != QChar(KVI_TEXT_UNDERLINE)) &&
				(c != QChar(KVI_TEXT_RESET))     &&
				(c != QChar(KVI_TEXT_REVERSE)))
			) {
				m_iBlockLen++;
				int xxx = (c.unicode() < 256 ? g_iInputFontCharWidth[c.unicode()] : fm.width(c));
				m_iBlockWidth += xxx;
				curXPos       += xxx;
				idx++;
			} else break;
		}
		return;
	} else {
		// Control code
		m_bControlBlock = true;
		m_iBlockLen = 1;
		m_iBlockWidth = g_iInputFontCharWidth[c.unicode()];
		switch( c.unicode() ) {
			case KVI_TEXT_BOLD:
				m_bCurBold    = !m_bCurBold;
				break;
			case KVI_TEXT_UNDERLINE:
				m_bCurUnderline = !m_bCurUnderline;
				break;
			case KVI_TEXT_RESET:
				m_iCurFore      = KVI_INPUT_DEF_FORE;
				m_iCurBack      = KVI_INPUT_DEF_BACK;
				m_bCurBold      = false;
				m_bCurUnderline = false;
				break;
			case KVI_TEXT_REVERSE: {
				char auxClr   = m_iCurFore;
				m_iCurFore    = m_iCurBack;
				m_iCurBack    = auxClr;
				break;
			}
			case KVI_TEXT_COLOR: {
				idx++;
				if( idx >= m_szTextBuffer.length() )
					return;
				unsigned char fore;
				unsigned char back;
				idx = getUnicodeColorBytes(m_szTextBuffer, idx, &fore, &back);

				if( fore != KVI_NOCHANGE ) {
					m_iCurFore = fore;
					if( back != KVI_NOCHANGE )
						m_iCurBack = back;
				} else {
					// Only a Ctrl+K
					m_iCurBack = KVI_INPUT_DEF_BACK;
					m_iCurFore = KVI_INPUT_DEF_FORE;
				}
				break;
			}
			case KVI_TEXT_TAB: // No need to do anything
				break;
			default: // Just ignore
				break;
		}
	}
}

void KviInput::runUpToTheFirstVisibleChar()
{
	register int idx = 0;
	while( idx < m_iFirstVisibleChar ) {
		unsigned short c = m_szTextBuffer[idx].unicode();
		if( c < 32 ) {
			switch( c ) {
				case KVI_TEXT_BOLD:
					m_bCurBold = !m_bCurBold;
					break;
				case KVI_TEXT_UNDERLINE:
					m_bCurUnderline = !m_bCurUnderline;
					break;
				case KVI_TEXT_RESET:
					m_iCurFore      = KVI_INPUT_DEF_FORE;
					m_iCurBack      = KVI_INPUT_DEF_BACK;
					m_bCurBold      = false;
					m_bCurUnderline = false;
					break;
				case KVI_TEXT_REVERSE: {
					char auxClr = m_iCurFore;
					m_iCurFore  = m_iCurBack;
					m_iCurBack  = auxClr;
					break;
				}
				case KVI_TEXT_COLOR: {
					idx++;
					if( (unsigned int) idx >= m_szTextBuffer.length() )
						return;
					unsigned char fore;
					unsigned char back;
					idx = getUnicodeColorBytes(m_szTextBuffer, idx, &fore, &back);
					if( fore != KVI_NOCHANGE )
						m_iCurFore = fore;
					else
						m_iCurFore = KVI_INPUT_DEF_FORE;
					if( back != KVI_NOCHANGE )
						m_iCurBack = back;
					else
						m_iCurBack = KVI_INPUT_DEF_BACK;
					break;
				}
				default:
					debug("[KviInput] Encountered invisible end of the string!");
					break;
			}
		}
		idx++;
	}
}

void KviInput::resizeEvent(QResizeEvent *)
{
	m_pMemBuffer->resize(width(), height());
}

void KviInput::mousePressEvent(QMouseEvent *e)
{
	if( e->button() & LeftButton ) {
		m_iCursorPosition = charIndexFromXPosition(e->pos().x());
		// Move the cursor to
		int anchorX = xPositionFromCharIndex(m_iCursorPosition);
		if( anchorX > (width() - KVI_INPUT_BORDER) )
			m_iFirstVisibleChar++;
		m_iSelectionAnchorChar = m_iCursorPosition;
		selectOneChar(-1);
		repaintWithCursorOn();
		killDragTimer();
		m_iDragTimer = startTimer(KVI_INPUT_DRAG_TIMEOUT);
	} else if( e->button() & RightButton ) {
		// Popup menu
		g_pInputPopup->clear();
		g_pInputPopup->insertItem(_i18n_("Copy"), this, SLOT(copy()), 0, 1);
		g_pInputPopup->setItemEnabled(1, hasSelection());
		g_pInputPopup->insertItem(_i18n_("Paste"), this, SLOT(paste()), 0, 2);
		g_pInputPopup->setItemEnabled(2, (g_pApp->clipboard()->text() != 0));
		g_pInputPopup->insertItem(_i18n_("Cut"), this, SLOT(cut()), 0, 3);
		g_pInputPopup->setItemEnabled(3, hasSelection());
		g_pInputPopup->insertSeparator();
		g_pInputPopup->insertItem(_i18n_("Select All"), this, SLOT(selectAll()), 0, 4);
		g_pInputPopup->setItemEnabled(4, !m_szTextBuffer.isEmpty());
		g_pInputPopup->insertItem(_i18n_("Clear"), this, SLOT(clear()), 0, 5);
		g_pInputPopup->setItemEnabled(5, !m_szTextBuffer.isEmpty());
		g_pInputPopup->popup(mapToGlobal(e->pos()));
	} else paste();
}

bool KviInput::hasSelection()
{
	return ((m_iSelectionBegin != -1) && (m_iSelectionEnd != -1));
}

void KviInput::copy()
{
	if( hasSelection() ) {
		g_pApp->clipboard()->setText(
			m_szTextBuffer.mid(m_iSelectionBegin, (m_iSelectionEnd - m_iSelectionBegin) + 1)
		);
	}
	repaintWithCursorOn();
}

void KviInput::moveCursorTo(int idx, bool bRepaint)
{
	if( idx < 0 ) idx = 0;
	if( (unsigned int) idx > m_szTextBuffer.length() )
		idx = m_szTextBuffer.length();
	if( idx > m_iCursorPosition) {
		while( m_iCursorPosition < idx ) {
			moveRightFirstVisibleCharToShowCursor();
			m_iCursorPosition++;
		}
	} else {
		m_iCursorPosition = idx;
		if( m_iFirstVisibleChar > m_iCursorPosition )
			m_iFirstVisibleChar = m_iCursorPosition;
	}
	if( bRepaint )
		repaintWithCursorOn();
}

void KviInput::cut()
{
	if( hasSelection() ) {
		m_szTextBuffer.remove(m_iSelectionBegin, (m_iSelectionEnd - m_iSelectionBegin) + 1);
		moveCursorTo(m_iSelectionBegin, false);
		selectOneChar(-1);
		repaintWithCursorOn();
	}
}

void KviInput::insertText(const QString &text)
{
	QString szText = text; // Make it non-const
	if( szText.isEmpty() )
		return;

	m_bUpdatesEnabled = false;
	cut();
	m_bUpdatesEnabled = true;

	if( szText.find('\n') == -1 ) {
		m_szTextBuffer.insert(m_iCursorPosition, szText);
		m_szTextBuffer.truncate(KVI_INPUT_MAX_BUFFER_SIZE);
		moveCursorTo(m_iCursorPosition + szText.length());
	} else {
		// Multiline paste; do not execute commands here
		QString szBlock;
		while( !szText.isEmpty() ) {
			int idx = szText.find('\n');
			if( idx != -1 ) {
				szBlock = szText.left(idx);
				szText.remove(0, idx + 1);
			} else {
				szBlock = szText;
				szText  = "";
			}

			m_szTextBuffer.insert(m_iCursorPosition,szBlock);
			m_szTextBuffer.truncate(KVI_INPUT_MAX_BUFFER_SIZE);

			unsigned int pos = 0;
			while( (pos < m_szTextBuffer.length()) && (m_szTextBuffer[pos] < 33) )
				pos++;
			if( (pos < m_szTextBuffer.length()) && (m_szTextBuffer[pos] == QChar('/')) )
				m_szTextBuffer.insert(pos, "\\");

			returnPressed(idx != -1);
		}
	}
}

void KviInput::paste()
{
	QString szText = g_pApp->getClipboardText();
	insertText(szText);
}

void KviInput::selectAll()
{
	if( m_szTextBuffer.length() > 0 ) {
		m_iSelectionBegin = 0;
		m_iSelectionEnd   = m_szTextBuffer.length() - 1;
	}
	end();
}

void KviInput::clear()
{
	m_szTextBuffer = "";
	selectOneChar(-1);
	home();
}

void KviInput::setText(const char *text)
{
	m_szTextBuffer = text;
	m_szTextBuffer.truncate(KVI_INPUT_MAX_BUFFER_SIZE);
	selectOneChar(-1);
	end();
}

void KviInput::mouseReleaseEvent(QMouseEvent *)
{
	copy();
	if( m_iDragTimer ) {
		m_iSelectionAnchorChar =-1;
		releaseMouse();
		killDragTimer();
	}
}

void KviInput::killDragTimer()
{
	if( m_iDragTimer ) {
		killTimer(m_iDragTimer);
		m_iDragTimer = 0;
	}
}

void KviInput::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_iCursorTimer ) {
		if( !hasFocus() || !isVisible() ) {
			killTimer(m_iCursorTimer);
			m_iCursorTimer = 0;
			m_bCursorOn    = false;
		} else m_bCursorOn = !m_bCursorOn;
		paintEvent(0);
	} else {
		// Drag timer
		handleDragSelection();
	}
}

void KviInput::handleDragSelection()
{
	if( m_iSelectionAnchorChar == -1 )
		return;

	QPoint pnt = mapFromGlobal(QCursor::pos());
	if( pnt.x() <= 0 ) {
		// Left side dragging
		if( m_iFirstVisibleChar > 0 )
			m_iFirstVisibleChar--;
		m_iCursorPosition = m_iFirstVisibleChar;
	} else if( pnt.x() >= width() ) {
		// Right side dragging; add a single character to the selection on the right
		if( (unsigned int) m_iCursorPosition < m_szTextBuffer.length() ) {
			moveRightFirstVisibleCharToShowCursor();
			m_iCursorPosition++;
		} // else: at the end of the selection; do not move anything
	} else {
		// Inside the window
		m_iCursorPosition = charIndexFromXPosition(pnt.x());
	}
	if( m_iCursorPosition == m_iSelectionAnchorChar )
		selectOneChar(-1);
	else {
		if( m_iCursorPosition > m_iSelectionAnchorChar ) {
			m_iSelectionBegin = m_iSelectionAnchorChar;
			m_iSelectionEnd   = m_iCursorPosition - 1;
		} else {
			m_iSelectionBegin = m_iCursorPosition;
			m_iSelectionEnd   = m_iSelectionAnchorChar - 1;
		}
	}
	repaintWithCursorOn();
}

void KviInput::returnPressed(bool bRepaint)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnIdleStart) || g_pEventManager->eventEnabled(KviEvent_OnIdleStop) ) {
		m_pParent->m_pFrm->textInput(&m_szTextBuffer);
	}
	QString *pHist = new QString(m_szTextBuffer);
	if( m_pHistory->count() > 0 ) {
		if( m_szTextBuffer.lower() != m_pHistory->at(0)->lower() )
			m_pHistory->insert(0, pHist);
	} else m_pHistory->insert(0, pHist);
	__range_valid(KVI_INPUT_HISTORY_ENTRIES > 1); // Absolutely needed! If not, pHist will be destroyed...
	if( m_pHistory->count() > KVI_INPUT_HISTORY_ENTRIES )
		m_pHistory->removeLast();
	m_iCurHistoryIdx    = -1;
	m_szTextBuffer      = "";
	selectOneChar(-1);
	m_iCursorPosition   = 0;
	m_iFirstVisibleChar = 0;
	if( bRepaint )
		repaintWithCursorOn();
	m_pUserParser->parseUserCommand(*pHist, m_pParent); // The parsed string must be NOT THE SAME AS m_szTextBuffer
}

void KviInput::focusInEvent(QFocusEvent *)
{
	if( m_iCursorTimer == 0 ) {
		m_iCursorTimer = startTimer(KVI_INPUT_BLINK_TIME);
		m_bCursorOn    = true;
		paintEvent(0);
	}
}

void KviInput::focusOutEvent(QFocusEvent *)
{
	if( m_iCursorTimer )
		killTimer(m_iCursorTimer);
	m_iCursorTimer = 0;
	m_bCursorOn    = false;
	paintEvent(0);
}

void KviInput::keyPressEvent(QKeyEvent *e)
{
	if( (e->key() == Qt::Key_Tab) || (e->key() == Qt::Key_BackTab) ) {
		completion(e->state() & ShiftButton);
		return;
	} else if( e->key() != Qt::Key_Shift )
		m_iLastCompletionIndex = -1;

	if( e->key() != Qt::Key_Backspace )
		m_bLastEventWasASubstitution = false;

	if( e->state() & ControlButton ) {
		switch( e->key() ) {
			case Qt::Key_K: {
				insertChar(KVI_TEXT_COLOR);
				int xPos = xPositionFromCharIndex(m_iCursorPosition);
				if( xPos > 24 )
					xPos -= 24;
				if( xPos + g_pColorWindow->width() > width() )
					xPos = width() - (g_pColorWindow->width() + 2);
				g_pColorWindow->move(mapToGlobal(QPoint(xPos, -35)));
				g_pColorWindow->popup(this);}
				break;
			case Qt::Key_B:
				insertChar(KVI_TEXT_BOLD);
				break;
			case Qt::Key_O:
				insertChar(KVI_TEXT_RESET);
				break;
			case Qt::Key_U:
				insertChar(KVI_TEXT_UNDERLINE);
				break;
			case Qt::Key_R:
				insertChar(KVI_TEXT_REVERSE);
				break;
			case Qt::Key_C:
				copy();
				break;
			case Qt::Key_X:
				cut();
				break;
			case Qt::Key_V:
				paste();
				break;
			default:
				e->ignore();
				break;
		}
		return;
	}

	if( e->state() & AltButton ) {
		// Qt::Key_Meta seems to substitute Key_Alt on some keyboards
		if( (e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta) ) {
			m_szAltKeyCode = "";
			return;
		} else if( (e->ascii() >= '0') && (e->ascii() <= '9') ) {
			m_szAltKeyCode += e->ascii();
			return;
		}
	}

	switch( e->key() ) {
		case Qt::Key_Right:
			if( (unsigned int) m_iCursorPosition < m_szTextBuffer.length() ) {
				moveRightFirstVisibleCharToShowCursor();
				if( e->state() & ShiftButton ) {
					// Grow the selection if needed
					if( (m_iSelectionBegin > -1) && (m_iSelectionEnd > -1) ) {
						if( m_iSelectionEnd == m_iCursorPosition - 1 )
							m_iSelectionEnd++;
						else if( m_iSelectionBegin == m_iCursorPosition )
							m_iSelectionBegin++;
						else
							selectOneChar(m_iCursorPosition);
					} else selectOneChar(m_iCursorPosition);
				} else {
					selectOneChar(-1);
				}
				m_iCursorPosition++;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Left:
			if( m_iCursorPosition > 0 ) {
				if( e->state() & ShiftButton ) {
					if( (m_iSelectionBegin > -1) && (m_iSelectionEnd > -1) ) {
						if( m_iSelectionBegin == m_iCursorPosition )
							m_iSelectionBegin--;
						else if( m_iSelectionEnd == m_iCursorPosition - 1)
							m_iSelectionEnd--;
						else
							selectOneChar(m_iCursorPosition - 1);
					} else selectOneChar(m_iCursorPosition - 1);
				} else {
					selectOneChar(-1);
				}
				m_iCursorPosition--;
				if( m_iFirstVisibleChar > m_iCursorPosition )
					m_iFirstVisibleChar--;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Backspace:
			if( (unsigned int) m_iCursorPosition <= m_szTextBuffer.length() ) {
				if( m_bLastEventWasASubstitution ) {
					// Restore the old string
					m_szTextBuffer         = m_szStringBeforeSubstitution;
					m_iCursorPosition      = m_iCursorPositionBeforeSubstitution;
					m_iLastCursorXPosition = m_iCursorXPositionBeforeSubstitution;
					m_iFirstVisibleChar    = m_iFirstVisibleCharBeforeSubstitution;

					insertChar(m_cDetonationChar);
					repaintWithCursorOn();
					m_bLastEventWasASubstitution = false;
					break;
				}
				if( hasSelection() &&
				    ((m_iSelectionEnd >= m_iCursorPosition) || (m_iSelectionBegin <= m_iCursorPosition))
				) {
					// Remove the selection
					m_szTextBuffer.remove(m_iSelectionBegin, (m_iSelectionEnd - m_iSelectionBegin) + 1);
					m_iCursorPosition = m_iSelectionBegin;
					if( (m_iFirstVisibleChar > 0) &&
					    (xPositionFromCharIndex(m_szTextBuffer.length()) <= width() - KVI_INPUT_BORDER)
					) {
						m_iFirstVisibleChar -= (m_iSelectionEnd - m_iSelectionBegin) + 1;
					} else if( m_iFirstVisibleChar > m_iCursorPosition )
						m_iFirstVisibleChar = m_iCursorPosition;
				} else if( m_iCursorPosition > 0 ) {
					m_iCursorPosition--;
					m_szTextBuffer.remove(m_iCursorPosition, 1);
					if( (m_iFirstVisibleChar > 0) &&
					    (xPositionFromCharIndex(m_szTextBuffer.length()) <= width() - KVI_INPUT_BORDER)
					) {
						m_iFirstVisibleChar--;
					} else if( m_iFirstVisibleChar > m_iCursorPosition )
						m_iFirstVisibleChar--;
				}
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Delete:
			if( (unsigned int) m_iCursorPosition < m_szTextBuffer.length() ) {
				if( hasSelection() &&
				    (m_iSelectionEnd == m_iCursorPosition - 1) || (m_iSelectionBegin == m_iCursorPosition)
				) {
					m_szTextBuffer.remove(m_iSelectionBegin, (m_iSelectionEnd - m_iSelectionBegin) + 1);
					m_iCursorPosition = m_iSelectionBegin;
					if( (m_iFirstVisibleChar > 0) &&
					    (xPositionFromCharIndex(m_szTextBuffer.length()) <= width() - KVI_INPUT_BORDER)
					) {
						m_iFirstVisibleChar -= (m_iSelectionEnd-m_iSelectionBegin) + 1;
					}
					else if( m_iFirstVisibleChar > m_iCursorPosition )
						m_iFirstVisibleChar = m_iCursorPosition;
				} else {
					m_szTextBuffer.remove(m_iCursorPosition, 1);
					if( (m_iFirstVisibleChar > 0) &&
					    (xPositionFromCharIndex(m_szTextBuffer.length()) < width() - KVI_INPUT_BORDER)
					) {
						m_iFirstVisibleChar--;
					}
				}
				moveRightFirstVisibleCharToShowCursor();
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Home:
			if( m_iCursorPosition > 0 ) {
				if( e->state() & ShiftButton ) {
					if( (m_iSelectionBegin == -1) && (m_iSelectionEnd == -1) )
						m_iSelectionEnd = m_iCursorPosition - 1;
					m_iSelectionBegin = 0;
				}
				home();
			}
			break;
		case Qt::Key_End:
			if( (unsigned int) m_iCursorPosition < m_szTextBuffer.length() ) {
				if( e->state() & ShiftButton ) {
					if( (m_iSelectionBegin == -1) && (m_iSelectionEnd == -1) )
						m_iSelectionBegin = m_iCursorPosition;
					m_iSelectionEnd = m_szTextBuffer.length();
				}
				end();
			}
			break;
		case Qt::Key_Up:
			if( m_pHistory->count() > 0 ) {
				if( m_iCurHistoryIdx < 0 ) {
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer     = *(m_pHistory->at(0));
					m_iCurHistoryIdx = 0;
				} else if( m_iCurHistoryIdx >= (int) (m_pHistory->count() - 1) ) {
					m_szTextBuffer   = m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx++;
					m_szTextBuffer = *(m_pHistory->at(m_iCurHistoryIdx));
				}
				selectOneChar(-1);
				if( g_pOptions->m_bInputHistoryCursorAtEnd )
					end();
				else
					home();
			}
			break;
		case Qt::Key_Down:
			if( m_pHistory->count() > 0 ) {
				if( m_iCurHistoryIdx < 0 ) {
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer     = *(m_pHistory->at(m_pHistory->count() - 1));
					m_iCurHistoryIdx   = m_pHistory->count() - 1;
				} else if( m_iCurHistoryIdx == 0 ) {
					m_szTextBuffer   = m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx--;
					m_szTextBuffer = *(m_pHistory->at(m_iCurHistoryIdx));
				}
				selectOneChar(-1);
				if( g_pOptions->m_bInputHistoryCursorAtEnd )
					end();
				else
					home();
			}
			break;
		case Qt::Key_Enter: // Fall-through
		case Qt::Key_Return:
			checkForSubstitution();
			m_bLastEventWasASubstitution = false;
			returnPressed();
			break;
		case Qt::Key_Alt: // Fall-through
		case Qt::Key_Meta:
			m_szAltKeyCode = "";
			break;
		case Qt::Key_Space:
			m_cDetonationChar = ' ';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Period:
			m_cDetonationChar = '.';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Comma:
			m_cDetonationChar = ',';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		default:
			if( e->ascii() > 0 )
				insertChar(e->ascii());
			else
				e->ignore();
			break;
	}
}

void KviInput::keyReleaseEvent(QKeyEvent *e)
{
	if( (e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta) ) {
		if( m_szAltKeyCode.hasData() ) {
			bool bOk;
			char ch = m_szAltKeyCode.toChar(&bOk);
			if( bOk && ch != 0 ) {
				insertChar(ch);
				e->accept();
			}
		}
		m_szAltKeyCode = "";
	}
	e->ignore();
}

void KviInput::completion(bool bShift)
{
	if( m_szTextBuffer.isEmpty() || (m_iCursorPosition == 0) )
		return;

	// Get the word before the cursor
	KviStr word = m_szTextBuffer.left(m_iCursorPosition);

	int idx  = word.findLastIdx(' ');
	int idx2 = word.findLastIdx(',');
	if( idx < idx2 )
		idx = idx2;
	idx2 = word.findLastIdx('(');
	if( idx < idx2 )
		idx = idx2;
	bool bFirstWordInLine = false;
	if( idx > -1 )
		word.cutLeft(idx + 1);
	else
		bFirstWordInLine = true;
	word.stripWhiteSpace();

	KviStr match;
	KviStr multiple;
	int iMatches = 0;

	if( (*(word.ptr()) == '/') || (*(word.ptr()) == g_pOptions->m_cPersonalCommandPrefix) ) {
		// Command completion
		word.cutLeft(1);
		if( word.isEmpty() )
			return;
		iMatches = m_pUserParser->completeCommand(word, match, multiple);
	} else if( *(word.ptr()) == '$' ) {
		// Function/identifier completion
		word.cutLeft(1);
		if( word.isEmpty() )
			return;
		iMatches = m_pUserParser->completeFunctionOrIdentifier(word, match, multiple);
	} else {
		// Empty word will end up here
		nickCompletion(bShift, word, bFirstWordInLine);
		repaintWithCursorOn();
		return;
	}

	if( iMatches != 1 ) {
		if( iMatches == 0 )
			m_pParent->output(KVI_OUT_INTERNAL, __tr("[Completion]: no matches"));
		else {
			multiple.toLower();
			m_pParent->output(KVI_OUT_INTERNAL, __tr("[Completion]: multiple matches: %s"), multiple.ptr());
		}
	}

	if( match.hasData() ) {
		match.toLower();
		m_iCursorPosition -= word.len();
		m_szTextBuffer.remove(m_iCursorPosition, word.len());
		m_szTextBuffer.insert(m_iCursorPosition, match.ptr());
		if( m_szTextBuffer.length() > KVI_INPUT_MAX_BUFFER_SIZE )
			m_szTextBuffer.setLength(KVI_INPUT_MAX_BUFFER_SIZE);
		moveCursorTo(m_iCursorPosition + match.len());
	}

	repaintWithCursorOn();
}

void KviInput::nickCompletion(bool bAddMask, KviStr &word, bool bFirstWordInLine)
{
	if( !m_pListBoxToControl )
		return;

	if( m_iLastCompletionIndex < 0 ) {
		// New completion session.
		// Save the buffer for the next time
		m_szLastCompletionBuffer          = m_szTextBuffer;
		m_iLastCompletionCursorPosition   = m_iCursorPosition;
		m_iLastCompletionCursorXPosition  = m_iLastCursorXPosition;
		m_iLastCompletionFirstVisibleChar = m_iFirstVisibleChar;
		if( word.isEmpty() )
			return; // New completion with an empty word
	} else {
		// Old completion... swap the buffers
		m_szTextBuffer         = m_szLastCompletionBuffer;
		m_iCursorPosition      = m_iLastCompletionCursorPosition;
		m_iLastCursorXPosition = m_iLastCompletionCursorXPosition;
		m_iFirstVisibleChar    = m_iLastCompletionFirstVisibleChar;
		// Re-extract
		word = m_szTextBuffer.left(m_iCursorPosition);
		// Trailing spaces are allowed this time!
		int iOriginalCursorPos = m_iCursorPosition;

		while( word.lastCharIs(' ') ) {
			m_iCursorPosition--;
			word.cutRight(1);
		}

		int idx  = word.findLastIdx(' ');
		int idx2 = word.findLastIdx(',');
		if( idx < idx2 )
			idx = idx2;
		idx2 = word.findLastIdx('(');
		if( idx < idx2 )
			idx = idx2;
		bFirstWordInLine = false;
		if( idx > -1 )
			word.cutLeft(idx + 1);
		else
			bFirstWordInLine = true;
		word.stripWhiteSpace();

		if( word.isEmpty() ) {
			// No word to complete
			m_iCursorPosition = iOriginalCursorPos;
			return;
		}
	}

	// OK, now look for a nick that matches
	selectOneChar(-1);

	int index = 0;
	for( KviIrcUser *u = m_pListBoxToControl->firstUser(); u; u = m_pListBoxToControl->nextUser() ) {
		if( index > m_iLastCompletionIndex ) {
			if( kvi_strEqualCIN(u->nick(), word.ptr(), word.len()) ) {
				// The first part matches word
				m_iLastCompletionIndex = index;
				KviStr missingPart = u->nick();
				if( bAddMask ) {
					missingPart.append('!');
					missingPart.append(u->username());
					missingPart.append('@');
					missingPart.append(u->host());
				}
				if( g_pOptions->m_bCompletionReplaceWholeWord ) {
					m_iCursorPosition -= word.len();
					m_szTextBuffer.remove(m_iCursorPosition, word.len());
				} else missingPart.cutLeft(word.len());
				if( bFirstWordInLine || (!g_pOptions->m_bApplyCompletionPostfixForFirstWordOnly) )
					missingPart += g_pOptions->m_szStringToAddAfterCompletedNick;
				m_szTextBuffer.insert(m_iCursorPosition, missingPart.ptr());
				if( m_szTextBuffer.length() > KVI_INPUT_MAX_BUFFER_SIZE )
					m_szTextBuffer.setLength(KVI_INPUT_MAX_BUFFER_SIZE);
				moveCursorTo(m_iCursorPosition + missingPart.len());
				// Repaint called from outside!
				return;
			}
		}
		index++;
	}
	// Nope... no matches...
	// Just reset to the old buffer and next time start from beginning
	m_iLastCompletionIndex = -1;
	// Repaint called from outside!
}

/**
 * Funky helpers
 */
void KviInput::end()
{
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iCursorPosition      = 0;
	m_iFirstVisibleChar    = 0;
	while( (unsigned int) m_iCursorPosition < m_szTextBuffer.length() ) {
		moveRightFirstVisibleCharToShowCursor();
		m_iCursorPosition++;
	}
	repaintWithCursorOn();
}

void KviInput::home()
{
	m_iFirstVisibleChar = 0;
	m_iCursorPosition   = 0;
	repaintWithCursorOn();
}

void KviInput::insertChar(char c)
{
	// Remove the selection
	if( (m_iSelectionBegin > -1) || (m_iSelectionEnd > -1) ) {
		if( (m_iSelectionEnd >= m_iCursorPosition) || (m_iSelectionBegin <= m_iCursorPosition) ) {
			m_szTextBuffer.remove(m_iSelectionBegin, (m_iSelectionEnd - m_iSelectionBegin) + 1);
			m_iCursorPosition = m_iSelectionBegin;
			if( (m_iFirstVisibleChar > 0) &&
				(xPositionFromCharIndex(m_szTextBuffer.length()) <= width() - KVI_INPUT_BORDER)
			) {
				m_iFirstVisibleChar -= (m_iSelectionEnd - m_iSelectionBegin) + 1;
			}
		}
	}
	selectOneChar(-1);
	if( m_szTextBuffer.length() < KVI_INPUT_MAX_BUFFER_SIZE ) {
		m_szTextBuffer.insert(m_iCursorPosition, c);
		moveRightFirstVisibleCharToShowCursor();
		m_iCursorPosition++;
		repaintWithCursorOn();
	} // TODO: beep to let the user know the message is too long?
}

void KviInput::moveRightFirstVisibleCharToShowCursor()
{
	QFontMetrics fm(g_pOptions->m_fntInput);
	QChar c = m_szTextBuffer.at(m_iCursorPosition);
	m_iLastCursorXPosition += (c.unicode() < 256) ? g_iInputFontCharWidth[c.unicode()] : fm.width(c);
	while( m_iLastCursorXPosition >= width() - KVI_INPUT_BORDER ) {
		c = m_szTextBuffer.at(m_iFirstVisibleChar);
		m_iLastCursorXPosition -= (c.unicode() < 256) ? g_iInputFontCharWidth[c.unicode()] : fm.width(c);
		m_iFirstVisibleChar++;
	}
}

void KviInput::repaintWithCursorOn()
{
	if( m_bUpdatesEnabled ) {
		m_bCursorOn = true;
		paintEvent(0);
	}
}

void KviInput::selectOneChar(int pos)
{
	m_iSelectionBegin = pos;
	m_iSelectionEnd   = pos;
}

int KviInput::charIndexFromXPosition(int xPos)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	int bufLen  = m_szTextBuffer.length();

	QFontMetrics fm(g_pOptions->m_fntInput);
	while( curChar < bufLen ) {
		QChar c = m_szTextBuffer.at(curChar);
		int widthCh = (c.unicode() < 256) ? g_iInputFontCharWidth[c.unicode()] : fm.width(c);
		if( xPos < (curXPos + (widthCh >> 1)) )
			return curChar;
		else if( xPos < (curXPos + widthCh) )
			return (curChar + 1);
		curXPos += widthCh;
		curChar++;
	}
	return curChar;
}

int KviInput::xPositionFromCharIndex(int chIdx)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	QFontMetrics fm(g_pOptions->m_fntInput);
	while( curChar < chIdx ) {
		QChar c = m_szTextBuffer.at(curChar);
		curXPos += (c.unicode() < 256) ? g_iInputFontCharWidth[c.unicode()] : fm.width(c);
		curChar++;
	}
	return curXPos;
}

void KviInput::checkForSubstitution()
{
	if( !g_pOptions->m_bUseStringSubstitution || m_szTextBuffer.isEmpty() || !m_iCursorPosition )
		return;

	KviStr szAuxTextBuffer = m_szTextBuffer;
	szAuxTextBuffer.prepend(' ');
	for( KviStrSubItem *m = g_pOptions->m_pStrSub->m_pList->first(); m; m = g_pOptions->m_pStrSub->m_pList->next() ) {
		KviStr szAuxOriginal = m->szOriginal;
		szAuxOriginal.prepend(' ');
		if( !kvi_strMatchRevCS(szAuxTextBuffer.ptr(), szAuxOriginal.ptr(), m_iCursorPosition) ) {
			// Save buffer in case of backspace
			m_szStringBeforeSubstitution          = m_szTextBuffer;
			m_iCursorPositionBeforeSubstitution   = m_iCursorPosition;
			m_iCursorXPositionBeforeSubstitution  = m_iLastCursorXPosition;
			m_iFirstVisibleCharBeforeSubstitution = m_iFirstVisibleChar;

			m_iCursorPosition -= m->szOriginal.len();
			m_szTextBuffer.remove(m_iCursorPosition, m->szOriginal.len());
			if( (m_iFirstVisibleChar > 0) &&
				(xPositionFromCharIndex(m_szTextBuffer.length()) <= width() - KVI_INPUT_BORDER)
			) {
				m_iFirstVisibleChar -= m->szOriginal.len();
			}
			m_szTextBuffer.insert(m_iCursorPosition, m->szSubstitute.ptr());
			if( m_szTextBuffer.length() > KVI_INPUT_MAX_BUFFER_SIZE )
				m_szTextBuffer.setLength(KVI_INPUT_MAX_BUFFER_SIZE);
			moveCursorTo(m_iCursorPosition + m->szSubstitute.len());
			m_bLastEventWasASubstitution = true;
			repaintWithCursorOn();
			return;
		}
	}
}

#include "m_kvi_input.moc"
