// =============================================================================
//
//      --- kvi_script_editorwidget.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptEditorWidget"

#define _KVI_SCRIPTEDITORWIDGET_CPP_

#include <qfontmetrics.h>
#include "qrichtext_p.h"

#include "kvi_app.h"
#include "kvi_completionbox.h"
#include "kvi_config.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_script_editorwidget.h"
#include "kvi_string.h"
#include "kvi_syntaxhighlighter.h"
#include "kvi_userparser.h"

KviScriptEditorWidget::KviScriptEditorWidget(QWidget *parent)
	: QTextEdit(parent)
{
	m_pFm            = 0;
	m_pHighlighter   = new KviSyntaxHighlighter(this);
	m_pCompletionBox = new KviCompletionBox(this);
	connect(m_pCompletionBox, SIGNAL(activated(const QString &)), this, SLOT(insertCompletion(const QString &)));
	connect(this, SIGNAL(textChanged()), this, SLOT(textChanged()));

	QFont fnt = KviConfig::getFixedFont();
	fnt.setPointSize(g_pOptions->m_fntView.pointSize());
	setCurrentFont(fnt);
	setFont(fnt);

	QPalette pal = palette();
	pal.setColor(                    QColorGroup::Text, *(g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_TEXT]));
	pal.setColor(                    QColorGroup::Base, *(g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ENABLEDBACK]));
	pal.setColor(QPalette::Disabled, QColorGroup::Base, *(g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_DISABLEDBACK]));
	setPalette(pal);

	setTextFormat(PlainText);
	setWordWrap(NoWrap);
	getCursorPosition(&m_currentParagraph, &m_currentIndex);
}

KviScriptEditorWidget::~KviScriptEditorWidget()
{
	if( m_pFm ) {
		delete m_pFm;
		m_pFm = 0;
	}
	delete m_pHighlighter;
	m_pHighlighter = 0;
}

void KviScriptEditorWidget::textChanged()
{
	int para, index;
	getCursorPosition(&para, &index);
	if( para != m_currentParagraph ) {
		m_currentParagraph = para;
		emit cursorRowChanged(m_currentParagraph);
	}
	if( index != m_currentIndex ) {
		m_currentIndex = index;
		emit cursorColChanged(m_currentIndex);
	}
}

void KviScriptEditorWidget::setFont(const QFont &font)
{
	if( m_pFm ) {
		delete m_pFm;
		m_pFm = 0;
	}
	m_pFm = new QFontMetrics(font);
	setTabStopWidth(m_pFm->width(' ') * KVI_TAB_WIDTH);
	QTextEdit::setFont(font);
}

KviStr KviScriptEditorWidget::getWordBeforeCursor()
{
	KviStr word;
	int para, index;
	getCursorPosition(&para, &index);
	QString pLine = text(para);
	if( pLine.isEmpty() || (index == 0))
		return word;

	// Get the word before the cursor
	word = pLine.left(index);
	int idx = word.findLastIdx(' ');
	if( idx > -1 )
		word.cutLeft(idx + 1);
	word.stripWhiteSpace();
	return word;
}

void KviScriptEditorWidget::completion()
{
	KviUserParser *p = g_pApp->anyUserParser();
	if( !p ) return;

	KviStr word = getWordBeforeCursor();
	if( word.isEmpty() ) return;

	KviStr match;
	KviStr multiple;
	int iMatches = 0;

	if( word.firstCharIs('$') ) {
		// Function/identifier completion
		word.cutLeft(1);
		iMatches = p->completeFunctionOrIdentifier(word, match, multiple);
	} else iMatches = p->completeCommand(word, match, multiple);

	if( iMatches != 1 ) {
		if( iMatches == 0 ) {
			emit message(_CHAR_2_QSTRING(__tr("[Completion]: no matches")));
			m_pCompletionBox->hide();
		} else {
			QString msg = __tr("[Completion]: %d matches");
			emit message(msg.sprintf(msg, iMatches));
			QStringList list = QStringList::split(", ", multiple.ptr());
			QStringList lwr;
			int height = m_pFm->lineSpacing();
			QTextCursor *cursor = textCursor();
			QRect rect(cursor->topParagraph()->rect());
			QPoint p = mapToGlobal(QPoint(x() + rect.x() + cursor->x() + 4, y() + rect.y() + cursor->y() + height));
			for( QStringList::Iterator iter = list.begin(); iter != list.end(); ++iter )
				lwr.append((*iter).lower());
			m_pCompletionBox->setItems(lwr);
			m_pCompletionBox->popup(p);
		}
	} else {
		match.toLower();
		insertCompletion(QString(match.ptr()));
	}
}

void KviScriptEditorWidget::insertCompletion(const QString &match)
{
	m_pCompletionBox->hide();
	if( !match.isEmpty() ) {
		int para, index;
		QString msg = __tr("Completion: \"%s\"");
		emit message(msg.sprintf(msg, match.ascii()));
		getCursorPosition(&para, &index);
		KviStr word = getWordBeforeCursor();
		if( word.isEmpty() ) return;

		if( word.firstCharIs('$') && match[0] != '$')
			word.cutLeft(1);
		QString line = text(para);
		line = line.replace(index - word.len(), word.len(), match);
		removeParagraph(para);
		insertParagraph(line, para);
		setCursorPosition(para, index - word.len() + match.length());
	}
}

void KviScriptEditorWidget::keyPressEvent(QKeyEvent *e)
{
	if( m_pCompletionBox->isVisible() ) {
		if( e->key() == Qt::Key_Space )
			m_pCompletionBox->hide();
		if( e->key() == Qt::Key_Return ) {
			m_pCompletionBox->activate();
			e->accept();
			return;
		}
		KviApplication::sendEvent(m_pCompletionBox, e);
	}
	if( e->state() & ControlButton ) {
		switch( e->key() ) {
			case Qt::Key_W:
				completion();
				return;
				break;
			case Qt::Key_K:
				insert("$K\\");
				return;
				break;
			case Qt::Key_B:
				insert("$B\\");
				return;
				break;
			case Qt::Key_O:
				insert("$O\\");
				return;
				break;
			case Qt::Key_U:
				insert("$U\\");
				return;
				break;
			case Qt::Key_R:
				insert("$R\\");
				return;
				break;
		}
	}
	QTextEdit::keyPressEvent(e);
	if( m_pCompletionBox->isVisible() )
		completion();
}

#include "m_kvi_script_editorwidget.moc"
